﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_SdHostStandardController.h"
#include <nn/nn_Abort.h>
#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_TIMER))
    #include <nn/os.h>
#endif
#include <nn/result/result_HandlingUtility.h>
#include <cstring>
#include "sdmmc_Timer.h"
#include "sdmmc_Log.h"

namespace nn { namespace sdmmc {
namespace detail {

#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
    #define NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED     if (IsRemoved()) { return ResultDeviceRemoved(); }
#else
    #define NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED
#endif

namespace
{
    // Controller 無反応タイムアウト (ms)
    const uint32_t ControllerReactionTimeoutMilliSeconds = 2000;

    // データ転送を除くコマンド通信完了待ちのタイムアウト (ms)
    const uint32_t CommandTimeoutMilliSeconds = 2000;

    #if (!defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))    // SDMA
        // データ転送進捗チェック間隔 (ms)
        const uint32_t DefaultCheckTransferIntervalMilliSeconds = 1500;
    #endif

    // Busy 解除待ちのタイムアウト (ms)
    const uint32_t BusyTimeoutMilliSeconds = 2000;

    // レジスタログ
    #define NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(basePointer, memberVariable) NN_DETAIL_SDMMC_DEBUG_LOG(#memberVariable "(%03Xh) 0x%08X\n",\
        static_cast<uint32_t>(reinterpret_cast<uintptr_t>(&(basePointer->memberVariable)) - reinterpret_cast<uintptr_t>(basePointer)), (basePointer->memberVariable))
    #define NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(basePointer, memberVariable) NN_DETAIL_SDMMC_DEBUG_LOG(#memberVariable "(%03Xh) 0x%04X\n",\
        static_cast<uint32_t>(reinterpret_cast<uintptr_t>(&(basePointer->memberVariable)) - reinterpret_cast<uintptr_t>(basePointer)), static_cast<uint32_t>(basePointer->memberVariable))
    #define NN_DETAIL_SDMMC_DEBUG_LOG_8BIT_REGISTER(basePointer, memberVariable) NN_DETAIL_SDMMC_DEBUG_LOG(#memberVariable "(%03Xh) 0x%02X\n",\
        static_cast<uint32_t>(reinterpret_cast<uintptr_t>(&(basePointer->memberVariable)) - reinterpret_cast<uintptr_t>(basePointer)), static_cast<uint32_t>(basePointer->memberVariable))

    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        // ADMA2 64-bit Address Descriptor 構造定義
        struct ADma2Descriptor
        {
            uint16_t    attribute;
            uint16_t    length;
            uint32_t    addressLow;
            uint32_t    addressHigh;
            uint32_t    reserved;
        };

        // ADMA2 Attribute 定義
        const uint16_t ADma2AttributeActTran    = 0x2U << 4;
        const uint16_t ADma2AttributeEnd        = 0x1U << 1;
        const uint16_t ADma2AttributeValid      = 0x1U << 0;
    #endif
}

#if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
    void SdHostStandardController::ResetBufferInfos() NN_NOEXCEPT
    {
        for (int i = 0; i < NumBufferInfos; i++)
        {
            m_BufferInfos[i].bufferAddress = 0;
            m_BufferInfos[i].bufferSize = 0;
        }
    }
#endif

#if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
    nn::dd::DeviceVirtualAddress SdHostStandardController::GetDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize) NN_NOEXCEPT
    {
        for (int i = 0; i < NumBufferInfos; i++)
        {
            if ((bufferAddress >= m_BufferInfos[i].bufferAddress)
                && ((bufferAddress + bufferSize) <= (m_BufferInfos[i].bufferAddress + m_BufferInfos[i].bufferSize))) {
                return m_BufferInfos[i].bufferDeviceVirtualAddress + (bufferAddress - m_BufferInfos[i].bufferAddress);
            }
        }
        NN_ABORT("DeviceVirtualAddress of Buffer is not known.\n");
        return 0;
    }
#endif

#if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
    size_t SdHostStandardController::MakeADma2Descriptors(const TransferData* pTransferData) NN_NOEXCEPT
    {
        ADma2Descriptor* pOutADma2Descriptors = reinterpret_cast<ADma2Descriptor*>(m_pADma2Descriptors);
        NN_ABORT_UNLESS_NOT_NULL(pOutADma2Descriptors);
        int maxNumADma2Descriptors = ADma2DescriptorsSize / sizeof(ADma2Descriptor);

        // データ転送最大ブロック数で打ち切る
        uint32_t numBlocks = (pTransferData->m_NumBlocks < REG_BLOCK_COUNT__MAX) ? pTransferData->m_NumBlocks : REG_BLOCK_COUNT__MAX;

        #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED) || defined(NN_DETAIL_SDMMC_BUFFER_ADDRESS_IS_PHYSICAL))
            // Descriptor Table 作成
            int i = 0;
            size_t transferredSize = 0;
            #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
                uint64_t currentAddress = GetDeviceVirtualAddress(reinterpret_cast<uintptr_t>(pTransferData->m_pBuffer), pTransferData->m_BlockSize * numBlocks);
            #else   // NN_DETAIL_SDMMC_BUFFER_ADDRESS_IS_PHYSICAL
                uint64_t currentAddress = reinterpret_cast<uintptr_t>(pTransferData->m_pBuffer);
            #endif
            size_t restSize = pTransferData->m_BlockSize * numBlocks;
            while (restSize > 0)
            {
                size_t currentSize = (restSize < 0x10000) ? restSize : 0x10000;
                transferredSize += currentSize;
                pOutADma2Descriptors[i].addressLow = static_cast<uint32_t>(currentAddress);
                pOutADma2Descriptors[i].addressHigh = static_cast<uint32_t>(currentAddress >> 32);
                pOutADma2Descriptors[i].length = static_cast<uint16_t>(currentSize);   // 0x10000 は 0 と設定
                pOutADma2Descriptors[i].attribute = ADma2AttributeActTran | ADma2AttributeValid;
                NN_DETAIL_SDMMC_DEBUG_LOG("ADMA2 %03d Address:0x%08X %08X Size:0x%08X\n",
                    i, pOutADma2Descriptors[i].addressHigh, pOutADma2Descriptors[i].addressLow, currentSize);
                NN_ABORT_UNLESS((currentAddress % sizeof(uint64_t)) == 0);  // 物理アドレスは 64bit アライン想定
                i++;
                if (i >= maxNumADma2Descriptors)
                {
                    break;
                }
                currentAddress += currentSize;
                restSize -= currentSize;
            }
        #else
            nn::dd::PhysicalMemoryInfo physicalMemoryInfo;
            nn::dd::InitializePhysicalMemoryInfo(&physicalMemoryInfo,
                pTransferData->m_pBuffer, pTransferData->m_BlockSize * numBlocks);

            // Descriptor Table 作成
            int i = 0;
            size_t transferredSize = 0;
            while (true)
            {
                // 物理アドレスが連続するサイズを取得
                Result result = nn::dd::QueryNextPhysicalMemoryInfo(&physicalMemoryInfo);
                if (nn::dd::ResultEndOfQuery::Includes(result))
                {
                    break;  // バッファ終端
                }
                NN_ABORT_UNLESS(result.IsSuccess());
                uint64_t physicalAddress = nn::dd::GetPhysicalAddress(&physicalMemoryInfo);
                size_t restSize = nn::dd::GetSize(&physicalMemoryInfo);
                while (restSize > 0)
                {
                    size_t currentSize = (restSize < 0x10000) ? restSize : 0x10000;
                    transferredSize += currentSize;
                    pOutADma2Descriptors[i].addressLow = static_cast<uint32_t>(physicalAddress);
                    pOutADma2Descriptors[i].addressHigh = static_cast<uint32_t>(physicalAddress >> 32);
                    pOutADma2Descriptors[i].length = static_cast<uint16_t>(currentSize);   // 0x10000 は 0 と設定
                    pOutADma2Descriptors[i].attribute = ADma2AttributeActTran | ADma2AttributeValid;
                    NN_DETAIL_SDMMC_DEBUG_LOG("ADMA2 %03d Address:0x%08X %08X Size:0x%08X\n",
                        i, pOutADma2Descriptors[i].addressHigh, pOutADma2Descriptors[i].addressLow, currentSize);
                    NN_ABORT_UNLESS((physicalAddress % sizeof(uint64_t)) == 0); // 物理アドレスは 64bit アライン想定
                    i++;
                    if (i >= maxNumADma2Descriptors)
                    {
                        break;
                    }
                    physicalAddress += currentSize;
                    restSize -= currentSize;
                }
            }
        #endif

        if (i >= maxNumADma2Descriptors)
        {
            // 次回に続きのデータ転送を行えるようブロックサイズに区切る
            size_t postSize = transferredSize % pTransferData->m_BlockSize;
            //NN_DETAIL_SDMMC_DEBUG_LOG("post size 0x%08X\n", postSize);
            transferredSize -= postSize;
            while (postSize > 0)
            {
                size_t length = pOutADma2Descriptors[i - 1].length;
                if (length == 0)
                {
                    length = 0x10000;
                }
                if (length == postSize)
                {
                    // 終端の Descriptor を取り除く
                    i--;
                    break;
                }
                else if (length > postSize)
                {
                    // 終端の Descriptor を削る
                    length -= postSize;
                    pOutADma2Descriptors[i - 1].length = static_cast<uint16_t>(length);
                    NN_DETAIL_SDMMC_DEBUG_LOG("ADMA2 %03d Address:0x%08X %08X Size:0x%08X updated\n",
                        i - 1, pOutADma2Descriptors[i - 1].addressHigh, pOutADma2Descriptors[i - 1].addressLow, length);
                    break;
                }
                else
                {
                    // 終端の Descriptor を取り除き、一つ前の Descriptor に進む
                    i--;
                    postSize -= length;
                }
            }
        }
        NN_DETAIL_SDMMC_DEBUG_LOG("ADMA2 numADmaDescriptors:%d\n", i);

        // 終端設定
        NN_ABORT_UNLESS(i > 0);
        pOutADma2Descriptors[i - 1].attribute |= ADma2AttributeEnd;

        // CPU で Descriptor Table を作成したため、キャッシュフラッシュ
        nn::dd::FlushDataCache(pOutADma2Descriptors, sizeof(pOutADma2Descriptors[0]) * i);

        return transferredSize;
    }
#endif

void SdHostStandardController::EnsureControl() NN_NOEXCEPT
{
    // レジスタダミーリードにより、これ以前のレジスタライトの実行を確かにする
    (void)m_pRegisters->clockControl;
}

Result SdHostStandardController::EnableInternalClock() NN_NOEXCEPT
{
    // Host Controller 内部クロック供給
    m_pRegisters->clockControl |= REG_CLOCK_CONTROL__INTERNAL_CLOCK_ENABLE;
    EnsureControl();
    ManualTimer timer(ControllerReactionTimeoutMilliSeconds);
    while (true)
    {
        // SoC 内部の処理のため、デバイスの抜けチェックは行わない

        if ((m_pRegisters->clockControl & REG_CLOCK_CONTROL__INTERNAL_CLOCK_STABLE) != 0)
        {
            break;
        }
        if (!(timer.Update()))
        {
            // 内部クロックが安定しないまま、タイムアウトを過ぎている
            return ResultInternalClockStableSwTimeout();
        }
        // us オーダで完了する処理のはずなので Sleep はしない
    }

    // 手動でバス設定
    m_pRegisters->hostControl2 &= (~REG_HOST_CONTROL_2__PRESET_VALUE_ENABLE);
    m_pRegisters->clockControl &= (~REG_CLOCK_CONTROL__CLOCK_GENERATOR_SELECT);

    // DMA 設定
    m_pRegisters->hostControl2 |= REG_HOST_CONTROL_2__HOST_VERSION_4_00_ENABLE;
    NN_ABORT_UNLESS((m_pRegisters->capabilities10 & REG_CAPABILITIES__64BIT_SYSTEM_ADDRESS_SUPPORT) != 0);
    m_pRegisters->hostControl2 |= REG_HOST_CONTROL_2__64BIT_ADDRESSING_ENABLE;
    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        // 64-bit Address ADMA2 転送
        NN_DETAIL_SDMMC_DEBUG_LOG("SDMMC uses 64-bit ADMA2\n");
        m_pRegisters->hostControl1 = (m_pRegisters->hostControl1 & (~REG_HOST_CONTROL_1__DMA_SELECT__MASK)) | REG_HOST_CONTROL_1__DMA_SELECT__ADMA2;
    #else
        // 64-bit Address SDMA 転送
        NN_DETAIL_SDMMC_DEBUG_LOG("SDMMC uses 64-bit SDMA\n");
        m_pRegisters->hostControl1 = (m_pRegisters->hostControl1 & (~REG_HOST_CONTROL_1__DMA_SELECT__MASK)) | REG_HOST_CONTROL_1__DMA_SELECT__SDMA;
    #endif

    // タイムアウト最大値
    m_pRegisters->timeoutControl = (m_pRegisters->timeoutControl & (~REG_TIMEOUT_CONTROL__DATA_TIMEOUT_COUNTER_VALUE__MASK)) | REG_TIMEOUT_CONTROL__DATA_TIMEOUT_COUNTER_VALUE__MAX;

    NN_RESULT_SUCCESS;
}

void SdHostStandardController::SetBusPower(BusPower busPower) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(IsSupportedBusPower(busPower));
    switch (busPower)
    {
    case BusPower_Off:
        m_pRegisters->powerControl &= (~REG_POWER_CONTROL__SD_BUS_POWER_FOR_VDD1);
        return;
    case BusPower_1_8V:
        m_pRegisters->powerControl = (m_pRegisters->powerControl & (~REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__MASK)) | REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__1_8V;
        m_pRegisters->powerControl |= REG_POWER_CONTROL__SD_BUS_POWER_FOR_VDD1;
        return;
    case BusPower_3_3V:
        m_pRegisters->powerControl = (m_pRegisters->powerControl & (~REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__MASK)) | REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__3_3V;
        m_pRegisters->powerControl |= REG_POWER_CONTROL__SD_BUS_POWER_FOR_VDD1;
        return;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
    Result SdHostStandardController::WaitInterrupt(uint32_t timeoutMilliSeconds) NN_NOEXCEPT
    {
        EnsureControl();
        nn::os::MultiWaitHolderType* signaledHolder = nn::os::TimedWaitAny(&m_Waiter, nn::TimeSpan::FromMilliSeconds(timeoutMilliSeconds));
        if (signaledHolder == &m_InterruptEventHolder)
        {
            NN_RESULT_SUCCESS;
        }
        else if (signaledHolder == &m_RemovedEventHolder)
        {
            return ResultDeviceRemoved();
        }
        else    // signaledHolder == NULL
        {
            return ResultWaitInterruptSwTimeout();
        }
    }
#endif

#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
    void SdHostStandardController::ClearInterrupt() NN_NOEXCEPT
    {
        EnsureControl();
        nn::os::ClearInterruptEvent(m_pInterruptEvent);
    }
#endif

void SdHostStandardController::EnableInterruptStatus() NN_NOEXCEPT
{
    // ステータスレジスタ許可（許可した後にクリア）
    m_pRegisters->normalInterruptStatusEnable |= REG_NORMAL_INTERRUPT__ISSUE_COMMAND__MASK;
    m_pRegisters->errorInterruptStatusEnable |= REG_ERROR_INTERRUPT__ISSUE_COMMAND__MASK;

    // ステータスクリア（1 を立てたビットだけがクリアされる）
    m_pRegisters->normalInterruptStatus = m_pRegisters->normalInterruptStatus;
    m_pRegisters->errorInterruptStatus = m_pRegisters->errorInterruptStatus;

    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        // INTC 割り込み要因クリア
        ClearInterrupt();

        // INTC アサート許可
        m_pRegisters->normalInterruptSignalEnable |= REG_NORMAL_INTERRUPT__ISSUE_COMMAND__MASK;
        m_pRegisters->errorInterruptSignalEnable |= REG_ERROR_INTERRUPT__ISSUE_COMMAND__MASK;
    #endif
}

void SdHostStandardController::DisableInterruptStatus() NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        // INTC アサート禁止
        m_pRegisters->errorInterruptSignalEnable &= (~REG_ERROR_INTERRUPT__ISSUE_COMMAND__MASK);
        m_pRegisters->normalInterruptSignalEnable &= (~REG_NORMAL_INTERRUPT__ISSUE_COMMAND__MASK);
    #endif

    // ステータスレジスタ禁止
    m_pRegisters->errorInterruptStatusEnable &= (~REG_ERROR_INTERRUPT__ISSUE_COMMAND__MASK);
    m_pRegisters->normalInterruptStatusEnable &= (~REG_NORMAL_INTERRUPT__ISSUE_COMMAND__MASK);
}

void SdHostStandardController::SetTransfer(uint32_t* pOutTransferredNumBlocks, const TransferData* pTransferData) NN_NOEXCEPT
{
    NN_ABORT_UNLESS((pTransferData->m_BlockSize != 0) && (pTransferData->m_NumBlocks != 0));

    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        // Descriptor Table 作成
        size_t transferredSize = MakeADma2Descriptors(pTransferData);

        // 今回のデータ転送ブロック数を取得
        NN_ABORT_UNLESS((transferredSize / pTransferData->m_BlockSize) <= REG_BLOCK_COUNT__MAX);
        uint16_t transferredNumBlocks = static_cast<uint16_t>(transferredSize / pTransferData->m_BlockSize);

        // ADMA アドレス設定
        m_pRegisters->admaSystemAddressLow = static_cast<uint32_t>(m_ADma2DescriptorsAddress);
        m_pRegisters->admaSystemAddressHigh = static_cast<uint32_t>(m_ADma2DescriptorsAddress >> 32);

        // ブロックサイズ設定
        NN_ABORT_UNLESS(pTransferData->m_BlockSize <= REG_BLOCK_SIZE__TRANSFER_BLOCK_SIZE__MAX);
        m_pRegisters->blockSize = static_cast<uint16_t>(pTransferData->m_BlockSize) << REG_BLOCK_SIZE__TRANSFER_BLOCK_SIZE__SHIFT;

    #else   // SDMA
        // データ転送最大ブロック数で打ち切る
        uint16_t numBlocks = (pTransferData->m_NumBlocks < REG_BLOCK_COUNT__MAX) ? static_cast<uint16_t>(pTransferData->m_NumBlocks) : REG_BLOCK_COUNT__MAX;

        #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
            uint64_t address = GetDeviceVirtualAddress(reinterpret_cast<uintptr_t>(pTransferData->m_pBuffer), pTransferData->m_BlockSize * numBlocks);
            uint16_t transferredNumBlocks = numBlocks;  // SMMU により連続して見える
        #elif (defined(NN_DETAIL_SDMMC_BUFFER_ADDRESS_IS_PHYSICAL))
            uint64_t address = reinterpret_cast<uintptr_t>(pTransferData->m_pBuffer);
            uint16_t transferredNumBlocks = numBlocks;  // 物理アドレスのため途切れない
        #else
            nn::dd::PhysicalMemoryInfo physicalMemoryInfo;
            nn::dd::InitializePhysicalMemoryInfo(&physicalMemoryInfo,
                pTransferData->m_pBuffer, pTransferData->m_BlockSize * numBlocks);

            // 物理アドレスが連続するサイズを取得
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::dd::QueryNextPhysicalMemoryInfo(&physicalMemoryInfo));
            uint64_t address = nn::dd::GetPhysicalAddress(&physicalMemoryInfo);
            size_t transferredSize = nn::dd::GetSize(&physicalMemoryInfo);
            NN_ABORT_UNLESS((transferredSize / pTransferData->m_BlockSize) <= REG_BLOCK_COUNT__MAX);
            uint16_t transferredNumBlocks = static_cast<uint16_t>(transferredSize / pTransferData->m_BlockSize);
            NN_ABORT_UNLESS(transferredNumBlocks >= 1); // 連続するサイズが 1 ブロック以上である想定
        #endif

        // SDMA アドレス設定
        NN_DETAIL_SDMMC_DEBUG_LOG("SDMA Address:0x%08X %08X\n", static_cast<uint32_t>(address >> 32), static_cast<uint32_t>(address));
        NN_ABORT_UNLESS((address % sizeof(uint64_t)) == 0);  // 64bit アライン想定
        m_pRegisters->admaSystemAddressLow = static_cast<uint32_t>(address);
        m_pRegisters->admaSystemAddressHigh = static_cast<uint32_t>(address >> 32);
        // 次の割り込みは SdmaBufferBoundary アドレス境界で発生する
        m_NextSdmaAddress = (address + SdmaBufferBoundary) & (~(static_cast<uint64_t>(SdmaBufferBoundary) - 1));

        // ブロックサイズ設定
        NN_ABORT_UNLESS(pTransferData->m_BlockSize <= REG_BLOCK_SIZE__TRANSFER_BLOCK_SIZE__MAX);
        m_pRegisters->blockSize = REG_BLOCK_SIZE__SDMA_BUFFER_BOUNDARY__DEF | (static_cast<uint16_t>(pTransferData->m_BlockSize) << REG_BLOCK_SIZE__TRANSFER_BLOCK_SIZE__SHIFT);
    #endif

    // ブロック数設定
    m_pRegisters->blockCount = transferredNumBlocks;
    if (pOutTransferredNumBlocks != nullptr)
    {
        *pOutTransferredNumBlocks = transferredNumBlocks;
    }

    // 転送モード設定
    // 0: DMA Disable, Block Count Disable,
    // Auto CMD Disable, Write, Single Block,
    // Response Error Check Disable, Response Interrupt Enable
    uint16_t transferMode = 0;
    transferMode |= REG_TRANSFER_MODE__DMA_ENABLE;
    if (pTransferData->m_IsMultiBlockTransfer)
    {
        #if (!defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))   // SDMA
            transferMode |= REG_TRANSFER_MODE__BLOCK_COUNT_ENABLE;
        #endif
        transferMode |= REG_TRANSFER_MODE__MULTI_BLOCK_SELECT;
    }
    if (pTransferData->m_TransferDirection == TransferDirection_ReadFromDevice)
    {
        transferMode |= REG_TRANSFER_MODE__DATA_TRANSFER_DIRECTION_READ;
    }
    #if (!defined(NN_DETAIL_SDMMC_MANUAL_CMD12_ENABLE))
        // 異常検出時のソフトによる CMD12 発行と衝突する危険があるが、再初期化を含むリトライで解消される想定
        if (pTransferData->m_IsStopTransmissionCommandEnable)
        {
            transferMode = (transferMode & (~REG_TRANSFER_MODE__AUTO_CMD_ENABLE__MASK)) | REG_TRANSFER_MODE__AUTO_CMD_ENABLE__AUTO_CMD12_ENABLE;
        }
    #endif
    m_pRegisters->transferMode = transferMode;
}

void SdHostStandardController::SetTransferForTuning() NN_NOEXCEPT
{
    // ブロックサイズ設定
    uint16_t tuningBlockSize;
    BusWidth busWidth = GetBusWidth();
    switch (busWidth)
    {
    case BusWidth_4Bit:
        tuningBlockSize = 64;
        break;
    case BusWidth_8Bit:
        tuningBlockSize = 128;
        break;
    case BusWidth_1Bit:
        NN_FALL_THROUGH;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_ABORT_UNLESS(tuningBlockSize <= REG_BLOCK_SIZE__TRANSFER_BLOCK_SIZE__MAX);
    m_pRegisters->blockSize = tuningBlockSize << REG_BLOCK_SIZE__TRANSFER_BLOCK_SIZE__SHIFT;

    // ブロック数設定
    m_pRegisters->blockCount = 1;

    // 転送モード設定
    // 0: DMA Disable（メモリに展開されないので禁止のまま）, Block Count Disable,
    // Auto CMD Disable, Write, Single Block,
    // Response Error Check Disable, Response Interrupt Enable
    m_pRegisters->transferMode = REG_TRANSFER_MODE__DATA_TRANSFER_DIRECTION_READ;
}

void SdHostStandardController::SetCommand(const Command* pCommand, bool isWithTransferData) NN_NOEXCEPT
{
    uint16_t command = 0;

    switch (pCommand->m_ResponseType)
    {
    case ResponseType_R0:
        command |= REG_COMMAND__RESPONSE_TYPE_SELECT__NO_RESPONSE;
        // Command Index Check は行わない
        // Command CRC Check は行わない
        break;
    case ResponseType_R1:
        NN_FALL_THROUGH;
    case ResponseType_R6:
        NN_FALL_THROUGH;
    case ResponseType_R7:
        if (pCommand->m_IsBusy)
        {
            command |= REG_COMMAND__RESPONSE_TYPE_SELECT__RESPONSE_LENGTH_48_BUSY;
        }
        else
        {
            command |= REG_COMMAND__RESPONSE_TYPE_SELECT__RESPONSE_LENGTH_48;
        }
        command |= REG_COMMAND__COMMAND_INDEX_CHECK_ENABLE;
        command |= REG_COMMAND__COMMAND_CRC_CHECK_ENABLE;
        break;
    case ResponseType_R2:
        command |= REG_COMMAND__RESPONSE_TYPE_SELECT__RESPONSE_LENGTH_136;
        // Command Index Check は行わない
        command |= REG_COMMAND__COMMAND_CRC_CHECK_ENABLE;
        break;
    case ResponseType_R3:
        command |= REG_COMMAND__RESPONSE_TYPE_SELECT__RESPONSE_LENGTH_48;
        // Command Index Check は行わない
        // Command CRC Check は行わない
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    if (isWithTransferData)
    {
        command |= REG_COMMAND__DATA_PRESENT_SELECT;
    }
    NN_ABORT_UNLESS(pCommand->m_CommandIndex <= (REG_COMMAND__COMMAND_INDEX__MASK >> REG_COMMAND__COMMAND_INDEX__SHIFT));
    command |= (pCommand->m_CommandIndex << REG_COMMAND__COMMAND_INDEX__SHIFT);

    m_pRegisters->argument1 = pCommand->m_CommandArgument;
    m_pRegisters->command = command;
    NN_DETAIL_SDMMC_CMD_LOG("CMD%u 0x%08X\n", pCommand->m_CommandIndex, pCommand->m_CommandArgument);
}

void SdHostStandardController::SetCommandForTuning(uint32_t commandIndex) NN_NOEXCEPT
{
    Command command(commandIndex, 0, ResponseType_R1, false);
    SetCommand(&command, true);
}

Result SdHostStandardController::ResetCmdDatLine() NN_NOEXCEPT
{
    m_pRegisters->softwareReset |= (REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_DAT | REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_CMD);
    EnsureControl();
    ManualTimer timer(ControllerReactionTimeoutMilliSeconds);
    while (true)
    {
        NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;

        if ((m_pRegisters->softwareReset & (REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_DAT | REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_CMD)) == 0)
        {
            NN_RESULT_SUCCESS;
        }
        if (!(timer.Update()))
        {
            // Abort が完了しないまま、タイムアウトを過ぎている
            return ResultAbortTransactionSwTimeout();
        }
        // us オーダで完了する処理のはずなので Sleep はしない
    }
}

Result SdHostStandardController::AbortTransaction() NN_NOEXCEPT
{
    NN_RESULT_DO(ResetCmdDatLine());
    NN_RESULT_SUCCESS;
}

Result SdHostStandardController::WaitWhileCommandInhibit(bool isWithDatLine) NN_NOEXCEPT
{
    // CMD ライン空き待ち
    EnsureControl();
    ManualTimer timer1(ControllerReactionTimeoutMilliSeconds);
    while (true)
    {
        NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;

        if ((m_pRegisters->presentState & REG_PRESENT_STATE__COMMAND_INHIBIT_CMD) == 0)
        {
            break;
        }
        if (!(timer1.Update()))
        {
            // CMD ライン空き待ちのまま、タイムアウトを過ぎている
            (void)AbortTransaction();
            return ResultCommandInhibitCmdSwTimeout();
        }
        // us オーダで完了する処理のはずなので Sleep はしない
    }

    if (isWithDatLine)
    {
        // DAT ライン空き待ち
        ManualTimer timer2(ControllerReactionTimeoutMilliSeconds);
        while (true)
        {
            NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;

            if ((m_pRegisters->presentState & REG_PRESENT_STATE__COMMAND_INHIBIT_DAT) == 0)
            {
                break;
            }
            if (!(timer2.Update()))
            {
                // DAT ライン空きを認識しない上、タイムアウトを過ぎている
                (void)AbortTransaction();
                return ResultCommandInhibitDatSwTimeout();
            }
            // us オーダで完了する処理のはずなので Sleep はしない
        }
    }

    NN_RESULT_SUCCESS;
}

Result SdHostStandardController::CheckAndClearInterruptStatus(uint16_t* pOutNormalInterruptStatus, uint16_t waitNormalInterruptStatus) NN_NOEXCEPT
{
    // REG_NORMAL_INTERRUPT__ERROR_INTERRUPT を検出した以後のエラーを扱うよう、
    // normalInterruptStatus -> errorInterruptStatus の順で保持する。
    volatile uint16_t normalInterruptStatus =  m_pRegisters->normalInterruptStatus;
    volatile uint16_t errorInterruptStatus =  m_pRegisters->errorInterruptStatus;
    volatile uint16_t autoCmdErrorStatus = m_pRegisters->autoCmdErrorStatus;    // AUTO_CMD_ERROR をクリアする前に保持

    if (pOutNormalInterruptStatus != nullptr)
    {
        *pOutNormalInterruptStatus = normalInterruptStatus;
    }

    if ((normalInterruptStatus & REG_NORMAL_INTERRUPT__ERROR_INTERRUPT) == 0)
    {
        if ((normalInterruptStatus & waitNormalInterruptStatus) != 0)
        {
            // 待っていた割り込み要因のみクリア（1 を立てたビットだけがクリアされる）
            m_pRegisters->normalInterruptStatus = (normalInterruptStatus & waitNormalInterruptStatus);
            NN_RESULT_SUCCESS;
        }
        else
        {
            // 待っていた割り込み要因はまだない
            return ResultNoWaitedInterrupt();
        }
    }

    // 保持したエラー割り込み要因をクリア
    m_pRegisters->errorInterruptStatus = errorInterruptStatus;

    // エラー Result として意味のある順にチェックする
    if ((errorInterruptStatus & REG_ERROR_INTERRUPT__COMMAND_INDEX_ERROR) != 0)
    {
        return ResultResponseIndexError();
    }
    if ((errorInterruptStatus & REG_ERROR_INTERRUPT__COMMAND_END_BIT_ERROR) != 0)
    {
        return ResultResponseEndBitError();
    }
    if ((errorInterruptStatus & REG_ERROR_INTERRUPT__COMMAND_CRC_ERROR) != 0)
    {
        return ResultResponseCrcError();
    }
    if ((errorInterruptStatus & REG_ERROR_INTERRUPT__COMMAND_TIMEOUT_ERROR) != 0)
    {
        return ResultResponseTimeoutError();
    }
    if ((errorInterruptStatus & REG_ERROR_INTERRUPT__DATA_END_BIT_ERROR) != 0)
    {
        return ResultDataEndBitError();
    }
    if ((errorInterruptStatus & REG_ERROR_INTERRUPT__DATA_CRC_ERROR) != 0)
    {
        return ResultDataCrcError();
    }
    if ((errorInterruptStatus & REG_ERROR_INTERRUPT__DATA_TIMEOUT_ERROR) != 0)
    {
        return ResultDataTimeoutError();
    }
    if ((errorInterruptStatus & REG_ERROR_INTERRUPT__AUTO_CMD_ERROR) != 0)
    {
        if (autoCmdErrorStatus & REG_AUTO_CMD_ERROR__AUTO_CMD_INDEX_ERROR)
        {
            return ResultAutoCommandResponseIndexError();
        }
        if ((autoCmdErrorStatus & REG_AUTO_CMD_ERROR__AUTO_CMD_END_BIT_ERROR) != 0)
        {
            return ResultAutoCommandResponseEndBitError();
        }
        if ((autoCmdErrorStatus & REG_AUTO_CMD_ERROR__AUTO_CMD_CRC_ERROR) != 0)
        {
            return ResultAutoCommandResponseCrcError();
        }
        if ((autoCmdErrorStatus & REG_AUTO_CMD_ERROR__AUTO_CMD_TIMEOUT_ERROR) != 0)
        {
            return ResultAutoCommandResponseTimeoutError();
        }
        NN_DETAIL_SDMMC_ERROR_LOG("There in no error corresponding Auto CMD Error: 0x%04X\n", autoCmdErrorStatus);
        return ResultSdHostStandardUnknownAutoCmdError();
    }
    // 想定していない REG_ERROR_INTERRUPT__ADMA_ERROR も含む
    NN_DETAIL_SDMMC_ERROR_LOG("There in no error corresponding Error Interrupt: 0x%04X\n", errorInterruptStatus);
    return ResultSdHostStandardUnknownError();
}

#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
    Result SdHostStandardController::WaitCommandComplete() NN_NOEXCEPT
    {
        Result result = WaitInterrupt(CommandTimeoutMilliSeconds);
        if (result.IsSuccess())
        {
            // Controller 側の割り込みをクリアした後に INTC の割り込みをクリア
            result = CheckAndClearInterruptStatus(nullptr, REG_NORMAL_INTERRUPT__COMMAND_COMPLETE);
            ClearInterrupt();
            if (result.IsSuccess())
            {
                NN_RESULT_SUCCESS;
            }
            else
            {
                (void)AbortTransaction();
                return result;
            }
        }
        else if (ResultDeviceRemoved::Includes(result))
        {
            return ResultDeviceRemoved();
        }
        else    // ResultWaitInterruptSwTimeout
        {
            // 何の状態変化も検出しないまま、タイムアウトを過ぎている
            (void)AbortTransaction();
            return ResultCommandCompleteSwTimeout();
        }
    }
#else
    Result SdHostStandardController::WaitCommandComplete() NN_NOEXCEPT
    {
        EnsureControl();
        ManualTimer timer(CommandTimeoutMilliSeconds);
        while (true)
        {
            Result result = CheckAndClearInterruptStatus(nullptr, REG_NORMAL_INTERRUPT__COMMAND_COMPLETE);
            if (result.IsSuccess())
            {
                NN_RESULT_SUCCESS;
            }
            else if (ResultNoWaitedInterrupt::Includes(result))
            {
                if (!(timer.Update()))
                {
                    // 何の状態変化も検出しないまま、タイムアウトを過ぎている
                    (void)AbortTransaction();
                    return ResultCommandCompleteSwTimeout();
                }
                // ポーリングを続ける
            }
            else
            {
                (void)AbortTransaction();
                return result;
            }
        }
    }
#endif

#if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        Result SdHostStandardController::WaitTransferComplete() NN_NOEXCEPT
        {
            // TODO: データサイズに応じたタイムアウト設定
            Result result = WaitInterrupt(4000);
            if (result.IsSuccess())
            {
                // Controller 側の割り込みをクリアした後に INTC の割り込みをクリア
                result = CheckAndClearInterruptStatus(nullptr, REG_NORMAL_INTERRUPT__TRANSFER_COMPLETE);
                ClearInterrupt();
                if (result.IsSuccess())
                {
                    NN_RESULT_SUCCESS;
                }
                else
                {
                    (void)AbortTransaction();
                    return result;
                }
            }
            else if (ResultDeviceRemoved::Includes(result))
            {
                return ResultDeviceRemoved();
            }
            else    // ResultWaitInterruptSwTimeout
            {
                // 何の状態変化も検出しないまま、タイムアウトを過ぎている
                (void)AbortTransaction();
                return ResultTransferCompleteSwTimeout();
            }
        }
    #else
        Result SdHostStandardController::WaitTransferComplete() NN_NOEXCEPT
        {
            // TODO: データサイズに応じたタイムアウト設定
            EnsureControl();
            ManualTimer timer(4000);
            while (true)
            {
                Result result = CheckAndClearInterruptStatus(nullptr, REG_NORMAL_INTERRUPT__TRANSFER_COMPLETE);
                if (result.IsSuccess())
                {
                    NN_RESULT_SUCCESS;
                }
                else if (ResultNoWaitedInterrupt::Includes(result))
                {
                    if (!(timer.Update()))
                    {
                        // 何の状態変化も検出しないまま、タイムアウトを過ぎている
                        (void)AbortTransaction();
                        return ResultTransferCompleteSwTimeout();
                    }
                    // ポーリングを続ける
                }
                else
                {
                    (void)AbortTransaction();
                    return result;
                }
            }
        }
    #endif
#else   // SDMA
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        Result SdHostStandardController::WaitTransferComplete() NN_NOEXCEPT
        {
            while (true)
            {
                uint16_t lastBlockCount = m_pRegisters->blockCount;
                Result result = WaitInterrupt(m_CheckTransferIntervalMilliSeconds);
                if (result.IsSuccess())
                {
                    // Controller 側の割り込みをクリアした後に INTC の割り込みをクリア
                    uint16_t normalInterruptStatus;
                    result = CheckAndClearInterruptStatus(&normalInterruptStatus, REG_NORMAL_INTERRUPT__TRANSFER_COMPLETE | REG_NORMAL_INTERRUPT__DMA_INTERRUPT);
                    ClearInterrupt();
                    if (result.IsSuccess())
                    {
                        if ((normalInterruptStatus & REG_NORMAL_INTERRUPT__TRANSFER_COMPLETE) != 0)
                        {
                            NN_RESULT_SUCCESS;
                        }
                        if ((normalInterruptStatus & REG_NORMAL_INTERRUPT__DMA_INTERRUPT) != 0)
                        {
                            NN_DETAIL_SDMMC_DEBUG_LOG("SDMA Address:0x%08X %08X\n", static_cast<uint32_t>(m_NextSdmaAddress >> 32), static_cast<uint32_t>(m_NextSdmaAddress));
                            m_pRegisters->admaSystemAddressLow = static_cast<uint32_t>(m_NextSdmaAddress);
                            m_pRegisters->admaSystemAddressHigh = static_cast<uint32_t>(m_NextSdmaAddress >> 32);
                            m_NextSdmaAddress += SdmaBufferBoundary;
                        }
                        // 割り込み待ちを続ける
                    }
                    else
                    {
                        (void)AbortTransaction();
                        return result;
                    }
                }
                else if (ResultDeviceRemoved::Includes(result))
                {
                    return ResultDeviceRemoved();
                }
                else    // ResultWaitInterruptSwTimeout
                {
                    if (lastBlockCount == m_pRegisters->blockCount)
                    {
                        // m_CheckTransferIntervalMilliSeconds 間に何の状態変化もない場合はタイムアウトとする
                        (void)AbortTransaction();
                        return ResultTransferCompleteSwTimeout();
                    }
                    // 割り込み待ちを続ける
                }
            }
        }
    #else
        Result SdHostStandardController::WaitTransferComplete() NN_NOEXCEPT
        {
            while (true)
            {
                uint16_t lastBlockCount = m_pRegisters->blockCount; // EnsureControl() も兼ねる
                ManualTimer timer(m_CheckTransferIntervalMilliSeconds);
                while (true)
                {
                    uint16_t normalInterruptStatus;
                    Result result = CheckAndClearInterruptStatus(&normalInterruptStatus, REG_NORMAL_INTERRUPT__TRANSFER_COMPLETE | REG_NORMAL_INTERRUPT__DMA_INTERRUPT);
                    if (result.IsSuccess())
                    {
                        if ((normalInterruptStatus & REG_NORMAL_INTERRUPT__TRANSFER_COMPLETE) != 0)
                        {
                            NN_RESULT_SUCCESS;
                        }
                        if ((normalInterruptStatus & REG_NORMAL_INTERRUPT__DMA_INTERRUPT) != 0)
                        {
                            NN_DETAIL_SDMMC_DEBUG_LOG("SDMA Address:0x%08X %08X\n", static_cast<uint32_t>(m_NextSdmaAddress >> 32), static_cast<uint32_t>(m_NextSdmaAddress));
                            m_pRegisters->admaSystemAddressLow = static_cast<uint32_t>(m_NextSdmaAddress);
                            m_pRegisters->admaSystemAddressHigh = static_cast<uint32_t>(m_NextSdmaAddress >> 32);
                            m_NextSdmaAddress += SdmaBufferBoundary;
                        }
                        // ポーリングを続ける
                    }
                    else if (ResultNoWaitedInterrupt::Includes(result))
                    {
                        if (!(timer.Update()))
                        {
                            if (lastBlockCount == m_pRegisters->blockCount)
                            {
                                // m_CheckTransferIntervalMilliSeconds 間に何の状態変化もない場合はタイムアウトとする
                                (void)AbortTransaction();
                                return ResultTransferCompleteSwTimeout();
                            }
                            break;  // 再度 m_CheckTransferIntervalMilliSeconds 待ちを始める
                        }
                        // ポーリングを続ける
                    }
                    else
                    {
                        (void)AbortTransaction();
                        return result;
                    }
                }
            }
        }
    #endif
#endif

Result SdHostStandardController::WaitWhileBusy() NN_NOEXCEPT
{
    // Busy 待ち
    EnsureControl();
    ManualTimer timer(BusyTimeoutMilliSeconds);
    while (true)
    {
        NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;

        if ((m_pRegisters->presentState & REG_PRESENT_STATE__DAT0_LINE_SIGNAL_LEVEL) != 0)
        {
            NN_RESULT_SUCCESS;
        }
        if (!(timer.Update()))
        {
            // Busy 解除を検出しないまま、タイムアウトを過ぎている
            (void)AbortTransaction();
            return ResultBusySwTimeout();
        }
        // ポーリングを続ける
    }
}

void SdHostStandardController::GetResponse(uint32_t* pOutResponse, size_t responseSize, ResponseType responseType) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutResponse);

    switch (responseType)
    {
    case ResponseType_R1:
        NN_FALL_THROUGH;
    case ResponseType_R3:
        NN_FALL_THROUGH;
    case ResponseType_R6:
        NN_FALL_THROUGH;
    case ResponseType_R7:
        NN_ABORT_UNLESS(responseSize >= sizeof(uint32_t));
        *pOutResponse = m_pRegisters->response10;
        NN_DETAIL_SDMMC_CMD_LOG("Resp 0x%08X\n", *pOutResponse);
        return;
    case ResponseType_R2:
        NN_ABORT_UNLESS(responseSize >= (sizeof(uint32_t) * 4));
        pOutResponse[0] = m_pRegisters->response10;
        pOutResponse[1] = m_pRegisters->response32;
        pOutResponse[2] = m_pRegisters->response54;
        pOutResponse[3] = m_pRegisters->response76;
        NN_DETAIL_SDMMC_CMD_LOG("Resp 0x%08X%08X%08X%08X\n", pOutResponse[3], pOutResponse[2], pOutResponse[1], pOutResponse[0]);
        return;
    case ResponseType_R0:
        NN_FALL_THROUGH;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

#if (defined(NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG))
    Result SdHostStandardController::IssueAbortCommandForDebug(uint32_t commandIndex, uint32_t commandArgument, int64_t abortTimingNanoSeconds) NN_NOEXCEPT
    {
        NN_RESULT_DO(WaitWhileCommandInhibit(false));

        if (abortTimingNanoSeconds > 0)
        {
            #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_TIMER))
                EnsureControl();
                nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(abortTimingNanoSeconds));
            #else
                NN_ABORT("NN_DETAIL_SDMMC_USE_NN_OS_FOR_TIMER isn't defined.\n");
            #endif
        }

        // 中断コマンド発行以降にデバイスと信号が衝突しないように今の通信をやめる
        NN_RESULT_DO(AbortTransaction());

        EnableInterruptStatus();

        // コマンド発行 (R1b コマンド固定)
        ResponseType responseType = ResponseType_R1;
        Command command(commandIndex, commandArgument, responseType, true);
        SetCommand(&command, false);

        // コマンド発行完了待ち
        Result result = WaitCommandComplete();

        DisableInterruptStatus();

        if (result.IsFailure())
        {
            return result;
        }

        // デバッグ用ログ出力のための取得
        uint32_t response;
        GetResponse(&response, sizeof(response), responseType);

        // Busy 完了待ち
        NN_RESULT_DO(WaitWhileBusy());

        NN_RESULT_SUCCESS;
    }
#endif

Result SdHostStandardController::IssueStopTransmissionCommandWithDeviceClock(uint32_t* pOutResponse) NN_NOEXCEPT
{
    NN_RESULT_DO(WaitWhileCommandInhibit(false));

    EnableInterruptStatus();

    // コマンド発行
    ResponseType responseType = ResponseType_R1;
    Command command(12, 0, responseType, true);
    SetCommand(&command, false);

    // コマンド発行完了待ち
    Result result = WaitCommandComplete();

    DisableInterruptStatus();

    if (result.IsFailure())
    {
        return result;
    }

    GetResponse(pOutResponse, sizeof(uint32_t), responseType);

    // Busy 完了待ち
    NN_RESULT_DO(WaitWhileBusy());

    NN_RESULT_SUCCESS;
}

Result SdHostStandardController::IssueCommandWithDeviceClock(const Command* pCommand, TransferData* pTransferData, uint32_t* pOutTransferredNumBlocks) NN_NOEXCEPT
{
    NN_RESULT_DO(WaitWhileCommandInhibit((pTransferData != nullptr) || pCommand->m_IsBusy));

    uint32_t transferredNumBlocks = 0;
    if (pTransferData != nullptr)
    {
        // データ転送設定（コマンド発行前に設定する必要がある）
        SetTransfer(&transferredNumBlocks, pTransferData);

        // pBuffer を含む領域が CPU アクセスされている可能性を考慮
        nn::dd::FlushDataCache(pTransferData->m_pBuffer, pTransferData->m_BlockSize * transferredNumBlocks);
    }

    EnableInterruptStatus();

    // コマンド発行
    SetCommand(pCommand, pTransferData != nullptr);

    #if (defined(NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG))
        if (m_IsAbortCommandEnable)
        {
            DisableInterruptStatus();
            (void)IssueAbortCommandForDebug(m_AbortCommandIndex, m_AbortCommandArgument, m_AbortTimingNanoSeconds);
            return ResultAbortCommandIssued();
        }
    #endif

    // コマンド発行完了待ち
    Result result = WaitCommandComplete();
    if (result.IsSuccess())
    {
        if (pCommand->m_ResponseType != ResponseType_R0)
        {
            m_LastResponseType = pCommand->m_ResponseType;
            GetResponse(m_LastResponse, sizeof(m_LastResponse), m_LastResponseType);
        }

        if (pTransferData != nullptr)
        {
            NN_DETAIL_SDMMC_CMD_LOG("Data %u(byte) x %u\n", static_cast<uint32_t>(pTransferData->m_BlockSize), transferredNumBlocks);

            // データ転送完了待ち
            result = WaitTransferComplete();
        }
    }

    DisableInterruptStatus();

    if (result.IsFailure())
    {
        return result;
    }

    if (pTransferData != nullptr)
    {
        if (pTransferData->m_TransferDirection == TransferDirection_ReadFromDevice)
        {
            // 最初の nn::dd::FlushDataCache 以降にキャッシュに取り込まれている可能性を考慮
            nn::dd::InvalidateDataCache(pTransferData->m_pBuffer, pTransferData->m_BlockSize * transferredNumBlocks);
        }

        if (pOutTransferredNumBlocks != nullptr)
        {
            *pOutTransferredNumBlocks = transferredNumBlocks;
        }

        if (pTransferData->m_IsStopTransmissionCommandEnable)
        {
            #if (defined(NN_DETAIL_SDMMC_MANUAL_CMD12_ENABLE))
                // CMD12 手動発行
                // このタイミングはデータ転送後のため、直前のコマンドから 8 clock あいており、Wait は不要
                NN_RESULT_DO(IssueStopTransmissionCommandWithDeviceClock(&m_LastStopTransmissionResponse));
            #else
                // CMD12 自動発行済み
                NN_DETAIL_SDMMC_CMD_LOG("Auto CMD12\n");
                m_LastStopTransmissionResponse = m_pRegisters->response76;
                NN_DETAIL_SDMMC_CMD_LOG("Resp 0x%08X\n", m_LastStopTransmissionResponse);
            #endif
        }
    }

    if (pCommand->m_IsBusy || (pTransferData != nullptr))
    {
        // Busy 完了待ち
        NN_RESULT_DO(WaitWhileBusy());
    }

    NN_RESULT_SUCCESS;
}

SdHostStandardController::SdHostStandardController(nn::dd::PhysicalAddress registersPhysicalAddress, size_t registersSize) NN_NOEXCEPT
{
    uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(registersPhysicalAddress, registersSize);
    NN_ABORT_UNLESS(registersAddress != 0);
    m_pRegisters = reinterpret_cast<SdHostStandardRegisters*>(registersAddress);

    #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
        ResetBufferInfos();
    #endif

    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        m_pRemovedEvent = nullptr;
    #endif

    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        m_pADma2Descriptors = nullptr;
        m_ADma2DescriptorsAddress = 0;
    #else
        m_NextSdmaAddress = 0;
        m_CheckTransferIntervalMilliSeconds = DefaultCheckTransferIntervalMilliSeconds;
    #endif

    m_DeviceClockFrequencyKHz = 0;
    m_IsPowerSavingEnable = false;
    m_IsDeviceClockEnable = false;

    #if (defined(NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG))
        m_AbortCommandIndex = 0;
        m_AbortCommandArgument = 0;
        m_AbortTimingNanoSeconds = 0;
        m_IsAbortCommandEnable = false;
    #endif

    m_LastResponseType = ResponseType_R0;
    std::memset(m_LastResponse, 0, sizeof(m_LastResponse));
    m_LastStopTransmissionResponse = 0;
}

void SdHostStandardController::LogRegisters() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, sdmaSystemAddress);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, blockSize);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, blockCount);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, argument1);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, transferMode);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, command);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, response10);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, response32);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, response54);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, response76);
    NN_DETAIL_SDMMC_DEBUG_LOG("bufferDataPort(%03Xh) skip\n",
        static_cast<uint32_t>(reinterpret_cast<uintptr_t>(&(m_pRegisters->bufferDataPort)) - reinterpret_cast<uintptr_t>(m_pRegisters)));
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, presentState);
    NN_DETAIL_SDMMC_DEBUG_LOG_8BIT_REGISTER(m_pRegisters, hostControl1);
    NN_DETAIL_SDMMC_DEBUG_LOG_8BIT_REGISTER(m_pRegisters, powerControl);
    NN_DETAIL_SDMMC_DEBUG_LOG_8BIT_REGISTER(m_pRegisters, blockGapControl);
    NN_DETAIL_SDMMC_DEBUG_LOG_8BIT_REGISTER(m_pRegisters, wakeupControl);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, clockControl);
    NN_DETAIL_SDMMC_DEBUG_LOG_8BIT_REGISTER(m_pRegisters, timeoutControl);
    NN_DETAIL_SDMMC_DEBUG_LOG_8BIT_REGISTER(m_pRegisters, softwareReset);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, normalInterruptStatus);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, errorInterruptStatus);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, normalInterruptStatusEnable);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, errorInterruptStatusEnable);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, normalInterruptSignalEnable);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, errorInterruptSignalEnable);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, autoCmdErrorStatus);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, hostControl2);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, capabilities10);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, capabilities32);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, maximumCurrentCapabilities10);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, maximumCurrentCapabilities32);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, forceEventForAutoCmdErrorStatus);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, forceEventForErrorInterruptStatus);
    NN_DETAIL_SDMMC_DEBUG_LOG_8BIT_REGISTER(m_pRegisters, admaErrorStatus);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, admaSystemAddressLow);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, admaSystemAddressHigh);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, PointerForVendorSpecificArea);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, slotInterruptStatus);
    NN_DETAIL_SDMMC_DEBUG_LOG_16BIT_REGISTER(m_pRegisters, hostControllerVersion);
}

void SdHostStandardController::Initialize() NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        nn::os::InitializeMultiWait(&m_Waiter);
        NN_ABORT_UNLESS_NOT_NULL(m_pInterruptEvent);
        nn::os::InitializeMultiWaitHolder(&m_InterruptEventHolder, m_pInterruptEvent);
        nn::os::LinkMultiWaitHolder(&m_Waiter, &m_InterruptEventHolder);
        if (m_pRemovedEvent != nullptr)
        {
            nn::os::InitializeMultiWaitHolder(&m_RemovedEventHolder, m_pRemovedEvent);
            nn::os::LinkMultiWaitHolder(&m_Waiter, &m_RemovedEventHolder);
        }
    #endif
}

void SdHostStandardController::Finalize() NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        if (m_pRemovedEvent != nullptr)
        {
            nn::os::UnlinkMultiWaitHolder(&m_RemovedEventHolder);
            nn::os::FinalizeMultiWaitHolder(&m_RemovedEventHolder);
        }
        nn::os::UnlinkMultiWaitHolder(&m_InterruptEventHolder);
        nn::os::FinalizeMultiWaitHolder(&m_InterruptEventHolder);
        nn::os::FinalizeMultiWait(&m_Waiter);
    #endif
}

#if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
    void SdHostStandardController::RegisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT
    {
        for (int i = 0; i < NumBufferInfos; i++)
        {
            if (m_BufferInfos[i].bufferAddress == 0)
            {
                m_BufferInfos[i].bufferAddress = bufferAddress;
                m_BufferInfos[i].bufferSize = bufferSize;
                m_BufferInfos[i].bufferDeviceVirtualAddress = bufferDeviceVirtualAddress;
                return;
            }
        }
        // ここに来たときは NumBufferInfos の数が足りないとき
        NN_ABORT("Out of NumBufferInfos\n");
    }
#endif

#if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
    void SdHostStandardController::UnregisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT
    {
        for (int i = 0; i < NumBufferInfos; i++)
        {
            if (m_BufferInfos[i].bufferAddress == bufferAddress)
            {
                NN_ABORT_UNLESS(m_BufferInfos[i].bufferSize == bufferSize);
                NN_ABORT_UNLESS(m_BufferInfos[i].bufferDeviceVirtualAddress == bufferDeviceVirtualAddress);
                m_BufferInfos[i].bufferAddress = 0;
                m_BufferInfos[i].bufferSize = 0;
                return;
            }
        }
        // ここに来たときは登録していないバッファを削除しようとしているとき
        NN_ABORT("bufferAddress is not found\n");
    }
#endif

void SdHostStandardController::SetWorkBuffer(void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        NN_ABORT_UNLESS_NOT_NULL(pWorkBuffer);
        NN_ABORT_UNLESS(workBufferSize >= ADma2DescriptorsSize);
        m_pADma2Descriptors = reinterpret_cast<ADma2Descriptor*>(pWorkBuffer);

        #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
            uint64_t address = GetDeviceVirtualAddress(reinterpret_cast<uintptr_t>(m_pADma2Descriptors), ADma2DescriptorsSize);
        #elif (defined(NN_DETAIL_SDMMC_BUFFER_ADDRESS_IS_PHYSICAL))
            uint64_t address = reinterpret_cast<uintptr_t>(m_pADma2Descriptors);
        #else
            // Descriptor Table 領域の物理アドレス情報を取得
            nn::dd::PhysicalMemoryInfo physicalMemoryInfo;
            nn::dd::InitializePhysicalMemoryInfo(&physicalMemoryInfo, m_pADma2Descriptors, ADma2DescriptorsSize);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::dd::QueryNextPhysicalMemoryInfo(&physicalMemoryInfo));
            uint64_t address = nn::dd::GetPhysicalAddress(&physicalMemoryInfo);
            size_t size = nn::dd::GetSize(&physicalMemoryInfo);
            NN_ABORT_UNLESS(size >= ADma2DescriptorsSize);  // Descriptor Table 領域は連続している
        #endif
        m_ADma2DescriptorsAddress = address;
        NN_DETAIL_SDMMC_DEBUG_LOG("ADMA2 Descriptors Address:0x%08X %08X\n", static_cast<uint32_t>(m_ADma2DescriptorsAddress >> 32), static_cast<uint32_t>(m_ADma2DescriptorsAddress));
        NN_ABORT_UNLESS((m_ADma2DescriptorsAddress % ADma2DescriptorsAlignment) == 0);
    #else   // SDMA
        NN_UNUSED(pWorkBuffer);
        NN_UNUSED(workBufferSize);
        NN_ABORT("SDMA requires no WorkBuffer.\n");
    #endif
}

BusPower SdHostStandardController::GetBusPower() const NN_NOEXCEPT
{
    if (((m_pRegisters->powerControl & REG_POWER_CONTROL__SD_BUS_POWER_FOR_VDD1)) == 0)
    {
        return BusPower_Off;
    }
    else
    {
        uint16_t sdBusVoltageSelectForVdd1 = m_pRegisters->powerControl & REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__MASK;
        switch (sdBusVoltageSelectForVdd1)
        {
        case REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__1_8V:
            return BusPower_1_8V;
        case REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__3_3V:
            return BusPower_3_3V;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void SdHostStandardController::SetBusWidth(BusWidth busWidth) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(IsSupportedBusWidth(busWidth));
    switch (busWidth)
    {
    case BusWidth_1Bit:
        m_pRegisters->hostControl1 &= (~REG_HOST_CONTROL_1__DATA_TRANSFER_WIDTH);
        m_pRegisters->hostControl1 &= (~REG_HOST_CONTROL_1__EXTENDED_DATA_TRANSFER_WIDTH);
        return;
    case BusWidth_4Bit:
        m_pRegisters->hostControl1 |= REG_HOST_CONTROL_1__DATA_TRANSFER_WIDTH;
        m_pRegisters->hostControl1 &= (~REG_HOST_CONTROL_1__EXTENDED_DATA_TRANSFER_WIDTH);
        return;
    case BusWidth_8Bit:
        m_pRegisters->hostControl1 |= REG_HOST_CONTROL_1__EXTENDED_DATA_TRANSFER_WIDTH;
        return;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

BusWidth SdHostStandardController::GetBusWidth() const NN_NOEXCEPT
{
    if ((m_pRegisters->hostControl1 & REG_HOST_CONTROL_1__EXTENDED_DATA_TRANSFER_WIDTH) != 0)
    {
        return BusWidth_8Bit;
    }
    else
    {
        if ((m_pRegisters->hostControl1 & REG_HOST_CONTROL_1__DATA_TRANSFER_WIDTH) != 0)
        {
            return BusWidth_4Bit;
        }
        else
        {
            return BusWidth_1Bit;
        }
    }
}

void SdHostStandardController::SetPowerSaving(bool isPowerSavingEnable) NN_NOEXCEPT
{
    m_IsPowerSavingEnable = isPowerSavingEnable;
    if (m_IsPowerSavingEnable)
    {
        if ((m_pRegisters->clockControl & REG_CLOCK_CONTROL__SD_CLOCK_ENABLE) != 0)
        {
            // クロック供給されていたら、必要なタイミングまで止める
            m_pRegisters->clockControl &= (~REG_CLOCK_CONTROL__SD_CLOCK_ENABLE);
        }
    }
    else
    {
        if ((m_IsDeviceClockEnable) && ((m_pRegisters->clockControl & REG_CLOCK_CONTROL__SD_CLOCK_ENABLE) == 0))
        {
            // 許可されているクロック供給が止まっていたら、常時供給する
            m_pRegisters->clockControl |= REG_CLOCK_CONTROL__SD_CLOCK_ENABLE;
        }
    }
}

void SdHostStandardController::EnableDeviceClock() NN_NOEXCEPT
{
    if ((!m_IsPowerSavingEnable) && ((m_pRegisters->clockControl & REG_CLOCK_CONTROL__SD_CLOCK_ENABLE) == 0))
    {
        // 省電力状態ではないのにクロックが止まっていたら、常時供給する
        m_pRegisters->clockControl |= REG_CLOCK_CONTROL__SD_CLOCK_ENABLE;
    }
    m_IsDeviceClockEnable = true;
}

void SdHostStandardController::DisableDeviceClock() NN_NOEXCEPT
{
    m_IsDeviceClockEnable = false;
    m_pRegisters->clockControl &= (~REG_CLOCK_CONTROL__SD_CLOCK_ENABLE);
}

void SdHostStandardController::ChangeCheckTransferInterval(uint32_t milliSeconds) NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        // ADMA2 でデータ転送の進捗チェックは行えない
        NN_UNUSED(milliSeconds);
    #else // SDMA
        m_CheckTransferIntervalMilliSeconds = milliSeconds;
    #endif
}

void SdHostStandardController::SetDefaultCheckTransferInterval() NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        // ADMA2 でデータ転送の進捗チェックは行えない
    #else // SDMA
        m_CheckTransferIntervalMilliSeconds = DefaultCheckTransferIntervalMilliSeconds;
    #endif
}

Result SdHostStandardController::IssueCommand(const Command* pCommand, TransferData* pTransferData, uint32_t* pOutTransferredNumBlocks) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_IsDeviceClockEnable);
    bool isClockDisabled = false;

    if ((m_pRegisters->clockControl & REG_CLOCK_CONTROL__SD_CLOCK_ENABLE) == 0)
    {
        isClockDisabled = true;
        m_pRegisters->clockControl |= REG_CLOCK_CONTROL__SD_CLOCK_ENABLE;
        EnsureControl();
        WaitClocks(8, m_DeviceClockFrequencyKHz);
    }

    Result result = IssueCommandWithDeviceClock(pCommand, pTransferData, pOutTransferredNumBlocks);

    WaitClocks(8, m_DeviceClockFrequencyKHz);   // コマンド発行間の 8 clock 保証 (コマンド制御後のため EnsureControl() は不要)

    if (isClockDisabled)
    {
        m_pRegisters->clockControl &= (~REG_CLOCK_CONTROL__SD_CLOCK_ENABLE);
    }

    return result;
}

Result SdHostStandardController::IssueStopTransmissionCommand(uint32_t* pOutResponse) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_IsDeviceClockEnable);
    bool isClockDisabled = false;

    if ((m_pRegisters->clockControl & REG_CLOCK_CONTROL__SD_CLOCK_ENABLE) == 0)
    {
        isClockDisabled = true;
        m_pRegisters->clockControl |= REG_CLOCK_CONTROL__SD_CLOCK_ENABLE;
        EnsureControl();
        WaitClocks(8, m_DeviceClockFrequencyKHz);
    }

    Result result = IssueStopTransmissionCommandWithDeviceClock(pOutResponse);

    WaitClocks(8, m_DeviceClockFrequencyKHz);   // コマンド発行間の 8 clock 保証 (コマンド制御後のため EnsureControl() は不要)

    if (isClockDisabled)
    {
        m_pRegisters->clockControl &= (~REG_CLOCK_CONTROL__SD_CLOCK_ENABLE);
    }

    return result;
}

void SdHostStandardController::GetLastResponse(uint32_t* pOutResponse, size_t responseSize, ResponseType responseType) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutResponse);
    NN_ABORT_UNLESS(responseType == m_LastResponseType);
    switch (responseType)
    {
    case ResponseType_R1:
        NN_FALL_THROUGH;
    case ResponseType_R3:
        NN_FALL_THROUGH;
    case ResponseType_R6:
        NN_FALL_THROUGH;
    case ResponseType_R7:
        NN_ABORT_UNLESS(responseSize >= sizeof(uint32_t));
        *pOutResponse = m_LastResponse[0];
        return;
    case ResponseType_R2:
        NN_ABORT_UNLESS(responseSize >= (sizeof(uint32_t) * 4));
        pOutResponse[0] = m_LastResponse[0];
        pOutResponse[1] = m_LastResponse[1];
        pOutResponse[2] = m_LastResponse[2];
        pOutResponse[3] = m_LastResponse[3];
        return;
    case ResponseType_R0:
        NN_FALL_THROUGH;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void SdHostStandardController::GetLastStopTransmissionResponse(uint32_t* pOutResponse, size_t responseSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutResponse);
    NN_ABORT_UNLESS(responseSize >= sizeof(uint32_t));
    *pOutResponse = m_LastStopTransmissionResponse;
}

} // namespace detail {
}} // namespace nn { namespace sdmmc {
