﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/sdmmc/sdmmc_Mmc.h>
#include "sdmmc_BaseDeviceAccessor.h"

namespace nn { namespace sdmmc {
namespace detail {

class MmcDevice : public BaseDevice
{
private:
    const uint16_t Rca = 0x2;   // 0 以外の任意の値

public:
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        virtual nn::os::EventType* GetRemovedEvent() const NN_NOEXCEPT NN_OVERRIDE
        {
            return nullptr; // 挿抜は検出しない
        }
    #endif

    virtual DeviceType GetDeviceType() const NN_NOEXCEPT NN_OVERRIDE
    {
        return DeviceType_Mmc;
    }

    void SetOcrAndHighCapacity(uint32_t ocr) NN_NOEXCEPT;

    virtual uint16_t GetRca() const NN_NOEXCEPT NN_OVERRIDE
    {
        return Rca;
    }
};

class MmcDeviceAccessor : public BaseDeviceAccessor
{
private:
    MmcDevice m_MmcDevice;
    void* m_pWorkBuffer;
    size_t m_WorkBufferSize;
    BusWidth m_MaxBusWidth;
    SpeedMode m_MaxSpeedMode;
    MmcPartition m_MmcPartition;
    bool m_IsInitialized;

    enum CommandSwitch {
        CommandSwitch_SetBitsProductionStateAwarenessEnable,
        CommandSwitch_ClearBitsAutoModeEnable,
        CommandSwitch_WriteProductionStateAwarenessNormal,
        CommandSwitch_WriteProductionStateAwarenessPreSolderingWrites,
        CommandSwitch_WriteProductionStateAwarenessPreSolderingPostWrites,
        CommandSwitch_SetBitsBkopsEnAutoEn,
        CommandSwitch_WriteBusWidth1Bit,
        CommandSwitch_WriteBusWidth4Bit,
        CommandSwitch_WriteBusWidth8Bit,
        CommandSwitch_WriteBusWidth8BitDdr,
        CommandSwitch_WriteHsTimingLegacySpeed,
        CommandSwitch_WriteHsTimingHighSpeed,
        CommandSwitch_WriteHsTimingHs200,
        CommandSwitch_WriteHsTimingHs400,
        CommandSwitch_WritePartitionAccessDefault,
        CommandSwitch_WritePartitionAccessRwBootPartition1,
        CommandSwitch_WritePartitionAccessRwBootPartition2
    };

    Result IssueCommandSendOpCond(uint32_t* pOutOcr, BusPower busPower) const NN_NOEXCEPT;
    Result IssueCommandSetRelativeAddr() const NN_NOEXCEPT;
    Result IssueCommandSwitch(CommandSwitch commandSwitch) const NN_NOEXCEPT;
    Result IssueCommandSendExtCsd(void* pOutExtendedCsdBuffer, size_t extendedCsdBufferSize) const NN_NOEXCEPT;
    Result IssueCommandEraseGroupStart(uint32_t sectorIndex) const NN_NOEXCEPT;
    Result IssueCommandEraseGroupEnd(uint32_t sectorIndex) const NN_NOEXCEPT;
    Result IssueCommandErase() const NN_NOEXCEPT;
    Result ChangeToReadyState(BusPower busPower) NN_NOEXCEPT;
    Result ExtendBusWidth(BusWidth maxBusWidth) NN_NOEXCEPT;
    Result EnableBkopsAuto() NN_NOEXCEPT;
    Result ChangeToHighSpeed(bool isClockup) NN_NOEXCEPT;
    Result ChangeToHs200() NN_NOEXCEPT;
    Result ChangeToHs400() NN_NOEXCEPT;
    Result ExtendBusSpeed(uint8_t mmcDevcieType, SpeedMode maxSpeedMode) NN_NOEXCEPT;
    Result StartupMmcDevice(BusWidth maxBusWidth, SpeedMode maxSpeedMode, void* pMmcWorkBuffer, size_t mmcWorkBufferSize) NN_NOEXCEPT;
    Result CancelToshibaMmcMode1() NN_NOEXCEPT;

protected:
    virtual Result OnActivate() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result OnReadWrite(uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, size_t bufferSize, bool isRead) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result ReStartup() NN_NOEXCEPT NN_OVERRIDE;

public:
    explicit MmcDeviceAccessor(IHostController* pHostController) NN_NOEXCEPT
        : BaseDeviceAccessor(pHostController)
    {
        m_pWorkBuffer = nullptr;
        m_WorkBufferSize = 0;
        m_MaxBusWidth = BusWidth_8Bit;          // 適当な値、設定した後しか参照されない
        m_MaxSpeedMode = SpeedMode_MmcHs400;    // 適当な値、設定した後しか参照されない
        m_MmcPartition = MmcPartition_Unknown;
        m_IsInitialized = false;
    }

    virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE;

    void SetMmcWorkBuffer(void* pMmcWorkBuffer, size_t mmcWorkBufferSize) NN_NOEXCEPT
    {
        m_pWorkBuffer = pMmcWorkBuffer;
        m_WorkBufferSize = mmcWorkBufferSize;
    }

    virtual Result GetSpeedMode(SpeedMode* pOutSpeedMode) const NN_NOEXCEPT NN_OVERRIDE;
    void PutMmcToSleep() NN_NOEXCEPT;
    void AwakenMmc() NN_NOEXCEPT;
    Result SelectMmcPartition(MmcPartition mmcPartition) NN_NOEXCEPT;
    Result EraseMmc() NN_NOEXCEPT;
    Result GetMmcBootPartitionCapacity(uint32_t* pOutNumSectors) const NN_NOEXCEPT;
    Result GetMmcExtendedCsd(void* pOutExtendedCsdBuffer, size_t extendedCsdBufferSize) const NN_NOEXCEPT;
};

} // namespace detail {
}} // namespace nn { namespace sdmmc {
