﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/nn_Abort.h>
#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE) || defined(NN_DETAIL_SDMMC_ERROR_LOGGER_ENABLE))
    #include <nn/os.h>
#endif
#include <nn/result/result_HandlingUtility.h>
#if (defined(NN_DETAIL_SDMMC_ERROR_LOGGER_ENABLE))
    #include <nn/util/util_FormatString.h>
    #include <nn/util/util_StringUtil.h>
#endif
#include "sdmmc_IDeviceAccessor.h"
#include "sdmmc_IHostController.h"

namespace nn { namespace sdmmc {
namespace detail {

#if (defined(NN_DETAIL_SDMMC_ERROR_LOGGER_ENABLE))
    class Logger
    {
    private:
        static const int LogLength = 32;
        static const int LogCountMax = 16;
        char m_Log[LogCountMax][LogLength];
        int m_LogIndex;

        void Clear()
        {
            for (int i = 0; i < LogCountMax; i++)
            {
                m_Log[i][0] = 0;
            }
            m_LogIndex = 0;
        }

        int Pop(char* pOutBuffer, int bufferLength) NN_NOEXCEPT
        {
            m_LogIndex--;
            if (m_LogIndex < 0)
            {
                m_LogIndex = LogCountMax - 1;
            }

            if (m_Log[m_LogIndex][0] == 0)
            {
                // Pop すべきログがない
                return 0;
            }

            // 終端のヌル文字を含まない要素数が返る
            int length = nn::util::Strlcpy(pOutBuffer, &m_Log[m_LogIndex][0], bufferLength);

            // Pop したログは消す
            m_Log[m_LogIndex][0] = 0;

            return length;
        }

    public:
        Logger() NN_NOEXCEPT
        {
            Clear();
        }

        void Push(const char* fmt, std::va_list vaList) NN_NOEXCEPT
        {
            // 切り詰められる前の要素数が返る
            int length = nn::util::VSNPrintf(&m_Log[m_LogIndex][0], LogLength, fmt, vaList);
            if (length >= LogLength)
            {
                length = LogLength - 1;   // 終端を切り詰める
            }
            m_Log[m_LogIndex][length] = 0;

            m_LogIndex++;
            if (m_LogIndex >= LogCountMax)
            {
                m_LogIndex = 0;
            }
        }

        void Push(const char* fmt, ...) NN_NOEXCEPT
        {
            std::va_list vaList;
            va_start(vaList, fmt);
            Push(fmt, vaList);
            va_end(vaList);
        }

        bool IsLogExist() NN_NOEXCEPT
        {
            int lastLogIndex = m_LogIndex - 1;
            if (lastLogIndex < 0)
            {
                lastLogIndex = LogCountMax - 1;
            }

            if (m_Log[lastLogIndex][0] != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        size_t GetAndClearLogs(char* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(pOutBuffer);
            NN_ABORT_UNLESS(bufferSize > 0);

            int totalLength = 0;
            while (true)
            {
                // 終端のヌル文字を含まない要素数が返る
                int length = Pop(pOutBuffer + totalLength, static_cast<int>(bufferSize) - totalLength);
                if (length == 0)
                {
                    // Pop すべきログがなかった
                    break;
                }
                if ((totalLength + length + 1) >= static_cast<int>(bufferSize))
                {
                    // ヌル文字を含めて格納しきれなかった
                    break;
                }
                totalLength += length;
                if ((totalLength + 2 + 1) >= static_cast<int>(bufferSize))
                {
                    // ヌル文字を含めて区切り文字を格納できない
                    break;
                }
                pOutBuffer[totalLength + 0] = ',';
                pOutBuffer[totalLength + 1] = ' ';
                totalLength += 2;
            }

            if (totalLength >= static_cast<int>(bufferSize))
            {
                totalLength = bufferSize - 1;   // 終端を切り詰める
            }
            pOutBuffer[totalLength] = 0;

            // バッファが足りず取得できなかったログもクリアする
            Clear();

            // 終端のヌル文字を含めて pOutBuffer に格納されたサイズを返す
            return static_cast<size_t>(totalLength + 1);
        }
    };
#endif

enum DeviceType
{
    DeviceType_Mmc,
    DeviceType_SdCard,
    DeviceType_GcAsic
};

// Device Status ビット定義
const uint32_t DeviceStatus_Bit_AddressOutOfRange   = 0x1U << 31;
const uint32_t DeviceStatus_Bit_AddressMisalign     = 0x1U << 30;
const uint32_t DeviceStatus_Bit_BlockLenError       = 0x1U << 29;
const uint32_t DeviceStatus_Bit_EraseSeqError       = 0x1U << 28;
const uint32_t DeviceStatus_Bit_EraseParam          = 0x1U << 27;
const uint32_t DeviceStatus_Bit_WpViolation         = 0x1U << 26;
//const uint32_t DeviceStatus_Bit_DeviceIsLocked      = 0x1U << 25;   // Status: カードロック状態 (Clear Condition: A)
const uint32_t DeviceStatus_Bit_LockUnlockFailed    = 0x1U << 24;
const uint32_t DeviceStatus_Bit_ComCrcError         = 0x1U << 23;
const uint32_t DeviceStatus_Bit_IllegalCommand      = 0x1U << 22;
const uint32_t DeviceStatus_Bit_DeviceEccFailed     = 0x1U << 21;
const uint32_t DeviceStatus_Bit_CcError             = 0x1U << 20;
const uint32_t DeviceStatus_Bit_Error               = 0x1U << 19;
const uint32_t DeviceStatus_Bit_CidCsdOverwrite     = 0x1U << 16;
const uint32_t DeviceStatus_Bit_WpEraseSkip         = 0x1U << 15;
const uint32_t DeviceStatus_Bit_EraseReset          = 0x1U << 13;
const uint32_t DeviceStatus_Pos_CurrentState        = 9;
const uint32_t DeviceStatus_Mask_CurrentState       = 0xFU << DeviceStatus_Pos_CurrentState;
//const uint32_t DeviceStatus_Bit_ReadyForData        = 0x1U << 8;    // Status: 空バッファの送信 (Clear Condition: A)
const uint32_t DeviceStatus_Bit_SwitchError         = 0x1U << 7;
//const uint32_t DeviceStatus_Bit_ExceptionEvent      = 0x1U << 6;    // Status: 例外発生 (Clear Condition: A)
const uint32_t DeviceStatus_Bit_AppCmd              = 0x1U << 5;     // Status: ACMD想定 (Clear Condition: C)
const uint32_t DeviceStatus_Mask_AllErrors = ( \
    DeviceStatus_Bit_AddressOutOfRange | \
    DeviceStatus_Bit_AddressMisalign | \
    DeviceStatus_Bit_BlockLenError | \
    DeviceStatus_Bit_EraseSeqError | \
    DeviceStatus_Bit_EraseParam | \
    DeviceStatus_Bit_WpViolation | \
    DeviceStatus_Bit_LockUnlockFailed | \
    DeviceStatus_Bit_ComCrcError | \
    DeviceStatus_Bit_IllegalCommand | \
    DeviceStatus_Bit_DeviceEccFailed | \
    DeviceStatus_Bit_CcError | \
    DeviceStatus_Bit_Error | \
    DeviceStatus_Bit_CidCsdOverwrite | \
    DeviceStatus_Bit_WpEraseSkip | \
    DeviceStatus_Bit_EraseReset | \
    DeviceStatus_Bit_SwitchError );

// Device Status の CURRENT_STATE 定義
enum DeviceState
{
    DeviceState_Idle = 0,
    DeviceState_Ready = 1,
    DeviceState_Ident = 2,
    DeviceState_Stby = 3,
    DeviceState_Tran = 4,
    DeviceState_Data = 5,
    DeviceState_Rcv = 6,
    DeviceState_Prg = 7,
    DeviceState_Dis = 8,
    DeviceState_Unknown = 0x10  // CURRENT_STATE は 4bit のため値に該当するものはない
};

class BaseDevice
{
private:
    uint32_t m_Ocr;
    uint8_t m_Cid[DeviceCidSize];
    uint16_t m_Csd[DeviceCsdSize / sizeof(uint16_t)];
    uint32_t m_MemoryCapacity;  // セクタ数
    bool m_IsHighCapacity;
    bool m_IsValidOcr;
    bool m_IsValidCid;
    bool m_IsValidCsd;
    bool m_IsValidHighCapacity;
    bool m_IsValidMemoryCapacity;
    bool m_IsActive;
    bool m_IsAwake;

public:
    #if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
        mutable nn::os::Mutex m_DeviceMutex;
    #endif

    void OnDeactivate() NN_NOEXCEPT
    {
        m_IsActive = false;
        m_IsValidOcr = false;
        m_IsValidCid = false;
        m_IsValidCsd = false;
        m_IsValidMemoryCapacity = false;
        m_IsValidHighCapacity = false;
    }

    #if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
        BaseDevice() NN_NOEXCEPT : m_DeviceMutex(true)
    #else
        BaseDevice() NN_NOEXCEPT
    #endif
    {
        m_IsAwake = true;
        m_Ocr = 0;
        m_MemoryCapacity = 0;
        m_IsHighCapacity = false;
        OnDeactivate();
    }

    void PutToSleep() NN_NOEXCEPT
    {
        m_IsAwake = false;
    }

    void Awaken() NN_NOEXCEPT
    {
        m_IsAwake = true;
    }

    bool IsAwake() const NN_NOEXCEPT
    {
        return m_IsAwake;
    }

    void SetActive() NN_NOEXCEPT
    {
        m_IsActive = true;
    }

    virtual void Deactivate() NN_NOEXCEPT
    {
        OnDeactivate();
    }

    bool IsActive() const NN_NOEXCEPT
    {
        return m_IsActive;
    }

    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        virtual nn::os::EventType* GetRemovedEvent() const NN_NOEXCEPT = 0;

        void InitializeRemovedEvent() NN_NOEXCEPT
        {
            nn::os::EventType* pRemovedEvent = GetRemovedEvent();
            if (pRemovedEvent != nullptr)
            {
                nn::os::InitializeEvent(pRemovedEvent, false,  nn::os::EventClearMode_ManualClear);
            }
        }

        void FinalizeRemovedEvent() NN_NOEXCEPT
        {
            nn::os::EventType* pRemovedEvent = GetRemovedEvent();
            if (pRemovedEvent != nullptr)
            {
                nn::os::FinalizeEvent(pRemovedEvent);
            }
        }

        void SignalRemovedEvent() NN_NOEXCEPT
        {
            nn::os::EventType* pRemovedEvent = GetRemovedEvent();
            if (pRemovedEvent != nullptr)
            {
                nn::os::SignalEvent(pRemovedEvent);
            }
        }

        void ClearRemovedEvent() NN_NOEXCEPT
        {
            nn::os::EventType* pRemovedEvent = GetRemovedEvent();
            if (pRemovedEvent != nullptr)
            {
                nn::os::ClearEvent(pRemovedEvent);
            }
        }

        bool IsRemoved() const NN_NOEXCEPT
        {
            nn::os::EventType* pRemovedEvent = GetRemovedEvent();
            if (pRemovedEvent != nullptr)
            {
                if (nn::os::TryWaitEvent(pRemovedEvent))
                {
                    return true;
                }
            }
            return false;
        }
    #endif

    Result CheckAccessible() const NN_NOEXCEPT
    {
        if (!IsAwake())
        {
            return ResultNotAwakened();
        }

        if (!IsActive())
        {
            return ResultNotActivated();
        }

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            if (IsRemoved())
            {
                return ResultDeviceRemoved();
            }
        #endif

        NN_RESULT_SUCCESS;
    }

    virtual DeviceType GetDeviceType() const NN_NOEXCEPT = 0;
    virtual uint16_t GetRca() const NN_NOEXCEPT = 0;

    void SetHighCapacity(bool isHighCapacity) NN_NOEXCEPT
    {
        m_IsHighCapacity = isHighCapacity;
        m_IsValidHighCapacity = true;
    }

    bool IsHighCapacity() const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_IsValidHighCapacity);
        return m_IsHighCapacity;
    }

    void SetOcr(uint32_t ocr) NN_NOEXCEPT
    {
        m_Ocr = ocr;
        m_IsValidOcr = true;
    }

    uint32_t GetOcr() const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_IsValidOcr);
        return m_Ocr;
    }

    void SetCid(void* pCid, size_t cidSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(cidSize >= DeviceCidSize);
        std::memcpy(m_Cid, pCid, DeviceCidSize);
        m_IsValidCid = true;
    }

    void GetCid(void* pOutCid, size_t cidSize) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_IsValidCid);
        NN_ABORT_UNLESS_NOT_NULL(pOutCid);
        NN_ABORT_UNLESS(cidSize >= DeviceCidSize);
        std::memcpy(pOutCid, m_Cid, DeviceCidSize);
    }

    void SetCsd(void* pCsd, size_t csdSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(csdSize >= DeviceCsdSize);
        std::memcpy(m_Csd, pCsd, DeviceCsdSize);
        m_IsValidCsd = true;
    }

    void GetCsd(void* pOutCsd, size_t csdSize) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_IsValidCsd);
        NN_ABORT_UNLESS_NOT_NULL(pOutCsd);
        NN_ABORT_UNLESS(csdSize >= DeviceCsdSize);
        std::memcpy(pOutCsd, m_Csd, DeviceCsdSize);
    }

    void GetLegacyCapacityParameters(uint8_t* pOutCSizeMult, uint8_t* pOutReadBlLen) const NN_NOEXCEPT;
    Result SetLegacyMemoryCapacity() NN_NOEXCEPT;

    void SetMemoryCapacity(uint32_t numSectors) NN_NOEXCEPT
    {
        m_MemoryCapacity = numSectors;
        m_IsValidMemoryCapacity = true;
    }

    uint32_t GetMemoryCapacity() const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_IsValidMemoryCapacity);
        return m_MemoryCapacity;
    }

    Result CheckDeviceStatus(uint32_t r1Response) const NN_NOEXCEPT;
    DeviceState GetDeviceState(uint32_t r1Response) const NN_NOEXCEPT;
};

class BaseDeviceAccessor : public IDeviceAccessor
{
private:
    IHostController* m_pHostController;
    BaseDevice* m_pBaseDevice;
    uint32_t m_NumActivationFailures;
    uint32_t m_NumActivationErrorCorrections;
    uint32_t m_NumReadWriteFailures;
    uint32_t m_NumReadWriteErrorCorrections;
    #if (defined(NN_DETAIL_SDMMC_ERROR_LOGGER_ENABLE))
        Logger m_ErrorLogger;
    #endif

    void ClearErrorInfo() NN_NOEXCEPT
    {
        m_NumActivationFailures = 0;
        m_NumActivationErrorCorrections = 0;
        m_NumReadWriteFailures = 0;
        m_NumReadWriteErrorCorrections = 0;
    }

protected:
    explicit BaseDeviceAccessor(IHostController* pHostController) NN_NOEXCEPT : m_pHostController(pHostController)
    {
        m_pBaseDevice = nullptr;
        ClearErrorInfo();
    }

    IHostController* GetHostController() const NN_NOEXCEPT
    {
        return m_pHostController;
    }

    void SetDevice(BaseDevice* pBaseDevice) NN_NOEXCEPT
    {
        m_pBaseDevice = pBaseDevice;
    }

    Result IssueCommandAndCheckR1(uint32_t* pOutResponse, uint32_t commandIndex, uint32_t commandArgument, bool isBusy, DeviceState expectedState, uint32_t deviceStatusBitsIgnored) const NN_NOEXCEPT;

    Result IssueCommandAndCheckR1(uint32_t commandIndex, uint32_t commandArgument, bool isBusy, DeviceState expectedState, uint32_t deviceStatusBitsIgnored) const NN_NOEXCEPT
    {
        uint32_t response;
        return IssueCommandAndCheckR1(&response, commandIndex, commandArgument, isBusy, expectedState, 0);
    }

    Result IssueCommandAndCheckR1(uint32_t commandIndex, uint32_t commandArgument, bool isBusy, DeviceState expectedState) const NN_NOEXCEPT
    {
        return IssueCommandAndCheckR1(commandIndex, commandArgument, isBusy, expectedState, 0);
    }

    Result IssueCommandGoIdleState() const NN_NOEXCEPT;
    Result IssueCommandAllSendCid(void* pOutCidBuffer, size_t cidBufferSize) const NN_NOEXCEPT;
    Result IssueCommandSelectCard() const NN_NOEXCEPT;
    Result IssueCommandSendCsd(void* pOutCsdBuffer, size_t csdBufferSize) const NN_NOEXCEPT;
    Result IssueCommandSendStatus(uint32_t* pOutDeviceStatus, uint32_t deviceStatusBitsIgnored) const NN_NOEXCEPT;

    Result IssueCommandSendStatus(uint32_t deviceStatusBitsIgnored) const NN_NOEXCEPT
    {
        uint32_t deviceStatus;
        return IssueCommandSendStatus(&deviceStatus, deviceStatusBitsIgnored);
    }

    Result IssueCommandSendStatus() const NN_NOEXCEPT
    {
        return IssueCommandSendStatus(0);
    }

    Result IssueCommandSetBlockLenToSectorSize() const NN_NOEXCEPT;
    Result IssueCommandMultipleBlock(uint32_t* pOutTransferredNumBlocks, uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, bool isRead) const NN_NOEXCEPT;
    Result ReadWriteSingly(uint32_t* pOutTransferredNumBlocks, uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, bool isRead) const NN_NOEXCEPT;
    Result ReadWriteMultiply(uint32_t sectorIndex, uint32_t numSectors, uint32_t divisionSectorIndexAlignment, void* pBuffer, size_t bufferSize, bool isRead) NN_NOEXCEPT;

    void CountUpActivationErrorCorrections() NN_NOEXCEPT
    {
        m_NumActivationErrorCorrections++;
    }

    void PushErrorTimeStamp() NN_NOEXCEPT
    {
        #if (defined(NN_DETAIL_SDMMC_ERROR_LOGGER_ENABLE))
            nn::os::Tick tick = nn::os::GetSystemTick();
            m_ErrorLogger.Push("%u", static_cast<uint32_t>(nn::os::ConvertToTimeSpan(tick).GetSeconds()));
        #endif
    }

    void PushErrorLog(bool isTimeStampEnabled, const char* fmt, ...) NN_NOEXCEPT
    {
        #if (defined(NN_DETAIL_SDMMC_ERROR_LOGGER_ENABLE))
            std::va_list vaList;
            va_start(vaList, fmt);
            m_ErrorLogger.Push(fmt, vaList);
            va_end(vaList);

            if (isTimeStampEnabled)
            {
                PushErrorTimeStamp();
            }
        #else
            NN_UNUSED(isTimeStampEnabled);
            NN_UNUSED(fmt);
        #endif
    }

    virtual Result OnActivate() NN_NOEXCEPT = 0;
    virtual Result OnReadWrite(uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, size_t bufferSize, bool isRead) NN_NOEXCEPT = 0;
    virtual Result ReStartup() NN_NOEXCEPT = 0;

public:
    #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
        virtual void RegisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT NN_OVERRIDE;
        virtual void UnregisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT NN_OVERRIDE;
    #endif
    virtual Result Activate() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Deactivate() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result ReadWrite(uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, size_t bufferSize, bool isRead) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result CheckConnection(SpeedMode* pOutHostSpeedMode, BusWidth* pOutHostBusWidth) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result GetMemoryCapacity(uint32_t* pOutNumSectors) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result GetDeviceStatus(uint32_t* pOutDeviceStatus) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result GetOcr(uint32_t* pOutOcr) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result GetRca(uint16_t* pOutRca) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result GetCid(void* pOutCid, size_t cidSize) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result GetCsd(void* pOutCsd, size_t csdSize) const NN_NOEXCEPT NN_OVERRIDE;
    #if (defined(NN_DETAIL_SDMMC_ISSUE_COMMAND_FOR_DEBUG))
        virtual Result IssueCommandForDebug(uint32_t* pOutResponse, uint32_t commandIndex, uint32_t commandArgument,
            DataTransfer* pDataTransfer, bool isBusy) NN_NOEXCEPT NN_OVERRIDE;
    #endif
    virtual void GetAndClearErrorInfo(ErrorInfo* pOutErrorInfo, size_t* pOutLogSize, char* pOutLogBuffer, size_t logBufferSize) NN_NOEXCEPT NN_OVERRIDE;
};

} // namespace detail {
}} // namespace nn { namespace sdmmc {
