﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/svc/svc_Base.h>
#include <nn/ro/detail/ro_RoExceptionInfo.h>
#include <nn/rocrt/rocrt.h>
#include <cstring>

namespace {

bool SearchSegmentHead(uintptr_t* pOutValue, uintptr_t targetAddress, nn::svc::MemoryState state)
{
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    bool isSuccess = false;

    for (uintptr_t addr = targetAddress;
         addr <= targetAddress;
         addr = static_cast<uintptr_t>(blockInfo.baseAddress - 1))
    {
        auto result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        if (blockInfo.state != state ||
            blockInfo.permission != nn::svc::MemoryPermission_ReadExecute)
        {
            break;
        }

        *pOutValue = blockInfo.baseAddress;
        isSuccess = true;
    }

    return isSuccess;
}

bool SearchSegmentTail(uintptr_t* pOutValue, uintptr_t targetAddress, nn::svc::MemoryState state)
{
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    bool isSuccess = false;

    for (uintptr_t addr = targetAddress;
         addr >= targetAddress;
         addr = blockInfo.baseAddress + blockInfo.size)
    {
        auto result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        if (blockInfo.state != state)
        {
            break;
        }

        *pOutValue = blockInfo.baseAddress + blockInfo.size - 1;
        isSuccess = true;
    }

    return isSuccess;
}

bool QueryModule(uintptr_t* pOutModuleAddress, size_t* pOutModuleSize, uintptr_t pc)
{
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    auto result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pc);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    if (blockInfo.permission != nn::svc::MemoryPermission_ReadExecute)
    {
        return false;
    }

    if (!((blockInfo.state == nn::svc::MemoryState_Code) ||
          (blockInfo.state == nn::svc::MemoryState_AliasCode)))
    {
        return false;
    }

    uintptr_t head = 0;
    uintptr_t tail = 0;
    nn::svc::MemoryState codeState = blockInfo.state;

    NN_ABORT_UNLESS(SearchSegmentHead(&head, pc, codeState));
    *pOutModuleAddress = head;

    NN_ABORT_UNLESS(SearchSegmentTail(&tail, pc, codeState));

    nn::svc::MemoryState state = (codeState == nn::svc::MemoryState_Code)
        ? nn::svc::MemoryState_CodeData
        : nn::svc::MemoryState_AliasCodeData;
    NN_ABORT_UNLESS(SearchSegmentTail(&tail, tail + 1, state));

    *pOutModuleSize = tail - head + 1;

    return true;
}

}

namespace nn { namespace ro {
namespace detail {

bool GetExceptionInfo(ExceptionInfo* pOutValue, uintptr_t pc)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    if (!QueryModule(&pOutValue->moduleAddress, &pOutValue->moduleSize, pc))
    {
        return false;
    }

    nn::rocrt::ModuleHeaderLocation* pLocation = reinterpret_cast<nn::rocrt::ModuleHeaderLocation*>(pOutValue->moduleAddress);
    nn::rocrt::ModuleHeader* header = reinterpret_cast<nn::rocrt::ModuleHeader*>(pOutValue->moduleAddress + pLocation->headerOffset);
    NN_ABORT_UNLESS(header->signature == nn::rocrt::ModuleHeaderVersion);
    pOutValue->infoOffset = header->exceptionInfoStartOffset + pLocation->headerOffset;
    pOutValue->infoSize = header->exceptionInfoEndOffset - header->exceptionInfoStartOffset;

    return true;
}

}
}}
