﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>

#include <nn/pinmux/pinmux.h>
#include <nn/pinmux/pinmux_ApiForHardwareTest.h>

#include <nn/pinmux/pinmux_IManager.sfdl.h>

#include "pinmux_GetManagerByHipc.h"

namespace nn { namespace pinmux {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
nn::sf::SharedPointer<nn::pinmux::IManager> g_PinmuxManager;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

inline ISession* GetInterface(PinmuxSession* pSession) NN_NOEXCEPT
{
    return static_cast<nn::pinmux::ISession*>(pSession->_sessionHandle);
}

}


void Initialize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(!g_PinmuxManager);

        g_PinmuxManager = GetManagerByHipc();
    }

    g_InitializeCount++;
}

void InitializeWith(nn::sf::SharedPointer<nn::pinmux::IManager> manager) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount == 0);

    g_PinmuxManager = manager;

    g_InitializeCount++;
}

// 以下は DFC と HIPC とで共通コード
void Finalize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(g_PinmuxManager);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_PinmuxManager = nullptr;
    }
}

// SF を外に出さないために外部にはハンドルとして渡すようにラップする。
void OpenSession(PinmuxSession* pOutSession, AssignablePinGroupName name) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_PinmuxManager);
    nn::sf::SharedPointer<nn::pinmux::ISession> session;
    auto result = g_PinmuxManager->OpenSession(&session, static_cast<std::int32_t>(name));
    NN_SDK_ASSERT( result.IsSuccess() );
    NN_UNUSED(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_sessionHandle = session.Detach();
}

// SessionHandle として外部に出したものを明示的に Close するための API
void CloseSession(PinmuxSession* pSession) NN_NOEXCEPT
{
    nn::sf::ReleaseSharedObject(static_cast<nn::pinmux::ISession*>(pSession->_sessionHandle));
}


// Session がもつ各機能のラップ版

void SetPinAssignment(PinmuxSession* pSession, PinAssignment assignment) NN_NOEXCEPT
{
    auto result = GetInterface(pSession)->SetPinAssignment(static_cast<int32_t>(assignment));
    NN_SDK_ASSERT( result.IsSuccess() );
    NN_UNUSED(result);
}

PinAssignment GetPinAssignment(PinmuxSession* pSession) NN_NOEXCEPT
{
    // 出力用の変数
    int32_t outAssignment;

    auto result = GetInterface(pSession)->GetPinAssignment(&outAssignment);

    NN_SDK_ASSERT( result.IsSuccess() );
    NN_UNUSED(result);

    return static_cast<nn::pinmux::PinAssignment>(outAssignment);
}

// 以下検査用の API
void OpenSessionForHardwareTest(PinmuxSession* pOutSession, AssignablePinGroupNameForHardwareTest name) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_PinmuxManager);
    nn::sf::SharedPointer<nn::pinmux::ISession> session;

    // PinmuxManager の OpenSession は共通のものを呼び出す
    auto result = g_PinmuxManager->OpenSession(&session, static_cast<std::int32_t>(name));
    NN_SDK_ASSERT( result.IsSuccess() );
    NN_UNUSED(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_sessionHandle = session.Detach();
}

void CloseSessionForHardwareTest(PinmuxSession* pSession) NN_NOEXCEPT
{
    nn::sf::ReleaseSharedObject(static_cast<nn::pinmux::ISession*>(pSession->_sessionHandle));
}

void SetPinAssignmentForHardwareTest(PinmuxSession* pSession, PinAssignmentForHardwareTest assignment) NN_NOEXCEPT
{
    auto result = GetInterface(pSession)->SetPinAssignmentForHardwareTest(static_cast<int32_t>(assignment));
    NN_SDK_ASSERT( result.IsSuccess() );
    NN_UNUSED(result);
}

}}
