﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_VirtualAddressMemoryApi.h>
#include <nn/os/detail/os_InternalCriticalSection.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_MemoryHeapManager.h"
#include "detail/os_VammManager.h"


namespace nn { namespace os {

void InitializeVirtualAddressMemory() NN_NOEXCEPT
{
    detail::GetVammManagerInstance()->InitializeIfEnabled();
}

//-----------------------------------------------------------------
// アドレス空間を獲得する
nn::Result AllocateAddressRegion(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( size > 0, NN_TEXT_OS("size > 0 ではありません（size=%zu）"), size );
    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryPageSize ), NN_TEXT_OS("size が MemoryPageSize の整数倍ではありません（size=%zu）"), size );

    auto result = detail::GetVammManagerInstance()->AllocateAddressRegion(pOutAddress, size);
    return result;
}

//-----------------------------------------------------------------
// アドレス空間を獲得し、物理メモリを割り当てる
nn::Result  AllocateMemory(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( size > 0, NN_TEXT_OS("size > 0 ではありません（size=%zu）"), size );
    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryPageSize ), NN_TEXT_OS("size が MemoryPageSize の整数倍ではありません（size=%zu）"), size );

    auto result = detail::GetVammManagerInstance()->AllocateMemory(pOutAddress, size);
    return result;
}

//-----------------------------------------------------------------
// 指定のアドレスに物理メモリを割り当てる
nn::Result  AllocateMemoryPages(uintptr_t address, size_t size)    NN_NOEXCEPT
{
    NN_SDK_REQUIRES( size > 0, NN_TEXT_OS("size > 0 ではありません（size=%zu）"), size );
    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryPageSize ), NN_TEXT_OS("size が MemoryPageSize の整数倍ではありません（size=%zu）"), size );
    NN_SDK_REQUIRES( detail::IsAligned( address, MemoryPageSize ), NN_TEXT_OS("address が MemoryPageSize の整数倍ではありません（address=0x%zx）"), address );

    auto result = detail::GetVammManagerInstance()->AllocateMemoryPages(address, size);
    return result;
}

//-----------------------------------------------------------------
// Reserved 領域にアドレス空間を返却する。物理メモリが割り当て済みなら解放する
nn::Result  FreeAddressRegion(uintptr_t address) NN_NOEXCEPT
{
    auto result = detail::GetVammManagerInstance()->FreeAddressRegion(address);
    return result;
}

//-----------------------------------------------------------------
// 物理メモリを解放する
nn::Result  FreeMemoryPages(uintptr_t address, size_t size)  NN_NOEXCEPT
{
    NN_SDK_REQUIRES( size > 0, NN_TEXT_OS("size > 0 ではありません（size=%zu）"), size );
    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryPageSize ), NN_TEXT_OS("size が MemoryPageSize の整数倍ではありません（size=%zu）"), size );
    NN_SDK_REQUIRES( detail::IsAligned( address, MemoryPageSize ), NN_TEXT_OS("address が MemoryPageSize の整数倍ではありません（address=0x%zx）"), address );

    auto result = detail::GetVammManagerInstance()->FreeMemoryPages(address, size);
    return result;
}

VirtualAddressMemoryResourceUsage GetVirtualAddressMemoryResourceUsage() NN_NOEXCEPT
{
    return detail::GetVammManagerInstance()->GetVirtualAddressMemoryResourceUsage();
}

bool IsVirtualAddressMemoryEnabled() NN_NOEXCEPT
{
    return detail::GetVammManagerInstance()->IsVirtualAddressMemoryEnabled();
}

}} // namespace nn::os

