﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/os/os_ThreadLocalStorageCommon.h>
#include <nn/os/os_ThreadLocalStorageApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_TlsManager.h"
#include "detail/os_ThreadManager.h"


//--------------------------------------------------------------------------
//  C++ 関数の定義
//--------------------------------------------------------------------------

namespace nn { namespace os {

//--------------------------------------------------------------------------
//  空き TLS スロットの確保
Result AllocateTlsSlot(TlsSlot* outTlsSlot, TlsDestructor function) NN_NOEXCEPT
{
    // TLS マネージャから、TLS スロットを１つ確保
    bool result = detail::GetTlsManagerInstance()->AllocateTlsSlot( outTlsSlot, function, false );

    if ( result == false )
    {
        return  ResultOutOfResource();
    }
    return  nn::ResultSuccess();
}


//--------------------------------------------------------------------------
//  SDK 内部用の空き TLS スロットの確保
Result SdkAllocateTlsSlot(TlsSlot* outTlsSlot, TlsDestructor function) NN_NOEXCEPT
{
    // TLS マネージャから、TLS スロットを１つ確保
    bool result = detail::GetTlsManagerInstance()->AllocateTlsSlot( outTlsSlot, function, true );

    if ( result == false )
    {
        return  ResultOutOfResource();
    }
    return  nn::ResultSuccess();
}


//--------------------------------------------------------------------------
//  TLS スロットの返却
void FreeTlsSlot(TlsSlot tlsSlot) NN_NOEXCEPT
{
    // TLS スロット番号が範囲内か？（符号なしなので最大値チェックのみ行なう）
    NN_SDK_REQUIRES( tlsSlot._innerValue < detail::TotalTlsSlotCountMax, NN_TEXT_OS("nn::os::FreeTlsSlot(): TLS スロット番号が異常です。") );

    // TLS マネージャに、TLS スロットを１つ返却
    detail::GetTlsManagerInstance()->FreeTlsSlot( tlsSlot );
}


//--------------------------------------------------------------------------
//  指定された TLS スロットから値を取得
uintptr_t GetTlsValue(TlsSlot tlsSlot) NN_NOEXCEPT
{
    uint32_t slotNum = tlsSlot._innerValue;

    // TLS スロット番号が範囲内か？（符号なしなので最大値チェックのみ行なう）
    NN_SDK_REQUIRES( slotNum < detail::TotalTlsSlotCountMax, NN_TEXT_OS("nn::os::GetTlsValue(): TLS スロット番号が異常です。") );

    // TLS スロットから値を取得
    return detail::GetCurrentThread()->_tlsValueArray[ slotNum ];
}


//--------------------------------------------------------------------------
//  指定された TLS スロットに値を設定
void SetTlsValue(TlsSlot tlsSlot, uintptr_t value) NN_NOEXCEPT
{
    uint32_t slotNum = tlsSlot._innerValue;

    // TLS スロット番号が範囲内か？（符号なしなので最大値チェックのみ行なう）
    NN_SDK_REQUIRES( slotNum < detail::TotalTlsSlotCountMax, NN_TEXT_OS("nn::os::SetTlsValue(): TLS スロット番号が異常です。") );

    // TLS スロットに値を設定
    detail::GetCurrentThread()->_tlsValueArray[ slotNum ] = value;
}


//--------------------------------------------------------------------------
//  現在確保済みの TLS スロット数を取得
int GetUsedTlsSlotCount() NN_NOEXCEPT
{
    return detail::GetTlsManagerInstance()->GetUsedTlsSlots();
}


}} // namespace nn::os

