﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/os/os_MultipleWaitApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_ThreadManager.h"
#include "detail/os_TimeoutHelper.h"
#include "detail/os_MultipleWaitHolderOfThread.h"
#include "detail/os_MultipleWaitHolderImpl.h"

#include <new>
#include <mutex>
#include <cstring>

namespace nn { namespace os {

//---------------------------------------------------------------------------
//  C++ ファイルスコープの定義
//---------------------------------------------------------------------------

namespace {

    // ユーザから渡されたスレッド名の長さを返す。ヌル文字はカウントされない。
    // スレッド名が ThreadNameLengthMax 以上の場合はアボートする。
    size_t CheckAndGetThreadNameLength(const char* name) NN_NOEXCEPT
    {
        const char* p = name;

        for (size_t length = 0; length < ThreadNameLengthMax; ++length)
        {
            if (*p == '\0')
            {
                return length;
            }
            ++p;
        }

        NN_ABORT(NN_TEXT_OS("nn::os::SetThreadName(): 指定されたスレッド名が長すぎます。\n（max=%d, name=\x22%s\x22）"), ThreadNameLengthMax, name);
    }

    // CreateThread の引数アサート
    inline void AssertArgumentsOfCreateThread(ThreadType* thread, void* stack, size_t stackSize, int priority) NN_NOEXCEPT
    {
        NN_UNUSED( thread );
        NN_UNUSED( stack );
        NN_UNUSED( stackSize );
        NN_UNUSED( priority );

        // スタックアドレスがアライメントされているか
        NN_SDK_REQUIRES( stack != NULL, NN_TEXT_OS("nn::os::CreateThread(): stack に NULL が指定されました。") );

        NN_SDK_REQUIRES( reinterpret_cast<uintptr_t>(stack) % ThreadStackAlignment == 0, NN_TEXT_OS("nn::os::CreateThread(): stack (=0x%p) が ThreadStackAlignment (=0x%zx) でアライメントされていません。"), stack, ThreadStackAlignment );

        // スタックサイズがアライメントされているか
        NN_SDK_REQUIRES( stackSize > 0, NN_TEXT_OS("nn::os::CreateThread(): stackSize に 0 が指定されました。") );

        NN_SDK_REQUIRES( stackSize % ThreadStackAlignment == 0, NN_TEXT_OS("nn::os::CreateThread(): stackSize (=0x%zx) が ThreadStackAlignment (=0x%zx) でアライメントされていません。"), stackSize, ThreadStackAlignment );
    }

}

//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//  スレッドの生成（idealCoreNumber あり）
Result CreateThread(ThreadType* thread, ThreadFunction function, void* argument, void* stack, size_t stackSize, int priority, int idealCoreNumber) NN_NOEXCEPT
{
    // CreateThread の引数アサート
    AssertArgumentsOfCreateThread(thread, stack, stackSize, priority);

    // idealCoreNumber のアサート
    NN_SDK_ASSERT(GetThreadAvailableCoreMask() & (1ULL << idealCoreNumber), NN_TEXT_OS("nn::os::CreateThread(): idealCoreNumber=%d に無効な値が指定されました。"), idealCoreNumber);

    // スレッドの生成（idealCoreNumber あり）
    return detail::GetThreadManagerInstance()->CreateThread(thread, function, argument, stack, stackSize, priority, idealCoreNumber);
}


//---------------------------------------------------------------------------
//  スレッドの生成（idealCoreNumber なし）
Result CreateThread(ThreadType* thread, ThreadFunction function, void* argument, void* stack, size_t stackSize, int priority) NN_NOEXCEPT
{
    // CreateThread の引数アサート
    AssertArgumentsOfCreateThread(thread, stack, stackSize, priority);

    // スレッドの生成（idealCoreNumber なし）
    return detail::GetThreadManagerInstance()->CreateThread(thread, function, argument, stack, stackSize, priority);
}


//---------------------------------------------------------------------------
//  スレッドの削除
void DestroyThread(ThreadType* thread) NN_NOEXCEPT
{
    detail::ThreadManager*  manager = detail::GetThreadManagerInstance();

    // 事前条件のチェック（オブジェクト参照前の確認）
    NN_SDK_REQUIRES( ( thread->_state == ThreadType::State_Initialized ) ||
                     ( thread->_state == ThreadType::State_Started ) ||
                     ( thread->_state == ThreadType::State_Exited ),
        NN_TEXT_OS("nn::os::DestroyThread(): スレッドの状態が削除可能な状態ではありません。") );
    NN_SDK_REQUIRES( thread != manager->GetMainThread(), NN_TEXT_OS("nn::os::DestroyThread(): メインスレッドは削除できません。") );

    // スレッドを削除
    manager->DestroyThread( thread );
}


//--------------------------------------------------------------------------
//  対象 Thread の実行を開始
void StartThread(ThreadType* thread) NN_NOEXCEPT
{
    // スレッド状態のチェック
    // thread に自スレッドを指定した場合もこれに引っかかる
    NN_SDK_REQUIRES( thread->_state == ThreadType::State_Initialized );

    // スレッド動作を開始
    detail::GetThreadManagerInstance()->StartThread( thread );
}


//--------------------------------------------------------------------------
//  対象スレッドのスレッド名の設定
void SetThreadName(ThreadType* thread, const char* threadName) NN_NOEXCEPT
{
    // スレッド状態のチェック
    NN_SDK_REQUIRES( ( thread->_state == ThreadType::State_Initialized ) ||
                     ( thread->_state == ThreadType::State_DestroyedBeforeStarted ) ||
                     ( thread->_state == ThreadType::State_Started ) ||
                     ( thread->_state == ThreadType::State_Exited ),
        NN_TEXT_OS("nn::os::SetThreadName(): スレッドが初期化されていません。") );

    if (threadName == NULL)
    {
        // 初期スレッド名を再設定
        detail::GetThreadManagerInstance()->SetInitialThreadNameUnsafe(thread);
        return;
    }

    // threadName の文字列長チェック
    size_t threadNameSize = CheckAndGetThreadNameLength(threadName) + 1;

    // 終端の NUL 文字を含めて memcpy でスレッド名をコピー。
    std::memcpy(thread->_threadNameBuffer, threadName, threadNameSize);

    SetThreadNamePointer(thread, thread->_threadNameBuffer);
}


//--------------------------------------------------------------------------
//  対象スレッドのスレッド名の格納アドレスの設定
void SetThreadNamePointer(ThreadType* thread, const char* threadName) NN_NOEXCEPT
{
    // スレッド状態のチェック
    NN_SDK_REQUIRES( ( thread->_state == ThreadType::State_Initialized ) ||
                     ( thread->_state == ThreadType::State_DestroyedBeforeStarted ) ||
                     ( thread->_state == ThreadType::State_Started ) ||
                     ( thread->_state == ThreadType::State_Exited ),
        NN_TEXT_OS("nn::os::SetThreadNamePointer(): スレッドが初期化されていません。") );

    if (threadName == NULL)
    {
        // 初期スレッド名を再設定
        detail::GetThreadManagerInstance()->SetInitialThreadNameUnsafe(thread);
        return;
    }

    thread->_namePointer = threadName;

    // デバッガ等に対して新しいスレッド名を通知
    detail::GetThreadManagerInstance()->NotifyThreadNameChanged( thread );
}


//--------------------------------------------------------------------------
//  対象スレッドのスレッド名の格納アドレスの取得
const char* GetThreadNamePointer(const ThreadType* thread) NN_NOEXCEPT
{
    // スレッド状態のチェック
    NN_SDK_REQUIRES( ( thread->_state == ThreadType::State_Initialized ) ||
                     ( thread->_state == ThreadType::State_DestroyedBeforeStarted ) ||
                     ( thread->_state == ThreadType::State_Started ) ||
                     ( thread->_state == ThreadType::State_Exited ),
        NN_TEXT_OS("nn::os::GetThreadNamePointer(): スレッドが初期化されていません。") );
    return thread->_namePointer;
}


//--------------------------------------------------------------------------
//  自スレッドの ThreadType オブジェクトの取得
ThreadType* GetCurrentThread() NN_NOEXCEPT
{
    // TLS から自スレッドの ThreadType オブジェクトを取り出して返す
    return  detail::GetCurrentThread();
}


//--------------------------------------------------------------------------
//  対象スレッドの優先度の変更
int ChangeThreadPriority(ThreadType* thread, int priority) NN_NOEXCEPT
{
    // スレッド状態のチェック（オブジェクト参照前のチェック）
    NN_SDK_REQUIRES( ( thread->_state == ThreadType::State_Initialized ) ||
                     ( thread->_state == ThreadType::State_DestroyedBeforeStarted ) ||
                     ( thread->_state == ThreadType::State_Started ) ||
                     ( thread->_state == ThreadType::State_Exited ),
        NN_TEXT_OS("nn::os::ChangeThreadPriority(): スレッドが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(thread->_csThread);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // 現在のベース優先度を保存しておく
        int previousPriority = thread->_basePriority;

        // スレッドの優先度を変更する
        bool result = detail::GetThreadManagerInstance()->ChangePriority( thread, priority );
        NN_SDK_ASSERT( result, NN_TEXT_OS("nn::os::ChangeThreadPriority(): スレッドの優先度変更に失敗しました（priority=%d）"), priority);
        NN_UNUSED(result);

        thread->_basePriority = priority;

        // 変更前の優先度を返す
        return  previousPriority;
    }
}


//--------------------------------------------------------------------------
//  対象スレッドの本来の優先度の取得
int GetThreadPriority(const ThreadType* thread) NN_NOEXCEPT
{
    // スレッド状態のチェック
    NN_SDK_REQUIRES( ( thread->_state == ThreadType::State_Initialized ) ||
                     ( thread->_state == ThreadType::State_DestroyedBeforeStarted ) ||
                     ( thread->_state == ThreadType::State_Started ) ||
                     ( thread->_state == ThreadType::State_Exited ),
        NN_TEXT_OS("nn::os::GetThreadPriority(): スレッドが初期化されていません。") );

    // ThreadType オブジェクトからベース優先度を取得して返す
    return thread->_basePriority;
}


//--------------------------------------------------------------------------
//  対象スレッドの現在の優先度の取得
int GetThreadCurrentPriority(const ThreadType* thread) NN_NOEXCEPT
{
    // スレッド状態のチェック
    NN_SDK_REQUIRES( ( thread->_state == ThreadType::State_Initialized ) ||
                     ( thread->_state == ThreadType::State_DestroyedBeforeStarted ) ||
                     ( thread->_state == ThreadType::State_Started ) ||
                     ( thread->_state == ThreadType::State_Exited ),
        NN_TEXT_OS("nn::os::GetThreadCurrentPriority(): スレッドが初期化されていません。") );

    // 現在の優先度を取得して返す
    return detail::GetThreadManagerInstance()->GetCurrentPriority( thread );
}


//--------------------------------------------------------------------------
//  同一優先度の他のスレッドに実行機会を譲渡
void YieldThread() NN_NOEXCEPT
{
    detail::GetThreadManagerInstance()->YieldThread();
}


//--------------------------------------------------------------------------
//  自スレッドを指定された時間だけ休止
void SleepThread(TimeSpan time) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( time.GetNanoSeconds() >= 0, NN_TEXT_OS("nn::os::SleepThread(): time に負の値が指定されました。") );

    detail::TimeoutHelper::Sleep( time );
}


//--------------------------------------------------------------------------
//  対象スレッド終了の待機
void WaitThread(ThreadType* thread) NN_NOEXCEPT
{
    // 自スレッドを指定した場合は動作停止
    NN_SDK_REQUIRES( thread != detail::GetCurrentThread(), NN_TEXT_OS("nn::os::WaitThread(): 自スレッドは指定できません。") );

    // スレッド状態のチェック
    NN_SDK_REQUIRES( ( thread->_state == ThreadType::State_Initialized ) ||
                     ( thread->_state == ThreadType::State_Started ) ||
                     ( thread->_state == ThreadType::State_Exited ) );

    // スレッド終了を待機
    detail::GetThreadManagerInstance()->WaitThread( thread );

    // ThreadType オブジェクトは確実に EXITED 状態である
}


//--------------------------------------------------------------------------
//  自スレッドが動作しているコア番号の取得
int GetCurrentCoreNumber() NN_NOEXCEPT
{
    // 現在のコア番号を取得して返す
    return detail::GetThreadManagerInstance()->GetCurrentCoreNumber();
}


//--------------------------------------------------------------------------
//  自スレッドが動作しているコア番号の取得
int GetCurrentProcessorNumber() NN_NOEXCEPT
{
    return GetCurrentCoreNumber();
}


//---------------------------------------------------------------------------
//  MultiWaitHolderType の初期化（Thread を関連付ける）
void InitializeMultiWaitHolder(MultiWaitHolderType* pHolder, ThreadType* thread) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( (thread->_state == ThreadType::State_Initialized) ||
                     (thread->_state == ThreadType::State_Started) ||
                     (thread->_state == ThreadType::State_Exited),
        NN_TEXT_OS("nn::oa::InitializeMultiWaitHolder(): 指定されたスレッドは待機できる状態ではありません。") );

    // 初期化処理（コンストラクタを呼び出す）
    // Get() がなくてもポインタは得られるが、TypedStorage のサイズと
    // アライメントチェックのため、Get() を使って実装しておく。
    new( &Get(pHolder->_holderImpl) ) detail::MultiWaitHolderOfThread( thread );

    //  構造体メンバの初期化
    pHolder->userData = 0;
}


//---------------------------------------------------------------------------
//  コアアフィニティの設定
void SetThreadCoreMask(ThreadType* thread, int idealCore, Bit64 affinityMask) NN_NOEXCEPT
{
    // 事前条件チェック
    NN_SDK_REQUIRES(idealCore==nn::os::IdealCoreDontCare || idealCore==nn::os::IdealCoreNoUpdate || idealCore==nn::os::IdealCoreUseDefaultValue ||
            (idealCore >=0 && idealCore < nn::os::detail::CoreAffinityMaskBitWidth), NN_TEXT_OS("nn::os::SetThreadCoreMask(): idealCore=%d の値が不正です。"), idealCore);
    if (idealCore != nn::os::IdealCoreUseDefaultValue)
    {
        NN_SDK_REQUIRES(affinityMask != 0, NN_TEXT_OS("nn::os::SetThreadCoreMask(): affinityMask には、有効なコアを 1 つ以上含む必要があります。"));
        NN_SDK_REQUIRES((affinityMask & ~GetThreadAvailableCoreMask()) == 0, NN_TEXT_OS("nn::os::SetThreadCoreMask(): affinityMask に無効なコアが含まれています。"));
    }
    if (idealCore >= 0)
    {
        NN_SDK_REQUIRES(affinityMask & (1ULL << idealCore), NN_TEXT_OS("nn::os::SetThreadCoreMask(): idealCore=%d で指定されたコアが affinityMask に含まれていません。"), idealCore);
    }

    detail::GetThreadManagerInstance()->SetThreadCoreMask(thread, idealCore, affinityMask);
}


//---------------------------------------------------------------------------
//  コアアフィニティの取得
void GetThreadCoreMask(int* pOutIdealCore, Bit64* pOutAffinityMask, const ThreadType* thread) NN_NOEXCEPT
{
    detail::GetThreadManagerInstance()->GetThreadCoreMask(pOutIdealCore, pOutAffinityMask, thread);
}


//---------------------------------------------------------------------------
//  割り当て可能なコアの取得
Bit64 GetThreadAvailableCoreMask() NN_NOEXCEPT
{
    return detail::GetThreadManagerInstance()->GetThreadAvailableCoreMask();
}

//---------------------------------------------------------------------------
//  スレッド ID の取得
ThreadId GetThreadId(const ThreadType* thread) NN_NOEXCEPT
{
    return detail::GetThreadManagerInstance()->GetThreadId(thread);
}

//---------------------------------------------------------------------------
//  自プロセスをクイック終了する。
//  本 API は開示されておらず、特定の SDK モジュールからのみ利用される。
//  静的オブジェクトのデストラクタを呼ばず、std::quick_exit() するのみです。
//
NN_NORETURN void QuickExit() NN_NOEXCEPT
{
    detail::GetThreadManagerInstance()->QuickExit();
}


//---------------------------------------------------------------------------

}} // namespace nn::os

