﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Result.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/os/os_NativeHandleTypes.h>
#include <nn/os/os_Result.h>
#include <nn/os/os_EventCommon.h>
#include <nn/os/os_EventApi.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/os/os_SystemEventApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_TimeoutHelper.h"
#include "detail/os_InterProcessEvent.h"
#include "detail/os_MultipleWaitHelper.h"
#include "detail/os_MultipleWaitHolderOfEvent.h"
#include "detail/os_MultipleWaitHolderOfInterProcessEvent.h"
#include "detail/os_MultipleWaitHolderBase.h"
#include "detail/os_MultipleWaitHolderImpl.h"


//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace os {

//---------------------------------------------------------------------------
//  SystemEventType オブジェクトの初期化
Result CreateSystemEvent(SystemEventType* event, EventClearMode clearMode, bool interProcess) NN_NOEXCEPT
{
    if (interProcess)
    {
        auto result = detail::CreateInterProcessEvent( &event->_interProcessEvent, clearMode );
        if (!result.IsSuccess())
        {
            return result;
        }
        event->_state = SystemEventType::State_InitializedAsInterProcessEvent;
    }
    else
    {
        InitializeEvent( &event->_event, false, clearMode );
        event->_state = SystemEventType::State_InitializedAsEvent;
    }
    return ResultSuccess();
}


//---------------------------------------------------------------------------
//  SystemEventType オブジェクトへのハンドルのアタッチ
void AttachSystemEvent(SystemEventType* event, NativeHandle readableHandle, bool isReadableHandleManaged, NativeHandle writableHandle, bool isWritableHandleManaged, EventClearMode clearMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( readableHandle != nn::os::InvalidNativeHandle ||
                     writableHandle != nn::os::InvalidNativeHandle,
        NN_TEXT_OS("nn::os::AttachSystemEvent(): readableHandle もしくは writableHandle のいずれかは有効なハンドルでなければなりません。") );

    detail::AttachInterProcessEvent( &event->_interProcessEvent, readableHandle, isReadableHandleManaged, writableHandle, isWritableHandleManaged, clearMode );
    event->_state = SystemEventType::State_InitializedAsInterProcessEvent;
}


//---------------------------------------------------------------------------
//  SystemEventType オブジェクトへの readableHandle のアタッチ
void AttachReadableHandleToSystemEvent(SystemEventType* event, NativeHandle readableHandle, bool isReadableHandleManaged, EventClearMode clearMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( readableHandle != nn::os::InvalidNativeHandle, NN_TEXT_OS("nn::os::AttachReadableHandleToSystemEvent(): readableHandle が有効なハンドルではありません。") );

    detail::AttachInterProcessEvent( &event->_interProcessEvent, readableHandle, isReadableHandleManaged, nn::os::InvalidNativeHandle, false, clearMode );
    event->_state = SystemEventType::State_InitializedAsInterProcessEvent;
}


//---------------------------------------------------------------------------
//  SystemEventType オブジェクトへの writableHandle のアタッチ
void AttachWritableHandleToSystemEvent(SystemEventType* event, NativeHandle writableHandle, bool isWritableHandleManaged, EventClearMode clearMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( writableHandle != nn::os::InvalidNativeHandle, NN_TEXT_OS("nn::os::AttachWritableHandleToSystemEvent(): writableHandle が有効なハンドルではありません。") );

    detail::AttachInterProcessEvent( &event->_interProcessEvent, nn::os::InvalidNativeHandle, false, writableHandle, isWritableHandleManaged, clearMode );
    event->_state = SystemEventType::State_InitializedAsInterProcessEvent;
}


//---------------------------------------------------------------------------
//  SystemEventType オブジェクトからのハンドルのデタッチ
NativeHandle DetachReadableHandleOfSystemEvent(SystemEventType* event) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( event->_state == SystemEventType::State_InitializedAsInterProcessEvent, NN_TEXT_OS("nn::os::DetachReadableHandleOfSystemEvent(): 指定されたシステムイベントはプロセス間同期用に初期化されていません。") );

    return detail::DetachReadableHandleOfInterProcessEvent( &event->_interProcessEvent );
}

NativeHandle DetachWritableHandleOfSystemEvent(SystemEventType* event) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( event->_state == SystemEventType::State_InitializedAsInterProcessEvent, NN_TEXT_OS("nn::os::DetachWritableHandleOfSystemEvent(): 指定されたシステムイベントはプロセス間同期用に初期化されていません。") );

    return detail::DetachWritableHandleOfInterProcessEvent( &event->_interProcessEvent );
}


//---------------------------------------------------------------------------
//  SystemEventType オブジェクトの破棄
void DestroySystemEvent(SystemEventType* event) NN_NOEXCEPT
{
    auto prevState = event->_state;

    // システムイベントを NotInitialized 状態へ（最初に行なう）
    event->_state  = SystemEventType::State_NotInitialized;

    switch (prevState)
    {
    case SystemEventType::State_InitializedAsInterProcessEvent:
            detail::DestroyInterProcessEvent( &event->_interProcessEvent );
            break;

    case SystemEventType::State_InitializedAsEvent:
            FinalizeEvent( &event->_event );
            break;

    default:
            NN_ABORT(NN_TEXT_OS("nn::os::DestroySystemEvent(): システムイベントが初期化されていません。"));
            break;
    }
}


//---------------------------------------------------------------------------
//  SystemEventType がシグナル状態になるまで待機
void WaitSystemEvent(SystemEventType* event) NN_NOEXCEPT
{
    switch (event->_state)
    {
    case SystemEventType::State_InitializedAsInterProcessEvent:
            detail::WaitInterProcessEvent( &event->_interProcessEvent );
            break;

    case SystemEventType::State_InitializedAsEvent:
            WaitEvent( &event->_event );
            break;

    default:
            NN_ABORT(NN_TEXT_OS("nn::os::WaitSystemEvent(): システムイベントが初期化されていません。"));
            break;
    }
}


//---------------------------------------------------------------------------
//  SystemEventType がシグナル状態になるまで待機（ポーリング）
bool TryWaitSystemEvent(SystemEventType* event) NN_NOEXCEPT
{
    switch (event->_state)
    {
    case SystemEventType::State_InitializedAsInterProcessEvent:
            return detail::TryWaitInterProcessEvent( &event->_interProcessEvent );

    case SystemEventType::State_InitializedAsEvent:
            return TryWaitEvent( &event->_event );

    default:
            NN_ABORT(NN_TEXT_OS("nn::os::TryWaitSystemEvent(): システムイベントが初期化されていません。"));
            break;
    }
}


//---------------------------------------------------------------------------
//  SystemEventType がシグナル状態になるまで待機（タイムアウト付き）
bool TimedWaitSystemEvent(SystemEventType* event, TimeSpan timeout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0, NN_TEXT_OS("nn::os::TimedWaitSystemEvent(): timeout 値が不正です。") );

    switch (event->_state)
    {
    case SystemEventType::State_InitializedAsInterProcessEvent:
            return detail::TimedWaitInterProcessEvent( &event->_interProcessEvent, timeout );

    case SystemEventType::State_InitializedAsEvent:
            return TimedWaitEvent( &event->_event, timeout );

    default:
            NN_ABORT(NN_TEXT_OS("nn::os::TimedWaitSystemEvent(): システムイベントが初期化されていません。"));
            break;
    }
}


//---------------------------------------------------------------------------
//  SystemEventType をシグナル状態へセット
void SignalSystemEvent(SystemEventType* event) NN_NOEXCEPT
{
    switch (event->_state)
    {
    case SystemEventType::State_InitializedAsInterProcessEvent:
            detail::SignalInterProcessEvent( &event->_interProcessEvent );
            break;

    case SystemEventType::State_InitializedAsEvent:
            SignalEvent( &event->_event );
            break;

    default:
            NN_ABORT(NN_TEXT_OS("nn::os::SignalSystemEvent(): システムイベントが初期化されていません。"));
            break;
    }
}


//---------------------------------------------------------------------------
//  SystemEventType を非シグナル状態へクリア
void ClearSystemEvent(SystemEventType* event) NN_NOEXCEPT
{
    switch (event->_state)
    {
    case SystemEventType::State_InitializedAsInterProcessEvent:
            detail::ClearInterProcessEvent( &event->_interProcessEvent );
            break;

    case SystemEventType::State_InitializedAsEvent:
            ClearEvent( &event->_event );
            break;

    default:
            NN_ABORT(NN_TEXT_OS("nn::os::ClearSystemEvent(): システムイベントが初期化されていません。"));
            break;
    }
}


//---------------------------------------------------------------------------
//  SystemEventType オブジェクトの ReadableHandle を取得する
NativeHandle GetReadableHandleOfSystemEvent(const SystemEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == SystemEventType::State_InitializedAsInterProcessEvent, NN_TEXT_OS("nn::os::GetReadableHandleOfSystemEvent(): 指定されたシステムイベントはプロセス間同期用に初期化されていません。") );

    return detail::GetReadableHandleOfInterProcessEvent( &event->_interProcessEvent );
}


//---------------------------------------------------------------------------
//  InterProcessEvent オブジェクトの WritableHandle を取得する
NativeHandle GetWritableHandleOfSystemEvent(const SystemEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == SystemEventType::State_InitializedAsInterProcessEvent, NN_TEXT_OS("nn::os::GetWritableHandleOfSystemEvent(): 指定されたシステムイベントはプロセス間同期用に初期化されていません。") );

    return detail::GetWritableHandleOfInterProcessEvent( &event->_interProcessEvent );
}


#if 1
//---------------------------------------------------------------------------
//  nvn.nss のリンク対策用。HR16 受取後に削除できるはず。
NativeHandle GetReadableHandleOfSystemEvent(SystemEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == SystemEventType::State_InitializedAsInterProcessEvent, NN_TEXT_OS("nn::os::GetReadableHandleOfSystemEvent(): 指定されたシステムイベントはプロセス間同期用に初期化されていません。") );

    return detail::GetReadableHandleOfInterProcessEvent( &event->_interProcessEvent );
}

NativeHandle GetWritableHandleOfSystemEvent(SystemEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == SystemEventType::State_InitializedAsInterProcessEvent, NN_TEXT_OS("nn::os::GetWritableHandleOfSystemEvent(): 指定されたシステムイベントはプロセス間同期用に初期化されていません。") );

    return detail::GetWritableHandleOfInterProcessEvent( &event->_interProcessEvent );
}
#endif

//---------------------------------------------------------------------------
//  MultiWaitHolderType の初期化（SystemEventType を関連付ける）
void InitializeMultiWaitHolder(MultiWaitHolderType* pHolder, SystemEventType* event) NN_NOEXCEPT
{
    switch (event->_state)
    {
    case SystemEventType::State_InitializedAsInterProcessEvent:
            new( &Get(pHolder->_holderImpl) ) detail::MultiWaitHolderOfInterProcessEvent( &event->_interProcessEvent );
            break;

    case SystemEventType::State_InitializedAsEvent:
            new( &Get(pHolder->_holderImpl) ) detail::MultiWaitHolderOfEvent( &event->_event );
            break;

    default:
            NN_ABORT(NN_TEXT_OS("nn::os::InitializeMultiWaitHolder(): システムイベントが初期化されていません。"));
            break;
    }

    // 構造体メンバの初期化
    pHolder->userData = 0;
}


}} // namespace nn::os

