﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/os/os_SemaphoreTypes.h>
#include <nn/os/os_SemaphoreApi.h>
#include <nn/os/os_MultipleWaitApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_TimeoutHelper.h"
#include "detail/os_MultipleWaitHelper.h"
#include "detail/os_MultipleWaitHolderOfSemaphore.h"
#include "detail/os_MultipleWaitHolderImpl.h"


//--------------------------------------------------------------------------
//  C++ 関数の定義
//--------------------------------------------------------------------------

namespace nn { namespace os {

//---------------------------------------------------------------------------
//  SemaphoreType オブジェクトのイニシャライズ
void InitializeSemaphore(SemaphoreType* semaphore, int initialCount, int maxCount) NN_NOEXCEPT
{
    // 事前条件チェック
    NN_SDK_REQUIRES( maxCount >= 1, NN_TEXT_OS("nn::os::InitializeSemaphore(): maxCount は 1 以上でなければなりません（maxCount= %d）"), maxCount );
    NN_SDK_REQUIRES( (initialCount >= 0) && (initialCount <= maxCount), NN_TEXT_OS("nn::os::InitializeSemaphore(): initialCount が不正です（initialCount= %d） "), initialCount );


    // 排他リソースを初期化
    new( &semaphore->_csSemaphore )  detail::InternalCriticalSection;
    new( &semaphore->_cvNotZero   )  detail::InternalConditionVariable;

    // IntrusiveList のコンストラクタを呼ぶ
    new( &semaphore->_multiWaitObjectList ) detail::MultiWaitObjectList;

    // メンバの初期化
    semaphore->_maxCount     = maxCount;
    semaphore->_count        = initialCount;

    // SemaphoreType を Initialized 状態へ（最後に行なう）
    semaphore->_state        = SemaphoreType::State_Initialized;
}


//---------------------------------------------------------------------------
//  SemaphoreType オブジェクトのファイナライズ
void FinalizeSemaphore(SemaphoreType* semaphore) NN_NOEXCEPT
{
    // SemaphoreType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( semaphore->_state == SemaphoreType::State_Initialized, NN_TEXT_OS("nn::os::FinalizeSemaphore(): 指定されたセマフォが初期化されていません。") );

    // 待機スレッドがいないことを確認
    NN_SDK_REQUIRES( Get(semaphore->_multiWaitObjectList).IsEmpty(), NN_TEXT_OS("nn::os::FinalizeSemaphore(): 指定されたセマフォに待機スレッドが残っています。") );


    // SemaphoreType を NotInitialized 状態へ変更（最初に行なう）
    semaphore->_state  = SemaphoreType::State_NotInitialized;

    // IntrusiveList のデストラクタを呼ぶ
    Get(semaphore->_multiWaitObjectList).~MultiWaitObjectList();

    // 排他リソースを破棄
    Get(semaphore->_csSemaphore).~InternalCriticalSection();
    Get(semaphore->_cvNotZero).~InternalConditionVariable();
}


//--------------------------------------------------------------------------
//  Semaphore の獲得（Ｐ操作）
void AcquireSemaphore(SemaphoreType* semaphore) NN_NOEXCEPT
{
    // SemaphoreType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( semaphore->_state == SemaphoreType::State_Initialized, NN_TEXT_OS("nn::os::AcquireSemaphore(): 指定されたセマフォが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(semaphore->_csSemaphore);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // セマフォカウンタが 0 なら待機
        while (semaphore->_count == 0)
        {
            Get(semaphore->_cvNotZero).Wait( &cs );
        }
        // セマフォを獲得
        semaphore->_count--;
    }
}


//  Semaphore の獲得（Ｐ操作）（ポーリング）
bool TryAcquireSemaphore(SemaphoreType* semaphore) NN_NOEXCEPT
{
    // SemaphoreType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( semaphore->_state == SemaphoreType::State_Initialized, NN_TEXT_OS("nn::os::TryAcquireSemaphore(): 指定されたセマフォが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(semaphore->_csSemaphore);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // セマフォカウンタが 0 なら false を返す
        if (semaphore->_count == 0)
        {
            return false;
        }

        // セマフォを獲得
        semaphore->_count--;
    }
    return true;
}


//  Semaphore の獲得（Ｐ操作）（タイムアウト付き）
bool TimedAcquireSemaphore(SemaphoreType* semaphore, TimeSpan timeout) NN_NOEXCEPT
{
    // SemaphoreType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( semaphore->_state == SemaphoreType::State_Initialized, NN_TEXT_OS("nn::os::TimedAcquireSemaphore(): 指定されたセマフォが初期化されていません。") );
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0, NN_TEXT_OS("nn::os::TimedAcquireSemaphore(): timeout 値が不正です。") );

    {   // クリティカルセクション
        detail::TimeoutHelper               tmout( timeout );
        detail::InternalCriticalSection&    cs = Get(semaphore->_csSemaphore);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // セマフォカウンタが 0 なら待機
        while (semaphore->_count == 0)
        {
            // タイムアウトならリターン
            if ( tmout.IsTimedout() )
            {
                return false;
            }

            Get(semaphore->_cvNotZero).TimedWait( &cs, tmout );
        }
        // セマフォを獲得
        semaphore->_count--;
    }

    return true;
}


//--------------------------------------------------------------------------
//  Semaphore を１カウント返却（Ｖ操作）
void ReleaseSemaphore(SemaphoreType* semaphore) NN_NOEXCEPT
{
    // SemaphoreType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( semaphore->_state == SemaphoreType::State_Initialized, NN_TEXT_OS("nn::os::ReleaseSemaphore(): 指定されたセマフォが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(semaphore->_csSemaphore);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // 増分後のカウンタは最大値以下でなければならない。
        NN_SDK_ASSERT( (semaphore->_count + 1) <= semaphore->_maxCount, NN_TEXT_OS("nn::os::ReleaseSemaphore(): セマフォカウンタが最大値（%d）を超えて返却されようとしました。"), semaphore->_maxCount );

        // セマフォカウンタを１つだけ返却
        semaphore->_count++;

        // セマフォ待機スレッド１つに NotZero 条件変数を通知
        Get(semaphore->_cvNotZero).Signal();

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(semaphore->_multiWaitObjectList).WakeupAllMultiWaitThreadsUnsafe();
    }
}


//--------------------------------------------------------------------------
//  Semaphore を複数カウント返却（Ｖ操作）
void ReleaseSemaphore(SemaphoreType* semaphore, int count) NN_NOEXCEPT
{
    // SemaphoreType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( semaphore->_state == SemaphoreType::State_Initialized, NN_TEXT_OS("nn::os::ReleaseSemaphore(): 指定されたセマフォが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(semaphore->_csSemaphore);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // 増分後のカウンタは最大値以下でなければならない。
        NN_SDK_ASSERT( (semaphore->_count + count) <= semaphore->_maxCount, NN_TEXT_OS("nn::os::ReleaseSemaphore(): セマフォカウンタが最大値（%d）を超えて返却されようとしました。"), semaphore->_maxCount );

        // セマフォカウンタを count 数だけ返却
        semaphore->_count += count;

        // セマフォ待機スレッド全てに NotZero 条件変数を通知
        Get(semaphore->_cvNotZero).Broadcast();

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(semaphore->_multiWaitObjectList).WakeupAllMultiWaitThreadsUnsafe();
    }
}


//--------------------------------------------------------------------------
//  Semaphore カウンタ値の取得
int GetCurrentSemaphoreCount(const SemaphoreType* semaphore) NN_NOEXCEPT
{
    // SemaphoreType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( semaphore->_state == SemaphoreType::State_Initialized, NN_TEXT_OS("nn::os::GetCurrentSemaphoreCount(): 指定されたセマフォが初期化されていません。") );

    // 現在のカウンタ値を返す
    return  semaphore->_count;
}


//---------------------------------------------------------------------------
//  MultiWaitHolderType の初期化（Semaphore を関連付ける）
void InitializeMultiWaitHolder(MultiWaitHolderType* pHolder, SemaphoreType* semaphore) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( semaphore->_state == SemaphoreType::State_Initialized, NN_TEXT_OS("nn::os::InitializeMultiWaitHolder(): 指定されたセマフォが初期化されていません。") );

    // 初期化処理（コンストラクタを呼び出す）
    // Get() がなくてもポインタは得られるが、TypedStorage のサイズと
    // アライメントチェックのため、Get() を使って実装しておく。
    new( &Get(pHolder->_holderImpl) ) detail::MultiWaitHolderOfSemaphore( semaphore );

    //  構造体メンバの初期化
    pHolder->userData = 0;
}


}} // namespace nn::os

