﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>

#include <nn/os/os_Config.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/os/os_ConditionVariableCommon.h>
#include <nn/os/os_ConditionVariableTypes.h>
#include <nn/os/os_ConditionVariableApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_ThreadManager.h"
#include "detail/os_TimeoutHelper.h"
#include "detail/os_MutexImpl.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <mutex>
#include <atomic>
#include "detail/os_GiantLock.h"
#endif

//---------------------------------------------------------------------------
// C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

namespace {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    // Win32 環境の場合は、NN_OS_USE_STD_CONDITION_VARIABLE_CLASS の
    // 定義／未定義に関わらず LazyInitialization のチェックを必ず行なう。
    // これは NN_OS_CONDITION_VARIABLE_INITIALIZER による静的初期化に対して、
    // OS 実装の破壊的変更による副作用を排除するためである。

    #if defined(NN_BUILD_CONFIG_ADDRESS_32)
        typedef std::atomic<uint8_t>    LazyAtomicFlag;
        static_assert(offsetof(ConditionVariableType, _needLazyInitialize) == 1, "");
    #elif defined(NN_BUILD_CONFIG_ADDRESS_64)
        typedef std::atomic<uint32_t>   LazyAtomicFlag;
        static_assert(offsetof(ConditionVariableType, _needLazyInitialize) == 4, "");
    #endif
    const uint8_t needLazyIsFalse = 0;
    const uint8_t needLazyIsTrue  = 1;

    // TORIAEZU: 静的初期化の遅延初期化を行なう（ダブルチェックロック）。
    // この対応は Windows 版かつ std::condition_variable_any 利用時のみ必要。
    // ::ConditionVariable 使用時は静的初期化が可能。
    // 参考 URL:
    //   http://yamasa.hatenablog.jp/entry/20100128/1264693781
    inline void CheckAndLazyInitializeConditionVariable(ConditionVariableType* cond) NN_NOEXCEPT
    {
        static_assert(sizeof(cond->_needLazyInitialize) == sizeof(LazyAtomicFlag), "");
        LazyAtomicFlag& needLazyInit = reinterpret_cast<LazyAtomicFlag&>(cond->_needLazyInitialize);

        auto needs = needLazyInit.load(std::memory_order_acquire);
        if (needs == needLazyIsTrue)
        {
            // GiantLock で同時呼び出しに対処しておく
            std::lock_guard<GiantLock> lock(*GetGiantLockInstance());

            needs = needLazyInit.load(std::memory_order_relaxed);
            if (needs == needLazyIsTrue)
            {
                // 呼び出し先で release バリアを構築
                InitializeConditionVariable(cond);
            }
        }
    }
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    inline void CheckAndLazyInitializeConditionVariable(ConditionVariableType* cond) NN_NOEXCEPT
    {
        NN_UNUSED(cond);
    }
#endif
}   // namespace

}   // namespace detail

//---------------------------------------------------------------------------
//  条件変数オブジェクトのイニシャライズ
void InitializeConditionVariable(ConditionVariableType* conditionVariable) NN_NOEXCEPT
{
    // 条件変数オブジェクトの初期化
    new( &conditionVariable->_condition ) detail::InternalConditionVariable;

    // 条件変数を Initialized 状態へ（最後に行なう）
    conditionVariable->_state  = ConditionVariableType::State_Initialized;

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    detail::LazyAtomicFlag& needLazyInit = reinterpret_cast<detail::LazyAtomicFlag&>(conditionVariable->_needLazyInitialize);
    needLazyInit.store(detail::needLazyIsFalse, std::memory_order_release);
#endif
}

//---------------------------------------------------------------------------
//  条件変数オブジェクトのファイナライズ
void FinalizeConditionVariable(ConditionVariableType* conditionVariable) NN_NOEXCEPT
{
    detail::CheckAndLazyInitializeConditionVariable(conditionVariable);

    // 条件変数オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( conditionVariable->_state == ConditionVariableType::State_Initialized, NN_TEXT_OS("nn::os::FinalizeConditionVariable(): 指定された条件変数が初期化されていません。") );

    // 条件変数オブジェクトを NotInitialized 状態へ
    conditionVariable->_state  = ConditionVariableType::State_NotInitialized;

    // 条件変数オブジェクトの破棄
    Get(conditionVariable->_condition).~InternalConditionVariable();
}

//---------------------------------------------------------------------------
//  条件変数にシグナル通知
void SignalConditionVariable(ConditionVariableType* conditionVariable) NN_NOEXCEPT
{
    detail::CheckAndLazyInitializeConditionVariable(conditionVariable);

    // 条件変数オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( conditionVariable->_state == ConditionVariableType::State_Initialized, NN_TEXT_OS("nn::os::SignalConditionVariable(): 指定された条件変数が初期化されていません。") );

    // 条件変数通知を発行
    Get(conditionVariable->_condition).Signal();
}

//---------------------------------------------------------------------------
//  条件変数にブロードキャスト通知
void BroadcastConditionVariable(ConditionVariableType* conditionVariable) NN_NOEXCEPT
{
    detail::CheckAndLazyInitializeConditionVariable(conditionVariable);

    // 条件変数オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( conditionVariable->_state == ConditionVariableType::State_Initialized, NN_TEXT_OS("nn::os::BroadcastConditionVariable(): 指定された条件変数が初期化されていません。") );

    // 条件変数通知を発行
    Get(conditionVariable->_condition).Broadcast();
}

//---------------------------------------------------------------------------
//  条件変数の待機
void WaitConditionVariable(ConditionVariableType* conditionVariable, MutexType *mutex) NN_NOEXCEPT
{
    detail::CheckAndLazyInitializeConditionVariable(conditionVariable);

    // 条件変数オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( conditionVariable->_state == ConditionVariableType::State_Initialized, NN_TEXT_OS("nn::os::WaitConditionVariable(): 指定された条件変数が初期化されていません。") );

    // ミューテックスが自スレッドによってロックされているか
    NN_SDK_REQUIRES( mutex->_state == MutexType::State_Initialized, NN_TEXT_OS("nn::os::WaitConditionVariable(): 指定されたミューテックスが初期化されていません。") );
    NN_ABORT_UNLESS( mutex->_ownerThread == detail::GetCurrentThread(), NN_TEXT_OS("nn::os::WaitConditionVariable(): 指定されたミューテックスの所有者が自スレッドではありません。"));

    // ミューテックスのネストカウンタが 1 か？（この時点で一度はロック済み）
    NN_SDK_ASSERT( mutex->_nestCount == 1, NN_TEXT_OS("nn::os::WaitConditionVariable(): ミューテックスが多重ロックされています。") );

    // この時点ではまだ Mutex を獲得している状態。
    // ロックレベルのチェック（Unlock）
    detail::PopAndCheckLockLevel( mutex );

    // ネストカウンタをデクリメントし、Mutex の所有者をクリア
    mutex->_nestCount--;
    if (mutex->_nestCount == 0)
    {
        mutex->_ownerThread = NULL;
    }

    // 条件変数待ちを発行（Mutex も一旦解放される）
    Get(conditionVariable->_condition).Wait( &Get(mutex->_mutex) );

    // この時点でも再度 Mutex を獲得している状態に戻っている
    // ロックレベルのチェック（Lock）
    detail::PushAndCheckLockLevel( mutex );

    // 再度 Mutex を獲得
    mutex->_nestCount++;
    mutex->_ownerThread = detail::GetCurrentThread();
}

//---------------------------------------------------------------------------
//  条件変数の待機（タイムアウト付き）
ConditionVariableStatus TimedWaitConditionVariable(ConditionVariableType* conditionVariable, MutexType *mutex, TimeSpan timeout) NN_NOEXCEPT
{
    detail::CheckAndLazyInitializeConditionVariable(conditionVariable);

    // 条件変数オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( conditionVariable->_state == ConditionVariableType::State_Initialized, NN_TEXT_OS("nn::os::TimedWaitConditionVariable(): 指定された条件変数が初期化されていません。") );

    // ミューテックスが自スレッドによってロックされているか
    NN_SDK_REQUIRES( mutex->_state == MutexType::State_Initialized, NN_TEXT_OS("nn::os::TimedWaitConditionVariable(): 指定されたミューテックスが初期化されていません。") );
    NN_ABORT_UNLESS( mutex->_ownerThread == detail::GetCurrentThread(), NN_TEXT_OS("nn::os::TimedWaitConditionVariable(): 指定されたミューテックスの所有者が自スレッドではありません。"));

    // タイムアウト値
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0, NN_TEXT_OS("nn::os::TimedWaitConditionVariable(): timeout 値が不正です。") );

    // ミューテックスのネストカウンタが 1 か？（この時点で一度はロック済み）
    NN_SDK_ASSERT( mutex->_nestCount == 1, NN_TEXT_OS("nn::os::TimedWaitConditionVariable(): ミューテックスが多重ロックされています。") );

    // この時点ではまだ Mutex を獲得している状態。
    // ロックレベルのチェック（Unlock）
    detail::PopAndCheckLockLevel( mutex );

    // ネストカウンタをデクリメントし、Mutex の所有者をクリア
    mutex->_nestCount--;
    if (mutex->_nestCount == 0)
    {
        mutex->_ownerThread = NULL;
    }

    // 条件変数待ちを発行（Mutex も一旦解放される）
    ConditionVariableStatus status;
    if (timeout == TimeSpan(0))
    {
        Get(mutex->_mutex).Leave();
        Get(mutex->_mutex).Enter();
        status = ConditionVariableStatus_Timeout;
    }
    else
    {
        detail::TimeoutHelper   tmout( timeout );
        status = Get(conditionVariable->_condition).TimedWait( &Get(mutex->_mutex), tmout );
    }

    // この時点でも再度 Mutex を獲得している状態に戻っている
    // ロックレベルのチェック（Lock）
    detail::PushAndCheckLockLevel( mutex );

    // 再度 Mutex を獲得
    mutex->_nestCount++;
    mutex->_ownerThread = detail::GetCurrentThread();

    // Mutex をロックしたまま戻る
    return status;
}

}}  // namespace nn::os

