﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/os/os_BarrierTypes.h>
#include <nn/os/os_BarrierApi.h>

#include "detail/os_Common.h"
#include "detail/os_Diag.h"


//---------------------------------------------------------------------------
// C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace os {

//---------------------------------------------------------------------------
//  BarrierType オブジェクトのイニシャライズ
void InitializeBarrier(BarrierType* barrier, int numThreads) NN_NOEXCEPT
{
    // 引数エラーチェック
    NN_SDK_REQUIRES( numThreads >= 1, NN_TEXT_OS("nn::os::InitializeBarrier(): numThreads=%d が不正です。"), numThreads );

    // 排他リソースの初期化
    new( &barrier->_csBarrier )  detail::InternalCriticalSection;
    new( &barrier->_cvGathered ) detail::InternalConditionVariable;

    // BarrierType オブジェクトを初期化
    barrier->_numThreads = 0;
    barrier->_maxThreads = numThreads;

    // state の更新は最後に行なう
    barrier->_state  = BarrierType::State_Initialized;
}

//---------------------------------------------------------------------------
//  BarrierType オブジェクトのファイナライズ
void FinalizeBarrier(BarrierType* barrier) NN_NOEXCEPT
{
    // BarrierType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( barrier->_state == BarrierType::State_Initialized, NN_TEXT_OS("nn::os::FinalizeBarrier(): 指定されたバリアが初期化されていません。") );

    // BarrierType オブジェクトに待ちスレッドがあれば異常停止
    NN_SDK_REQUIRES( barrier->_numThreads == 0, NN_TEXT_OS("nn::os::FinalizeBarrier(): 指定されたバリアに待機スレッドが残っています。") );

    // BarrierType を NotInitialized 状態へ変更（最初に行なう）
    barrier->_state  = BarrierType::State_NotInitialized;

    // 排他リソースの破棄
    Get(barrier->_csBarrier).~InternalCriticalSection();
    Get(barrier->_cvGathered).~InternalConditionVariable();
}

//---------------------------------------------------------------------------
//  Barrier の待合せ
void AwaitBarrier(BarrierType* barrier) NN_NOEXCEPT
{
    // BarrierType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( barrier->_state == BarrierType::State_Initialized, NN_TEXT_OS("nn::os::AwaitBarrier(): 指定されたバリアが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(barrier->_csBarrier);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // 前の同期が終わるまで待機
        while ( barrier->_numThreads >= barrier->_maxThreads )
        {
            Get(barrier->_cvGathered).Wait( &cs );
        }

        // 抜ける条件を計算
        const auto  upToCount = barrier->_maxThreads * 2 - 1;

        // 待ち合わせスレッド数をインクリメント
        barrier->_numThreads++;

        // スレッド数が指定数に満たない場合は待ちに入る
        if ( barrier->_numThreads < barrier->_maxThreads )
        {
            // スレッド数が満たされるまで待機
            while ( barrier->_numThreads < barrier->_maxThreads )
            {
                Get(barrier->_cvGathered).Wait( &cs );
            }

            // 抜ける時もインクリメント
            barrier->_numThreads++;
            if (barrier->_numThreads == upToCount)
            {
                // 最後に到達したスレッドを起床
                Get(barrier->_cvGathered).Broadcast();
            }
        }
        else
        {
            // 自分が最後の到達スレッドなので、他のスレッドを起床させる
            Get(barrier->_cvGathered).Broadcast();

            // 自分より先に到達したスレッド群が全て抜けるまで待機
            while ( barrier->_numThreads < upToCount )
            {
                Get(barrier->_cvGathered).Wait( &cs );
            }

            // 最後に抜けるスレッドがクリアする
            barrier->_numThreads = 0;

            // 「前の同期待ち」をしていたスレッドを起床
            Get(barrier->_cvGathered).Broadcast();
        }
    }
}

}}  // namespace nn::os

