﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Config.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------

#if defined(NN_OS_CPU_ARM_AARCH64_ARMV8A) // 64bit のみ対応

class VammManagerImplByHorizon
{
public:
    static void GetReservedRegionImpl(uintptr_t* pOutStart, uintptr_t* pOutSize)
    {
        Bit64 start;
        Bit64 size;

        // Reserved 領域を範囲をカーネルから取得
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::GetInfo(&start, svc::InfoType_ReservedRegionAddress, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::GetInfo(&size,  svc::InfoType_ReservedRegionSize,    svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));

        *pOutStart = start;
        *pOutSize =  size;
    }

    static nn::Result AllocatePhysicalMemoryImpl(uintptr_t address, size_t size)
    {
        auto result = svc::MapPhysicalMemory(address, size);
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH( svc::ResultOutOfResource )
            {
                NN_RESULT_THROW( os::ResultOutOfResource() );
            }
            NN_RESULT_CATCH( svc::ResultOutOfMemory )
            {
                NN_RESULT_THROW( os::ResultOutOfMemory() );
            }
            NN_RESULT_CATCH( svc::ResultLimit )
            {
                NN_RESULT_THROW( os::ResultOutOfMemory() );
            }
            NN_RESULT_CATCH_ALL
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS( result );
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    static nn::Result FreePhysicalMemoryImpl(uintptr_t address, size_t size)
    {
        auto result = svc::UnmapPhysicalMemory(address, size);
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH( svc::ResultInvalidCurrentMemory )
            {
                NN_RESULT_THROW( os::ResultBusy() );
            }
            NN_RESULT_CATCH( svc::ResultOutOfResource )
            {
                NN_RESULT_THROW( os::ResultOutOfResource() );
            }
            NN_RESULT_CATCH( svc::ResultOutOfMemory )
            {
                // 発生しない？
                NN_RESULT_THROW( os::ResultOutOfMemory() );
            }
            NN_RESULT_CATCH( svc::ResultLimit )
            {
                NN_RESULT_THROW( os::ResultOutOfMemory() );
            }
            NN_RESULT_CATCH_ALL
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS( result );
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    static size_t GetExtraSystemResourceAssignedSize()
    {
        Bit64 v;
        auto r = svc::GetInfo(&v, svc::InfoType_ExtraResourceSize, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0);
        return r.IsSuccess() ? static_cast<size_t>(v): 0;
    }

    static size_t GetExtraSystemResourceUsedSize()
    {
        Bit64 v;
        auto r = svc::GetInfo(&v, svc::InfoType_ExtraResourceUsage, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0);
        return r.IsSuccess() ? static_cast<size_t>(v): 0;
    }

    static bool IsVirtualAddressMemoryEnabled()
    {
        return GetExtraSystemResourceAssignedSize() > 0;
    }
};

typedef VammManagerImplByHorizon VammManagerImpl;

#endif

//--------------------------------------------------------------------------

}  // namespace detail
}} // namespace nn::os

