﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>
#include <algorithm>
#include <nn/os/os_Config.h>
#include <nn/os/os_ReaderWriterLock.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/util/util_TypedStorage.h>

#include <nn/lmem/lmem_UnitHeap.h>

#include "os_VammManagerTypes.h"
#include "os_RbTree.h"

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include "os_DefaultUserExceptionHandlerImpl.h"
#endif

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include "os_VammManagerImpl-os.win32.h"
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "os_VammManagerImpl-os.horizon.h"
#else
    #error   "未サポートの OS 種別が指定されています。"
#endif


namespace nn { namespace os {
namespace detail {

// 64bit のみ対応
#if defined(NN_OS_CPU_ARM_AARCH64_ARMV8A) || defined(NN_BUILD_CONFIG_CPU_X64)

namespace
{

    enum AddressAllocationResult
    {
        AddressAllocationResult_Success,
        AddressAllocationResult_OutOfMemory,
        AddressAllocationResult_OutOfSpace,
    };

    class AddressRegion : public IntrusiveRbTreeBaseNode<AddressRegion>
    {
    public:
        AddressRegion(uintptr_t begin, size_t size) NN_NOEXCEPT
        : m_AddressBegin(begin), m_Size(size)
        {
        }
        uintptr_t   GetAddressBegin() const NN_NOEXCEPT
        {
            return m_AddressBegin;
        }
        uintptr_t   GetAddressEnd() const NN_NOEXCEPT
        {
            return m_AddressBegin + m_Size;
        }
        size_t      GetSize() const NN_NOEXCEPT
        {
            return m_Size;
        }
        bool IsContaine(uintptr_t address) const NN_NOEXCEPT
        {
            if( address < m_AddressBegin )
            {
                return false;
            }
            if( address < GetAddressEnd() )
            {
                return true;
            }

            return false;
        }
        bool IsContaine(uintptr_t address, size_t size) const NN_NOEXCEPT
        {
            uintptr_t end = address + size;

            if( ! (address <= end) )
            {
                return false;
            }
            if( ! (GetAddressBegin() <= address) )
            {
                return false;
            }
            if( ! (end <= GetAddressEnd()) )
            {
                return false;
            }

            return true;
        }

    private:
        uintptr_t   m_AddressBegin;
        size_t      m_Size;

    };
    struct AddressRegionCompare
    {
        int operator() (const AddressRegion& a, const AddressRegion& b) NN_NOEXCEPT
        {
            if( a.GetAddressBegin() < b.GetAddressBegin() )
            {
                return -1;
            }
            else if( a.GetAddressBegin() > b.GetAddressBegin() )
            {
                return 1;
            }
            return 0;
        }
    };
    template <class Mutex>
    class SharedLock
    {
    public:
        explicit SharedLock(Mutex* p) NN_NOEXCEPT : m_pMutex(p)
        {
            m_pMutex->AcquireReadLock();
        }
        ~SharedLock() NN_NOEXCEPT
        {
            m_pMutex->ReleaseReadLock();
        }
    private:
        Mutex*  m_pMutex;
    };

    typedef IntrusiveRbTree<AddressRegion, AddressRegionCompare> AddressRegionTree;

    class DynamicUnitHeap
    {
    private:
        static const size_t BufferUnitSize = MemoryPageSize;

    public:
        DynamicUnitHeap(uintptr_t address, size_t size, size_t unitSize) NN_NOEXCEPT
        : m_Begin(address), m_End(address + size)
        {
            VammManagerImpl::AllocatePhysicalMemoryImpl(m_Begin, BufferUnitSize);
            m_CurrentLimit = m_Begin + BufferUnitSize;

            m_Heap = lmem::CreateUnitHeap(
                reinterpret_cast<void*>(m_Begin),
                BufferUnitSize,
                unitSize,
                lmem::CreationOption_NoOption,
                8,
                &m_Head );
        }
        void* Allocate() NN_NOEXCEPT
        {
            void* p = lmem::AllocateFromUnitHeap(m_Heap);
            if( p == NULL )
            {
                Extend();
                p = lmem::AllocateFromUnitHeap(m_Heap);
            }

            return p;
        }
        void Free(void* p) NN_NOEXCEPT
        {
            lmem::FreeToUnitHeap(m_Heap, p);
        }

    private:
        void Extend() NN_NOEXCEPT
        {
            NN_ABORT_UNLESS( m_CurrentLimit < m_End );

            auto result = VammManagerImpl::AllocatePhysicalMemoryImpl(m_CurrentLimit, BufferUnitSize);
            if( result.IsSuccess() )
            {
                m_CurrentLimit += BufferUnitSize;

                lmem::ExtendUnitHeapArea(m_Heap, BufferUnitSize);
            }
        }

    private:
        uintptr_t               m_Begin;
        uintptr_t               m_CurrentLimit;
        uintptr_t               m_End;
        lmem::HeapCommonHead    m_Head;
        lmem::HeapHandle        m_Heap;
    };
}
// anonymous namespace


class AddressRegionManager
{
private:
    static const size_t UnitHeapRegionSize = 1 * 1024 * 1024 * 1024 - 2 * 1024 * 1024; // SIGLO-75986
    static const size_t Size4K  =           4 * 1024;
    static const size_t Size64K =          64 * 1024;
    static const size_t Size2M  =    2 * 1024 * 1024;
    static const size_t Size4M  =    4 * 1024 * 1024;
    static const size_t Size32M =   32 * 1024 * 1024;
    static const size_t Size1G  = 1024 * 1024 * 1024;

public:
    AddressRegionManager(uintptr_t begin, size_t size) NN_NOEXCEPT
    : m_Begin(begin), m_Size(size), m_Heap(begin, UnitHeapRegionSize, sizeof(AddressRegion))
    {
        // DynamicUnitHeap 用の領域
        // 既に DynamicUnitHeap のコンストラクタで領域は使用されているので後追いで領域を登録する
        m_Tree.Insert( new(m_Heap.Allocate()) AddressRegion(m_Begin, UnitHeapRegionSize) );

        // ダミーの終端
        // GetMax でこれが取れる
        m_Tree.Insert( new(m_Heap.Allocate()) AddressRegion(m_Begin + m_Size, 0) );
    }

    AddressAllocationResult Allocate(AddressRegion** ppOutRegion, size_t size) NN_NOEXCEPT
    {
        void* p = m_Heap.Allocate();
        if( p == NULL )
        {
            return AddressAllocationResult_OutOfMemory;
        }

        const size_t alignment = SelectAlignment(size);

        // 終端から前に向かって空き領域を探す
        auto pRegion = m_Tree.GetMax();

        for(;;)
        {
            auto pPrev = m_Tree.Prev(pRegion);
            if( pPrev == NULL )
            {
                break;
            }

            const uintptr_t spaceBegin = pPrev->GetAddressEnd()     + MemoryPageSize;
            const uintptr_t spaceEnd   = pRegion->GetAddressBegin() - MemoryPageSize;
            size_t freeSpaceSize = spaceEnd - spaceBegin;

            // まずサイズで簡易チェック
            if( freeSpaceSize >= size )
            {
                // アライメント込みできちんとチェック
                const uintptr_t regionBegin = util::align_up(spaceBegin, alignment);
                const uintptr_t regionEnd   = regionBegin + size;

                if( regionEnd <= spaceEnd )
                {
                    AddressRegion* pNew = new(p) AddressRegion(regionBegin, size);
                    m_Tree.Insert(pNew);
                    *ppOutRegion = pNew;
                    return AddressAllocationResult_Success;
                }
            }

            pRegion = pPrev;
        }

        return AddressAllocationResult_OutOfSpace;
    }

    bool IsAlreadyAllocated(uintptr_t address, size_t size) NN_NOEXCEPT
    {
        AddressRegion key(address, 0);

        // NearFind は address 以上の最近のエントリを返す
        auto pFound = m_Tree.NearFind(&key);
        if( pFound == NULL )
        {
            // ダミーの終端を登録しているので NULL が返るなら管理範囲を超えている
            return false;
        }

        // 「以上」を返すので pFound に対象領域が含まれるのは一致した場合のみ
        if( pFound->GetAddressBegin() == address )
        {
            return size <= pFound->GetSize();
        }

        // 一致しなかったら一つ前のエントリに含まれている
        // もしくはエントリをまたがっている
        pFound = m_Tree.Prev(pFound);

        if( pFound == NULL )
        {
            return false;
        }

        return pFound->IsContaine(address, size);
    }

    AddressRegion* Find(uintptr_t address) NN_NOEXCEPT
    {
        AddressRegion key(address, 0);
        return m_Tree.Find(&key);
    }

    void Free(AddressRegion* pRegion) NN_NOEXCEPT
    {
        m_Tree.Remove(pRegion);
        m_Heap.Free(pRegion);
    }

private:
    size_t SelectAlignment(size_t size) NN_NOEXCEPT
    {
        if( size < Size4M )
        {
            if( size < Size2M )
            {
                return Size64K;
            }
            else
            {
                return Size2M;
            }
        }
        else
        {
            if( size < Size32M )
            {
                return Size4M;
            }
            else if( size < Size1G )
            {
                return Size32M;
            }
            else
            {
                return Size1G;
            }
        }
    }

private:
    uintptr_t               m_Begin;
    uintptr_t               m_Size;
    AddressRegionTree       m_Tree;
    DynamicUnitHeap         m_Heap;
};


namespace
{
    util::TypedStorage<AddressRegionManager, sizeof(AddressRegionManager), NN_ALIGNOF(AddressRegionManager)>
        g_AddressRegionManagerStorage;
}
// anonymous namespace

#endif

//--------------------------------------------------------------------------


// 64bit のみ対応
#if defined(NN_OS_CPU_ARM_AARCH64_ARMV8A) || defined(NN_BUILD_CONFIG_CPU_X64)

VammManager::VammManager() NN_NOEXCEPT
: m_pRegionManager(NULL)
{
    // リザーブ領域を取得
    VammManagerImpl::GetReservedRegionImpl(&m_ReservedRegionStart, &m_ReservedRegionSize);
}

void VammManager::InitializeIfEnabled() NN_NOEXCEPT
{
    std::lock_guard<ReaderWriterLock> lock(m_Lock);

    if( m_pRegionManager == NULL )
    {
        if( IsVirtualAddressMemoryEnabled() )
        {
            m_pRegionManager = new(&g_AddressRegionManagerStorage.storage) AddressRegionManager(m_ReservedRegionStart, m_ReservedRegionSize);
        }
    }
}

nn::Result VammManager::AllocateAddressRegion(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT
{
    uintptr_t address;

    {
        std::lock_guard<ReaderWriterLock> lock(m_Lock);
        NN_SDK_ASSERT(m_pRegionManager != NULL, NN_TEXT_OS("仮想アドレスメモリ管理が有効になっていません。"));

        AddressRegion* pRegion;
        auto result = m_pRegionManager->Allocate(&pRegion, size);
        if( result != AddressAllocationResult_Success )
        {
            if( result == AddressAllocationResult_OutOfSpace )
            {
                return ResultOutOfVirtualAddressSpace();
            }
            else
            {
                return ResultOutOfMemory();
            }
        }

        address = pRegion->GetAddressBegin();
    }

    *pOutAddress = address;

    NN_RESULT_SUCCESS;
}

nn::Result VammManager::AllocateMemory(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT
{
    uintptr_t address;

    {
        std::lock_guard<ReaderWriterLock> lock(m_Lock);
        NN_SDK_ASSERT(m_pRegionManager != NULL, NN_TEXT_OS("仮想アドレスメモリ管理が有効になっていません。"));

        AddressRegion* pRegion;
        auto result = m_pRegionManager->Allocate(&pRegion, size);
        if( result != AddressAllocationResult_Success )
        {
            if( result == AddressAllocationResult_OutOfSpace )
            {
                return ResultOutOfVirtualAddressSpace();
            }
            else
            {
                return ResultOutOfMemory();
            }
        }

        auto r2 = VammManagerImpl::AllocatePhysicalMemoryImpl(pRegion->GetAddressBegin(), size);
        if( r2.IsFailure() )
        {
            m_pRegionManager->Free(pRegion);
            return r2;
        }

        address = pRegion->GetAddressBegin();
    }

    *pOutAddress = address;

    NN_RESULT_SUCCESS;
}

nn::Result VammManager::AllocateMemoryPages(uintptr_t address, size_t size) NN_NOEXCEPT
{
    SharedLock<ReaderWriterLock> lock(&m_Lock);
    NN_SDK_ASSERT(m_pRegionManager != NULL, NN_TEXT_OS("仮想アドレスメモリ管理が有効になっていません。"));

    // address, size が確保済みのアドレス空間内にあるかチェック
    if( !m_pRegionManager->IsAlreadyAllocated(address, size) )
    {
        return ResultInvalidParameter();
    }

    // 物理メモリを割り当て
    NN_RESULT_DO(VammManagerImpl::AllocatePhysicalMemoryImpl(address, size));

    NN_RESULT_SUCCESS;
}

nn::Result VammManager::FreeAddressRegion(uintptr_t address) NN_NOEXCEPT
{
    std::lock_guard<ReaderWriterLock> lock(m_Lock);
    NN_SDK_ASSERT(m_pRegionManager != NULL, NN_TEXT_OS("仮想アドレスメモリ管理が有効になっていません。"));

    auto pRegion = m_pRegionManager->Find(address);
    if( pRegion == NULL )
    {
        return ResultInvalidParameter();
    }

    // 物理メモリが割り当て済みなら解放
    NN_RESULT_DO(VammManagerImpl::FreePhysicalMemoryImpl(address, pRegion->GetSize()));

    // アドレス空間を解放
    m_pRegionManager->Free(pRegion);

    NN_RESULT_SUCCESS;
}

nn::Result VammManager::FreeMemoryPages(uintptr_t address, size_t size) NN_NOEXCEPT
{
    SharedLock<ReaderWriterLock> lock(&m_Lock);
    NN_SDK_ASSERT(m_pRegionManager != NULL, NN_TEXT_OS("仮想アドレスメモリ管理が有効になっていません。"));

    // address, size が確保済みのアドレス空間内にあるかチェック
    if( !m_pRegionManager->IsAlreadyAllocated(address, size) )
    {
        return ResultInvalidParameter();
    }

    // 物理メモリを解放
    NN_RESULT_DO(VammManagerImpl::FreePhysicalMemoryImpl(address, size));

    NN_RESULT_SUCCESS;
}

VirtualAddressMemoryResourceUsage VammManager::GetVirtualAddressMemoryResourceUsage() NN_NOEXCEPT
{
    const size_t assignedSize   = VammManagerImpl::GetExtraSystemResourceAssignedSize();
    size_t usedSize             = VammManagerImpl::GetExtraSystemResourceUsedSize();

    // 512 KB の下駄をはかせる
    usedSize = std::min(assignedSize, usedSize + 512 * 1024);

    VirtualAddressMemoryResourceUsage v;
    v.assignedSize  = assignedSize;
    v.usedSize      = usedSize;
    return v;
}

bool VammManager::IsVirtualAddressMemoryEnabled() const NN_NOEXCEPT
{
    return VammManagerImpl::IsVirtualAddressMemoryEnabled();
}

#else // 32bit は未対応

VammManager::VammManager() NN_NOEXCEPT
{
    NN_UNUSED(m_ReservedRegionStart);
    NN_UNUSED(m_ReservedRegionSize);
    NN_UNUSED(m_pRegionManager);
}

void VammManager::InitializeIfEnabled() NN_NOEXCEPT
{
}

nn::Result VammManager::AllocateAddressRegion(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(pOutAddress);
    NN_UNUSED(size);
    return nn::os::ResultNotSupported();
}

nn::Result VammManager::AllocateMemory(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(pOutAddress);
    NN_UNUSED(size);
    return nn::os::ResultNotSupported();
}

nn::Result VammManager::AllocateMemoryPages(uintptr_t address, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(address);
    NN_UNUSED(size);
    return nn::os::ResultNotSupported();
}

nn::Result VammManager::FreeAddressRegion(uintptr_t address) NN_NOEXCEPT
{
    NN_UNUSED(address);
    return nn::os::ResultNotSupported();
}

nn::Result VammManager::FreeMemoryPages(uintptr_t address, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(address);
    NN_UNUSED(size);
    return nn::os::ResultNotSupported();
}

VirtualAddressMemoryResourceUsage VammManager::GetVirtualAddressMemoryResourceUsage() NN_NOEXCEPT
{
    VirtualAddressMemoryResourceUsage v = { 0, 0 };
    return v;
}

bool VammManager::IsVirtualAddressMemoryEnabled() const NN_NOEXCEPT
{
    return false;
}

#endif

}   // namespace detail
}}  // namespace nn::os
