﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_ThreadLocalStorageCommon.h>
#include <nn/os/os_SdkThreadTypes.h>
#include <nn/os/detail/os_InternalCriticalSection.h>


namespace nn { namespace os {
namespace detail {

//---------------------------------------------------------------------------
// 各 TLS のスロットカウント等を返す
//
const int TotalTlsSlotCountMax  = TlsSlotCountMax + SdkTlsSlotCountMax;
const int SdkInternalTlsCount   = sizeof(SdkInternalTlsType) / sizeof(uintptr_t);

//---------------------------------------------------------------------------
//  TlsManager クラスです
//

class TlsManager
{
private:
    // コピー、ムーブを禁止する
    NN_DISALLOW_COPY( TlsManager );
    NN_DISALLOW_MOVE( TlsManager );

public:
    // コンストラクタ
    TlsManager()        NN_NOEXCEPT;

    // 全ての TLS デストラクタを順番に呼出す
    void    InvokeTlsDestructors()  NN_NOEXCEPT;

    // TLS スロットを確保し、デストラクタを登録する
    bool    AllocateTlsSlot(TlsSlot* outTlsSlot, TlsDestructor pDestructor, bool sdkInternal)  NN_NOEXCEPT;

    // TLS スロットを返却する
    void    FreeTlsSlot(TlsSlot tlsSlot)    NN_NOEXCEPT;

    // TLS スロット数を返す
    int     GetMaxTlsSlots()  const NN_NOEXCEPT
    {
        return TlsSlotCountMax;
    }
    int     GetUsedTlsSlots() const NN_NOEXCEPT
    {
        return m_NumUsedTlsSlots;
    }

private:
    int     SearchUnusedTlsSlotUnsafe(bool sdkInternal) const NN_NOEXCEPT;

private:
    // TLS 管理用
    int                     m_NumUsedTlsSlots;  // ユーザーが使用中のスロット数（SDK が使用中のものはカウントしない）
    TlsDestructor           m_TlsDestructorArray[ TotalTlsSlotCountMax ];
    InternalCriticalSection m_TlsCriticalSection;
};

//---------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::os

