﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Config.h>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Tick.h>
#include "os_TickManager.h"
#include <algorithm>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include "os_TimeoutHelper-os.win32.h"
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "os_TimeoutHelper-os.horizon.h"
#else
    #error   "未サポートの OS 種別が指定されています。"
#endif


//--------------------------------------------------------------------------
//  C++ 向けのプロトタイプ宣言
//--------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
//  TimeoutHelper クラス
//
class TimeoutHelper
{
public:
    //--------------------------------------------------
    // コンストラクタ
    explicit TimeoutHelper(TimeSpan timeout) NN_NOEXCEPT
    {
        // タイムアウトが 0 の場合はすぐに返る
        if (timeout == 0)
        {
            m_absoluteEndTick = Tick ( 0 );
            return;
        }

        // タイムアウトが成立する絶対時間を計算しておく
        // 指定時間よりも早くタイムアウトすることを防ぐために +1 tick しておく
        uint64_t currTick        = detail::GetTickManagerInstance()->GetTick().GetInt64Value();
        uint64_t timeoutTick     = detail::GetTickManagerInstance()->ConvertToTick(timeout).GetInt64Value();
        uint64_t absoluteEndTick = currTick + timeoutTick + 1;

        // オーバーフローしないよう調整
        m_absoluteEndTick = Tick(std::min<uint64_t>((std::numeric_limits<int64_t>::max)(), absoluteEndTick));
    }

    //--------------------------------------------------
    // 単純スリープ
    static void Sleep(TimeSpan time) NN_NOEXCEPT
    {
        TimeoutHelperImpl::Sleep( time );
    }

    //--------------------------------------------------
    // タイムアウト成立か否かを返す
    bool    IsTimedout()   const NN_NOEXCEPT
    {
        // m_absoluteEndTick==0 なら必ずタイムアウト成立
        if (m_absoluteEndTick.GetInt64Value() == 0)
        {
            return true;
        }

        // 現在時間を取得
        Tick  currTick = GetTickManagerInstance()->GetTick();

        // タイムアウト成立かどうかを確認
        return  currTick >= m_absoluteEndTick;
    }

    //--------------------------------------------------
    // タイムアウト成立までの残り時間をターゲット依存の時間型で返す
    // 既にタイムアウトが成立している場合は 0 を返す
    InternalTargetTimeSpan  GetLeftTimeOnTarget()   const NN_NOEXCEPT;

private:
    Tick                m_absoluteEndTick;
};

}   // namespace detail
}}  // namespace nn::os

