﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Config.h>

#include <cstdlib>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/util/util_IntrusiveList.h>

#include <nn/os/os_Types.h>
#include <nn/os/os_ThreadCommon.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/detail/os_InternalCriticalSection.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include "os_ThreadManager-os.win32.h"
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "os_ThreadManager-os.horizon.h"
#else
    #error   "未サポートの OS 種別が指定されています。"
#endif


namespace nn { namespace os {
namespace detail {

//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//  ThreadManager クラスの定義
//

class ThreadManager
{
    // AllThreadsList 専用の NodeTraits の定義
    class ThreadListNodeTraits
    {
    private:
        friend class util::IntrusiveList<ThreadType, ThreadListNodeTraits>;

        static util::IntrusiveListNode& GetNode(ThreadType& ref) NN_NOEXCEPT
        {
            return Get(ref._allThreadsListNode);
        }

        static const util::IntrusiveListNode& GetNode(const ThreadType& ref) NN_NOEXCEPT
        {
            return Get(ref._allThreadsListNode);
        }

        static ThreadType& GetItem(util::IntrusiveListNode& node) NN_NOEXCEPT
        {
            return *reinterpret_cast<ThreadType*>(reinterpret_cast<char*>(&node) - offsetof(ThreadType, _allThreadsListNode));
        }

        static const ThreadType& GetItem(const util::IntrusiveListNode& node) NN_NOEXCEPT
        {
            return *reinterpret_cast<const ThreadType*>(reinterpret_cast<const char*>(&node) - offsetof(ThreadType, _allThreadsListNode));
        }
    };

    typedef util::IntrusiveList<ThreadType, ThreadListNodeTraits>   AllThreadsList;

    // コピー、ムーブを禁止する
    NN_DISALLOW_COPY( ThreadManager );
    NN_DISALLOW_MOVE( ThreadManager );

public:
    // コンストラクタ
    ThreadManager()     NN_NOEXCEPT;

    // スレッド終了時の後処理
    void    CleanupThread() NN_NOEXCEPT;

    // Thread 数を返す（メインスレッドを含む）
    int     GetThreadCountForDebug()    const NN_NOEXCEPT
    {
        return m_numCreatedThreads;
    }


    // Thread 生成（idealCoreNumber あり）
    nn::Result  CreateThread(ThreadType* thread, ThreadFunction function, void* argument, void* stack, size_t stackSize, int priority, int idealCoreNumber) NN_NOEXCEPT;

    // Thread 生成（idealCoreNumber なし）
    nn::Result  CreateThread(ThreadType* thread, ThreadFunction function, void* argument, void* stack, size_t stackSize, int priority) NN_NOEXCEPT;

    // Thread 削除
    void    DestroyThread(ThreadType* thread)  NN_NOEXCEPT;

    // Thread 開始
    void    StartThread(ThreadType *thread) NN_NOEXCEPT;

    // Thread 終了を待機
    void    WaitThread(ThreadType *thread) NN_NOEXCEPT;

    // Thread の Yield
    void    YieldThread() NN_NOEXCEPT
    {
        m_impl.YieldThread();
    }

    // Thread の優先度変更
    bool    ChangePriority(ThreadType *thread, int priority)    NN_NOEXCEPT
    {
        return m_impl.ChangePriority(thread, priority);
    }

    // Thread の現在優先度を返す
    int     GetCurrentPriority(const ThreadType *thread) const NN_NOEXCEPT
    {
        return m_impl.GetCurrentPriority(thread);
    }

    // 現在の ThreadType オブジェクトを取得して返す
    ThreadType* GetCurrentThread() const NN_NOEXCEPT
    {
        return  m_impl.GetCurrentThread();
    }

    // スレッドの Suspend/Resume 関連
    int     SuspendThread(ThreadType* thread) NN_NOEXCEPT;
    int     ResumeThread(ThreadType* thread) NN_NOEXCEPT;
    void    GetThreadContext(ThreadContextInfo* context, const ThreadType* thread) NN_NOEXCEPT;


public:
    // Thread に初期スレッド名をセットする
    void    SetInitialThreadNameUnsafe(ThreadType* thread) NN_NOEXCEPT;

    // Thread 名を下位の OS レイヤーに登録する
    void    NotifyThreadNameChanged(const ThreadType* thread) const NN_NOEXCEPT
    {
        m_impl.NotifyThreadNameChangedImpl(thread);
    }

    // 現在の ThreadType オブジェクトを TLS に設定する
    void    SetCurrentThread(const ThreadType* thread) const NN_NOEXCEPT
    {
        m_impl.SetCurrentThread( thread );
    }

    //  自スレッドが動作しているプロセッサ番号の取得
    int     GetCurrentCoreNumber() const NN_NOEXCEPT
    {
        return m_impl.GetCurrentCoreNumber();
    }

    // コアアフィニティの設定
    void    SetThreadCoreMask(ThreadType* thread, int idealCore, Bit64 affinityMask) const NN_NOEXCEPT
    {
        m_impl.SetThreadCoreMask(thread, idealCore, affinityMask);
    }

    // コアアフィニティの取得
    void    GetThreadCoreMask(int* pOutIdealCore, Bit64* pOutAffinityMask, const ThreadType* thread) const NN_NOEXCEPT
    {
        m_impl.GetThreadCoreMask(pOutIdealCore, pOutAffinityMask, thread);
    }

    // 割り当て可能なコアの取得
    nn::Bit64    GetThreadAvailableCoreMask() const NN_NOEXCEPT
    {
        return m_impl.GetThreadAvailableCoreMask();
    }

    //  全スレッドの TLS 値をゼロにする
    void SetZeroToAllThreadsTlsSafe(int slotNum) NN_NOEXCEPT;

    //  全スレッドリストへ Thread をリンクし、スレッド数をインクリメント
    void PushBackToAllThreadsListUnsafe(ThreadType* thread) NN_NOEXCEPT
    {
        m_AllThreadsList.push_back( *thread );
        ++m_numCreatedThreads;
        m_totalThreadStackSize += thread->_stackSize;
    }

    //  全スレッドリストから Thread をアンリンクし、スレッド数をデクリメント
    void EraseFromAllThreadsListUnsafe(ThreadType* thread) NN_NOEXCEPT
    {
        m_AllThreadsList.erase( m_AllThreadsList.iterator_to( *thread ) );
        --m_numCreatedThreads;
        m_totalThreadStackSize -= thread->_stackSize;
    }

    //  全スレッドリスト操作＋スレッド数制御 のスレッドセーフ版
    void PushBackToAllThreadsListSafe(ThreadType* thread) NN_NOEXCEPT
    {
        std::lock_guard<InternalCriticalSection> lock( m_criticalSection );
        PushBackToAllThreadsListUnsafe(thread);
    }
    void EraseFromAllThreadsListSafe(ThreadType* thread) NN_NOEXCEPT
    {
        std::lock_guard<InternalCriticalSection> lock( m_criticalSection );
        EraseFromAllThreadsListUnsafe(thread);
    }

    // Thread オブジェクトの初期スレッド名を設定し ThreadManager の管理下におく
    void PlaceThreadObjectUnderThreadManagerSafe(ThreadType* thread) NN_NOEXCEPT
    {
        SetInitialThreadNameUnsafe( thread );
        {
            std::lock_guard<InternalCriticalSection> lock( m_criticalSection );
            PushBackToAllThreadsListUnsafe( thread );
        }
    }

    //  ThreadType オブジェクトの確保
    ThreadType* AllocateThreadType() const NN_NOEXCEPT
    {
        void* p = std::malloc( sizeof(ThreadType) );
        return reinterpret_cast<ThreadType*>( p );
    }

    //  ThreadType オブジェクトの開放
    void FreeThreadType(ThreadType* p) const NN_NOEXCEPT
    {
        free( p );
    }

    //  m_mainThread を返す
    const ThreadType* GetMainThread() const NN_NOEXCEPT
    {
        return &m_mainThread;
    }

    // スレッドの総スタックサイズ
    size_t GetTotalThreadStackSize() const NN_NOEXCEPT
    {
        return m_totalThreadStackSize;
    }

    // 自プロセスの終了
    NN_NORETURN void QuickExit() NN_NOEXCEPT
    {
        m_impl.QuickExit();
    }

    // RYNDA 向け
    NN_NORETURN void ExitProcess() NN_NOEXCEPT
    {
        m_impl.ExitProcessImpl();
    }

    // スレッドの ID を取得
    ThreadId GetThreadId(const ThreadType* thread) NN_NOEXCEPT
    {
        return m_impl.GetThreadId(thread);
    }

private:
    bool CreateAliasStackUnsafe(ThreadType* thread) NN_NOEXCEPT;
    void DeleteAliasStackUnsafe(ThreadType* thread) NN_NOEXCEPT;

private:
    ThreadManagerImpl       m_impl;
    ThreadType              m_mainThread;           // MainThread オブジェクト
    InternalCriticalSection m_criticalSection;      // OS ライブラリ内排他用
    AllThreadsList          m_AllThreadsList;       // 全スレッドを繋ぐリスト
    size_t                  m_totalThreadStackSize; // Thread の総スタックサイズ
    int                     m_numCreatedThreads;    // 作成されたスレッド数
};

//---------------------------------------------------------------------------

}  // namespace detail
}} // namespace nn::os

