﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/util/util_TypedStorage.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_Utility.h"
#include "os_ThreadManager.h"
#include "os_TlsManager.h"
#include "os_MultipleWaitHelper.h"
#include "os_StackGuardManager.h"

#include <nn/os/detail/os_InternalCriticalSection.h>
#include <nn/os/os_ThreadApi.private.h>

#include <cstdint>
#include <new>
#include <mutex>
#include <cstring>

//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace os {

//--------------------------------------------------------------------------

namespace detail {

    //-----------------------------------------------------------------------
    // ThreadType オブジェクトのセットアップ（スレッドの生成は行なわない）
    void SetupThreadObjectUnsafe(ThreadType* thread, ThreadFunction function, void* argument, void* stack, size_t stackSize, int priority) NN_NOEXCEPT
    {
        // 排他リソースの初期化
        new( &thread->_csThread ) InternalCriticalSection;
        new( &thread->_cvThread ) InternalConditionVariable;

        // IntrusiveList のコンストラクタを呼ぶ
        new( &thread->_multiWaitObjectList ) MultiWaitObjectList;
        new( &thread->_allThreadsListNode  ) util::IntrusiveListNode;

        // メンバの初期化
        thread->_threadFunction  = function;
        thread->_argument        = argument;
        thread->_stackIsAliased  = false;
        thread->_originalStack   = stack;
        thread->_stack           = stack;
        thread->_stackSize       = stackSize;
        thread->_basePriority    = priority;
        thread->_lockHistory     = 0;
        thread->_autoRegistered  = false;
        thread->_suspendCount    = 0;
        thread->_initialFiber    = NULL;
        thread->_currentFiber    = NULL;
        thread->_threadNameBuffer[0]    = '\0';
        thread->_namePointer            = thread->_threadNameBuffer;
        std::memset(thread->_tlsValueArray, 0, sizeof(thread->_tlsValueArray));

        // ThreadType 状態の設定
        thread->_state           = ThreadType::State_Initialized;
    }

namespace {

    //-----------------------------------------------------------------------
    // スレッド関数の入口／出口
    // 引数 thread には自スレッドの ThreadType オブジェクトへのポインタが渡される
    void InvokeThreadByThreadManager(ThreadType *thread) NN_NOEXCEPT
    {
        ThreadManager*  threadManager = GetThreadManagerInstance();

        // 自スレッド用の TLS にオブジェクトポインタを格納しておく。
        // この情報は、自スレッドからしか参照できない。
        threadManager->SetCurrentThread( thread );

        // デバッガ等に対して新しいスレッド名を通知
        threadManager->NotifyThreadNameChanged( thread );

        // ここで自スレッドの state 変数で条件変数待機する
        {   // クリティカルセクション
            InternalCriticalSection&    cs = Get(thread->_csThread);
            std::unique_lock<InternalCriticalSection>   lock( cs );

            // スレッド状態がまだ Initialize なら待機
            while (thread->_state == ThreadType::State_Initialized)
            {
                Get(thread->_cvThread).Wait( &cs );
            }

            // StartThread されていれば、スレッド関数を呼出す
            // DestroyedBeforeStarted の場合はスレッド関数を呼ばずに終了処理へ
            if (thread->_state == ThreadType::State_Started)
            {
                // この場合だけロックを解除してからスレッド関数を呼ぶ
                lock.unlock();

                // スレッド関数エントリを呼ぶ
                thread->_threadFunction( thread->_argument );

                // この後、アンロック状態のままロック区間を抜ける
            }
        }

        // スレッドの後処理
        threadManager->CleanupThread();
    }

}   // namespace


//----------------------------------------------------------------------
//  スレッド終了時の後処理
void ThreadManager::CleanupThread() NN_NOEXCEPT
{
    // 登録されている TLS デストラクタを順に呼出す
    GetTlsManagerInstance()->InvokeTlsDestructors();

    // TLS から自スレッドの ThreadType オブジェクトを取り出す
    ThreadType* thread = GetCurrentThread();

    {   // クリティカルセクション
        InternalCriticalSection&    cs = Get(thread->_csThread);
        std::lock_guard<InternalCriticalSection>    lock( cs );

        // スレッド状態を更新
        thread->_state  = ThreadType::State_Exited;

        // 終了待ちの全スレッドに cvThread 条件変数を通知
        Get(thread->_cvThread).Broadcast();

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(thread->_multiWaitObjectList).WakeupAllMultiWaitThreadsUnsafe();
    }
}

//-------------------------------------------------------------------------
//  既存のスタック領域を元領域として、スタックのエイリアスを構築する。
//  エイリアスされたスタックの前後にはガード用のページが存在する。
bool ThreadManager::CreateAliasStackUnsafe( ThreadType* thread ) NN_NOEXCEPT
{
    // TORIAEZU: 64 回までリトライする
    for (int i=0; i<64; ++i)
    {
        // スタックガード用の空間から空き空間を獲得する。
        // 第１引数にはガード領域を含まないスタックサイズを渡し、
        // 返値にはスタックをマップすべきガードを含まない先頭アドレスが返る。
        void* newStack = GetStackGuardManagerInstance()->AllocateStackGuardSpace( thread->_stackSize );
        if ( !newStack )
        {
            // ここで取れない場合は即失敗
            NN_ABORT(NN_TEXT_OS("nn::os::CreateThread(): スタックガード用の空間を確保できませんでした。"));
            return false;
        }

        // スタックのエイリアスを構築する
        auto isMapSuccess = m_impl.MapAliasStack( newStack, thread->_stack, thread->_stackSize );
        if ( isMapSuccess )
        {
            // 前後にガード空間があるかチェック
            if (!detail::GetStackGuardManagerInstance()->CheckGuardSpace(reinterpret_cast<uintptr_t>(newStack), thread->_stackSize))
            {
                // なかった場合、アンマップしてリトライ
                bool ret = m_impl.UnmapAliasStack(newStack, thread->_stack, thread->_stackSize);
                NN_SDK_ASSERT( ret, NN_TEXT_OS("nn::os::CreateThread(): スタック領域のエイリアス解除に失敗しました。"));
                NN_UNUSED(ret);
            }
            else
            {
                // エイリアスされたスタック情報を設定
                thread->_stackIsAliased  = true;
                thread->_stack           = newStack;
                return true;
            }
        }
        else
        {
            // 既に使用済みの領域だった場合リトライする
        }
    }
    return false;
}

//-------------------------------------------------------------------------
//  エイリアススタックを解除する。
void ThreadManager::DeleteAliasStackUnsafe( ThreadType* thread ) NN_NOEXCEPT
{
    // SVC を発行して、スタックのエイリアスを解除する
    bool ret = m_impl.UnmapAliasStack(thread->_stack, thread->_originalStack, thread->_stackSize);
    NN_SDK_ASSERT( ret, NN_TEXT_OS("nn::os::DestroyThread(): スタック領域のエイリアス解除に失敗しました。"));
    NN_UNUSED( ret );

    // エイリアス解除されたスタック情報を設定
    thread->_stackIsAliased  = false;
    thread->_stack           = thread->_originalStack;
}


//---------------------------------------------------------------------------
// Thread の生成（idealCoreNumber あり）
Result  ThreadManager::CreateThread(ThreadType* thread, ThreadFunction function, void* argument, void* stack, size_t stackSize, int priority, int idealCoreNumber) NN_NOEXCEPT
{
    // ThreadType オブジェクトのセットアップ
    SetupThreadObjectUnsafe(thread, function, argument, stack, stackSize, priority);

    // スタック領域をエイリアスする
    {
        auto ret = CreateAliasStackUnsafe( thread );

        NN_ABORT_UNLESS( ret, NN_TEXT_OS("nn::os::CreateThread() スタック領域のエイリアス構築に失敗しました。"));
    }

    // スレッドの生成
    // スレッドは一旦動き出すが InvokeThreadByThreadManager() の中で停止する
    auto result = m_impl.CreateThread(thread, &InvokeThreadByThreadManager, idealCoreNumber);
    if ( !result.IsSuccess() )
    {
        // 巻戻し
        DeleteAliasStackUnsafe( thread );
        thread->_state = ThreadType::State_NotInitialized;

        return result;
    }

    // ThreadType オブジェクトを ThreadManager の管理下に置く
    PlaceThreadObjectUnderThreadManagerSafe( thread );

    NN_RESULT_SUCCESS;
}


//---------------------------------------------------------------------------
// Thread の生成（idealCoreNumber なし）
Result  ThreadManager::CreateThread(ThreadType* thread, ThreadFunction function, void* argument, void* stack, size_t stackSize, int priority) NN_NOEXCEPT
{
    int idealCoreNumber = m_impl.GetDefaultCoreNumber();
    return CreateThread(thread, function, argument, stack, stackSize, priority, idealCoreNumber);
}


//---------------------------------------------------------------------------
// Thread 開始
void ThreadManager::StartThread(ThreadType *thread) NN_NOEXCEPT
{
    {   // クリティカルセクション
        InternalCriticalSection&    cs = Get(thread->_csThread);
        std::lock_guard<InternalCriticalSection>    lock( cs );

        // スレッドの状態が変化していないことの確認（必須）
        NN_SDK_REQUIRES( thread->_state == ThreadType::State_Initialized, NN_TEXT_OS("nn::os::StartThread(): スレッドの状態が実行開始可能な状態ではありません。") );

        // 対象スレッドを実行開始
        m_impl.StartThread(thread);
        thread->_state = ThreadType::State_Started;

        // 対象スレッドへ条件変数通知
        Get(thread->_cvThread).Signal();
    }
}


//---------------------------------------------------------------------------
// Thread 終了を待機
void ThreadManager::WaitThread(ThreadType *thread) NN_NOEXCEPT
{
    // ThreadType::State の値だけでは、スレッドの実コンテキストが本当に終了したのか
    // 判別ができないので、結局実装依存のスレッド終了待ちを呼ぶ。
    m_impl.WaitForExitThread( thread );

    {   // スレッドのクリティカルセクション
        InternalCriticalSection&    cs = Get(thread->_csThread);
        std::lock_guard<InternalCriticalSection>    lock( cs );

        if ( thread->_stackIsAliased )
        {
            // スタック領域のエイリアスを解除する
            DeleteAliasStackUnsafe( thread );
        }
    }
}


//---------------------------------------------------------------------------
// Thread 削除
void ThreadManager::DestroyThread(ThreadType* thread) NN_NOEXCEPT
{
    {   // スレッドのクリティカルセクション
        InternalCriticalSection&    cs = Get(thread->_csThread);
        std::lock_guard<InternalCriticalSection>    lock( cs );

        // スレッドが Initialized 状態の場合
        if (thread->_state == ThreadType::State_Initialized)
        {
            // StartThread されてないスレッドを DestroyThread した場合の処理。
            // スレッド状態を DestroyedBeforeStarted に変更してから
            // スレッドの実行を開始。対象スレッドは起動してすぐ終了処理へ移行
            thread->_state = ThreadType::State_DestroyedBeforeStarted;
            m_impl.StartThread(thread);

            // 対象スレッドへ条件変数通知
            Get(thread->_cvThread).Signal();
        }
    }

    // Thread の State が Initialized および Started の場合は、ここで Exited に
    // なるまで待機する。すでに Exited の場合でも、スレッドの実コンテキストが
    // 本当に終了したかを保障するためには、結局ここでスレッド終了待ちをする
    // 必要がある。
    m_impl.WaitForExitThread( thread );

    {   // スレッドのクリティカルセクション
        InternalCriticalSection&    cs = Get(thread->_csThread);
        std::lock_guard<InternalCriticalSection>    lock( cs );

        // 通常はこの段階で thread->_state == ThreadType::State_Exited だが、
        // 強制終了などでスレッドが強制削除されている場合、対象スレッドの
        // オブジェクト状態は State_Initialized か State_Started 状態のまま
        // となる。ただし、そのような場合でも以降の処理を行なって問題ない。

        // スタック領域がエイリアスされていれば解除する
        if ( thread->_stackIsAliased )
        {
            DeleteAliasStackUnsafe( thread );
        }

        // スレッドの実コンテキストの削除処理
        // もし複数のスレッドから DestroyThread() が呼ばれているような場合、
        // この中の Assert で失敗する。
        m_impl.DestroyThreadUnsafe( thread );

        // ThreadType の状態を NotInitialized にする
        thread->_state    = ThreadType::State_NotInitialized;

        // IntrusiveList のデストラクタを呼ぶ
        Get(thread->_multiWaitObjectList).~MultiWaitObjectList();

        // Thread 名を初期化
        thread->_threadNameBuffer[0]  = '\0';

        {   // ThreadManager のクリティカルセクション
            std::lock_guard<InternalCriticalSection> lock2( m_criticalSection );

            // 全スレッドを繋ぐリストから削除
            EraseFromAllThreadsListUnsafe( thread );
        }
    }

    // 排他リソースの破棄
    Get(thread->_csThread).~InternalCriticalSection();
    Get(thread->_cvThread).~InternalConditionVariable();
}


//---------------------------------------------------------------------------
// スレッドの Suspend/Resume 関連
//
int ThreadManager::SuspendThread(ThreadType* thread) NN_NOEXCEPT
{
    std::lock_guard<InternalCriticalSection> lock(Get(thread->_csThread));

    auto prevSuspendCount = thread->_suspendCount;
    NN_SDK_REQUIRES( prevSuspendCount < ThreadSuspendCountMax );
    thread->_suspendCount = prevSuspendCount + 1;

    if (prevSuspendCount == 0)
    {
        m_impl.SuspendThreadUnsafe(thread);
    }
    return prevSuspendCount;
}

int ThreadManager::ResumeThread(ThreadType* thread) NN_NOEXCEPT
{
    std::lock_guard<InternalCriticalSection> lock(Get(thread->_csThread));

    auto prevSuspendCount = thread->_suspendCount;
    if (prevSuspendCount > 0)
    {
        thread->_suspendCount = prevSuspendCount - 1;
        if (prevSuspendCount == 1)
        {
            m_impl.ResumeThreadUnsafe(thread);
        }
    }
    return prevSuspendCount;
}

void ThreadManager::GetThreadContext(ThreadContextInfo* context, const ThreadType* thread) NN_NOEXCEPT
{
    std::lock_guard<InternalCriticalSection> lock(Get(thread->_csThread));
    NN_SDK_REQUIRES( thread->_suspendCount > 0 );

    m_impl.GetThreadContextUnsafe(context, thread);
}

//---------------------------------------------------------------------------
// 全スレッドの当該 TLS スロットの TLS 値をゼロに設定
void ThreadManager::SetZeroToAllThreadsTlsSafe(int slotNum) NN_NOEXCEPT
{
    std::lock_guard<InternalCriticalSection> lock( m_criticalSection );

    for ( const ThreadType& thread : m_AllThreadsList )
    {
        thread._tlsValueArray[ slotNum ] = 0;
    }
}

//---------------------------------------------------------------------------
// 指定されたスレッドに初期スレッド名を設定する
#define NN_OS_DETAIL_MAIN_THREAD_NAME   "MainThread"
#define NN_OS_DETAIL_THREAD_NAME_PREFIX "Thread_0x"

void ThreadManager::SetInitialThreadNameUnsafe(ThreadType* thread) NN_NOEXCEPT
{
    if (thread == &m_mainThread)
    {
        // メインスレッドだけ専用のデフォルトスレッド名を設定
        static_assert( sizeof(thread->_threadNameBuffer) >= sizeof(NN_OS_DETAIL_MAIN_THREAD_NAME),
            "実装エラー：thread->_threadNameBuffer[] のサイズが小さすぎます）");
        std::memcpy(thread->_threadNameBuffer, NN_OS_DETAIL_MAIN_THREAD_NAME, sizeof(NN_OS_DETAIL_MAIN_THREAD_NAME));
    }
    else
    {
        const size_t prefixSize = sizeof(NN_OS_DETAIL_THREAD_NAME_PREFIX) - 1;
        uint64_t function = reinterpret_cast<uint64_t>(thread->_threadFunction);

        static_assert((prefixSize + sizeof(function) * 2 + 1) <= sizeof(thread->_threadNameBuffer),
            "内部エラー：初期スレッド名の文字数が長すぎます。");

        std::memcpy(thread->_threadNameBuffer, NN_OS_DETAIL_THREAD_NAME_PREFIX, prefixSize);
        char* p = thread->_threadNameBuffer + prefixSize;
        ExpandUnsignedValueToAscii(p, function, 64);
    }

    thread->_namePointer = thread->_threadNameBuffer;
}

//---------------------------------------------------------------------------
// ThreadManager のコンストラクタ
ThreadManager::ThreadManager() NN_NOEXCEPT
    : m_impl( &m_mainThread ),
      m_totalThreadStackSize( 0 ),
      m_numCreatedThreads( 0 )
{
    // スレッドの状態を Started に
    m_mainThread._state = ThreadType::State_Started;

    // メインスレッドをカレントスレッドとして設定
    SetCurrentThread( &m_mainThread );

    // メインスレッドの ThreadType オブジェクトをセットアップ
    PlaceThreadObjectUnderThreadManagerSafe( &m_mainThread );
}


}  // namespace detail
}} // namespace nn::os

