﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include "os_Diag.h"
#include "os_Common.h"
#include "os_ThreadManager.h"
#include "os_ThreadPointerInterface-os.horizon.h"
#include <nn/nn_BitTypes.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Thread.h>
#include <nn/svc/svc_Handle.h>
#include <nn/svc/svc_Synchronization.h>
#include <nn/svc/svc_Result.h>

#include <cstring>

namespace nn { namespace os { namespace detail {

//--------------------------------------------------------------------------
//  プロセスの優先度定数の定義が正しいかの静的チェック

static_assert( UserThreadPriorityOffset >= 0 &&
               UserThreadPriorityOffset < NativeThreadPriorityRangeSize &&
               (ReservedThreadPriorityRangeSize + SystemThreadPriorityRangeSize)
                                           == NativeThreadPriorityRangeSize &&
               (LowestSystemThreadPriority - HighestSystemThreadPriority + 1)
                                           == SystemThreadPriorityRangeSize &&
               HighestSystemThreadPriority <= HighestThreadPriority &&
                      LowestThreadPriority <= LowestSystemThreadPriority,
               "システムプロセス用のスレッド優先度の設定値が異常です。");

//---------------------------------------------------------------------------
// コンストラクタ呼出し前に nnosInitialize() の中で設定される
uintptr_t   g_OsBootParamter;

//---------------------------------------------------------------------------

namespace {

inline void SetThreadPointerForCompilerTls() NN_NOEXCEPT
{
    uintptr_t tp = nn::svc::GetThreadLocalRegion()->thread_pointer;
#if defined NN_BUILD_CONFIG_CPU_ARM64
    asm volatile(" msr tpidr_el0, %0" :: "r"(tp) : "memory");
#else
    asm volatile(" mcr p15, 0, %0, c13, c0, 2" :: "r"(tp) : "memory");
#endif
}

// Siglo ⇔ Horizon での優先度の変換
int32_t ConvertToHorizonPriority(int sigloPriority)
{
    int32_t horizonPriority = sigloPriority + UserThreadPriorityOffset;

    NN_SDK_REQUIRES((horizonPriority >= HighestNativeThreadPriority) &&
                    (horizonPriority <= LowestNativeThreadPriority) ,
        NN_TEXT_OS("指定されたスレッド優先度が規定の範囲内ではありません(priority=%d)"), sigloPriority);

    return horizonPriority;
}

int ConvertToSigloPriority(int horizonPriority)
{
    NN_SDK_REQUIRES((horizonPriority >= HighestNativeThreadPriority) &&
                    (horizonPriority <= LowestNativeThreadPriority) ,
        NN_TEXT_OS("内部エラー：対象スレッドのカーネル優先度が異常です(kernel priority=%d)"), horizonPriority);

    return horizonPriority - UserThreadPriorityOffset;
}

struct Arglist
{
    void (*function)(ThreadType*);
};
union ArglistU
{
    Arglist arg;
    std::max_align_t dummy;
};

void InvokeThread(uintptr_t p) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_CPU_ARM) || defined(NN_BUILD_CONFIG_CPU_ARM64)
    // 第 1 引数 p は r0 レジスタ（AArch64 では x0）にアサインされており、
    // そこに自スレッドの ThreadType オブジェクト先頭アドレスが格納されている。
    // この値は専用の TLS にセットされるまで r0 に保持し続けなければならない。
    // ※）TMA などのデバッグモニタが TLS から ThreadType* のアドレスを
    //     取得するが、これが NULL の場合は r0 の内容を利用するため。
    {
#if defined NN_BUILD_CONFIG_CPU_ARM64
        register uintptr_t               tp  asm("x0") = p;
        register svc::ThreadLocalRegion* tlr asm("x1");

        asm volatile(" mrs %0, tpidrro_el0;"
                     " add %0, %0, %2;"
                     " str %1, [%0];"
                     " dsb sy"
                        : "=&r"(tlr) : "r"(tp), "i"(offsetof(svc::ThreadLocalRegion, pThreadType)) : "memory");
#else
        register uintptr_t               tp  asm("r0") = p;
        register svc::ThreadLocalRegion* tlr asm("r1");

        asm volatile(" mrc p15, 0, %0, c13, c0, 3;"
                     " add %0, %0, %2;"
                     " str %1, [%0];"
                     " dsb sy"
                        : "=&r"(tlr) : "r"(tp), "i"(offsetof(svc::ThreadLocalRegion, pThreadType)) : "memory");
#endif
    }
#endif
    if (nnmuslThreadPointerInitializer)
    {
        (*nnmuslThreadPointerInitializer)();
    }
    SetThreadPointerForCompilerTls();

    ThreadType* thread  = reinterpret_cast<ThreadType*>(p);

    // ThreadId をキャッシュしておく
    Bit64 threadId;
    NN_ABORT_UNLESS_RESULT_SUCCESS( svc::GetThreadId(&threadId, svc::Handle(thread->_handle)) );
    svc::GetThreadLocalRegion()->threadId = threadId;

    // 親スレッドから継承されてきた fpcr を設定
    Arglist* args = &(reinterpret_cast<ArglistU*>(reinterpret_cast<uintptr_t>(thread->_stack) + thread->_stackSize) - 1)->arg;

    // ここの function はユーザ関数エントリではなく、
    // ThreadManager が用意したスレッド関数ラッパーエントリを指す。
    args->function( thread );
    if (nnmuslThreadPointerDestroyer)
    {
        (*nnmuslThreadPointerDestroyer)();
    }
    nn::svc::ExitThread();
}

// CreateThread() 時のエイリアス元スタック領域のメモリ状態のチェック
void CheckDetailsForInvalidCurrentMemory( uintptr_t stack, size_t stackSize ) NN_NOEXCEPT
{
    bool isAlive = true;
    svc::MemoryInfo memoryInfo;
    svc::PageInfo pageInfo;

    for ( uintptr_t p = stack; p < stack + stackSize; p = memoryInfo.baseAddress + memoryInfo.size )
    {
        {
            auto result = svc::QueryMemory( &memoryInfo, &pageInfo, p );
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }

        if ( memoryInfo.state != svc::MemoryState_Normal &&
             memoryInfo.state != svc::MemoryState_CodeData &&
             memoryInfo.state != svc::MemoryState_AliasCodeData )
        {
            isAlive = false;

            NN_SDK_LOG( NN_TEXT_OS("nn::os::CreateThread() に指定されたスタック領域のメモリ状態が事前条件違反です。(Addr=0x%p, Size=0x%016llx, Stat=0x%x)\n"), memoryInfo.baseAddress, memoryInfo.size, memoryInfo.state );

#if defined(NN_SDK_BUILD_RELEASE)
            break;
#endif // defined(NN_SDK_BUILD_RELEASE)
        }

        if ( memoryInfo.attribute != 0 )
        {
            isAlive = false;

            NN_SDK_LOG( NN_TEXT_OS("nn::os::CreateThread() に指定されたスタック領域のメモリ属性が事前条件違反です。(Addr=0x%p, Size=0x%016llx, Attr=0x%x)\n"), memoryInfo.baseAddress, memoryInfo.size, memoryInfo.attribute );

#if defined(NN_SDK_BUILD_RELEASE)
            break;
#endif // defined(NN_SDK_BUILD_RELEASE)
        }

        if ( memoryInfo.permission != svc::MemoryPermission_ReadWrite )
        {
            isAlive = false;

            NN_SDK_LOG( NN_TEXT_OS("nn::os::CreateThread() に指定されたスタック領域のアクセス権が事前条件違反です。(Addr=0x%p, Size=0x%016llx, Perm=0x%x)\n"), memoryInfo.baseAddress, memoryInfo.size, memoryInfo.permission );

#if defined(NN_SDK_BUILD_RELEASE)
            break;
#endif // defined(NN_SDK_BUILD_RELEASE)
        }
    }

    NN_ABORT_UNLESS( isAlive, "" );
}

}   // namespace


//---------------------------------------------------------------------------
//  コンストラクタ
//  メインスレッドの ThreadType オブジェクトもここで中身を埋める
ThreadManagerImplByHorizon::ThreadManagerImplByHorizon(ThreadType* mainThread) NN_NOEXCEPT
{
    // スタック情報を直接参照できないので、ガードを検出する。
    nn::svc::MemoryInfo  mi;
    nn::svc::PageInfo pi;
    auto result = nn::svc::QueryMemory(&mi, &pi, reinterpret_cast<uintptr_t>(__builtin_frame_address(0)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::svc::Handle threadHandle( g_OsBootParamter );
    int32_t priority;
    result = nn::svc::GetThreadPriority(&priority, threadHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // TORIAEZU:
    // 第２引数にはメインスレッドの開始エントリアドレスを指定する。
    // _start が妥当だが、crt0 への参照を断ち切るため一旦 NULL にしておく。
    // 将来的に正しいエントリアドレスを代入したい場合は、_start のアドレスを
    // スタック経由でもらうなどの方法で実装してもよい。
    SetupThreadObjectUnsafe(
            mainThread,
            reinterpret_cast<ThreadFunction>(NULL), NULL,
            reinterpret_cast<void*>(mi.baseAddress), mi.size,
            ConvertToSigloPriority(priority));

    mainThread->_handle = g_OsBootParamter;

    // ThreadId をキャッシュしておく
    Bit64 threadId;
    NN_ABORT_UNLESS_RESULT_SUCCESS( svc::GetThreadId(&threadId, threadHandle) );
    svc::GetThreadLocalRegion()->threadId = threadId;

    SetThreadPointerForCompilerTls();
}

//-------------------------------------------------------------------------
//  既存のスタック領域を元領域として、スタックのエイリアスをマップする。
bool ThreadManagerImplByHorizon::MapAliasStack(const void* newStack, const void* originalStack, size_t stackSize) const NN_NOEXCEPT
{
    // SVC を発行して、スタックのエイリアスを構築する
    auto result = svc::MapMemory(
                        reinterpret_cast<uintptr_t>(newStack),
                        reinterpret_cast<uintptr_t>(originalStack),
                        stackSize);

    if ( result.IsFailure() )
    {
        if ( nn::svc::ResultInvalidCurrentMemory::Includes( result ) )
        {
            // toAddr と size の領域のチェック
            svc::MemoryInfo memoryInfo;
            svc::PageInfo pageInfo;
            {
                auto result = svc::QueryMemory( &memoryInfo, &pageInfo, reinterpret_cast<uintptr_t>(newStack) );
                NN_ABORT_UNLESS_RESULT_SUCCESS( result );
            }
            if ( memoryInfo.state != svc::MemoryState_Free )
            {
                return false;
            }
            else
            {
                // From メモリの異常を細かく調べる
                CheckDetailsForInvalidCurrentMemory( reinterpret_cast<uintptr_t>(originalStack), stackSize );
            }
        }
        return false;
    }

    return true;
}

//-------------------------------------------------------------------------
//  マップされているエイリアススタックをアンマップする。
bool ThreadManagerImplByHorizon::UnmapAliasStack(const void* newStack, const void* originalStack, size_t stackSize) const NN_NOEXCEPT
{
    // SVC を発行して、スタックのエイリアスを解除する
    auto result = svc::UnmapMemory(
                        reinterpret_cast<uintptr_t>(newStack),
                        reinterpret_cast<uintptr_t>(originalStack),
                        stackSize);

    return result.IsSuccess();
}

// CreateThread（coreNumber あり）
Result ThreadManagerImplByHorizon::CreateThread(ThreadType* thread, void (*function)(ThreadType*), int coreNumber) NN_NOEXCEPT
{
    // スレッド生成用 引数リストを構築
    // スタックの先頭を一時的に使用。スレッド開始時はボトムから使用。
    Arglist* args = &(reinterpret_cast<ArglistU*>(reinterpret_cast<uintptr_t>(thread->_stack) + thread->_stackSize) - 1)->arg;
    args->function  = function;

    // コア番号の指定は暫定
    int count = 0;
retry:
    nn::svc::Handle handle;
    Result result = nn::svc::CreateThread(&handle,
            reinterpret_cast<nn::svc::ThreadFunc>(&InvokeThread),
            reinterpret_cast<uintptr_t>(thread),
            reinterpret_cast<uintptr_t>(args),
            ConvertToHorizonPriority(thread->_basePriority), coreNumber);

    // Result コードの判別
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultOutOfResource )
        {
            if (++count < 10)
            {
                // こちらはカーネルで資源回収中であるため、
                // 一定時間後であれば成功する可能性がある。
                // ただ、念のため一定回数でリトライを断念する。
                os::SleepThread( TimeSpan::FromMilliSeconds(10) );
                goto retry;
            }
            NN_RESULT_THROW( os::ResultOutOfResource() );
        }
        NN_RESULT_CATCH( svc::ResultLimit )
        {
            // こちらはリソース上限に達しているため、リトライはしない。
            NN_ABORT_UNLESS_RESULT_SUCCESS( os::ResultResourceLimit() );
        }
        NN_RESULT_CATCH( svc::ResultInvalidPointer )
        {
            NN_ABORT(NN_TEXT_OS("nn::os::CreateThread(): nn::svc::ResultInvalidPointer を検知しました。"));
        }
        NN_RESULT_CATCH( svc::ResultInvalidPriority )
        {
            NN_ABORT(NN_TEXT_OS("nn::os::CreateThread(): 指定された優先度が不正です（priority=%d）"), thread->_basePriority);
        }
        NN_RESULT_CATCH( svc::ResultInvalidCoreNumber )
        {
            NN_ABORT(NN_TEXT_OS("nn::os::CreateThread(): 指定されたコア番号が不正です（codeNumber=%d）"), coreNumber);
        }
        NN_RESULT_CATCH( svc::ResultMaxHandle )
        {
            NN_ABORT(NN_TEXT_OS("nn::os::CreateThread(): ハンドル不足です。"));
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    // スレッドのハンドル情報を残す
    thread->_handle = static_cast<nnHandle>(handle).value;

    NN_RESULT_SUCCESS;
}

void ThreadManagerImplByHorizon::StartThread(const ThreadType* thread) NN_NOEXCEPT
{
    nn::svc::Handle handle(thread->_handle);

    // スレッドの実行開始
    auto result = nn::svc::StartThread(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );
}

void ThreadManagerImplByHorizon::DestroyThreadUnsafe(ThreadType* thread) NN_NOEXCEPT
{
    nn::svc::Handle handle(thread->_handle);

    auto result = nn::svc::CloseHandle(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

    thread->_handle = 0;
}

void ThreadManagerImplByHorizon::WaitForExitThread(ThreadType* thread) NN_NOEXCEPT
{
    for (;;)
    {
        nn::svc::Handle handle(thread->_handle);
        int32_t index;
        auto result = nn::svc::WaitSynchronization(&index, &handle, 1, nn::svc::WAIT_INFINITE);
        if (result.IsSuccess())
        {
            return;
        }
        if (result <= svc::ResultCancelled())
        {
            continue;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS( result );
    }
}

void ThreadManagerImplByHorizon::YieldThread() NN_NOEXCEPT
{
    nn::svc::SleepThread( svc::YieldType_WithCoreMigration );
}

bool ThreadManagerImplByHorizon::ChangePriority(ThreadType* thread, int priority) NN_NOEXCEPT
{
    nn::svc::Handle handle(thread->_handle);

    auto result = nn::svc::SetThreadPriority(handle, ConvertToHorizonPriority(priority));
    if (result <= nn::svc::ResultInvalidPriority())
    {
        NN_ABORT(NN_TEXT_OS("指定された優先度が不正です（priority=%d）"), priority);
    }

    return result.IsSuccess();
}

int ThreadManagerImplByHorizon::GetCurrentPriority(const ThreadType* thread) const NN_NOEXCEPT
{
    nn::svc::Handle handle(thread->_handle);
    int32_t priority;

    auto result = nn::svc::GetThreadPriority(&priority, handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

    return ConvertToSigloPriority(priority);
}

uint64_t ThreadManagerImplByHorizon::GetThreadId(const ThreadType* thread) const NN_NOEXCEPT
{
    // 自スレッドならキャッシュしておいた内容を見る
    if (thread == this->GetCurrentThread())
    {
        return svc::GetThreadLocalRegion()->threadId;
    }

    Bit64 threadId;
    NN_ABORT_UNLESS_RESULT_SUCCESS( svc::GetThreadId(&threadId, svc::Handle(thread->_handle)) );
    return threadId;
}

void ThreadManagerImplByHorizon::SuspendThreadUnsafe(ThreadType* thread) NN_NOEXCEPT
{
    svc::Handle handle(thread->_handle);
    auto result = svc::SetThreadActivity(handle, svc::ThreadActivity_Paused);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void ThreadManagerImplByHorizon::ResumeThreadUnsafe(ThreadType* thread) NN_NOEXCEPT
{
    svc::Handle handle(thread->_handle);
    auto result = svc::SetThreadActivity(handle, svc::ThreadActivity_Runnable);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void ThreadManagerImplByHorizon::GetThreadContextUnsafe(ThreadContextInfo* dstContext, const ThreadType* thread) NN_NOEXCEPT
{
    svc::Handle         handle(thread->_handle);
    svc::ThreadContext  svcContext;
    auto result = svc::GetThreadContext3(&svcContext, handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

#if defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    std::memcpy(dstContext->r, svcContext.r, sizeof(Bit32) * 13);
    dstContext->r[13]   = svcContext.sp;
    dstContext->r[14]   = svcContext.lr;
    dstContext->r[15]   = svcContext.pc;
    dstContext->cpsr    = svcContext.cpsr;

    std::memcpy(dstContext->fpuRegisters, svcContext.fpuRegisters, sizeof(Bit64) * 16);
    dstContext->fpscr   = svcContext.fpscr;
    dstContext->fpexc   = svcContext.fpexc;
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    std::memcpy(dstContext->r, svcContext.r, sizeof(Bit64) * 29);
    dstContext->r[29]   = svcContext.fp;
    dstContext->r[30]   = svcContext.lr;
    dstContext->sp      = svcContext.sp;
    dstContext->pc      = svcContext.pc;
    dstContext->pstate  = svcContext.pstate;

    std::memcpy(dstContext->v, svcContext.v, sizeof(Bit128) * 32);
    dstContext->fpcr    = svcContext.fpcr;
    dstContext->fpsr    = svcContext.fpsr;
#else
    #error "Unknown ARM's AArch type"
#endif
}


int ThreadManagerImplByHorizon::GetCurrentCoreNumber() const NN_NOEXCEPT
{
    return nn::svc::GetCurrentProcessorNumber();
}

static_assert(nn::os::IdealCoreDontCare        == nn::svc::IdealCoreDontCare,        "");
static_assert(nn::os::IdealCoreUseDefaultValue == nn::svc::IdealCoreUseProcessValue, "");
static_assert(nn::os::IdealCoreNoUpdate        == nn::svc::IdealCoreNoUpdate,        "");

void ThreadManagerImplByHorizon::SetThreadCoreMask(ThreadType* thread, int idealCore, Bit64 affinityMask) const NN_NOEXCEPT
{
    nn::svc::Handle handle(thread->_handle);
    auto result = nn::svc::SetThreadCoreMask(handle, static_cast<int32_t>(idealCore), affinityMask);
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );
}

void ThreadManagerImplByHorizon::GetThreadCoreMask(int* pOutIdealCore, Bit64* pOutAffinityMask, const ThreadType* thread) const NN_NOEXCEPT
{
    int32_t idealCore;
    Bit64   affinityMask;

    nn::svc::Handle handle(thread->_handle);
    auto result = nn::svc::GetThreadCoreMask(&idealCore, &affinityMask, handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

    if (pOutIdealCore)
    {
        *pOutIdealCore = static_cast<int>(idealCore);
    }
    if (pOutAffinityMask)
    {
        *pOutAffinityMask = affinityMask;
    }
}

nn::Bit64 ThreadManagerImplByHorizon::GetThreadAvailableCoreMask() const NN_NOEXCEPT
{
    nn::Bit64 coreMask;
    auto result = nn::svc::GetInfo(&coreMask, nn::svc::InfoType_CoreMask, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0);
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

    return coreMask;
}

}}} // namespace nn::os::detail

