﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>
#include <atomic>

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_LightEventTypes.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_TimeoutHelper.h"
#include "os_LightEventImpl-os.win32.h"


namespace nn { namespace os { namespace detail {

namespace {

// LightEvent の broadcastCounter は 24bit しかないが、
// LightEvent をパルス的に Signal -> Clear した時に、
// 待機中スレッドが broadcastCounter の一致で起床できないケースは
// 非常に希であり、実質的に問題にならない。

inline uint32_t GetBroadcastCounterUnsafe(LightEventType* p) NN_NOEXCEPT
{
    uint32_t upper = p->_broadcastCounterUpper;
    NN_STATIC_ASSERT(sizeof(upper) >= sizeof(p->_broadcastCounterUpper) + sizeof(p->_broadcastCounterLower));

    return (upper << (sizeof(p->_broadcastCounterLower) * 8)) | p->_broadcastCounterLower;
}

inline void IncrementBroadcastCounterUnsafe(LightEventType* p) NN_NOEXCEPT
{
    if (0 == (++p->_broadcastCounterLower))
    {
        ++p->_broadcastCounterUpper;
    }
}

inline bool AtomicAutoClearLightEvent(LightEventType* event) NN_NOEXCEPT
{
    // CAS 的に SignaledAuto を NotSignaledAuto に書き換えて成否を返す。
    int8_t expected = LightEventType::SignaledAuto;
    int8_t desired  = LightEventType::NotSignaledAuto;
    return std::atomic_compare_exchange_strong(&event->_signalState, &expected, desired);
}

} // namespace

//-----------------------------------------------------------------------------

void InitializeLightEventImpl(LightEventType* event, bool initiallySignaled, EventClearMode clearMode) NN_NOEXCEPT
{
    new( &event->_mutex ) detail::InternalCriticalSection;
    new( &event->_cond )  detail::InternalConditionVariable;

    event->_broadcastCounterLower = 0;
    event->_broadcastCounterUpper = 0;

    int8_t signalState;
    if (clearMode == EventClearMode_AutoClear)
    {
        signalState = initiallySignaled ? LightEventType::SignaledAuto
                                        : LightEventType::NotSignaledAuto;
    }
    else
    {
        signalState = initiallySignaled ? LightEventType::SignaledManual
                                        : LightEventType::NotSignaledManual;
    }
    std::atomic_init(&event->_signalState, signalState);
}

void FinalizeLightEventImpl(LightEventType* event) NN_NOEXCEPT
{
    event->_signalState = LightEventType::NotInitialized;

    Get(event->_mutex).~InternalCriticalSection();
    Get(event->_cond).~InternalConditionVariable();
}

void SignalLightEventImpl(LightEventType* event) NN_NOEXCEPT
{
    auto signalState = std::atomic_load_explicit(&event->_signalState, std::memory_order_acquire);
    if (signalState == LightEventType::NotSignaledAuto)
    {
        std::lock_guard<detail::InternalCriticalSection> lk(Get(event->_mutex));
        event->_signalState = LightEventType::SignaledAuto;
        Get(event->_cond).Signal();
    }
    else if (signalState == LightEventType::NotSignaledManual)
    {
        std::lock_guard<detail::InternalCriticalSection> lk(Get(event->_mutex));
        event->_signalState = LightEventType::SignaledManual;
        IncrementBroadcastCounterUnsafe(event);
        Get(event->_cond).Broadcast();
    }
}

void ClearLightEventImpl(LightEventType* event) NN_NOEXCEPT
{
    auto signalState = std::atomic_load_explicit(&event->_signalState, std::memory_order_acquire);
    if (signalState == LightEventType::SignaledManual)
    {
        std::lock_guard<detail::InternalCriticalSection> lk(Get(event->_mutex));
        event->_signalState = LightEventType::NotSignaledManual;
    }
    else if (signalState == LightEventType::SignaledAuto)
    {
        std::lock_guard<detail::InternalCriticalSection> lk(Get(event->_mutex));
        event->_signalState = LightEventType::NotSignaledAuto;
    }
}

void WaitLightEventImpl(LightEventType* event) NN_NOEXCEPT
{
    for (;;)
    {
        auto signalState = std::atomic_load_explicit(&event->_signalState, std::memory_order_acquire);
        if (signalState == LightEventType::NotSignaledManual)
        {
            // フラグがセットされるまで待機する、フラグはクリアしない。
            std::lock_guard<detail::InternalCriticalSection> lk(Get(event->_mutex));
            auto currentCounter = GetBroadcastCounterUnsafe(event);
            while (event->_signalState == LightEventType::NotSignaledManual)
            {
                // ManualClear の場合、カウンタが進んでいたら起床する
                // AutoClear   の場合、カウンタが進むことはない
                if (currentCounter != GetBroadcastCounterUnsafe(event))
                {
                    break;
                }
                Get(event->_cond).Wait( &Get(event->_mutex) );
            }
            return;
        }
        else if (signalState == LightEventType::NotSignaledAuto)
        {
            // フラグがセットされるまで待機する、フラグはクリアする。
            std::lock_guard<detail::InternalCriticalSection> lk(Get(event->_mutex));
            while (event->_signalState == LightEventType::NotSignaledAuto)
            {
                Get(event->_cond).Wait( &Get(event->_mutex) );
            }
            event->_signalState = LightEventType::NotSignaledAuto;
            return;
        }
        else if (signalState == LightEventType::SignaledAuto)
        {
            // SignaledAuto の場合のみ CAS 的に NotSignaledAuto に書き換える。
            // 成功したら return し、失敗したら signalState を評価し直す。
            if (AtomicAutoClearLightEvent(event))
            {
                return;
            }
            continue;
        }
        else if (signalState == LightEventType::SignaledManual)
        {
            // フラグはセットされているので待機しない、クリアもしない。
            return;
        }
    }
}

bool TryWaitLightEventImpl(LightEventType* event) NN_NOEXCEPT
{
    auto signalState = std::atomic_load_explicit(&event->_signalState, std::memory_order_acquire);
    if (signalState == LightEventType::SignaledAuto)
    {
        // SignaledAuto の場合のみ CAS 的に NotSignaledAuto に書き換える。
        return AtomicAutoClearLightEvent(event);
    }
    else
    {
        // 他の 3 状態の場合は状態は変化しないので評価結果のみ返す
        return signalState == LightEventType::SignaledManual;
    }
}

bool TimedWaitLightEventImpl(LightEventType* event, TimeSpan timeout) NN_NOEXCEPT
{
    detail::TimeoutHelper tmout( timeout );
    for (;;)
    {
        auto signalState = std::atomic_load_explicit(&event->_signalState, std::memory_order_acquire);
        if (signalState == LightEventType::NotSignaledManual)
        {
            // フラグがセットされるまで待機する、フラグはクリアしない。
            std::lock_guard<detail::InternalCriticalSection> lk(Get(event->_mutex));
            auto currentCounter = GetBroadcastCounterUnsafe(event);
            while (event->_signalState == LightEventType::NotSignaledManual)
            {
                // ManualClear の場合、カウンタが進んでいたら起床する
                // AutoClear   の場合、カウンタが進むことはない
                if (currentCounter != GetBroadcastCounterUnsafe(event))
                {
                    break;
                }
                auto ret = Get(event->_cond).TimedWait( &Get(event->_mutex), tmout );
                if (ret == ConditionVariableStatus_Timeout)
                {
                    return false;
                }
            }
            return true;
        }
        else if (signalState == LightEventType::NotSignaledAuto)
        {
            // フラグがセットされるまで待機する、フラグはクリアする。
            std::lock_guard<detail::InternalCriticalSection> lk(Get(event->_mutex));
            while (event->_signalState == LightEventType::NotSignaledAuto)
            {
                auto ret = Get(event->_cond).TimedWait( &Get(event->_mutex), tmout );
                if (ret == ConditionVariableStatus_Timeout)
                {
                    return false;
                }
            }
            event->_signalState = LightEventType::NotSignaledAuto;
            return true;
        }
        else if (signalState == LightEventType::SignaledAuto)
        {
            // SignaledAuto の場合のみ CAS 的に NotSignaledAuto に書き換える。
            // 成功したら return し、失敗したら signalState を評価し直す。
            if (AtomicAutoClearLightEvent(event))
            {
                return true;
            }
            continue;
        }
        else if (signalState == LightEventType::SignaledManual)
        {
            // フラグはセットされているので待機しない、クリアもしない。
            return true;
        }
    }
}

//-----------------------------------------------------------------------------

}}} // namespace nn::os::detail

