﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <new>
#include <mutex>
#include <atomic>
#include <cstring>

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/svc/svc_Base.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_TimeoutHelper.h"


#define NN_OS_REQUIRES_INITIALIZED(p) \
        NN_SDK_REQUIRES( (p)->_isInitialized, NN_TEXT_OS("nn::os::%s(): 指定された LightEvent が初期化されていません。"), NN_CURRENT_FUNCTION_NAME)

namespace nn { namespace os { namespace detail {

//-----------------------------------------------------------------------------

namespace {

const int32_t NotSignaledAndNoWaiter = LightEventType::NotSignaledAndNoWaiter;
const int32_t NotSignaledAndWaiter   = LightEventType::NotSignaledAndWaiter;
const int32_t Signaled               = LightEventType::Signaled;

// svc の仕様を満たすために必要な条件など
NN_STATIC_ASSERT(NotSignaledAndNoWaiter == NotSignaledAndWaiter - 1);
NN_STATIC_ASSERT(Signaled               == NotSignaledAndWaiter + 1);
NN_STATIC_ASSERT(sizeof(LightEventType) == sizeof(uint32_t) * 3);


inline bool SvcWaitForAddressIfEqual(LightEventType* p) NN_NOEXCEPT
{
    auto result = svc::WaitForAddress(reinterpret_cast<uintptr_t>(&p->_signalState), svc::ArbitrationType_WaitIfEqual, NotSignaledAndWaiter, -1);
    if (result <= svc::ResultInvalidState())
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return true;
}

inline bool SvcWaitForAddressIfEqual(bool* pOutIsTimedout, LightEventType* p, int64_t tmout) NN_NOEXCEPT
{
    auto result = svc::WaitForAddress(reinterpret_cast<uintptr_t>(&p->_signalState), svc::ArbitrationType_WaitIfEqual, NotSignaledAndWaiter, tmout);
    bool isTimedout = (result <= svc::ResultTimeout());
    *pOutIsTimedout = isTimedout;
    if (isTimedout || result <= svc::ResultInvalidState())
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return true;
}

inline bool SvcSignalToAddressForAutoClear(LightEventType* p, int32_t expected) NN_NOEXCEPT
{
    auto result = svc::SignalToAddress(reinterpret_cast<uintptr_t>(&p->_signalState), svc::SignalType_UpdateByCountIfEqual, expected, 1);
    if (result <= svc::ResultInvalidState())
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return true;
}

inline bool SvcSignalToAddressForManualClear(LightEventType* p, int32_t expected) NN_NOEXCEPT
{
    auto result = svc::SignalToAddress(reinterpret_cast<uintptr_t>(&p->_signalState), svc::SignalType_IncrementIfEqual, expected, -1);
    if (result <= svc::ResultInvalidState())
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return true;
}

inline bool CompareAndSwap(LightEventType* p, int32_t expectedValue, int32_t desiredValue) NN_NOEXCEPT
{
    // CAS 的に expected を desired に書き換えて成否を返す。
    int32_t expected = expectedValue;
    int32_t desired  = desiredValue;
    return std::atomic_compare_exchange_strong(&p->_signalState, &expected, desired);
}

}   // namespace

//-----------------------------------------------------------------------------

inline void InitializeLightEventImpl(LightEventType* p, bool initiallySignaled, EventClearMode clearMode) NN_NOEXCEPT
{
    std::memset(p, 0, sizeof(LightEventType));

    p->_isAutoClear = (EventClearMode_AutoClear == clearMode);
    int32_t state = initiallySignaled ? Signaled : NotSignaledAndNoWaiter;
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_GCC)
    p->_signalState = state;
#else
    std::atomic_init(&p->_signalState, state);
#endif
    p->_isInitialized = true;
}


inline void FinalizeLightEventImpl(LightEventType* p) NN_NOEXCEPT
{
    p->_isInitialized = false;
}


inline void SignalLightEventImpl(LightEventType* p) NN_NOEXCEPT
{
    // Signal 操作によって他スレッドが他コア上で Wait を抜けた時には、
    // 本コアでのこれまでのストア完了結果を観測できることを期待するため、
    // ここは最低限 Store バリアの配置が必要。
    //
    // 同時に他コアでのストア結果を待ってから p->_signalState を観測した方が
    // 無駄な CAS 失敗等を抑えることができるため、ループの先頭に Load バリア
    // を入れておく方がよい。
    //
    // 上記両方を満たすため、memory_order_seq_cst のメモリフェンスをループの
    // 先頭に入れている（実コード生成は NX64, NX32 共に "dmb ish" となる）。

    if (p->_isAutoClear)
    {
        for (;;)
        {
            std::atomic_thread_fence(std::memory_order_seq_cst);
            auto state = p->_signalState.load(std::memory_order_relaxed);
            switch (state)
            {
                case NotSignaledAndNoWaiter:
                {
                    if (CompareAndSwap(p, state, Signaled))
                    {
                        return;
                    }
                    break;
                }
                case NotSignaledAndWaiter:
                {
                    if (SvcSignalToAddressForAutoClear(p, state))
                    {
                        return;
                    }
                    break;
                }
                case Signaled:
                {
                    // Do nothing
                    return;
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }
    else // ManualClear
    {
        for (;;)
        {
            std::atomic_thread_fence(std::memory_order_seq_cst);
            auto state = p->_signalState.load(std::memory_order_relaxed);
            switch (state)
            {
                case NotSignaledAndNoWaiter:
                {
                    if (CompareAndSwap(p, state, Signaled))
                    {
                        return;
                    }
                    break;
                }
                case NotSignaledAndWaiter:
                {
                    if (SvcSignalToAddressForManualClear(p, state))
                    {
                        return;
                    }
                    break;
                }
                case Signaled:
                {
                    // Do nothing
                    return;
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }
}


inline void ClearLightEventImpl(LightEventType* p) NN_NOEXCEPT
{
    std::atomic_thread_fence(std::memory_order_acquire);

    // 失敗してもリトライ不要
    CompareAndSwap(p, Signaled, NotSignaledAndNoWaiter);

    std::atomic_thread_fence(std::memory_order_release);
}


inline void WaitLightEventImpl(LightEventType* p) NN_NOEXCEPT
{
    // この Wait 関数を抜けた段階で、他コアで Signal されるまでにストア
    // されたメモリ内容を正しく観測できることを期待しているはずなので、
    // 本関数を抜ける時点で Load バリアの配置が必要。
    //
    // また、本関数で AutoClear した内容を他コアで観測できるようにするため、
    // Store バリアも配置しておく必要がある。ARMv8 ではバリア付き命令が
    // 使用されるため必須ではないが、ARMv7 等では必要になる。
    //
    // 上記両方を満たすため、memory_order_seq_cst のメモリフェンスを本関数の
    // 末尾に入れている（実コード生成は NX64, NX32 共に "dmb ish" となる）。

    NN_UTIL_SCOPE_EXIT
    {
        std::atomic_thread_fence(std::memory_order_seq_cst);
    };

    if (p->_isAutoClear)
    {
        for (;;)
        {
            auto state = p->_signalState.load(std::memory_order_acquire);
            switch (state)
            {
                case NotSignaledAndNoWaiter:
                {
                    if (!CompareAndSwap(p, state, NotSignaledAndWaiter))
                    {
                        break;
                    }
                    NN_FALL_THROUGH;
                }
                case NotSignaledAndWaiter:
                {
                    if (SvcWaitForAddressIfEqual(p))
                    {
                        return;
                    }
                    break;
                }
                case Signaled:
                {
                    if (CompareAndSwap(p, state, NotSignaledAndNoWaiter))
                    {
                        return;
                    }
                    break;
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }
    else // ManualClear
    {
        for (;;)
        {
            auto state = p->_signalState.load(std::memory_order_acquire);
            switch (state)
            {
                case NotSignaledAndNoWaiter:
                {
                    if (!CompareAndSwap(p, state, NotSignaledAndWaiter))
                    {
                        break;
                    }
                    NN_FALL_THROUGH;
                }
                case NotSignaledAndWaiter:
                {
                    // Result によらずリトライしない
                    SvcWaitForAddressIfEqual(p);
                    return;
                }
                case Signaled:
                {
                    // Do nothing
                    return;
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }
}


inline bool TryWaitLightEventImpl(LightEventType* p) NN_NOEXCEPT
{
    // このメモリフェンスについては WaitLightEventImpl() のコメントを参照。
    NN_UTIL_SCOPE_EXIT
    {
        std::atomic_thread_fence(std::memory_order_seq_cst);
    };

    if (p->_isAutoClear)
    {
        return CompareAndSwap(p, Signaled, NotSignaledAndNoWaiter);
    }
    else // ManualClear
    {
        auto state = p->_signalState.load(std::memory_order_acquire);
        return Signaled == state;
    }
}


inline bool TimedWaitLightEventImpl(LightEventType* p, TimeSpan timeout) NN_NOEXCEPT
{
    // このメモリフェンスについては WaitLightEventImpl() のコメントを参照。
    NN_UTIL_SCOPE_EXIT
    {
        std::atomic_thread_fence(std::memory_order_seq_cst);
    };

    if (p->_isAutoClear)
    {
        detail::TimeoutHelper tmout( timeout );
        for (;;)
        {
            auto state = p->_signalState.load(std::memory_order_acquire);
            switch (state)
            {
                case NotSignaledAndNoWaiter:
                {
                    if (!CompareAndSwap(p, state, NotSignaledAndWaiter))
                    {
                        break;
                    }
                    NN_FALL_THROUGH;
                }
                case NotSignaledAndWaiter:
                {
                    bool isTimedout;
                    if (SvcWaitForAddressIfEqual(&isTimedout, p, tmout.GetLeftTimeOnTarget().GetNanoSeconds()))
                    {
                        return true;
                    }
                    if (isTimedout)
                    {
                        return false;
                    }
                    break;
                }
                case Signaled:
                {
                    if (CompareAndSwap(p, state, NotSignaledAndNoWaiter))
                    {
                        return true;
                    }
                    break;
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }
    else // ManualClear
    {
        detail::TimeoutHelper tmout( timeout );
        for (;;)
        {
            auto state = p->_signalState.load(std::memory_order_acquire);
            switch (state)
            {
                case NotSignaledAndNoWaiter:
                {
                    if (!CompareAndSwap(p, state, NotSignaledAndWaiter))
                    {
                        break;
                    }
                    NN_FALL_THROUGH;
                }
                case NotSignaledAndWaiter:
                {
                    // Result によらずリトライしない
                    bool isTimedout;
                    SvcWaitForAddressIfEqual(&isTimedout, p, tmout.GetLeftTimeOnTarget().GetNanoSeconds());
                    return !isTimedout;
                }
                case Signaled:
                {
                    // Do nothing
                    return true;
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }
}

//-----------------------------------------------------------------------------

}}} // namespace nn::os::detail

