﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/os/os_Config.h>
#include <nn/os/detail/os_InternalCriticalSection-os.horizon.h>
#include <nn/os/os_Thread.h>
#include <nn/svc/svc_Base.h>
#include "os_Diag.h"
#include "os_Common.h"
#include "os_ThreadManager.h"

namespace nn { namespace os { namespace detail {

namespace {

#define NN_OS_LIKELY(exp)   __builtin_expect(!!(exp), true)
#define NN_OS_UNLIKELY(exp) __builtin_expect(!!(exp), false)

#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A)
#define NN_OS_DMB_UNLESS_ARMV8A()   asm volatile("dmb ish");
#else
#define NN_OS_DMB_UNLESS_ARMV8A()   do { } while(NN_STATIC_CONDITION(0))
#endif

Bit32 LoadEx(Bit32* ptr) NN_NOEXCEPT
{
    Bit32 val;
#if defined(NN_OS_FOR_SYSTEMCORE)
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || \
    defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
    asm volatile("ldrex %0, [%1]" : "=&r"(val) : "r"(ptr) : "memory");
#elif defined(NN_OS_CPU_ARM_AARCH64)
    asm volatile("ldxr %w0, [%1]" : "=&r"(val) : "r"(ptr) : "memory");
#endif
#else // defined(NN_OS_FOR_SYSTEMCORE)
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A)
    asm volatile("ldrex %0, [%1]" : "=&r"(val) : "r"(ptr) : "memory");
#elif defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
    asm volatile("ldaex %0, [%1]" : "=&r"(val) : "r"(ptr) : "memory");
#elif defined(NN_OS_CPU_ARM_AARCH64)
    asm volatile("ldaxr %w0, [%1]" : "=&r"(val) : "r"(ptr) : "memory");
#endif
#endif // defined(NN_OS_FOR_SYSTEMCORE)
    return val;
}

int StoreEx(Bit32 val, Bit32* ptr) NN_NOEXCEPT
{
    int result;
#if defined(NN_OS_FOR_SYSTEMCORE)
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || \
    defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
    asm volatile("strex %0, %1, [%2]" : "=&r"(result) : "r"(val), "r"(ptr) : "memory");
#elif defined(NN_OS_CPU_ARM_AARCH64)
    asm volatile("stxr %w0, %w1, [%2]" : "=&r"(result) : "r"(val), "r"(ptr) : "memory");
#endif
#else // defined(NN_OS_FOR_SYSTEMCORE)
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A)
    asm volatile("strex %0, %1, [%2]" : "=&r"(result) : "r"(val), "r"(ptr) : "memory");
#elif defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
    asm volatile("stlex %0, %1, [%2]" : "=&r"(result) : "r"(val), "r"(ptr) : "memory");
#elif defined(NN_OS_CPU_ARM_AARCH64)
    asm volatile("stlxr %w0, %w1, [%2]" : "=&r"(result) : "r"(val), "r"(ptr) : "memory");
#endif
#endif // defined(NN_OS_FOR_SYSTEMCORE)
    return result;
}

void Clrex() NN_NOEXCEPT
{
    asm volatile("clrex" ::: "memory");
}

}   // namespace

//-----------------------------------------------------------------------------
// ■ this->m_ThreadHandle のアトミック操作に関する説明
//
// - m_ThreadHandle[29:0] がこのロックを保持しているスレッドのハンドル即値
// - m_ThreadHandle[29:0] == 0 ならこのロックを保持しているスレッドはいない
// - m_ThreadHandle[30]   == 1 ならこのロックを待機している他スレッドあり
//-----------------------------------------------------------------------------

InternalCriticalSectionImplByHorizon::InternalCriticalSectionImplByHorizon()  NN_NOEXCEPT
    : m_ThreadHandle(0)
{
#if !defined(NN_SDK_BUILD_RELEASE)
    svc::MemoryInfo memoryInfo;
    svc::PageInfo pageInfo;

    auto result = svc::QueryMemory( &memoryInfo, &pageInfo, reinterpret_cast<uintptr_t>(&m_ThreadHandle) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

    NN_SDK_ASSERT( !(memoryInfo.attribute & svc::MemoryAttribute_Uncached), NN_TEXT_OS("非キャッシュ領域に同期オブジェクトを置こうとしています。(address=0x%p)"), &m_ThreadHandle );
#endif
}

void InternalCriticalSectionImplByHorizon::Initialize()  NN_NOEXCEPT
{
    this->m_ThreadHandle = 0;
}

void InternalCriticalSectionImplByHorizon::Enter() NN_NOEXCEPT
{
    Bit32 currentHandle = detail::GetCurrentThread()->_handle;

    NN_SDK_ASSERT( (m_ThreadHandle & ~nn::svc::Handle::WaitMask) != currentHandle, NN_TEXT_OS("同一スレッドによる多重ロックを検出しました。"));

    Bit32 value = LoadEx(&m_ThreadHandle);
    for (;;)
    {
        if (NN_OS_LIKELY(value == 0))
        {
            if (NN_OS_UNLIKELY(StoreEx(currentHandle, &m_ThreadHandle) != 0))
            {
                value = LoadEx(&m_ThreadHandle);
                continue;
            }
            break;
        }
        if (NN_OS_LIKELY((value & nn::svc::Handle::WaitMask) == 0))
        {
            if (NN_OS_UNLIKELY(StoreEx((value | nn::svc::Handle::WaitMask), &m_ThreadHandle)) != 0)
            {
                value = LoadEx(&m_ThreadHandle);
                continue;
            }
        }

        auto result = nn::svc::ArbitrateLock(
                nn::svc::Handle(value & ~nn::svc::Handle::WaitMask),
                reinterpret_cast<uintptr_t>(&m_ThreadHandle),
                currentHandle);
        NN_SDK_ASSERT(result.IsSuccess());
        NN_UNUSED(result);

        value = LoadEx(&m_ThreadHandle);
        if (NN_OS_LIKELY((value & ~nn::svc::Handle::WaitMask) == currentHandle))
        {
            Clrex();
            break;
        }
    }
    NN_OS_DMB_UNLESS_ARMV8A();
}

bool InternalCriticalSectionImplByHorizon::TryEnter() NN_NOEXCEPT
{
#if 0
    Bit32 currentHandle = detail::GetCurrentThread()->_handle;
    return __sync_bool_compare_and_swap(&m_ThreadHandle, 0, currentHandle);
#else
    Bit32 currentHandle = detail::GetCurrentThread()->_handle;
    for (;;)
    {
        Bit32 value = LoadEx(&m_ThreadHandle);
        if (NN_OS_UNLIKELY(value != 0))
        {
            break;
        }
        NN_OS_DMB_UNLESS_ARMV8A();
        if (NN_OS_LIKELY(StoreEx(currentHandle, &m_ThreadHandle) == 0))
        {
            return true;
        }
    }
    Clrex();
    NN_OS_DMB_UNLESS_ARMV8A();
    return false;
#endif
}

void InternalCriticalSectionImplByHorizon::Leave() NN_NOEXCEPT
{
#if 0
    Bit32 currentHandle = detail::GetCurrentThread()->_handle;
    Bit32 value = __sync_val_compare_and_swap(&m_ThreadHandle, currentHandle, 0);
#else
    Bit32 currentHandle = detail::GetCurrentThread()->_handle;
    Bit32 value = LoadEx(&m_ThreadHandle);
    for (;;)
    {
        if (NN_OS_UNLIKELY(value != currentHandle))
        {
            Clrex();
            break;
        }
        NN_OS_DMB_UNLESS_ARMV8A();
        if (NN_OS_LIKELY(StoreEx(0, &m_ThreadHandle) == 0))
        {
            break;
        }
        value = LoadEx(&m_ThreadHandle);
    }
    NN_OS_DMB_UNLESS_ARMV8A();
#endif

    NN_SDK_ASSERT((value | nn::svc::Handle::WaitMask) == (currentHandle | nn::svc::Handle::WaitMask));
    if (value & nn::svc::Handle::WaitMask)
    {
        auto result = nn::svc::ArbitrateUnlock(reinterpret_cast<uintptr_t>(&m_ThreadHandle));
        NN_SDK_ASSERT(result.IsSuccess());
        NN_UNUSED(result);
    }
}

bool InternalCriticalSectionImplByHorizon::IsLockedByCurrentThread() const NN_NOEXCEPT
{
    Bit32 currentHandle = detail::GetCurrentThread()->_handle;
    return (m_ThreadHandle & ~nn::svc::Handle::WaitMask) == currentHandle;
}

}}} // namespace nn::os::detail

