﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_ConditionVariableCommon.h>
#include "os_TimeoutHelper.h"
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Result.h>
#include "os_Diag.h"
#include "os_Common.h"

namespace nn { namespace os { namespace detail {

InternalConditionVariableImplByHorizon::InternalConditionVariableImplByHorizon() NN_NOEXCEPT
    : m_Value(0)
{
#if !defined(NN_SDK_BUILD_RELEASE)
    svc::MemoryInfo memoryInfo;
    svc::PageInfo pageInfo;

    auto result = svc::QueryMemory( &memoryInfo, &pageInfo, reinterpret_cast<uintptr_t>(&m_Value) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

    NN_SDK_ASSERT( !(memoryInfo.attribute & svc::MemoryAttribute_Uncached), NN_TEXT_OS("nn::os::InitializeConditionVariable(): 非キャッシュ領域に同期オブジェクトを置こうとしています。(address=0x%p)"), &m_Value );
#endif
}

void InternalConditionVariableImplByHorizon::Initialize() NN_NOEXCEPT
{
    this->m_Value = 0;
}

void InternalConditionVariableImplByHorizon::Signal() NN_NOEXCEPT
{
    nn::svc::SignalProcessWideKey(reinterpret_cast<uintptr_t>(&m_Value), 1);
}
void InternalConditionVariableImplByHorizon::Broadcast() NN_NOEXCEPT
{
    nn::svc::SignalProcessWideKey(reinterpret_cast<uintptr_t>(&m_Value), -1);
}
void InternalConditionVariableImplByHorizon::Wait(InternalCriticalSection* p) NN_NOEXCEPT
{
    NN_SDK_ASSERT( (p->Get()->m_ThreadHandle & ~nn::svc::Handle::WaitMask) == nn::os::GetCurrentThread()->_handle, NN_TEXT_OS("nn::os::WaitConditionVariable(): 指定されたロックの所有者が自スレッドではありません。"));

    Bit32 currentHandle = nn::os::GetCurrentThread()->_handle;
    Result result = nn::svc::WaitProcessWideKeyAtomic(
            reinterpret_cast<uintptr_t>(&p->Get()->m_ThreadHandle), reinterpret_cast<uintptr_t>(&m_Value), currentHandle, -1);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

ConditionVariableStatus InternalConditionVariableImplByHorizon::TimedWait(InternalCriticalSection* p, const TimeoutHelper& timeout) NN_NOEXCEPT
{
    NN_SDK_ASSERT( (p->Get()->m_ThreadHandle & ~nn::svc::Handle::WaitMask) == nn::os::GetCurrentThread()->_handle, NN_TEXT_OS("nn::os::TimedWaitConditionVariable(): 指定されたロックの所有者が自スレッドではありません。"));

    TimeSpan left = timeout.GetLeftTimeOnTarget();
    if (left > 0)
    {
        Bit32 currentHandle = nn::os::GetCurrentThread()->_handle;
        Result result = nn::svc::WaitProcessWideKeyAtomic(
                reinterpret_cast<uintptr_t>(&p->Get()->m_ThreadHandle), reinterpret_cast<uintptr_t>(&m_Value), currentHandle, left.GetNanoSeconds());

        if (result.IsSuccess())
        {
            return ConditionVariableStatus_NoTimeout;
        }
        if (result <= nn::svc::ResultTimeout())
        {
            p->Enter();
            return ConditionVariableStatus_Timeout;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    return ConditionVariableStatus_Timeout;
}

}}} // namespace nn::os::detail

