﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/ncm/ncm_IContentStorage.h>
#include <nn/ncm/ncm_ContentStorage.h>

namespace nn { namespace ncm {

    ContentStorage::ContentStorage() NN_NOEXCEPT{}

    ContentStorage::ContentStorage(sf::SharedPointer<IContentStorage> interfac) NN_NOEXCEPT : m_Interface(interfac){}

    ContentStorage::ContentStorage(ContentStorage&& rvalue) NN_NOEXCEPT
    {
        m_Interface = std::move(rvalue.m_Interface);
    }

    ContentStorage& ContentStorage::operator=(ContentStorage&& rvalue) NN_NOEXCEPT
    {
        ContentStorage(std::move(rvalue)).swap(*this);

        return *this;
    }

    void ContentStorage::swap(ContentStorage& other) NN_NOEXCEPT
    {
        std::swap(m_Interface, other.m_Interface);
    }

    PlaceHolderId ContentStorage::GeneratePlaceHolderId() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        PlaceHolderId outId;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GeneratePlaceHolderId(&outId));

        return outId;
    }

    Result ContentStorage::CreatePlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, int64_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        return m_Interface->CreatePlaceHolder(placeHolderId, contentId, size);
    }

    Result ContentStorage::SetPlaceHolderSize(PlaceHolderId placeHolderId, int64_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        return m_Interface->SetPlaceHolderSize(placeHolderId, size);
    }

    Result ContentStorage::DeletePlaceHolder(PlaceHolderId id) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        return m_Interface->DeletePlaceHolder(id);
    }

    Result ContentStorage::HasPlaceHolder(bool* outValue, PlaceHolderId id) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        return m_Interface->HasPlaceHolder(outValue, id);
    }

    Result ContentStorage::WritePlaceHolder(PlaceHolderId id, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        return m_Interface->WritePlaceHolder(id, offset, sf::InBuffer(reinterpret_cast<const char*>(buffer), size));
    }

    Result ContentStorage::Register(PlaceHolderId placeHolderId, ContentId contentId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        return m_Interface->Register(placeHolderId, contentId);
    }

    Result ContentStorage::RevertToPlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, ContentId postContentId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        return m_Interface->RevertToPlaceHolder(placeHolderId, contentId, postContentId);
    }

    Result ContentStorage::Delete(ContentId id) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        return m_Interface->Delete(id);
    }

    Result ContentStorage::Has(bool* outValue, ContentId id) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        return m_Interface->Has(outValue, id);
    }

    void ContentStorage::GetPath(Path* outValue, ContentId id) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetPath(outValue, id));
    }

    Result ContentStorage::GetFreeSpaceSize(int64_t* outValue) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        return m_Interface->GetFreeSpaceSize(outValue);
    }

    Result ContentStorage::GetTotalSpaceSize(int64_t* outValue) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        return m_Interface->GetTotalSpaceSize(outValue);
    }

    void ContentStorage::GetPlaceHolderPath(Path* outValue, PlaceHolderId id) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetPlaceHolderPath(outValue, id));
    }

    Result ContentStorage::CleanupAllPlaceHolder() NN_NOEXCEPT
    {
        return m_Interface->CleanupAllPlaceHolder();
    }

    Result ContentStorage::ListPlaceHolder(int* outCount, PlaceHolderId outList[], int count) const NN_NOEXCEPT
    {
        return m_Interface->ListPlaceHolder(outCount, sf::OutArray<PlaceHolderId>(outList, static_cast<size_t>(count)));
    }

    Result ContentStorage::GetContentCount(int* outCount) const NN_NOEXCEPT
    {
        return m_Interface->GetContentCount(outCount);
    }

    Result ContentStorage::ListContentId(int* outCount, ContentId outList[], int count, int offset) const NN_NOEXCEPT
    {
        return m_Interface->ListContentId(outCount, sf::OutArray<ContentId>(outList, static_cast<size_t>(count)), offset);
    }

    Result ContentStorage::DisableForcibly() NN_NOEXCEPT
    {
        return m_Interface->DisableForcibly();
    }

    Result ContentStorage::GetSize(int64_t* outValue, ContentId id) const NN_NOEXCEPT
    {
        return m_Interface->GetSizeFromContentId(outValue, id);
    }

    Result ContentStorage::GetSize(int64_t* outValue, PlaceHolderId id) const NN_NOEXCEPT
    {
        return m_Interface->GetSizeFromPlaceHolderId(outValue, id);
    }

    Result ContentStorage::ReadContentIdFile(void* buffer, size_t size, ContentId id, int64_t offset) const NN_NOEXCEPT
    {
        return m_Interface->ReadContentIdFile(sf::OutBuffer(reinterpret_cast<char*>(buffer), size), id, offset);
    }

    Result ContentStorage::GetRightsId(RightsId* outValue, PlaceHolderId id) const NN_NOEXCEPT
    {
        return m_Interface->GetRightsIdFromPlaceHolderId(outValue, id);
    }

    Result ContentStorage::GetRightsId(RightsId* outValue, ContentId id) const NN_NOEXCEPT
    {
        return m_Interface->GetRightsIdFromContentId(outValue, id);
    }

    Result ContentStorage::WriteContentForDebug(ContentId id, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Interface);

        return m_Interface->WriteContentForDebug(id, offset, sf::InBuffer(reinterpret_cast<const char*>(buffer), size));
    }

    Result ContentStorage::FlushPlaceHolder() NN_NOEXCEPT
    {
        return m_Interface->FlushPlaceHolder();
    }

    Result ContentStorage::RepairInvalidFileAttribute() NN_NOEXCEPT
    {
        return m_Interface->RepairInvalidFileAttribute();
    }

}}
