﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_ContentMetaExtendedData.h>

namespace nn { namespace ncm {
    namespace {
        void ConvertInstallContentMetaHeaderToContentMetaHeader(
            ContentMetaHeader* out, const InstallContentMetaHeader& source) NN_NOEXCEPT
        {
            out->extendedHeaderSize = source.extendedHeaderSize;
            out->contentMetaCount = source.contentMetaCount;
            out->contentCount = source.contentCount;
            out->attributes = source.attributes;
        }

        void ConvertPackageContentMetaHeaderToContentMetaHeader(
            ContentMetaHeader* out, const PackagedContentMetaHeader& source) NN_NOEXCEPT
        {
            out->extendedHeaderSize = source.extendedHeaderSize;
            out->contentMetaCount = source.contentMetaCount;
            out->contentCount = source.contentCount;
            out->attributes = source.attributes;
        }

        void ConvertPackageContentMetaHeaderToInstallContentMetaHeader(
            InstallContentMetaHeader* out, const PackagedContentMetaHeader& source) NN_NOEXCEPT
        {
            std::memcpy(out, &source, sizeof(InstallContentMetaHeader));
        }
        Result FindDeltaIndex(int* outIndex, const PatchMetaExtendedDataReader& reader, uint32_t sourceVersion, uint32_t destinationVersion) NN_NOEXCEPT
        {
            auto header = reader.GetHeader();
            for (int i = 0; i < static_cast<int>(header->deltaCount); ++i)
            {
                auto delta = reader.GetPatchDeltaHeader(i);
                // sourceVersion == 0 の場合、先頭のものを返すとしておく。テスト用
                if ((delta->delta.sourceVersion == sourceVersion || sourceVersion == 0) && delta->delta.destinationVersion == destinationVersion)
                {
                    *outIndex = i;
                    NN_RESULT_SUCCESS;
                }
            }
            NN_RESULT_THROW(ResultDeltaNotFound());
        }
        int CountContentExceptForMeta(const PatchMetaExtendedDataReader& reader, int deltaIndex) NN_NOEXCEPT
        {
            int count = 0;
            auto delta = reader.GetPatchDeltaHeader(deltaIndex);
            for (int i = 0; i < static_cast<int>(delta->contentCount); ++i)
            {
                auto info = reader.GetPatchDeltaPackagedContentInfo(deltaIndex, i);
                if (info->info.type != ContentType::Meta)
                {
                    count++;
                }
            }
            return count;
        }
    }

    size_t PackagedContentMetaReader::CalculateConvertInstallContentMetaSize() const NN_NOEXCEPT
    {
        auto header = GetHeader();
        if (header->type == ContentMetaType::Patch)
        {
            return RecalculateSizeTemplate<InstallContentMetaHeader, InstallContentInfo>(GetHeader()->extendedHeaderSize, GetHeader()->contentCount + 1 - CountDeltaFragments(), GetHeader()->contentMetaCount, GetExtendedDataSize(), false);
        }
        else
        {
            return RecalculateSizeTemplate<InstallContentMetaHeader, InstallContentInfo>(GetHeader()->extendedHeaderSize, GetHeader()->contentCount + 1, GetHeader()->contentMetaCount, GetExtendedDataSize(), false);
        }
    }

    size_t PackagedContentMetaReader::CalculateConvertContentMetaSize() const NN_NOEXCEPT
    {
        return RecalculateSizeTemplate<ContentMetaHeader, ContentInfo>(GetHeader()->extendedHeaderSize, GetHeader()->contentCount + 1, GetHeader()->contentMetaCount, 0, false);
    }

    Result PackagedContentMetaReader::CalculateConvertFragmentOnlyInstallContentMetaSize(size_t* outSize, uint32_t sourceVersion) const NN_NOEXCEPT
    {
        // 特定の更新差分に関する fragment だけ含みたいので、CountFragment は利用できない
        PatchMetaExtendedDataReader reader(GetExtendedData(), GetExtendedDataSize());
        auto key = GetKey();
        int index;
        NN_RESULT_DO(FindDeltaIndex(&index, reader, sourceVersion, key.version));
        auto fragmentCount = CountContentExceptForMeta(reader, index);
        *outSize = RecalculateSizeTemplate<InstallContentMetaHeader, InstallContentInfo>(GetHeader()->extendedHeaderSize, fragmentCount + 1, 0, GetExtendedDataSize(), false);
        NN_RESULT_SUCCESS;
    }

    void PackagedContentMetaReader::ConvertToInstallContentMeta(void* outValue, size_t size, const InstallContentInfo& meta) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(size >= CalculateConvertInstallContentMetaSize());

        uintptr_t address = reinterpret_cast<uintptr_t>(outValue);
        InstallContentMetaHeader header = {};
        ConvertPackageContentMetaHeaderToInstallContentMetaHeader(&header, *GetHeader());
        header.contentCount += 1; // メタの分を追加
        if (header.type == ContentMetaType::Patch)
        {
            header.contentCount -= CountDeltaFragments();
        }

        std::memcpy(reinterpret_cast<void*>(address), &header, sizeof(header));
        address += sizeof(header);
        std::memcpy(reinterpret_cast<void*>(address), reinterpret_cast<void*>(GetExtendedHeaderAddress()), GetHeader()->extendedHeaderSize);
        address += GetHeader()->extendedHeaderSize;

        std::memcpy(reinterpret_cast<void*>(address), &meta, sizeof(meta));
        address += sizeof(meta);
        for (int i = 0; i < CountContent(); i++)
        {
            if (header.type == ContentMetaType::Patch)
            {
                auto contentInfo = GetContentInfo(i);
                if (contentInfo->info.type == ContentType::DeltaFragment)
                {
                    continue;
                }
            }
            InstallContentInfo data = {};
            InstallContentInfo::MakeFrom(&data, *GetContentInfo(i), header.type);
            std::memcpy(reinterpret_cast<void*>(address), &data, sizeof(data));
            address += sizeof(data);
        }
        for (int i = 0; i < CountContentMeta(); i++)
        {
            std::memcpy(reinterpret_cast<void*>(address), GetContentMetaInfo(i), sizeof(ContentMetaInfo));
            address += sizeof(ContentMetaInfo);
        }
    }

    Result PackagedContentMetaReader::ConvertToFragmentOnlyInstallContentMeta(void* outValue, size_t size, const InstallContentInfo& meta, uint32_t sourceVersion) const NN_NOEXCEPT
    {
        // 特定の更新差分に関する fragment だけ含みたいので、CountFragment は利用できない
        size_t requiredSize;
        NN_RESULT_DO(CalculateConvertFragmentOnlyInstallContentMetaSize(&requiredSize, sourceVersion));
        NN_ABORT_UNLESS(size >= requiredSize);

        PatchMetaExtendedDataReader reader(GetExtendedData(), GetExtendedDataSize());
        auto key = GetKey();
        int index;
        NN_RESULT_DO(FindDeltaIndex(&index, reader, sourceVersion, key.version));
        auto delta = reader.GetPatchDeltaHeader(index);

        uintptr_t address = reinterpret_cast<uintptr_t>(outValue);
        InstallContentMetaHeader header = {};
        ConvertPackageContentMetaHeaderToInstallContentMetaHeader(&header, *GetHeader());
        header.installTypeReserve = ContentInstallType::FragmentOnly;

        auto fragmentCount = CountContentExceptForMeta(reader, index);
        header.contentCount = static_cast<uint16_t>(fragmentCount) + 1;

        std::memcpy(reinterpret_cast<void*>(address), &header, sizeof(header));
        address += sizeof(header);
        std::memcpy(reinterpret_cast<void*>(address), reinterpret_cast<void*>(GetExtendedHeaderAddress()), GetHeader()->extendedHeaderSize);
        address += GetHeader()->extendedHeaderSize;

        std::memcpy(reinterpret_cast<void*>(address), &meta, sizeof(meta));
        address += sizeof(meta);

        int count = 0;
        for (int i = 0; i < static_cast<int>(delta->contentCount); i++)
        {
            // CountContentExceptForMeta とロジックがそろっている必要がある
            auto info = reader.GetPatchDeltaPackagedContentInfo(index, i);
            if (info->info.type != ContentType::Meta)
            {
                InstallContentInfo data = {};
                InstallContentInfo::MakeFrom(&data, *info, header.type);
                std::memcpy(reinterpret_cast<void*>(address), &data, sizeof(data));
                address += sizeof(data);
                count++;
            }
        }
        NN_SDK_ASSERT_EQUAL(count, fragmentCount);
        NN_UNUSED(count);

        NN_RESULT_SUCCESS;
    }
    int16_t PackagedContentMetaReader::CountDeltaFragments() const NN_NOEXCEPT
    {
        int16_t count = 0;
        for (int i = 0; i < CountContent(); ++i)
        {
            auto contentInfo = GetContentInfo(i);
            if (contentInfo->info.type == ContentType::DeltaFragment)
            {
                count++;
            }
        }
        return count;
    }

    void PackagedContentMetaReader::ConvertToContentMeta(void* outValue, size_t size, const ContentInfo& meta) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(size >= CalculateConvertContentMetaSize());

        uintptr_t address = reinterpret_cast<uintptr_t>(outValue);
        auto originalHeader = GetHeader();
        ContentMetaHeader header = {};
        ConvertPackageContentMetaHeaderToContentMetaHeader(&header, *originalHeader);
        header.contentCount += 1; // メタの分を追加
        if (originalHeader->type == ContentMetaType::Patch)
        {
            header.contentCount -= CountDeltaFragments();
        }

        std::memcpy(reinterpret_cast<void*>(address), &header, sizeof(header));
        address += sizeof(header);
        std::memcpy(reinterpret_cast<void*>(address), reinterpret_cast<void*>(GetExtendedHeaderAddress()), GetHeader()->extendedHeaderSize);
        address += GetHeader()->extendedHeaderSize;

        std::memcpy(reinterpret_cast<void*>(address), &meta, sizeof(meta));
        address += sizeof(meta);
        for (int i = 0; i < CountContent(); i++)
        {
            if (originalHeader->type == ContentMetaType::Patch)
            {
                auto contentInfo = GetContentInfo(i);
                if (contentInfo->info.type == ContentType::DeltaFragment)
                {
                    continue;
                }
            }
            std::memcpy(reinterpret_cast<void*>(address), &GetContentInfo(i)->info, sizeof(ContentInfo));
            address += sizeof(ContentInfo);
        }
        for (int i = 0; i < CountContentMeta(); i++)
        {
            std::memcpy(reinterpret_cast<void*>(address), GetContentMetaInfo(i), sizeof(ContentMetaInfo));
            address += sizeof(ContentMetaInfo);
        }
    }

    size_t InstallContentMetaReader::CalculateConvertSize() const NN_NOEXCEPT
    {
        return RecalculateSizeTemplate<ContentMetaHeader, ContentInfo>(GetHeader()->extendedHeaderSize, GetHeader()->contentCount, GetHeader()->contentMetaCount, GetExtendedDataSize(), false);
    }

    void InstallContentMetaReader::ConvertToContentMeta(void* outValue, size_t size) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(size >= CalculateConvertSize());

        uintptr_t address = reinterpret_cast<uintptr_t>(outValue);
        ContentMetaHeader header = {};
        ConvertInstallContentMetaHeaderToContentMetaHeader(&header, *GetHeader());

        std::memcpy(reinterpret_cast<void*>(address), &header, sizeof(header));
        address += sizeof(header);
        std::memcpy(reinterpret_cast<void*>(address), reinterpret_cast<void*>(GetExtendedHeaderAddress()), GetHeader()->extendedHeaderSize);
        address += GetHeader()->extendedHeaderSize;

        for (int i = 0; i < CountContent(); i++)
        {
            std::memcpy(reinterpret_cast<void*>(address), &GetContentInfo(i)->info, sizeof(ContentInfo));
            address += sizeof(ContentInfo);
        }
        for (int i = 0; i < CountContentMeta(); i++)
        {
            std::memcpy(reinterpret_cast<void*>(address), GetContentMetaInfo(i), sizeof(ContentMetaInfo));
            address += sizeof(ContentMetaInfo);
        }
    }
}}
