﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SignedSystemPartition.h>

#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ServiceName.h>
#include <nn/ncm/ncm_IContentStorage.h>
#include <nn/ncm/ncm_ContentStorageImpl.h>
#include <nn/ncm/ncm_IContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentMetaDatabaseImpl.h>
#include <nn/ncm/ncm_OnMemoryContentMetaDatabaseImpl.h>
#include <nn/ncm/ncm_ContentManagerImpl.h>
#include <nn/ncm/ncm_ContentManagerSaveDataInfo.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_ReadOnlyContentStorageImpl.h>
#include <nn/ncm/detail/ncm_Log.h>
#include <nn/ncm/ncm_AutoBuffer.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ncm/ncm_MaxCount.h>
#include <nn/ncm/ncm_PathString.h>

#include <nn/sf/sf_ObjectFactory.h>

#include "ncm_FileSystemUtility.h"

namespace nn { namespace ncm {
    namespace {
        // TODO: kvdb のデータベース読み書きを最適化してメモリを削減する
        NN_ALIGNAS(4096) char g_SystemContentMetaDatabaseHeap[512 * 1024];
        NN_ALIGNAS(4096) char g_CardContentMetaDatabaseHeap[512 * 1024];
        NN_ALIGNAS(4096) char g_UserContentMetaDatabaseHeap[2560 * 1024];

        ContentMetaMemoryResource g_SystemContentMetaMemoryResource(g_SystemContentMetaDatabaseHeap, sizeof(g_SystemContentMetaDatabaseHeap));
        ContentMetaMemoryResource g_CardContentMetaMemoryResource(g_CardContentMetaDatabaseHeap, sizeof(g_CardContentMetaDatabaseHeap));
        ContentMetaMemoryResource g_UserContentMetaMemoryResource(g_UserContentMetaDatabaseHeap, sizeof(g_UserContentMetaDatabaseHeap));

        typedef kvdb::BoundedString<32> RootPath;

        void ReplaceMountName(char* outPath, const char* newMountName, const char* path) NN_NOEXCEPT
        {
            std::strcpy(outPath, newMountName);
            std::strcat(outPath, std::strchr(path, ':'));
        }

        template<typename FuncT>
        nn::Result TraverseContentId(sf::SharedPointer<IContentStorage> &contentStorage, FuncT func)
        {
            const int BufferSize = 64;
            nn::ncm::ContentId contentIds[BufferSize];
            int32_t totalCount;
            NN_RESULT_DO(
                contentStorage->GetContentCount(&totalCount));

            for(int32_t bufferOffset = 0; bufferOffset < totalCount; bufferOffset += BufferSize)
            {
                int32_t count;
                NN_RESULT_DO(
                    contentStorage->ListContentId(
                        &count,
                        sf::OutArray<nn::ncm::ContentId>(contentIds, BufferSize),
                        bufferOffset));

                for(int i = 0; i < count; i++)
                {
                    NN_RESULT_DO(
                        func(contentIds[i]));
                }
            }

            NN_RESULT_SUCCESS;
        }

        nn::Result EnsureBuildInSystemSaveDataFlag()
        {
            uint32_t currentFlag = 0;
            NN_RESULT_DO(fs::GetSaveDataFlags(&currentFlag, BuildInSystemSystemSaveDataId));
            if (currentFlag != BuildInSystemSystemSaveDataFlag)
            {
                return fs::SetSaveDataFlags(BuildInSystemSystemSaveDataId, fs::SaveDataSpaceId::System, BuildInSystemSystemSaveDataFlag);
            }
            NN_RESULT_SUCCESS;
        }
        nn::Result GetContentMetaDatabaseNotActiveResult(StorageId storageId) NN_NOEXCEPT
        {
            switch(storageId)
            {
            case StorageId::GameCard:
                NN_RESULT_THROW(ResultGameCardContentMetaDatabaseNotActive());
            case StorageId::BuiltInSystem:
                NN_RESULT_THROW(ResultBuiltInSystemContentMetaDatabaseNotActive());
            case StorageId::BuiltInUser:
                NN_RESULT_THROW(ResultBuiltInUserContentMetaDatabaseNotActive());
            case StorageId::SdCard:
                NN_RESULT_THROW(ResultSdCardContentMetaDatabaseNotActive());
            default:
                NN_RESULT_THROW(ResultUnknownContentMetaDatabaseNotActive());
            }
        }
        nn::Result GetContentStorageNotActiveResult(StorageId storageId) NN_NOEXCEPT
        {
            switch(storageId)
            {
            case StorageId::GameCard:
                NN_RESULT_THROW(ResultGameCardContentStorageNotActive());
            case StorageId::BuiltInSystem:
                NN_RESULT_THROW(ResultBuiltInSystemContentStorageNotActive());
            case StorageId::BuiltInUser:
                NN_RESULT_THROW(ResultBuiltInUserContentStorageNotActive());
            case StorageId::SdCard:
                NN_RESULT_THROW(ResultSdCardContentStorageNotActive());
            default:
                NN_RESULT_THROW(ResultUnknownContentStorageNotActive());
            }
        }
    }

    Result ContentManagerImpl::GetContentStorageRoot(ContentManagerImpl::ContentStorageRoot** outValue, StorageId id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(IsUniqueStorage(id), ResultUnknownStorage());

        for (auto& root : m_StorageRoot)
        {
            if (root.storageId == id)
            {
                *outValue = &root;
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_THROW(ResultUnknownStorage());
    }

    Result ContentManagerImpl::GetContentMetaDatabaseRoot(ContentManagerImpl::ContentMetaDatabaseRoot** outValue, StorageId id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(IsUniqueStorage(id), ResultUnknownStorage());

        for (auto& root : m_MetaRoot)
        {
            if (root.storageId == id)
            {
                *outValue = &root;
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_THROW(ResultUnknownStorage());
    }

    Result ContentManagerImpl::BuildContentMetaDataBase(StorageId storageId) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(storageId == StorageId::BuildInSystem, "BuildContentMetaDataBase support only BuildInSystem.");

        NN_RESULT_DO(BuildContentMetaDataBaseFromFile(storageId, fs::BisPartitionId::System, "cnmtdb.arc"));

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::BuildContentMetaDataBaseFromFile(StorageId storageId, fs::BisPartitionId bisPartitionId, const char* filePath) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        ContentMetaDatabaseRoot* root;
        NN_RESULT_DO(GetContentMetaDatabaseRoot(&root, storageId));

        const auto SystemPartitionMountName = detail::CreateUniqueMountName();

        PathString databasePathInSaveData;
        databasePathInSaveData.AssignFormat("%s/%s", root->path, "imkvdb.arc");

        PathString databasePathInSystemPartition;
        databasePathInSystemPartition.AssignFormat("%s:/%s", SystemPartitionMountName.string, filePath);

        NN_RESULT_DO(fs::MountBis(SystemPartitionMountName.string, bisPartitionId));
        NN_UTIL_SCOPE_EXIT
        {
            fs::Unmount(SystemPartitionMountName.string);
        };

        NN_RESULT_DO(fs::MountSystemSaveData(root->mountName, root->info.spaceId, root->info.id));
        NN_UTIL_SCOPE_EXIT
        {
            fs::Unmount(root->mountName);
        };

        NN_RESULT_DO(
            detail::EnsureDirectoryRecursively(root->path));

        NN_RESULT_DO(
            detail::CopyFile(databasePathInSaveData, databasePathInSystemPartition));

        NN_RESULT_DO(
            fs::CommitSaveData(root->mountName));

        NN_RESULT_SUCCESS;
    }


    Result ContentManagerImpl::InitializeContentStorageRoot(ContentStorageRoot* outValue, StorageId storageId, fs::ContentStorageId contentStorageId) NN_NOEXCEPT
    {
        outValue->storageId = storageId;
        outValue->contentStorageId = contentStorageId;
        outValue->storage = nullptr;

        std::strcpy(outValue->mountName, detail::CreateUniqueMountName().string);
        util::SNPrintf(outValue->path, sizeof(outValue->path), "%s:/", outValue->mountName);

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::InitializeGameCardContentStorageRoot(ContentStorageRoot* outValue) NN_NOEXCEPT
    {
        outValue->storageId = StorageId::Card;
        outValue->storage = nullptr;

        std::strcpy(outValue->mountName, detail::CreateUniqueMountName().string);
        util::SNPrintf(outValue->path, sizeof(outValue->path), "%s:", outValue->mountName);

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::InitializeContentMetaDatabaseRoot(
        ContentMetaDatabaseRoot* outValue,
        StorageId storageId,
        const SystemSaveDataInfo& info,
        int maxContentMetaCount,
        ContentMetaMemoryResource* memoryResource) NN_NOEXCEPT
    {
        outValue->storageId = storageId;
        outValue->info = info;
        outValue->maxContentMetaCount = maxContentMetaCount;
        outValue->memoryResource = memoryResource;
        outValue->db = nullptr;
        outValue->kvs = util::nullopt;

        std::strcpy(outValue->mountName, detail::CreateUniqueMountName().string);
        outValue->mountName[0] = '#'; // マウント名の頭文字に '@' は使えないので書き換える
        util::SNPrintf(outValue->path, sizeof(outValue->path), "%s:/meta", outValue->mountName);

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::InitializeGameCardContentMetaDatabaseRoot(ContentMetaDatabaseRoot* outValue, int maxContentMetaCount, ContentMetaMemoryResource* memoryResource) NN_NOEXCEPT
    {
        outValue->storageId = StorageId::Card;
        outValue->maxContentMetaCount = maxContentMetaCount;
        outValue->memoryResource = memoryResource;
        outValue->db = nullptr;
        outValue->kvs = util::nullopt;

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::Initialize() NN_NOEXCEPT
    {
        ContentManagerConfig config = {};
        NN_RESULT_DO(Initialize(config));
        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::Initialize(const ContentManagerConfig &config) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        if (m_IsInitialized)
        {
            NN_RESULT_SUCCESS;
        }

        for (auto& root : m_StorageRoot)
        {
            root.storageId = StorageId::None;
        }

        for (auto& root : m_MetaRoot)
        {
            root.storageId = StorageId::None;
        }

        // BuildInSystem
        NN_RESULT_DO(InitializeContentStorageRoot(&m_StorageRoot[m_StorageRootIndex++], StorageId::BuildInSystem, fs::ContentStorageId::System));
        auto result = VerifyContentStorage(StorageId::BuildInSystem);
        if (result.IsFailure())
        {
            NN_SDK_LOG("[NCM] Invalid BuildInSystem content storage as 0x%08x. Create.\n", result.GetInnerValueForDebug());
            NN_RESULT_DO(CreateContentStorage(StorageId::BuildInSystem));
        }
        NN_RESULT_DO(ActivateContentStorage(StorageId::BuildInSystem));

        const SystemSaveDataInfo buildInSystemSaveData =
        {
            BuildInSystemSystemSaveDataId,
            BuildInSystemSystemSaveDataSize,
            BuildInSystemSystemSaveDataJournalSize,
            BuildInSystemSystemSaveDataFlag,
            fs::SaveDataSpaceId::System
        };
        NN_RESULT_DO(InitializeContentMetaDatabaseRoot(
            &m_MetaRoot[m_MetaRootIndex++], StorageId::BuildInSystem, buildInSystemSaveData,
            SystemMaxContentMetaCount, &g_SystemContentMetaMemoryResource));
        result = VerifyContentMetaDatabase(StorageId::BuildInSystem);
        if (result.IsFailure())
        {
            NN_SDK_LOG("[NCM] Invalid BuildInSystem content meta database as 0x%08x. Create.\n", result.GetInnerValueForDebug());

            NN_SDK_LOG("[NCM] EnableBuildingContentMetaDatabase:%d\n", config.EnableBuildingContentMetaDatabase);

            NN_RESULT_DO(CreateContentMetaDatabase(StorageId::BuildInSystem));

            if (config.EnableBuildingContentMetaDatabase)
            {
                NN_RESULT_DO(BuildContentMetaDataBase(StorageId::BuildInSystem));
                NN_RESULT_DO(VerifyContentMetaDatabase(StorageId::BuildInSystem));
            }

            if (config.EnableImportContentMetaDatabaseFromSdCard)
            {
                // セーフモード用の ncm (ncm.for-safemode) で、
                // かつ、FS が SD カード上の SignedSystemPartition をマウントしている場合のみ
                // 予め作成しておいたコンテンツメタデータベースをインポートする
                if (fs::IsSignedSystemPartitionOnSdCardValid())
                {
                    NN_RESULT_DO(BuildContentMetaDataBaseFromFile(StorageId::BuildInSystem, fs::BisPartitionId::System, "cnmtdb.arc"));
                }
            }
        }

        result = EnsureBuildInSystemSaveDataFlag();
        if (result.IsFailure())
        {
            // InitializeSdev 時に system パーティションが壊れている場合があるため、失敗時は無視する
            NN_SDK_LOG("[NCM] Failed to ensure BuildInSystem save data flag as 0x%08x. (ignored)\n", result.GetInnerValueForDebug());
        }

        NN_RESULT_DO(ActivateContentMetaDatabase(StorageId::BuildInSystem));

        // BuildInUser
        NN_RESULT_DO(InitializeContentStorageRoot(&m_StorageRoot[m_StorageRootIndex++], StorageId::BuildInUser, fs::ContentStorageId::User));

        const SystemSaveDataInfo buildInUserSaveData =
        {
            BuildInUserSystemSaveDataId,
            BuildInUserSystemSaveDataSize,
            BuildInUserSystemSaveDataJournalSize,
            BuildInUserSystemSaveDataFlag,
            fs::SaveDataSpaceId::System
        };
        NN_RESULT_DO(InitializeContentMetaDatabaseRoot(
            &m_MetaRoot[m_MetaRootIndex++], StorageId::BuildInUser, buildInUserSaveData,
            UserMaxContentMetaCount, &g_UserContentMetaMemoryResource));

        // SdCard
        NN_RESULT_DO(InitializeContentStorageRoot(&m_StorageRoot[2], StorageId::SdCard, fs::ContentStorageId::SdCard));

        const SystemSaveDataInfo sdCardSaveData =
        {
            SdCardSystemSaveDataId,
            SdCardSystemSaveDataSize,
            SdCardSystemSaveDataJournalSize,
            SdCardSystemSaveDataFlag,
            fs::SaveDataSpaceId::SdSystem
        };
        NN_RESULT_DO(InitializeContentMetaDatabaseRoot(
            &m_MetaRoot[2], ncm::StorageId::SdCard, sdCardSaveData,
            SdCardMaxContentMetaCount, &g_UserContentMetaMemoryResource));

        // GameCard
        // TODO: 初期化タイミング検討
        NN_RESULT_DO(InitializeGameCardContentStorageRoot(&m_StorageRoot[3]));
        NN_RESULT_DO(InitializeGameCardContentMetaDatabaseRoot(&m_MetaRoot[3], CardMaxContentMetaCount, &g_CardContentMetaMemoryResource));

        m_IsInitialized = true;
        NN_RESULT_SUCCESS;
    }

    ContentManagerImpl::~ContentManagerImpl() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        for (auto& root : m_StorageRoot)
        {
            InactivateContentStorage(root.storageId);
        }

        for (auto& root : m_MetaRoot)
        {
            InactivateContentMetaDatabase(root.storageId);
        }
    }

    Result ContentManagerImpl::CreateContentStorage(StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        ContentStorageRoot* root;
        NN_RESULT_DO(GetContentStorageRoot(&root, storageId));
        NN_RESULT_DO(fs::MountContentStorage(root->mountName, root->contentStorageId));
        NN_UTIL_SCOPE_EXIT
        {
            fs::Unmount(root->mountName);
        };

        NN_RESULT_DO(detail::EnsureDirectoryRecursively(root->path));
        NN_RESULT_DO(ContentStorageImpl::InitializeBase(root->path));

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::CreateContentMetaDatabase(StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        NN_RESULT_THROW_UNLESS(storageId != StorageId::Card, ResultUnknownStorage());

        ContentMetaDatabaseRoot* root;
        NN_RESULT_DO(GetContentMetaDatabaseRoot(&root, storageId));

        NN_RESULT_DO(EnsureAndMountSystemSaveData(root->mountName, root->info));
        NN_UTIL_SCOPE_EXIT
        {
            fs::Unmount(root->mountName);
        };

        NN_RESULT_DO(detail::EnsureDirectoryRecursively(root->path));

        NN_RESULT_DO(fs::CommitSaveData(root->mountName));

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::VerifyContentStorage(StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        ContentStorageRoot* root;
        NN_RESULT_DO(GetContentStorageRoot(&root, storageId));

        char path[128];
        auto mountName = detail::CreateUniqueMountName();
        ReplaceMountName(path, mountName.string, root->path);

        NN_RESULT_DO(fs::MountContentStorage(mountName.string, root->contentStorageId));
        NN_UTIL_SCOPE_EXIT
        {
            fs::Unmount(mountName.string);
        };
        NN_RESULT_DO(ContentStorageImpl::VerifyBase(path));

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::VerifyContentMetaDatabase(StorageId storageId) NN_NOEXCEPT
    {
        if (storageId == StorageId::Card)
        {
            // ゲームカードはオンメモリで必ず正しいので、いつも成功を返す
            NN_RESULT_SUCCESS;
        }

        std::lock_guard<os::Mutex> locker(m_Mutex);

        ContentMetaDatabaseRoot* root;
        NN_RESULT_DO(GetContentMetaDatabaseRoot(&root, storageId));

        bool doUnmount = false;

        // マウントした状態でマウント関数を呼ぶことは禁止されている
        // 本 API 内でマウントした場合は、アンマウントするようにして、 API 呼び出し前と同じ状態に戻せるようにする
        if (!root->db)
        {
            NN_RESULT_DO(fs::MountSystemSaveData(root->mountName, root->info.spaceId, root->info.id));
            doUnmount = true;
        }
        NN_UTIL_SCOPE_EXIT
        {
            if (doUnmount)
            {
                fs::Unmount(root->mountName);
            }
        };

        bool hasDirectory;
        NN_RESULT_DO(detail::HasDirectory(&hasDirectory, root->path));
        NN_RESULT_THROW_UNLESS(hasDirectory, ResultInvalidContentMetaDatabase());

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::OpenContentStorage(sf::Out<sf::SharedPointer<IContentStorage>> outValue, StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        ContentStorageRoot* root;
        NN_RESULT_DO(GetContentStorageRoot(&root, storageId));
        NN_RESULT_THROW_UNLESS(root->storage, GetContentStorageNotActiveResult(storageId));

        *outValue = root->storage;

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::OpenContentMetaDatabase(sf::Out<sf::SharedPointer<IContentMetaDatabase>> outValue, StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        ContentMetaDatabaseRoot* root;
        NN_RESULT_DO(GetContentMetaDatabaseRoot(&root, storageId));
        NN_RESULT_THROW_UNLESS(root->db, GetContentMetaDatabaseNotActiveResult(storageId));

        *outValue = root->db;

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::CleanupContentMetaDatabase(StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        NN_RESULT_DO(InactivateContentMetaDatabase(storageId));

        ContentMetaDatabaseRoot* root;
        NN_RESULT_DO(GetContentMetaDatabaseRoot(&root, storageId));

        NN_RESULT_DO(fs::DeleteSaveData(root->info.spaceId, root->info.id));

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::EnsureAndMountSystemSaveData(const char* mountName, const SystemSaveDataInfo& info) const NN_NOEXCEPT
    {
        const uint64_t saveDataOwnerId = 0;
        fs::DisableAutoSaveDataCreation();
        NN_RESULT_TRY(fs::MountSystemSaveData(mountName, info.spaceId, info.id))
            NN_RESULT_CATCH(fs::ResultTargetNotFound)
            {
                NN_RESULT_DO(fs::CreateSystemSaveData(info.spaceId, info.id, saveDataOwnerId, info.size, info.journalSize, info.flag));
                NN_RESULT_DO(fs::MountSystemSaveData(mountName, info.spaceId, info.id));
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::ActivateContentStorage(StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        ContentStorageRoot* root;
        NN_RESULT_DO(GetContentStorageRoot(&root, storageId));

        if(! root->storage)
        {
            if (storageId == nn::ncm::StorageId::Card)
            {
                fs::GameCardHandle handle;
                NN_RESULT_DO(fs::GetGameCardHandle(&handle));
                NN_RESULT_DO(fs::MountGameCardPartition(root->mountName, handle, fs::GameCardPartition::Secure));
            }
            else
            {
                NN_RESULT_DO(fs::MountContentStorage(root->mountName, root->contentStorageId));
            }
            bool isSuccess = false;
            NN_UTIL_SCOPE_EXIT
            {
                if (!isSuccess)
                {
                    fs::Unmount(root->mountName);
                }
            };

            if (storageId == nn::ncm::StorageId::Card)
            {
                auto contentStorage = sf::CreateSharedObjectEmplaced<IContentStorage, ReadOnlyContentStorageImpl>();
                NN_RESULT_DO(contentStorage.GetImpl().Initialize(root->path, MakeFlatContentFilePath));
                root->storage = contentStorage;
            }
            else if (storageId == nn::ncm::StorageId::BuildInSystem)
            {
                auto contentStorage = sf::CreateSharedObjectEmplaced<IContentStorage, ContentStorageImpl>();
                NN_RESULT_DO(contentStorage.GetImpl().Initialize(root->path, MakeFlatContentFilePath, MakeFlatPlaceHolderFilePath, false));
                root->storage = contentStorage;
            }
            else if (storageId == nn::ncm::StorageId::SdCard)
            {
                auto contentStorage = sf::CreateSharedObjectEmplaced<IContentStorage, ContentStorageImpl>();
                NN_RESULT_DO(contentStorage.GetImpl().Initialize(root->path, MakeSha256HierarchicalContentFilePath_ForFat16KCluster, MakeSha256HierarchicalPlaceHolderFilePath_ForFat16KCluster, true));
                root->storage = contentStorage;
            }
            else
            {
                auto contentStorage = sf::CreateSharedObjectEmplaced<IContentStorage, ContentStorageImpl>();
                NN_RESULT_DO(contentStorage.GetImpl().Initialize(root->path, MakeSha256HierarchicalContentFilePath_ForFat16KCluster, MakeSha256HierarchicalPlaceHolderFilePath_ForFat16KCluster, false));
                root->storage = contentStorage;
            }

            isSuccess = true;
        }

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::InactivateContentStorage(StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        ContentStorageRoot* root;
        NN_RESULT_DO(GetContentStorageRoot(&root, storageId));

        if (root->storage)
        {
            root->storage->DisableForcibly();
            root->storage = nullptr;
            fs::Unmount(root->mountName);
        }

        NN_RESULT_SUCCESS;

    }

    Result ContentManagerImpl::ActivateContentMetaDatabase(StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        ContentMetaDatabaseRoot* root;
        NN_RESULT_DO(GetContentMetaDatabaseRoot(&root, storageId));

        if(! root->db)
        {
            root->kvs.emplace();
            if (storageId == nn::ncm::StorageId::Card)
            {
                NN_RESULT_DO(root->kvs->Initialize(root->maxContentMetaCount, root->memoryResource->Get()));
                root->db = sf::CreateSharedObjectEmplaced<IContentMetaDatabase, OnMemoryContentMetaDatabaseImpl>(&(*root->kvs));
            }
            else
            {
                NN_RESULT_DO(fs::MountSystemSaveData(root->mountName, root->info.spaceId, root->info.id));
                bool isSuccess = false;
                NN_UTIL_SCOPE_EXIT
                {
                    if (!isSuccess)
                    {
                        fs::Unmount(root->mountName);
                    }
                };

                NN_RESULT_DO(root->kvs->Initialize(root->path, root->maxContentMetaCount, root->memoryResource->Get()));
                NN_RESULT_DO(root->kvs->Load());
                root->db = sf::CreateSharedObjectEmplaced<IContentMetaDatabase, ContentMetaDatabaseImpl>(&(*root->kvs), root->mountName);
                isSuccess = true;
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerImpl::InactivateContentMetaDatabase(StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        ContentMetaDatabaseRoot* root;
        NN_RESULT_DO(GetContentMetaDatabaseRoot(&root, storageId));

        if (root->db)
        {
            root->db->DisableForcibly();
            root->db = nullptr;
            root->kvs = util::nullopt;
            if (storageId != StorageId::Card)
            {
                fs::Unmount(root->mountName);
            }
        }

        NN_RESULT_SUCCESS;
    }

}}
