﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ncm/ncm_ContentInfo.h>
#include <nn/ncm/ncm_ContentInfoUtil.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>

namespace nn { namespace ncm {
namespace {
    // SD カードに保存するファイルを暗号化に使われるファイルヘッダサイズ
    const int64_t AdditionalSizeForEncryption = 16 * 1024;

    // 連結ファイルを構成する実ファイルの最大サイズ (4GiB)
    const int64_t ConcatenationFileActualSizeMax = 4 * 1024 * 1024 * 1024ll;

    int64_t CalculateAdditionalContentSize(int64_t fileSize, int64_t clusterSize) NN_NOEXCEPT
    {
        int64_t size = 0;
        size += AdditionalSizeForEncryption;
        // クラスタサイズ分の下駄をはかせる。512GB までの SD カードのクラスタサイズは最大 256KiB
        size += ((fileSize / ConcatenationFileActualSizeMax) + 1) * clusterSize; // 4GiB 越えのファイル分割の際発生する領域
        size += clusterSize; // nca が階層化されるので、その分のディレクトリエントリ
        size += clusterSize; // Registered が階層化されるので、その分のディレクトリエントリ
        size += clusterSize; // Placeholder が階層化されるので、その分のディレクトリエントリ

        return size;
    }

    template <class FuncT>
    Result ForEachContentInfo(const ContentMetaKey& key, const ContentMetaDatabase* pDb, FuncT func) NN_NOEXCEPT
    {
        const int NumInfo = 16;
        ncm::ContentInfo infoList[NumInfo];
        int offset = 0;
        while (NN_STATIC_CONDITION(true))
        {
            int count;
            NN_RESULT_DO(pDb->ListContentInfo(&count, infoList, NumInfo, key, offset));
            for (int i = 0; i < count; i++)
            {
                bool isBreak = false;
                NN_RESULT_DO(func(&isBreak, infoList[i]));
                if (isBreak)
                {
                    break;
                }
            }
            if (count != NumInfo)
            {
                break;
            }
            offset += count;
        }

        NN_RESULT_SUCCESS;
    }
}

    int64_t CalculateRequiredSize(int64_t fileSize, int64_t clusterSize) NN_NOEXCEPT
    {
        int64_t requiredSize = fileSize;
        requiredSize += CalculateAdditionalContentSize(fileSize, clusterSize);
        return requiredSize;
    }

    int64_t CalculateRequiredSizeForExtension(int64_t fileExtensionSize, int64_t clusterSize) NN_NOEXCEPT
    {
        int64_t requiredSize = fileExtensionSize;
        requiredSize += ((fileExtensionSize / ConcatenationFileActualSizeMax) + 1) * clusterSize;
        return requiredSize;
    }

    Result EstimateRequiredSize(int64_t* outValue, const ncm::ContentMetaKey& key, const ContentMetaDatabase* pDb) NN_NOEXCEPT
    {
        int64_t size = 0;
        NN_RESULT_DO(ForEachContentInfo(key, pDb, [&size](bool* outStopForEach, const ncm::ContentInfo& info) -> Result
            {
                size += ncm::CalculateRequiredSize(info.GetSize(), MaxClusterSize);
                *outStopForEach = false;
                NN_RESULT_SUCCESS;
            }));
        *outValue = size;
        NN_RESULT_SUCCESS;
    }
}}
