﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <mutex>

#include <nn/nn_SdkLog.h>
#include <nn/util/util_Optional.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>

#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_ContentLocationResolverImpl.h>
#include <nn/lr/lr_RegisteredLocationResolverImpl.h>
#include <nn/lr/lr_RedirectOnlyLocationResolverImpl.h>
#include <nn/lr/lr_LocationResolverManagerImpl.h>
#include <nn/lr/lr_AddOnContentLocationResolverImpl.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_StdAllocationPolicy.h>
#include <new>

namespace nn { namespace lr {

namespace {
    typedef sf::ObjectFactory<nn::sf::StdAllocationPolicy<std::allocator>> ContentLocationResolverFactory;
    typedef sf::ObjectFactory<nn::sf::StdAllocationPolicy<std::allocator>> RedirectOnlyLocationResolverFactory;
}

    Result LocationResolverManagerImpl::OpenLocationResolver(sf::Out<sf::SharedPointer<ILocationResolver>> outValue, Bit8 storageId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        auto resolver = m_LocationResolverMap.Find(storageId);
        if(! resolver)
        {
            if(storageId == static_cast<Bit8>(ncm::StorageId::Host))
            {
                m_LocationResolverMap[storageId] = RedirectOnlyLocationResolverFactory::CreateSharedEmplaced<ILocationResolver, RedirectOnlyLocationResolverImpl>();
            }
            else
            {
                auto tmpResolver = ContentLocationResolverFactory::CreateSharedEmplaced<ILocationResolver, ContentLocationResolverImpl>(static_cast<ncm::StorageId>(storageId));
                NN_RESULT_DO(tmpResolver->Refresh());
                m_LocationResolverMap[storageId] = std::move(tmpResolver);
            }
            resolver = m_LocationResolverMap.Find(storageId);
        }

        *outValue = *resolver;

        NN_RESULT_SUCCESS;
    }
    Result LocationResolverManagerImpl::OpenRegisteredLocationResolver(sf::Out<sf::SharedPointer<IRegisteredLocationResolver>> outValue) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);
        if(m_RegisteredLocationResolver == nullptr)
        {
            m_RegisteredLocationResolver = ContentLocationResolverFactory::CreateSharedEmplaced<IRegisteredLocationResolver, RegisteredLocationResolverImpl>();
        }
        *outValue = m_RegisteredLocationResolver;

        NN_RESULT_SUCCESS;
    }

    Result LocationResolverManagerImpl::RefreshLocationResolver(Bit8 storageId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);

        auto resolver = m_LocationResolverMap.Find(storageId);
        NN_RESULT_THROW_UNLESS(resolver, ResultUnknownResolver());

        if(storageId != static_cast<Bit8>(ncm::StorageId::Host))
        {
            NN_RESULT_DO((*resolver)->Refresh());
        }

        NN_RESULT_SUCCESS;
    }

    Result LocationResolverManagerImpl::OpenAddOnContentLocationResolver(sf::Out<sf::SharedPointer<IAddOnContentLocationResolver>> outValue) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> locker(m_Mutex);
        if (m_AddOnContentLocationResolver == nullptr)
        {
            m_AddOnContentLocationResolver = ContentLocationResolverFactory::CreateSharedEmplaced<IAddOnContentLocationResolver, AddOnContentLocationResolverImpl>();
        }
        *outValue = m_AddOnContentLocationResolver;

        NN_RESULT_SUCCESS;
    }
}}
