﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_Result.h>

#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_ContentLocationResolverImpl.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>

namespace nn { namespace lr {

    Result ContentLocationResolverImpl::ResolveProgramPath(sf::Out<Path> outValue, ncm::ProgramId id) NN_NOEXCEPT
    {
        if (GetRedirectedPath(&outValue, &m_ProgramRedirector, id))
        {
            NN_RESULT_SUCCESS;
        }

        ncm::ContentId contentId;
        NN_RESULT_TRY(m_Meta.GetLatestProgram(&contentId, id))
            NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
            {
                NN_RESULT_THROW(ResultProgramNotFound());
            }
        NN_RESULT_END_TRY

        GetContentPath(&outValue, contentId);

        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::RedirectProgramPath(ncm::ProgramId id, const Path& path) NN_NOEXCEPT
    {
        m_ProgramRedirector.SetRedirection(id, path);

        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::RedirectApplicationProgramPath(ncm::ProgramId id, const Path& path) NN_NOEXCEPT
    {
        m_ProgramRedirector.SetRedirection(id, path, static_cast<int>(RedirectionFlag::Application));

        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::ResolveDataPath(sf::Out<lr::Path> outValue, ncm::DataId id) NN_NOEXCEPT
    {
        ncm::ContentId contentId;
        NN_RESULT_DO(m_Meta.GetLatestData(&contentId, id));

        GetContentPath(&outValue, contentId);

        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::ResolveApplicationControlPath(sf::Out<lr::Path> outValue, ncm::ProgramId id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(GetRedirectedPath(&outValue, &m_ControlRedirector, id), ResultControlNotFound());
        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::RedirectApplicationControlPath(ncm::ProgramId id, const Path& path) NN_NOEXCEPT
    {
        m_ControlRedirector.SetRedirection(id, path, static_cast<int>(RedirectionFlag::Application));

        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::ResolveApplicationHtmlDocumentPath(sf::Out<lr::Path> outValue, ncm::ProgramId id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(GetRedirectedPath(&outValue, &m_HtmlDocumentRedirector, id), ResultHtmlDocumentNotFound());
        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::RedirectApplicationHtmlDocumentPath(ncm::ProgramId id, const Path& path) NN_NOEXCEPT
    {
        m_HtmlDocumentRedirector.SetRedirection(id, path, static_cast<int>(RedirectionFlag::Application));

        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::ResolveApplicationLegalInformationPath(sf::Out<lr::Path> outValue, ncm::ProgramId id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(GetRedirectedPath(&outValue, &m_LegalInformationRedirector, id), ResultLegalInformationNotFound());
        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::RedirectApplicationLegalInformationPath(ncm::ProgramId id, const Path& path) NN_NOEXCEPT
    {
        m_LegalInformationRedirector.SetRedirection(id, path, static_cast<int>(RedirectionFlag::Application));

        NN_RESULT_SUCCESS;
    }

    void ContentLocationResolverImpl::GetContentPath(sf::Out<Path>* outValue, ncm::ContentId contentId) NN_NOEXCEPT
    {
        ncm::Path ncmPath;
        m_Storage.GetPath(&ncmPath, contentId);

        Path path;
        util::Strlcpy(path.string, ncmPath.string, sizeof(path.string));

        outValue->Set(path);
    }

    bool ContentLocationResolverImpl::GetRedirectedPath(sf::Out<Path>* outValue, LocationRedirector* redirector, ncm::ProgramId id) NN_NOEXCEPT
    {
        Path path;
        if (redirector->FindRedirection(&path, id))
        {
            outValue->Set(path);
            return true;
        }
        return false;
    }

    Result ContentLocationResolverImpl::Refresh() NN_NOEXCEPT
    {
        ncm::ContentMetaDatabase meta;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&meta, m_StorageId));
        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, m_StorageId));

        m_Meta = std::move(meta);
        m_Storage = std::move(storage);

        ClearRedirection(static_cast<int>(RedirectionFlag::None));

        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::ClearApplicationRedirection() NN_NOEXCEPT
    {
        ClearRedirection(static_cast<int>(RedirectionFlag::Application));
        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::EraseProgramRedirection(ncm::ProgramId id) NN_NOEXCEPT
    {
        m_ProgramRedirector.EraseRedirection(id);
        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::EraseApplicationControlRedirection(ncm::ProgramId id) NN_NOEXCEPT
    {
        m_ControlRedirector.EraseRedirection(id);
        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::EraseApplicationHtmlDocumentRedirection(ncm::ProgramId id) NN_NOEXCEPT
    {
        m_HtmlDocumentRedirector.EraseRedirection(id);
        NN_RESULT_SUCCESS;
    }

    Result ContentLocationResolverImpl::EraseApplicationLegalInformationRedirection(ncm::ProgramId id) NN_NOEXCEPT
    {
        m_LegalInformationRedirector.EraseRedirection(id);
        NN_RESULT_SUCCESS;
    }

    void ContentLocationResolverImpl::ClearRedirection(int flag) NN_NOEXCEPT
    {
        m_ProgramRedirector.ClearRedirection(flag);
        m_ControlRedirector.ClearRedirection(flag);
        m_HtmlDocumentRedirector.ClearRedirection(flag);
        m_LegalInformationRedirector.ClearRedirection(flag);
    }
}}
