﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_FrameHeap.h>

namespace nn { namespace lmem { namespace detail {

/**
 * @brief       フレームヒープのフリーエリアの先頭アドレスを取得します。
 * @param[in]   heap    フレームヒープのハンドル。
 * @return      フレームヒープのフリーエリアの先頭アドレスを返します。
 */
void* GetNextBlockFront( HeapHandle heap );

/**
 * @brief       フレームヒープのフリーエリアの末尾アドレスを取得します。
 * @param[in]   heap    フレームヒープのハンドル
 * @return      フレームヒープのフリーエリアの末尾アドレス +1 を返します。
 */
void* GetNextBlockRear( HeapHandle heap );

/* =======================================================================
     公開関数
   ======================================================================== */

/**
 * @brief       フレームヒープを作成します。
 * @param[in]   startAddress    ヒープ領域の先頭アドレス
 * @param[in]   size            ヒープ領域のサイズ
 * @param[in]   optFlag         オプションフラグ
 * @return      関数が成功した場合、作成されたフレームヒープのハンドルが返ります。@n
 *              関数が失敗すると、nullptr が返ります。
 * @details     ヒープの管理領域（HeapHead）はヒープ内部に作成されます。
                スレッドセーフではありません。@n
                スレッドセーフにする場合、ヒープの属性を指定する引数を追加するようにするか、あるいは、属性をセットする関数で制御してもらうか、何かしらの対策が必要です。
 */
HeapHandle CreateFrameHeap( void* startAddress, size_t size, int option ) NN_NOEXCEPT;

/**
 * @brief       フレームヒープを作成します。
 * @param[in]   startAddress    ヒープ領域の先頭アドレス
 * @param[in]   size            ヒープ領域のサイズ
 * @param[in]   option          オプションフラグ
 * @param[in]   pHeapHead       ヒープ共通ヘッダ
 * @return      関数が成功した場合、作成されたフレームヒープのハンドルが返ります。@n
 *              関数が失敗すると、nullptr が返ります。
 * @details     ヒープの管理領域（HeapHead）はヒープ外部に作成され、引数で与えられます。
                スレッドセーフではありません。@n
                スレッドセーフにする場合、ヒープの属性を指定する引数を追加するようにするか、あるいは、属性をセットする関数で制御してもらうか、何かしらの対策が必要です。
 */
HeapHandle CreateFrameHeap( void* startAddress, size_t size, int option, HeapCommonHead* pHeapHead ) NN_NOEXCEPT;

/**
 * @brief       フレームヒープを破棄します。
 * @param[in]   heap    フレームヒープのハンドル。
 * @return      破棄したヒープが占めていた領域へのポインタを返します。
 */
void DestroyFrameHeap( HeapHandle heap ) NN_NOEXCEPT;

/**
 * @brief       フレームヒープからメモリブロックを確保します。
 * @param[in]   heap        フレームヒープのハンドル
 * @param[in]   size        確保するメモリブロックのサイズ(バイト単位)
 * @param[in]   alignment   確保するメモリブロックのアライメント@n
                            正または負の 2 のべき乗が指定できます@n
 * @return      メモリブロックの確保が成功した場合、確保したメモリブロックへのポインタが返ります。@n
                失敗した場合、NULLが返ります。
 */
void* AllocFromFrameHeap( HeapHandle heap, size_t size, int alignment ) NN_NOEXCEPT;

/**
 * @brief       フレームヒープへメモリブロックを返却します。
 * @param[in]   heap    フレームヒープのハンドル
 * @param[in]   mode    メモリブロックの返却方法
 */
void FreeToFrameHeap( HeapHandle heap, FreeMode mode ) NN_NOEXCEPT;

/**
 * @brief       フレームヒープ内の割り当て可能な最大サイズを取得します。
 * @param[in]   heap        フレームヒープのハンドル
 * @param[in]   alignment   確保するメモリブロックのアライメント@n
                            正または負の 2 のべき乗が指定できます@n
 * @return      フレームヒープ内の割り当て可能な最大サイズを返します(バイト単位)。
 */
size_t GetAllocatableSizeForFrameHeap( HeapHandle heap, int alignment ) NN_NOEXCEPT;

/**
 * @brief       フレームヒープの現在のメモリ使用状態を記録します。@n
                後で記録したメモリ使用状況に戻すことができます。
 * @param[in]   heap    フレームヒープのハンドル
 * @return      フレームヒープの状態が返ります。
 */
FrameHeapState GetFrameHeapState( HeapHandle heap ) NN_NOEXCEPT;

/**
 * @brief       フレームヒープのメモリブロックを記録された状態に従って返却します。
 * @param[in]   heap:     フレームヒープのハンドル。
 * @param[in]   state:  記録した状態
 */
void RestoreFrameHeapState( HeapHandle heap, const FrameHeapState& pState ) NN_NOEXCEPT;

/**
 * @brief       フレームヒープの空き領域をヒープ領域から開放し、その分ヒープ領域を縮小します。
 * @param[in]   heap        フレームヒープのハンドル
 * @param[in]   adjustMode  縮小する方向
 * @return      関数が成功した場合、縮小した領域を MemoryRange 型で返します。@n
                失敗した場合、 MemoryRange::size が 0 の MemoryRange 型を返します。
 */
MemoryRange AdjustFrameHeap( HeapHandle heap, AdjustMode adjustMode ) NN_NOEXCEPT;

/**
 * @brief       フレームヒープから確保されたメモリブロックのサイズを変更します。
 * @param[in]   heap        フレームヒープのハンドル。
 * @param[in]   pBlock      サイズを変更するメモリブロックへのポインタ
 * @param[in]   newSize     新しく割り当てるサイズ(バイト単位)。@n
                            4未満の値を指定された場合は、4が指定されたものとして処理します
 * @return      関数が成功した場合、変更されたメモリブロックのサイズを返します(バイト単位)。
                関数が失敗した場合、0 が返ります。
 */
size_t ResizeForMemoryBlockFrameHeap( HeapHandle heap, void* pBlock, size_t newSize ) NN_NOEXCEPT;

/**
 * @brief       フレームヒープ内部の情報を表示します。
 * @param[in]   heap    フレームヒープのハンドル
 */
void DumpFrameHeap( HeapHandle heap );

}}} // nn::mem::detail
