﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief ヒープ共通の処理の内部実装について定義します。
 */

#include <nn/lmem/lmem_Common.h>

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

// デバッグ時のみ有効
#define NN_DETAIL_HEAP_DEBUG

#endif

namespace nn { namespace lmem { namespace detail {

#define NN_DETAIL_CREATE_HEAP_SIGNATURE(a,b,c,d) \
    ( ((a & 0xFF) << 24) | ((b & 0xFF) << 16) | ((c & 0xFF) << 8) | ((d & 0xFF) << 0) )

// 拡張ヒープのシグネチャ
const int MemExpHeapSignature = NN_DETAIL_CREATE_HEAP_SIGNATURE('E','X','P','H');

// フレームヒープのシグネチャ
const int MemFrameHeapSignature = NN_DETAIL_CREATE_HEAP_SIGNATURE('F','R','M','H');

// ユニットヒープのシグネチャ
const int MemUnitHeapSignature = NN_DETAIL_CREATE_HEAP_SIGNATURE('U','N','T','H');

/**
 * @brief メモリ毎に呼び出されるコールバック関数の型です。
 */
typedef void (*HeapVisitor)(void* pMemBlock, HeapHandle heap, uintptr_t userParam);

/**
 * @brief   ヒープを Scoped Lock するためのクラスです。
 * @details オプションでロックするかどうかを判定します。
 */
class HeapLocker
{
private:
    HeapHandle m_HeapHandle;
public:
    explicit HeapLocker(HeapHandle heapHandle) NN_NOEXCEPT
        : m_HeapHandle(heapHandle)
    {
        if(m_HeapHandle->attribute & CreationOption_ThreadSafe)
        {
            nn::os::LockMutex(&(m_HeapHandle->mutex));
        }
    }
    ~HeapLocker() NN_NOEXCEPT
    {
        if(m_HeapHandle->attribute & CreationOption_ThreadSafe)
        {
            nn::os::UnlockMutex(&(m_HeapHandle->mutex));
        }
    }
};

/**
 * @brief       ヒープヘッダの初期化を行います。
 * @param[out]  pOutHeader  ヒープヘッダへのポインタ
 * @param[in]   signature   シグネチャ
 * @param[in]   heapStart   ヒープメモリの開始アドレス
 * @param[in]   heapEnd     ヒープメモリの終了アドレス + 1
 * @param[in]   option      ヒープオプション
 */
void InitHeapHead(HeapHead* pOutHeader,
                  uint32_t  signature,
                  void*     heapStart,
                  void*     heapEnd,
                  int  option) NN_NOEXCEPT;

/**
 * @brief       ヒープ共通の後始末を行います。
 * @param[in]   pHeader ヒープヘッダへのポインタ
 */
void FinalizeHeap(HeapHead* pHeader) NN_NOEXCEPT;

/**
 * @brief       ヒープヘッダの情報を表示します。
 * @param[in]   pHeader ヒープヘッダへのポインタ
 */
void DumpHeapHead(HeapHead const* pHeapHead) NN_NOEXCEPT;

/**
 * @brief MemoryRange 構造体を作成し、返却します。
 */
inline MemoryRange MakeMemoryRange(void* address, size_t size) NN_NOEXCEPT
{
    MemoryRange memoryRange;
    memoryRange.begin = reinterpret_cast<uintptr_t>(address);
    memoryRange.size = size;
    return memoryRange;
}

/* ========================================================================
    外部公開用関数
   ======================================================================== */

/**
 * @brief       ヒープの作成時やメモリブロックの確保・解放時にメモリにセットする値をセットします。
 * @param[in]   type    取得する値の種類
 * @param[in]   value   セットする値
 * @return      以前の、メモリブロックの確保時にメモリにセットする値を返します。
 * @details      この関数はデバッグ用の関数です。最終 ROM 版(FINALROM)ライブラリでは常に 0 を返します。
 */
uint32_t SetHeapFillVal( int type, uint32_t value ) NN_NOEXCEPT;

/**
 * @brief       ヒープの作成時やメモリブロックの確保・解放時にメモリにセットする値を取得します。
 * @param[in]   type    取得する値の種類
 * @return      指定された種類のメモリにセットする値を返します。
 * @details      この関数はデバッグ用の関数です。最終 ROM 版(FINALROM)ライブラリでは常に 0 を返します。
 */
uint32_t GetHeapFillVal( int type ) NN_NOEXCEPT;

/**
 * @brief       ヒープに割り当てられているメモリサイズ(ヘッダ部分も含む全体のメモリサイズ)を取得します。
 * @param[in]   handle  ヒープハンドル
 * @return      ヒープに割り当てられているメモリサイズ(ヘッダ部分も含む全体のメモリサイズ)
                を返します。
 */
size_t GetHeapTotalSize(HeapHandle handle) NN_NOEXCEPT;

/**
 * @brief       ヒープが利用するメモリ領域の開始アドレスを取得します。
 * @param[in]   handle  ヒープハンドル
 * @return      ヒープが利用するメモリ領域の開始アドレスを返します。
 */
inline void* GetHeapStartAddress(HeapHandle handle) NN_NOEXCEPT
{
    return handle->pHeapStart;
}

/**
    @brief      指定したアドレスがヒープに含まれているか調べます。
    @param[in]  heapHandle  ヒープハンドル
    @param[in]  address     調べる対象のアドレス
    @return     指定したアドレスがヒープ領域内なら、 true を返します。
*/
bool HasAddress( HeapHandle heapHandle, void* address ) NN_NOEXCEPT;

}}}
