﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <cstdlib>
#include <cstring>
#include <cerrno>

#include <nn/TargetConfigs/build_Base.h>
#include <nn/TargetConfigs/build_Compiler.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/mem.h>
#include <nn/init/init_Malloc.h>

#include "detail/init_Macro.h"

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    // 文字コードエンコーディングに関する警告の抑制
    #pragma warning( disable : 4566 )
#endif



//-----------------------------------------------------------------------------

namespace {

    void*   g_MallocRegionAddress   = NULL;
    size_t  g_MallocRegionSize      = 0;

    // コンストラクタは起動しないようにしておく
    nn::util::TypedStorage<nn::mem::StandardAllocator,sizeof(nn::mem::StandardAllocator),NN_ALIGNOF(nn::mem::StandardAllocator)>    g_MallocAllocator;

}   // anonymous


namespace nn { namespace init {

//-----------------------------------------------------------------------------
//  InitializeAllocator() 関数の定義
//
void InitializeAllocator(void* address, size_t size) NN_NOEXCEPT
{
    InitializeAllocator(address, size, false);
}

void InitializeAllocator(void* address, size_t size, bool isCacheEnable) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    NN_UNUSED(address);
    NN_UNUSED(size);
    NN_UNUSED(isCacheEnable);
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_SDK_ASSERT(g_MallocRegionSize == 0);
    NN_SDK_ASSERT(size > 0);

    new( &Get(g_MallocAllocator) ) nn::mem::StandardAllocator;
    Get(g_MallocAllocator).Initialize(address, size, isCacheEnable);
    g_MallocRegionAddress   = address;
    g_MallocRegionSize      = size;
#endif
}

//-----------------------------------------------------------------------------
//  アロケータの各種サイズ取得関数の定義
//
nn::mem::StandardAllocator* GetAllocator() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    return NULL;
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_SDK_ASSERT(g_MallocRegionSize > 0);
    return &(Get(g_MallocAllocator));
#endif
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
//-----------------------------------------------------------------------------
//  デフォルトの nninitStartup() で登録される thread_local 用の
//  デフォルトアロケータとデフォルトデアロケータの関数定義
//
namespace detail {

void* DefaultAllocatorForThreadLocal(size_t size, size_t alignment) NN_NOEXCEPT
{
    if (g_MallocRegionSize == 0)
    {
        return NULL;
    }
    return Get(g_MallocAllocator).Allocate(size, alignment);
}

void DefaultDeallocatorForThreadLocal(void* p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    if (p)
    {
        Get(g_MallocAllocator).Free(p);
    }
}

}   // namespace detail
#endif

}}  // namespace nn::init

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
//-----------------------------------------------------------------------------
//  malloc(), free(), calloc(), realloc(), aligned_alloc, malloc_usable_size 関数の定義
//
NN_INIT_EXTERN_C void* malloc(size_t size)
{
    if (g_MallocRegionSize == 0)
    {
        return NULL;
    }

    void* p = Get(g_MallocAllocator).Allocate(size);
    if ( p == NULL )
    {
        errno = ENOMEM;
    }

    return p;
}

NN_INIT_EXTERN_C void free(void* p)
{
    if (p)
    {
        Get(g_MallocAllocator).Free(p);
    }
}

NN_INIT_EXTERN_C void* calloc(size_t num, size_t size)
{
    size_t sum = num * size;
    void*  p   = std::malloc(sum);

    if ( p != NULL )
    {
        std::memset(p, 0, sum);
    }
    else
    {
        errno = ENOMEM;
    }

    return p;
}

NN_INIT_EXTERN_C void* realloc(void* p, size_t newSize)
{
    if (g_MallocRegionSize == 0)
    {
        return NULL;
    }

    // メモリブロックのサイズを変更する
    // Reallocate() の仕様は realloc() の仕様と同じため、そのまま呼ぶ
    void* r = Get(g_MallocAllocator).Reallocate(p, newSize);
    if ( r == NULL )
    {
        errno = ENOMEM;
    }

    return r;
}

NN_INIT_EXTERN_C void* aligned_alloc(size_t alignment, size_t size)
{
    if (g_MallocRegionSize == 0)
    {
        return NULL;
    }

    void* p = Get(g_MallocAllocator).Allocate(size, alignment);
    if ( p == NULL )
    {
        errno = ENOMEM;
    }

    return p;
}

NN_INIT_EXTERN_C size_t malloc_usable_size(void* p)
{
    if ( p == NULL )
    {
        errno = ENOMEM;
        return 0;
    }

    return Get(g_MallocAllocator).GetSizeOf(p);
}

//-----------------------------------------------------------------------------
#endif  // NN_BUILD_CONFIG_OS_HORIZON

