﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>

#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>

#include <nn/os/os_SdkMutex.h>
#include <nn/sf/sf_ShimLibraryUtility.h>

#include <nn/htclow.h>
#include <nn/htclow/htclow_ResultPrivate.h>
#include <nn/htclow/detail/htclow_IDebugServiceObject.h>
#include <nn/htclow/detail/htclow_Log.h>
#include <nn/htclow/detail/htclow_DebugApi.h>
#include <nn/htclow/detail/htclow_DebugServiceName.h>

namespace nn {
namespace htclow {
namespace detail {

namespace
{
    const int NumberOfClients = 1;
    sf::SimpleAllInOneHipcClientManager<NumberOfClients> g_ClientManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
    sf::ShimLibraryObjectHolder<detail::IDebugServiceObject> g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

    nn::os::SdkMutexType g_InitializeMutex = NN_OS_SDK_MUTEX_INITIALIZER();

    bool g_Initialized = false;
    nn::sf::SharedPointer<detail::IDebugServiceObject> g_Service;
}

void InitializeForDebugApi() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_InitializeMutex)> lock(g_InitializeMutex);
    if (g_Initialized)
    {
        return;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ClientManager.InitializeShimLibraryHolder(&g_Holder, nn::htclow::detail::DebugServiceName));
    g_Initialized = true;

    g_Service = g_Holder.GetObject();
}

void InitializeForDebugApiWith(nn::sf::SharedPointer<IDebugServiceObject>&& manager) NN_NOEXCEPT
{
    std::lock_guard<decltype(g_InitializeMutex)> lock(g_InitializeMutex);
    if (g_Initialized)
    {
        NN_ABORT("This function cannot be called after library initialization.\n");
    }

    g_Holder.InitializeHolderDirectly(std::move(manager));
    g_Initialized = true;

    g_Service = g_Holder.GetObject();
}

nn::Result SendFromTest(const void* pBuffer, size_t sendSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    const auto buffer = nn::sf::InBuffer(reinterpret_cast<const char*>(pBuffer), sendSize);
    return g_Service->SendFromTest(buffer);
}

nn::Result ReceiveFromTest(void* pBuffer, size_t receiveSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    const auto buffer = nn::sf::OutBuffer(reinterpret_cast<char*>(pBuffer), receiveSize);
    return g_Service->ReceiveFromTest(buffer);
}

nn::Result OpenDriver(DriverType driverType) NN_NOEXCEPT
{
    return g_Service->OpenDriver(driverType);
}

void CloseDriver() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Service->CloseDriver());
}

size_t GetDefaultAllocatorUsage() NN_NOEXCEPT
{
    uint64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Service->GetDefaultAllocatorUsage(&size));

    NN_ABORT_UNLESS(size <= std::numeric_limits<size_t>::max());
    return static_cast<size_t>(size);
}

void PrintDefaultAllocatorUsage() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Service->PrintDefaultAllocatorUsage());
}

}}}
