﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/fs/fs_Result.h>
#include <nn/fssystem/utilTool/fs_BucketTreeRange.h>

namespace nn { namespace fssystem { namespace utilTool {

Result BucketTreeRangeArray::ReadData(fs::IStorage* pDataStorage, int64_t dataSize, int64_t readOffset, void* readBuffer, size_t readSize, fs::IStorage* pGroundStorage) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDataStorage);
    NN_SDK_REQUIRES_GREATER_EQUAL(dataSize, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(readOffset, 0);

    if( readSize == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_SDK_REQUIRES_NOT_NULL(readBuffer);

    // 読み込むデータがないので 0 埋め
    if( dataSize <= readOffset )
    {
        std::memset(readBuffer, 0, readSize);
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW_UNLESS(0 < m_Count, fs::ResultNotFound());

    // 必要なデータをストレージから読み込むクラス
    class DataStorageStream
    {
    public:
        DataStorageStream(void* buffer, int64_t size) NN_NOEXCEPT
            : m_Buffer(reinterpret_cast<char*>(buffer))
            , m_BufferSize(size)
            , m_FilledSize(0)
            , m_Position(0)
        {
        }

        ~DataStorageStream() NN_NOEXCEPT
        {
            const auto size = m_BufferSize - m_FilledSize;
            // 余った領域は 0 埋め
            if( 0 < size )
            {
                std::memset(m_Buffer + m_FilledSize, 0, size_t(size));
            }
        }

        void SetPosition(int64_t position) NN_NOEXCEPT
        {
            m_Position = position;
        }

        int64_t GetPosition() const NN_NOEXCEPT
        {
            return m_Position;
        }

        Result Write(fs::IStorage* pStorage, int64_t offset, int64_t size) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_LESS_EQUAL(0, offset);
            NN_SDK_ASSERT_LESS(0, size);

            const auto readSize = GetReadableSize(size);
            if( readSize == 0 )
            {
                NN_RESULT_SUCCESS;
            }

            NN_RESULT_DO(pStorage->Read(offset, m_Buffer + m_FilledSize, readSize));

            m_FilledSize += readSize;
            m_Position += readSize;

            NN_RESULT_SUCCESS;
        }

        Result Advance(int64_t size, fs::IStorage* pGroundStorage) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_LESS(0, size);

            const auto fillSize = GetReadableSize(size);
            if( fillSize == 0 )
            {
                NN_RESULT_SUCCESS;
            }

            if( pGroundStorage != nullptr )
            {
                NN_RESULT_DO(pGroundStorage->Read(m_Position, m_Buffer + m_FilledSize, fillSize));
            }
            else
            {
                std::memset(m_Buffer + m_FilledSize, 0, fillSize);
            }

            m_FilledSize += fillSize;
            m_Position += fillSize;

            NN_RESULT_SUCCESS;
        }

        bool IsFull() const NN_NOEXCEPT
        {
            return m_FilledSize == m_BufferSize;
        }

        int64_t GetFilledSize() const NN_NOEXCEPT
        {
            return m_FilledSize;
        }

    private:
        size_t GetReadableSize(int64_t size) const NN_NOEXCEPT
        {
            return static_cast<size_t>(std::min(m_BufferSize - m_FilledSize, size));
        }

        char* const m_Buffer;
        const int64_t m_BufferSize;
        int64_t m_FilledSize;
        int64_t m_Position;
    };

    DataStorageStream stream(readBuffer, readSize);
    stream.SetPosition(readOffset);

    const auto beginIter = begin();
    const auto endIter = end();

    // readOffset を含む Range を検索
    auto iter = std::upper_bound(
        beginIter,
        endIter,
        readOffset,
        [](int64_t value, const BucketTreeRange& range) NN_NOEXCEPT
        {
            return value < range.physicalOffset;
        }
    );
    if( beginIter < iter )
    {
        --iter;
    }

    for( ; iter < endIter; ++iter )
    {
        const auto range = *iter;
        auto rangeSize = range.size;
        auto readableOffset = std::max(readOffset, stream.GetPosition());
        auto remainingSize = int64_t(readSize) - stream.GetFilledSize();

        // これ以上先には読出し対象領域はない
        if( readableOffset + remainingSize <= range.physicalOffset )
        {
            break;
        }

        // 今回読み出す領域のデータを指すエントリが見つかった
        if( readableOffset < range.physicalOffset + rangeSize )
        {
            // 差分データ上にテーブルから参照されない隙間の領域がある
            if( readableOffset < range.physicalOffset )
            {
                // バッファを pGroundStorage のデータもしくは 0 で埋めて進め、オフセットを補正
                const auto fixOffset = range.physicalOffset - readableOffset;
                NN_RESULT_DO(stream.Advance(fixOffset, pGroundStorage));
                readableOffset += fixOffset;
            }

            auto virtualOffset = range.virtualOffset;

            // 現エントリが指す領域の途中から読出し対象領域に差し掛かっている
            if( stream.GetPosition() <= readableOffset )
            {
                const auto fixOffset = readableOffset - range.physicalOffset;
                stream.SetPosition(readableOffset);
                virtualOffset += fixOffset;
                rangeSize -= fixOffset;
            }

            // 読出し対象領域
            NN_RESULT_DO(stream.Write(pDataStorage, virtualOffset, rangeSize));
            if( stream.IsFull() )
            {
                NN_RESULT_SUCCESS;
            }
        }
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

}}}
