﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/fs_BucketTreeUtility.h>
#include <nn/fssystem/utilTool/fs_BucketTreeAnalyzer.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace fssystem { namespace utilTool {

namespace {

typedef BucketTree::NodeHeader NodeHeader;
typedef detail::BucketTreeNode<const int64_t*> NodeList;
typedef detail::BucketTreeNode<const void*> EntryList;

NN_STATIC_ASSERT(std::is_pod<NodeList>::value);
NN_STATIC_ASSERT(std::is_pod<EntryList>::value);

inline ptrdiff_t GetDistance(const void* ptr1, const void* ptr2) NN_NOEXCEPT
{
    return std::distance(
        reinterpret_cast<const char*>(ptr1), reinterpret_cast<const char*>(ptr2));
}

}

NN_DEFINE_STATIC_CONSTANT(const size_t BucketTreeAnalyzer::TextLength);

BucketTreeAnalyzer::BucketTreeAnalyzer() NN_NOEXCEPT
    : m_NodeStorage()
    , m_EntryStorage()
    , m_TextBuffer()
{
}

Result BucketTreeAnalyzer::Initialize(
                               fs::SubStorage* pStorageArray,
                               int storageCount,
                               size_t nodeSize,
                               size_t entrySize,
                               int entryCount,
                               int64_t startOffset,
                               const char* const pEntryHeader,
                               FormatFunction formatEntry,
                               FormatFunction formatData
                           ) NN_NOEXCEPT
{
    static const char Header[] = "           Address|Layer|";
    static const size_t HeaderLength = sizeof(Header); // '\0' 含む

    NN_SDK_REQUIRES_NOT_NULL(pStorageArray);
    NN_SDK_REQUIRES_MINMAX(storageCount, StorageIndex_Max - 1, StorageIndex_Max);
    NN_SDK_REQUIRES_GREATER(nodeSize, static_cast<size_t>(0));
    NN_SDK_REQUIRES_GREATER(entrySize, static_cast<size_t>(0));
    NN_SDK_REQUIRES_GREATER_EQUAL(startOffset, 0);
    NN_SDK_REQUIRES_LESS(
        strnlen(pEntryHeader, TextLength - HeaderLength),
        TextLength - HeaderLength
    );
    NN_UNUSED(storageCount);

    static const size_t TextBufferSize = 2048;
    m_TextBuffer.reset(new char[TextBufferSize]);
    NN_RESULT_THROW_UNLESS(m_TextBuffer != nullptr, fs::ResultAllocationMemoryFailedNew());

    // テーブルファイルのヘッダ部分を書き出し
    int tableOffset;
    {
        char header[TextLength];
        tableOffset = util::SNPrintf(header, TextLength, "%s%s\n", Header, pEntryHeader);
        NN_SDK_ASSERT_LESS(static_cast<size_t>(tableOffset), TextLength);

        NN_RESULT_DO(pStorageArray[StorageIndex_TableFile].Write(0, header, tableOffset));
    }

    const auto entryOffset = HeaderLength + strnlen(pEntryHeader, TextLength - HeaderLength);
    for( size_t i = 0; i < entryOffset - 1; ++i )
    {
        m_Separator[i] = '-';
    }
    m_Separator[entryOffset - 1] = '\n';
    m_Separator[entryOffset] = '\0';

    // エントリファイル先頭の分割線書き出し
    NN_RESULT_DO(pStorageArray[StorageIndex_EntryFile].Write(0, m_Separator, entryOffset));

    // ノードストレージの初期化
    {
        int64_t storageSize = 0;
        NN_RESULT_DO(pStorageArray[StorageIndex_TableFile].GetSize(&storageSize));

        m_NodeStorage.Initialize(
            fs::SubStorage(
                &pStorageArray[StorageIndex_TableFile],
                tableOffset,
                storageSize - tableOffset
            ),
            pStorageArray[StorageIndex_NodeFile],
            nodeSize,
            startOffset,
            m_Separator
        );
    }

    // エントリストレージの初期化
    {
        int64_t storageSize = 0;
        NN_RESULT_DO(pStorageArray[StorageIndex_EntryFile].GetSize(&storageSize));

        const auto nodeOffset =
            BucketTreeBuilder::QueryNodeStorageSize(nodeSize, entrySize, entryCount);

        const bool isOutputData = (storageCount == StorageIndex_Max);

        m_EntryStorage.Initialize(
            fs::SubStorage(
                &pStorageArray[StorageIndex_EntryFile],
                entryOffset,
                storageSize - entryOffset
            ),
            isOutputData ? pStorageArray[StorageIndex_DataFile] : fs::SubStorage(),
            m_TextBuffer.get(),
            TextBufferSize,
            nodeSize,
            entrySize,
            startOffset + nodeOffset,
            formatEntry,
            formatData,
            isOutputData
        );
    }

    NN_RESULT_SUCCESS;
}

Result BucketTreeAnalyzer::NodeStorage::Write(
                                            int64_t offset,
                                            const void* buffer,
                                            size_t size
                                        ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(size == m_NodeSize, fs::ResultInvalidSize());

    const NodeList& node = *reinterpret_cast<const NodeList*>(buffer);
    const NodeHeader& header = node.header;

    fs::SubStorage* pOutput;

    // ヘッダ書き出し
    {
        char text[TextLength];
        const auto addr = m_StartOffset + offset;
        const auto layer = (offset == 0) ? 1 : 2;
        const size_t length = util::SNPrintf(
            text, TextLength, "0x%016llX|   L%d|(index:%d count:%d)\n",
            addr, layer, header.index, header.count);
        NN_SDK_ASSERT_LESS(length, TextLength);

        if( offset == 0 )
        {
            // m_TableStorage への最初（で最後）の書き出し
            m_Position = 0;

            pOutput = &m_TableStorage;
        }
        else
        {
            // m_NodeStorage への最初の書き出し
            if( offset == static_cast<int64_t>(m_NodeSize) )
            {
                m_Position = 0;
            }

            pOutput = &m_NodeStorage;
        }

        NN_RESULT_DO(pOutput->Write(m_Position, m_pSeparator, m_SeparatorLength));
        m_Position += m_SeparatorLength;

        NN_RESULT_DO(pOutput->Write(m_Position, text, length));
        m_Position += length;
    }

    auto iter = node.GetBegin();
    auto addr = m_StartOffset + offset + sizeof(NodeHeader);

    // オフセット書き出し
    for( int i = 0, count = header.count; i < count; ++i )
    {
        char text[TextLength];
        const size_t length = util::SNPrintf(
            text, TextLength, "0x%016llX|     |0x%016llX\n", addr, *iter);
        NN_SDK_ASSERT_LESS(length, TextLength);

        NN_RESULT_DO(pOutput->Write(m_Position, text, length));
        m_Position += length;

        ++iter;
        addr += sizeof(int64_t);
    }

    auto stride = static_cast<size_t>(GetDistance(buffer, iter));
    if( stride < m_NodeSize )
    {
        // パディング書き出し
        if( *iter == 0 )
        {
            char text[TextLength];
            const size_t length = util::SNPrintf(
                text, TextLength, "0x%016llX|     |(padding)\n", addr);
            NN_SDK_ASSERT_LESS(length, TextLength);

            NN_RESULT_DO(pOutput->Write(m_Position, text, length));
            m_Position += length;
        }
        // L1 内の L2 オフセット書き出し
        else
        {
            // L1 ノード以外はこちらに来ない
            NN_RESULT_THROW_UNLESS(offset == 0, fs::ResultDataCorrupted());

            // L2 オフセットの書き始め
            char text[TextLength];
            size_t length = util::SNPrintf(
                text, TextLength, "0x%016llX|   L2|0x%016llX\n", addr, *iter);
            NN_SDK_ASSERT_LESS(length, TextLength);

            for( ; ; )
            {
                NN_RESULT_DO(pOutput->Write(m_Position, text, length));
                m_Position += length;

                ++iter;
                addr += sizeof(int64_t);

                stride += sizeof(int64_t);
                NN_SDK_ASSERT_LESS_EQUAL(stride, m_NodeSize);

                if( m_NodeSize <= stride )
                {
                    break;
                }
                NN_RESULT_THROW_UNLESS(0 < *iter, fs::ResultDataCorrupted());

                length = util::SNPrintf(
                    text, TextLength, "0x%016llX|     |0x%016llX\n", addr, *iter);
                NN_SDK_ASSERT_LESS(length, TextLength);
            }
        }
    }

    NN_RESULT_SUCCESS;
}

Result BucketTreeAnalyzer::NodeStorage::Flush() NN_NOEXCEPT
{
    NN_RESULT_DO(m_TableStorage.Flush());
    NN_RESULT_DO(m_NodeStorage.Flush());
    NN_RESULT_SUCCESS;
}

Result BucketTreeAnalyzer::EntryStorage::Write(
                                            int64_t offset,
                                            const void* buffer,
                                            size_t size
                                        ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(size == m_NodeSize, fs::ResultInvalidSize());

    const EntryList& node = *reinterpret_cast<const EntryList*>(buffer);
    const NodeHeader& header = node.header;
    auto addr = m_StartOffset + offset;

    // ヘッダ書き出し
    {
        char text[TextLength];
        const size_t length = util::SNPrintf(
            text, TextLength, "0x%016llX|Entry|(index:%d count:%d)\n",
            addr, header.index, header.count);
        NN_SDK_ASSERT_LESS(length, TextLength);

        NN_RESULT_DO(m_EntryStorage.Write(m_EntryPos, text, length));
        m_EntryPos += length;
    }

    util::ConstBytePtr ptr(node.GetBegin());
    addr += sizeof(NodeHeader);

    for( int i = 0, count = header.count; i < count; ++i )
    {
        // エントリ書き出し
        {
            size_t length = util::SNPrintf(
                m_TextBuffer, m_TextBufferSize, "0x%016llX|     |", addr);
            NN_SDK_ASSERT_LESS(length, m_TextBufferSize);

            length += m_FormatEntry(m_TextBuffer + length, m_TextBufferSize - length, ptr.Get());

            NN_RESULT_DO(m_EntryStorage.Write(m_EntryPos, m_TextBuffer, length));
            m_EntryPos += length;
        }

        // データ書き出し
        if( m_IsOutputData )
        {
            const size_t length = m_FormatData(m_TextBuffer, m_TextBufferSize, ptr.Get());

            NN_RESULT_DO(m_DataStorage.Write(m_DataPos, m_TextBuffer, length));
            m_DataPos += length;
        }

        ptr.Advance(m_EntrySize);
        addr += m_EntrySize;
    }

    // パディング書き出し
    auto stride = static_cast<size_t>(GetDistance(buffer, ptr.Get()));
    if( stride < m_NodeSize )
    {
        NN_RESULT_THROW_UNLESS(*ptr.Get<int64_t>() == 0, fs::ResultDataCorrupted());

        char text[TextLength];
        const size_t length = util::SNPrintf(
            text, TextLength, "0x%016llX|     |(padding)\n", addr);
        NN_SDK_ASSERT_LESS(length, TextLength);

        NN_RESULT_DO(m_EntryStorage.Write(m_EntryPos, text, length));
        m_EntryPos += length;
    }

    NN_RESULT_SUCCESS;
}

Result BucketTreeAnalyzer::EntryStorage::Flush() NN_NOEXCEPT
{
    NN_RESULT_DO(m_EntryStorage.Flush());
    NN_RESULT_DO(m_DataStorage.Flush());
    NN_RESULT_SUCCESS;
}

}}}
