﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_FormatString.h>
#include <nn/fssystem/utilTool/fs_BinaryMatch.h>
#include <nn/fssystem/utilTool/fs_BinaryRegionFile.h>

namespace nn { namespace fssystem { namespace utilTool {

NN_DEFINE_STATIC_CONSTANT(const uint32_t BinaryRegionFileHeader::Signature);
NN_DEFINE_STATIC_CONSTANT(const uint32_t BinaryRegionFileHeader::Version);

const char* const BinaryRegionFileHeader::Extension = ".brf";

// ヘッダー部分を設定
void BinaryRegionFileHeader::SetHeader(const char* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_EQUAL(size, static_cast<size_t>(NcaFsHeader::Size));

    const uint32_t Pattern = 4133765235; // 適当な値で xor する

    crypto::GenerateSha256Hash(this->headerHash, sizeof(this->headerHash), buffer, size);

    // 難読化しておく（必要ある？そもそも保存しない方がいい？）
    for( size_t i = 0; i < sizeof(this->header); i += sizeof(uint32_t) )
    {
        *reinterpret_cast<uint32_t*>(this->header + i) =
            *reinterpret_cast<const uint32_t*>(buffer + i) ^ Pattern;
    }
}

// データ部分を設定
void BinaryRegionFileHeader::SetData(int64_t offset, const char* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(size, static_cast<size_t>(0));

    crypto::GenerateSha256Hash(this->dataHash, sizeof(this->dataHash), buffer, size);

    this->dataOffset = offset;
    this->regionSize = static_cast<uint32_t>(size);
}

// ファイル名を生成
void BinaryRegionFileHeader::GenerateFileName(char* outBuffer, size_t outSize) const NN_NOEXCEPT
{
    const auto HashSize = sizeof(this->headerHash);

    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES_GREATER(outSize, HashSize * 2 + 4);
    NN_SDK_REQUIRES_EQUAL(this->signature, Signature);
    NN_UNUSED(outSize);

    // headerHash と dataHash の xor をファイル名にする
    for( size_t i = 0; i < HashSize; ++i )
    {
        const auto value = static_cast<uint8_t>(this->headerHash[i]) ^ static_cast<uint8_t>(this->dataHash[i]);
        nn::util::SNPrintf(outBuffer + i * 2, outSize - i * 2, "%02x", value);
    }
    std::strcpy(outBuffer + HashSize * 2, Extension);
}

// 読み込んだファイルのリージョンハッシュをチェック
bool BinaryRegionFile::CheckRegion(char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, static_cast<size_t>(0));

    typedef BinaryRegionArray::Region Region;

    const Header& header = *reinterpret_cast<Header*>(buffer);
    if( header == m_Header )
    {
        const auto totalSize = sizeof(Header) + sizeof(Region) * header.regionCount;
        if( totalSize == bufferSize )
        {
            m_Header.regionCount = header.regionCount;
            m_Buffer = buffer + sizeof(Header);

            return true;
        }
    }
    return false;
}

// リージョンハッシュを設定
void BinaryRegionFile::SetRegion(const BinaryRegionArray& region) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(region.IsValid() && !region.empty());
    NN_SDK_REQUIRES(m_Buffer == nullptr);

    m_Header.regionCount = static_cast<uint32_t>(region.size());
}

// リージョンハッシュを取得
const BinaryRegionArray BinaryRegionFile::GetRegion() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    NN_SDK_REQUIRES_GREATER(m_Header.regionCount, static_cast<uint32_t>(0));

    typedef BinaryRegionArray RegionArray;
    typedef RegionArray::Region Region;

    return RegionArray(reinterpret_cast<Region*>(m_Buffer), static_cast<int>(m_Header.regionCount));
}

}}}
