﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/util/util_IntUtil.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/utilTool/fs_BinaryMatchPrivate.h>

namespace nn { namespace fssystem { namespace utilTool {

namespace {

inline size_t GetMinimumSize(size_t x, size_t y) NN_NOEXCEPT
{
    return std::min(x, y);
}

template< typename T1, typename T2 >
inline size_t GetMinimumSize(T1 x, T2 y) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(x, static_cast<T1>(0));
    NN_SDK_REQUIRES_GREATER_EQUAL(y, static_cast<T2>(0));
    return static_cast<size_t>(std::min(static_cast<int64_t>(x), static_cast<int64_t>(y)));
}

template< typename T >
inline bool Contains(const T& container, typename T::const_reference element) NN_NOEXCEPT
{
    const auto distance = std::distance<typename T::const_pointer>(container.data(), &element);
    return (0 <= distance) && (distance < static_cast<ptrdiff_t>(container.size()));
}

const size_t ConflictCountMax = 64;

}

NN_DEFINE_STATIC_CONSTANT(const int64_t BinaryMatchResult::InvalidOffset);

NN_DEFINE_STATIC_CONSTANT(const int BinaryBlockHash::BlockSize);
NN_DEFINE_STATIC_CONSTANT(const int BinaryBlockHash::ValueCount);
NN_DEFINE_STATIC_CONSTANT(const int BinaryBlockHash::RotateCount);

NN_DEFINE_STATIC_CONSTANT(const int BinaryRegionHash::BinCount);
NN_DEFINE_STATIC_CONSTANT(const int BinaryRegionHash::HashSize);

NN_DEFINE_STATIC_CONSTANT(const size_t BinaryMatch::RegionSizeMin);

// 連続する BinaryMatch の結果をまとめる
void BinaryMatchResult::Merge(std::vector<BinaryMatchResult>* pResults) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pResults);

    auto& results = *pResults;
    std::sort(results.begin(), results.end());

    const auto resultCount = results.size();
    if( 1 < resultCount )
    {
        size_t mergeCount = 0;
        {
            auto range1 = results[0];
            auto merge = range1;

            for( size_t i = 1; i < resultCount; ++i )
            {
                const auto& range2 = results[i];

                // データが連続している
                if( range1.CanMerge(range2) )
                {
                    merge.size += range2.size;
                }
                else
                {
                    results[mergeCount] = merge;
                    ++mergeCount;

                    merge = range2;
                }

                range1 = range2;
            }

            results[mergeCount] = merge;
            ++mergeCount;
        }

        results.resize(mergeCount);
    }
}

// 連続する BinaryMatch の結果をチェックする
nn::Result BinaryMatchResult::Verify(const std::vector<BinaryMatchResult>& results) NN_NOEXCEPT
{
    if( !results.empty() )
    {
        int64_t oldOffset = -1;
        int64_t newOffset = results[0].newOffset;
        NN_RESULT_THROW_UNLESS(0 <= newOffset, fs::ResultInvalidOffset());

        for( const auto& result : results )
        {
            NN_RESULT_THROW_UNLESS(newOffset == result.newOffset, fs::ResultInvalidOffset());
            NN_RESULT_THROW_UNLESS(0 < result.size, fs::ResultInvalidSize());

            if( result.detail == BinaryMatchDetail_Match )
            {
                NN_RESULT_THROW_UNLESS((0 <= result.oldOffset) && (oldOffset != result.oldOffset), fs::ResultInvalidOffset());
                NN_RESULT_THROW_UNLESS(result.storageIndex == 0, fs::ResultInvalidArgument());

                oldOffset = result.oldOffset + result.size;
            }
            else
            {
                NN_RESULT_THROW_UNLESS(result.oldOffset == BinaryMatchResult::InvalidOffset, fs::ResultInvalidOffset());
                NN_RESULT_THROW_UNLESS(result.storageIndex == 1, fs::ResultInvalidArgument());
                NN_RESULT_THROW_UNLESS((0 <= result.detail) && (result.detail < BinaryMatchDetail_Max), fs::ResultInvalidArgument());

                oldOffset = BinaryMatchResult::InvalidOffset;
            }

            newOffset = result.newOffset + result.size;
        }
    }
    NN_RESULT_SUCCESS;
}

// SetPhase() と GetInfo() は別スレッドで呼び出されるので、相互の排他制御が必要。
// 複数の値を排他する必要があるので mutex を使用する。
//
// 一方、SetPhase() と SetValue() は別スレッドから呼び出すことはないなので、相互の排他制御は不要。
// （セッター単体も別スレッドから同時に呼び出すことはない想定）
void BinaryMatchProgress::SetPhase(Phase phase, int64_t total) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(phase, Phase::MakeRegionHash, Phase::Count);
    NN_SDK_REQUIRES_GREATER_EQUAL(total, 0);

    std::lock_guard<os::Mutex> lock(m_Mutex);

    m_Phase = int(phase);
    m_Total = total;
    m_Value = 0;
}

// GetInfo() と SetValue() は別スレッドで呼び出されるので、相互の排他制御が必要。
// m_Value の排他ができていれば十分なので atomic を使用する。
const BinaryMatchProgress::Info BinaryMatchProgress::GetInfo() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);

    Info info = { m_Phase, m_Total, m_Value };
    return info;
}

// コンストラクタです。
BinaryMatch::BinaryMatch(size_t blockSize, size_t regionSize, int64_t matchSize) NN_NOEXCEPT
    : m_BlockSize(blockSize)
    , m_RegionSize(regionSize)
    , m_MatchSize(matchSize)
    , m_WindowSize(std::numeric_limits<int64_t>::max())
    , m_OldStorage()
    , m_OldStorageSize(0)
    , m_NewStorage()
    , m_NewStorageSize(0)
    , m_RegionBuffer()
    , m_Regions()
    , m_pRegionSet()
    , m_Hints()
    , m_ExcludeRangeOld()
    , m_ExcludeRangeNew()
    , m_Result()
    , m_ResultCountCache(0)
    , m_pProgress(AllocateShared<Progress>()) // エラーハンドリングは Run() で行う
{
    NN_SDK_REQUIRES(0 < blockSize && util::ispow2(blockSize));
    NN_SDK_REQUIRES(blockSize < regionSize && util::is_aligned(regionSize, blockSize));
    NN_SDK_REQUIRES_GREATER_EQUAL(regionSize, RegionSizeMin);
    NN_SDK_REQUIRES(util::IsIntValueRepresentable<int>(regionSize / blockSize));
    NN_SDK_REQUIRES_GREATER_EQUAL(matchSize, 0);

    static const auto BlockSize = BinaryBlockHash::BlockSize;
    static const auto HashSize = BinaryRegionHash::HashSize;
    NN_SDK_REQUIRES_EQUAL((regionSize / BlockSize) % (HashSize / 2 * NN_BITSIZEOF(int8_t)), static_cast<size_t>(0));
}

// デストラクタです。
BinaryMatch::~BinaryMatch() NN_NOEXCEPT
{
}

// 比較しない領域を追加します。
nn::Result BinaryMatch::AddExcludeRange(std::vector<ExcludeRange>* pRanges, const ExcludeRange& range) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pRanges); // クラス外部から渡されないので REQUIRES にはしない
    NN_SDK_REQUIRES_GREATER_EQUAL(range.offset, 0);
    NN_SDK_REQUIRES(util::is_aligned(range.offset, m_BlockSize));
    NN_SDK_REQUIRES_GREATER(range.size, 0);
    NN_SDK_REQUIRES(util::is_aligned(range.size, m_BlockSize));

    const auto iter = std::upper_bound(
        pRanges->begin(),
        pRanges->end(),
        range.offset,
        [](int64_t offset, const ExcludeRange& value) NN_NOEXCEPT
        {
            return offset < value.offset;
        }
    );

    // 挿入予定位置より前の範囲と被っていないかチェック
    if( pRanges->begin() < iter )
    {
        const auto& front = *(iter - 1);

        NN_RESULT_THROW_UNLESS(
            front.offset + front.size <= range.offset, fs::ResultInvalidOffset());
    }

    // 挿入予定位置より後ろの範囲と被っていないかチェック
    if( iter < pRanges->end() )
    {
        const auto& back = *iter;

        NN_RESULT_THROW_UNLESS(
            range.offset + range.size <= back.offset, fs::ResultInvalidOffset());
    }

    pRanges->insert(iter, range);

    NN_RESULT_SUCCESS;
}

// マッチングを実行します。
nn::Result BinaryMatch::Run(fs::SubStorage oldStorage, fs::SubStorage newStorage, size_t shiftSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_pProgress != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());
    NN_RESULT_THROW_UNLESS((0 < shiftSize) && (shiftSize <= m_BlockSize) && util::ispow2(shiftSize), fs::ResultInvalidAlignment());

    m_Result.clear();
    m_ResultCountCache = 0;

    int64_t newStorageSize = 0;
    NN_RESULT_DO(newStorage.GetSize(&newStorageSize));

    if( newStorageSize == 0 )
    {
        NN_RESULT_SUCCESS;
    }

    int64_t oldStorageSize = 0;
    NN_RESULT_DO(oldStorage.GetSize(&oldStorageSize));

    // 外部から設定されたリージョンハッシュの正当性をチェック
    if( m_Regions.IsValid() )
    {
        const auto& front = m_Regions.front();

        NN_RESULT_THROW_UNLESS(front.offset == 0, fs::ResultInvalidOffset());
        NN_RESULT_THROW_UNLESS((front.size % static_cast<int64_t>(m_RegionSize)) == 0, fs::ResultInvalidSize());

        const auto& back = m_Regions.back();
        const auto endOffset = back.offset + back.size;

        NN_RESULT_THROW_UNLESS(
            (endOffset <= oldStorageSize) && (oldStorageSize < endOffset + static_cast<int64_t>(m_RegionSize)),
            fs::ResultInvalidSize()
        );
    }

    // データ比較の実行
    if( (static_cast<int64_t>(m_RegionSize) <= oldStorageSize) &&
        (static_cast<int64_t>(m_RegionSize) <= newStorageSize) )
    {
        // m_RegionSize * (2 work + 2 text) + m_BlockSize
        const auto bufferSize = m_RegionSize * 4 + m_BlockSize;

        std::unique_ptr<char[]> buffer(new char[bufferSize]);
        NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultAllocationMemoryFailedNew());

        m_OldStorage = oldStorage;
        m_OldStorageSize = oldStorageSize;
        m_NewStorage = newStorage;
        m_NewStorageSize = newStorageSize;
        NN_UTIL_SCOPE_EXIT
        {
            m_OldStorage = fs::SubStorage();
            m_OldStorageSize = 0;
            m_NewStorage = fs::SubStorage();
            m_NewStorageSize = 0;
        };

        NN_RESULT_DO(MakeRegionHash(buffer.get()));

        // ヒント情報を元にデータを比較
        if( !m_Hints.empty() )
        {
            NN_RESULT_DO(CompareDataWithHint(buffer.get()));

            if( !m_Result.empty() )
            {
                Result::Merge(&m_Result);
            }
        }

        // NOTE: CompareRange でストレージ末尾まで正しく処理されるように番兵追加
        m_Result.push_back(Result::MakeUnknown(newStorageSize, 0));
        m_ResultCountCache = m_Result.size();
        {
            NN_RESULT_DO(CompareData(buffer.get(), shiftSize, Progress::Phase::BinaryMatch1stPass));
            NN_RESULT_DO(CompareData(buffer.get(), shiftSize, Progress::Phase::BinaryMatch2ndPass));

            NN_SDK_ASSERT(!m_Result.empty());
            NN_SDK_ASSERT_EQUAL(m_Result.back().size, 0);
            NN_SDK_ASSERT_EQUAL(m_Result.back().newOffset, newStorageSize);
        }
        m_Result.pop_back(); // 番兵削除
        // 以降 m_ResultCountCache は使用しないのでそのままにする
    }

    // ストレージ全体を比較した結果を取得
    MakeResult(newStorageSize);
    NN_RESULT_DO(Result::Verify(m_Result));

    NN_RESULT_SUCCESS;
}

// リージョンハッシュを生成します。
nn::Result BinaryMatch::MakeRegionHash(char* workBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);

    NN_RESULT_THROW_UNLESS(
        util::IsIntValueRepresentable<int>(m_OldStorageSize / m_RegionSize),
        fs::ResultInvalidSize()
    );

    m_pProgress->SetPhase(Progress::Phase::MakeRegionHash, m_OldStorageSize);

    const int regionCount = static_cast<int>(m_OldStorageSize / m_RegionSize);

    // 内部でリージョンハッシュを生成
    if( !m_Regions.IsValid() && (0 < regionCount) )
    {
        m_RegionBuffer.reset(new Region[regionCount]);
        NN_RESULT_THROW_UNLESS(m_RegionBuffer != nullptr, fs::ResultAllocationMemoryFailedNew());
        m_Regions.Reserve(m_RegionBuffer.get(), regionCount);

        const auto blockCount = static_cast<int>(m_RegionSize / m_BlockSize);
        char* buffers[] = { workBuffer, workBuffer + m_RegionSize };

        // 先頭リージョンのリージョンハッシュを計算
        {
            NN_RESULT_DO(m_OldStorage.Read(0, buffers[0], m_RegionSize));

            Region region = {};
            region.hash.Make(buffers[0], blockCount, m_BlockSize);
            region.size = m_RegionSize;

            m_Regions.push_back(region);
        }

        // リージョン群のリージョンハッシュを計算
        for( int i = 1; i < regionCount; ++i )
        {
            m_pProgress->SetValue(static_cast<int64_t>(m_RegionSize) * i);

            std::swap(buffers[0], buffers[1]);

            const auto offset = i * static_cast<int64_t>(m_RegionSize);
            NN_RESULT_DO(m_OldStorage.Read(offset, buffers[0], m_RegionSize));

            RegionHash hash = {};
            hash.Make(buffers[0], blockCount, m_BlockSize);

            auto& previousRegion = m_Regions.back();

            // 直前の領域と完全に一致するならまとめる
            if( (previousRegion.hash == hash) && std::equal(buffers[0], buffers[0] + m_RegionSize, buffers[1]) )
            {
                previousRegion.size += m_RegionSize;
            }
            else
            {
                Region region = {};
                region.hash = hash;
                region.offset = offset;
                region.size = m_RegionSize;

                m_Regions.push_back(region);
            }
        }

        m_Regions.shrink_to_fit();
    }

    m_pRegionSet.reset(new RegionSet(int(m_Regions.size()), int(m_RegionSize / m_BlockSize)));
    NN_RESULT_THROW_UNLESS(RegionSet::IsValid(m_pRegionSet), fs::ResultAllocationMemoryFailedNew());

    if( !m_Regions.empty() )
    {
        m_pRegionSet->Initialize(m_Regions.data());
    }

    m_pProgress->SetValue(m_OldStorageSize);

    NN_RESULT_SUCCESS;
}

// ヒントを元に新旧データを比較します。
nn::Result BinaryMatch::CompareDataWithHint(char* workBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES(!m_Regions.empty());

    m_pProgress->SetPhase(Progress::Phase::FileBaseMatch, static_cast<int64_t>(m_Hints.size()));

    const auto hashCount = static_cast<int>(m_RegionSize / m_BlockSize);
    const auto oldBuffer = workBuffer;
    const auto newBuffer = workBuffer + m_RegionSize;
    int64_t endOffset = 0;

    std::vector<Candidate> candidates;
    candidates.reserve(1);

    const auto count = m_Hints.size();
    for( size_t i = 0; i < count; ++i )
    {
        NN_RESULT_THROW_UNLESS(
            (i == 0) || (m_Hints[i - 1].newOffset < m_Hints[i].newOffset),
            fs::ResultInvalidOffset()
        );

        auto hint = m_Hints[i];

        NN_RESULT_THROW_UNLESS(
            util::is_aligned(hint.oldOffset, m_BlockSize) &&
            util::is_aligned(hint.newOffset, m_BlockSize) &&
            (hint.oldOffset + hint.oldSize <= m_OldStorageSize) &&
            (hint.newOffset + hint.newSize <= m_NewStorageSize),
            fs::ResultInvalidOffset()
        );
        NN_RESULT_THROW_UNLESS(
            util::is_aligned(hint.oldSize, m_BlockSize) && util::is_aligned(hint.newSize, m_BlockSize),
            fs::ResultInvalidSize()
        );

        // 指定されたオフセット以降で一番近いリージョンハッシュまでの距離を算出
        const int64_t fraction = hint.oldOffset % static_cast<int64_t>(m_RegionSize);
        const int64_t distance = (fraction == 0) ? 0 : (static_cast<int64_t>(m_RegionSize) - fraction);

        // 比較するデータの領域を調整
        hint.oldOffset += distance;
        hint.oldSize -= distance;
        hint.newOffset += distance;
        hint.newSize -= distance;

        if( (endOffset <= hint.newOffset) &&
            (hint.newOffset + static_cast<int64_t>(m_RegionSize) <= m_NewStorageSize) &&
            (hint.oldOffset + static_cast<int64_t>(m_RegionSize) <= m_OldStorageSize) )
        {
            // oldOffset に対応するリージョンハッシュを取得
            const auto& region = *std::lower_bound(
                m_Regions.begin(),
                m_Regions.end(),
                hint.oldOffset,
                [](const Region& region, int64_t offset) NN_NOEXCEPT
                {
                    return region.offset < offset;
                }
            );
            NN_SDK_ASSERT(Contains(m_Regions, region));
            NN_SDK_ASSERT(region.IsInside(hint.oldOffset));

            NN_RESULT_DO(m_NewStorage.Read(hint.newOffset, newBuffer, m_RegionSize));

            // ハッシュが一致したので前後のデータを比較する
            if( region.hash.IsEqual(newBuffer, hashCount, m_BlockSize) )
            {
                candidates.emplace_back(region, hint.oldOffset, hint.newOffset, 0);

                NN_RESULT_DO(CompareTogether(
                    &candidates,
                    hint.newOffset,
                    newBuffer,
                    oldBuffer,
                    hint.newOffset - endOffset,
                    m_NewStorageSize - hint.newOffset
                ));

                if( !candidates.empty() )
                {
                    // 最長要素を取得
                    const auto result = std::max_element(candidates.begin(), candidates.end())->GetResult();

                    // 最小マッチサイズに満たなければなかったことにする
                    if( m_MatchSize <= result.size )
                    {
                        m_Result.push_back(result);

                        endOffset = result.GetNewEndOffset();
                    }

                    candidates.clear();
                }
            }
        }

        m_pProgress->SetValue(static_cast<int64_t>(i + 1));
    }

    NN_RESULT_SUCCESS;
}

// 新旧データを比較します。
nn::Result BinaryMatch::CompareData(char* workBuffer, size_t shiftSize, Progress::Phase phase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES((0 < shiftSize) && (shiftSize <= m_BlockSize) && util::ispow2(shiftSize));
    NN_SDK_REQUIRES(!m_Regions.empty());

    m_pProgress->SetPhase(phase, m_NewStorageSize);

    CompareRange compareRange(m_Result, m_RegionSize);
    if( compareRange.IsValid() )
    {
        const auto textBuffer = workBuffer + m_RegionSize * 2;

        HashQueueArray hashQueueArray(static_cast<int>(m_BlockSize / shiftSize), static_cast<int>(m_RegionSize / m_BlockSize));
        NN_RESULT_THROW_UNLESS(hashQueueArray.IsValid(), fs::ResultAllocationMemoryFailedNew());
        auto pHashQueue = &hashQueueArray[0];

        // 新データ側の最初の比較領域を設定する
        NN_RESULT_DO(m_NewStorage.Read(compareRange.GetOffset(), textBuffer + m_RegionSize, m_RegionSize));
        pHashQueue->Reset(textBuffer + m_RegionSize, m_BlockSize);

        size_t textOffset = m_RegionSize;
        int64_t mismatchSize = 0;

        std::vector<Candidate> candidates;
        candidates.reserve(m_pRegionSet->GetDuplicateCountMax());

        // 順にパターンと比較
        while( compareRange.IsValid() )
        {
            Result longestResult = {};
            RegionRange regionRange;

            if( !pHashQueue->IsEqualToPrevious() )
            {
                // 現在の領域とリージョンハッシュが一致する領域群を探索
                regionRange = m_pRegionSet->Find(pHashQueue->GetRegionHash());
            }

            // リージョンハッシュが一致する領域が見つかった場合、厳密比較を行う
            if( !regionRange.empty() )
            {
                NN_SDK_ASSERT(candidates.empty());

                // 候補の抽出（ハッシュの衝突数が多いものは 2 パス目に後回し）
                if( (regionRange.size() <= ConflictCountMax) || (phase == Progress::Phase::BinaryMatch2ndPass) )
                {
                    const auto newOffset = compareRange.GetOffset();

                    // 同じリージョンハッシュを持つ領域の範囲は各領域に対してあらかじめ計算されているので、その範囲を候補領域とする
                    for( auto pRegion : regionRange )
                    {
                        const auto& region = *pRegion;
                        const auto oldOffset = region.offset;

                        // ウィンドウサイズに収まっている領域を比較
                        if( std::abs(oldOffset - newOffset) < m_WindowSize )
                        {
                            candidates.emplace_back(region, oldOffset, newOffset, 0);
                        }
                    }
                }

                if( !candidates.empty() )
                {
                    if( candidates.size() == 1 )
                    {
                        NN_RESULT_DO(CompareTogether(
                            &candidates,
                            compareRange.GetOffset(),
                            textBuffer + textOffset,
                            workBuffer,
                            mismatchSize,
                            compareRange.GetSize()
                        ));
                    }
                    else
                    {
                        NN_RESULT_DO(CompareSeparate(
                            &candidates,
                            compareRange.GetOffset(),
                            workBuffer,
                            mismatchSize,
                            compareRange.GetSize()
                        ));
                    }

                    if( !candidates.empty() )
                    {
                        // 最長要素を取得
                        longestResult = std::max_element(candidates.begin(), candidates.end())->GetResult();

                        // 最小マッチサイズに満たなければなかったことにする
                        if( longestResult.size < m_MatchSize )
                        {
                            longestResult.size = 0;
                        }

                        if( 0 < longestResult.size )
                        {
                            m_Result.push_back(longestResult);
                        }

                        candidates.clear();
                    }
                }
            }

            int64_t endOffset = -1;

            // 現在の領域を一致分前進しリージョンハッシュを刷新する
            if( 0 < longestResult.size )
            {
                endOffset = longestResult.GetNewEndOffset();
            }
            // 現在の領域をすべてチェックし終えたので次へ
            else if( compareRange.GetSize() <= static_cast<int64_t>(m_RegionSize) )
            {
                endOffset = compareRange.GetEndOffset();
            }

            if( 0 <= endOffset )
            {
                compareRange.MoveNext(endOffset);

                const auto offset = compareRange.GetOffset();
                NN_SDK_ASSERT_LESS_EQUAL(offset, compareRange.GetEndOffset());
                const auto size = GetMinimumSize(compareRange.GetSize(), m_RegionSize);

                NN_RESULT_DO(m_NewStorage.Read(offset, textBuffer + m_RegionSize, size));

                pHashQueue = &hashQueueArray[0];

                if( m_RegionSize <= size )
                {
                    pHashQueue->Reset(textBuffer + m_RegionSize, m_BlockSize);
                }

                textOffset = m_RegionSize;
                mismatchSize = 0;
            }
            // 現在の領域を 1 ブロック前進しリージョンハッシュを更新する
            else
            {
                // 必要ならストレージから新たにデータを読み込む
                if( textOffset == m_RegionSize )
                {
                    std::memcpy(textBuffer, textBuffer + m_RegionSize, m_RegionSize);

                    const auto offset = compareRange.GetOffset() + static_cast<int64_t>(m_RegionSize);
                    NN_SDK_ASSERT_LESS_EQUAL(offset, compareRange.GetEndOffset());

                    // shiftSize < m_BlockSize の時に (m_BlockSize - shiftSize) 分足が出るので余分に読み込む
                    const auto size = GetMinimumSize(compareRange.GetEndOffset() - offset, m_RegionSize + m_BlockSize);

                    NN_RESULT_DO(m_NewStorage.Read(offset, textBuffer + m_RegionSize, size));

                    textOffset = 0;
                }

                NN_SDK_ASSERT_LESS_EQUAL(m_RegionSize + textOffset, m_RegionSize * 2);
                pHashQueue->Push(textBuffer + m_RegionSize + textOffset, m_BlockSize);

                compareRange.Advance(shiftSize);

                textOffset += shiftSize;
                mismatchSize += shiftSize;

                pHashQueue = &hashQueueArray[static_cast<int>((textOffset % m_BlockSize) / shiftSize)];

                // hashQueueArray[(mismatchSize % m_BlockSize) / shiftSize] の遅延初期化
                if( mismatchSize < static_cast<int64_t>(m_BlockSize) )
                {
                    NN_SDK_ASSERT_GREATER(textOffset, static_cast<size_t>(0));
                    pHashQueue->Reset(textBuffer + textOffset, m_BlockSize);
                }
            }

            m_pProgress->SetValue(compareRange.GetOffset());
        }

        // マッチ結果を最適化
        if( m_ResultCountCache != m_Result.size() )
        {
            Result::Merge(&m_Result);

            m_ResultCountCache = m_Result.size();
        }
    }

    m_pProgress->SetValue(m_NewStorageSize);

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

// ハッシュとデータを同時に比較します。
nn::Result BinaryMatch::CompareTogether(
                            std::vector<Candidate>* pCandidates,
                            int64_t dataOffset,
                            const char* textBuffer,
                            char* workBuffer,
                            int64_t mismatchSize,
                            int64_t compareSize
                        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCandidates);
    NN_SDK_REQUIRES_EQUAL(pCandidates->size(), static_cast<size_t>(1));
    NN_SDK_REQUIRES_GREATER_EQUAL(dataOffset, 0);
    NN_SDK_REQUIRES_NOT_NULL(textBuffer);
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(mismatchSize, 0);
    NN_SDK_REQUIRES_GREATER(compareSize, 0);

    auto& candidates = *pCandidates;

    // 後方の領域を比較
    NN_RESULT_DO(CompareNextData(&candidates, dataOffset, textBuffer, workBuffer, compareSize));

    // 最長候補を取得
    const auto candidateSize = std::max_element(candidates.begin(), candidates.end())->GetSize();

    // 最長候補が最小マッチサイズ以上になる可能性があれば前方の領域を調べる
    if( m_MatchSize <= mismatchSize + candidateSize )
    {
        if( 0 < mismatchSize )
        {
            for( auto& candidate : candidates )
            {
                // 前方の領域を調べて最長候補以上になる可能性がある
                if( candidateSize <= mismatchSize + candidate.GetSize() )
                {
                    candidate.ResetRegion(m_RegionSize);
                }
                else
                {
                    candidate.Exclude();
                }
            }

            candidates.erase(
                std::remove_if(
                    candidates.begin(),
                    candidates.end(),
                    [](const Candidate& obj) NN_NOEXCEPT
                    {
                        return obj.IsExcluded();
                    }
                ),
                candidates.end()
            );
            NN_SDK_ASSERT(!candidates.empty());

            // 前方の領域を比較
            NN_RESULT_DO(ComparePreviousData(&candidates, dataOffset, workBuffer, mismatchSize));
        }
    }
    else
    {
        candidates.clear();
    }

    NN_RESULT_SUCCESS;
}

// 一致する領域の前方を比較します。
nn::Result BinaryMatch::ComparePreviousData(
                            std::vector<Candidate>* pCandidates,
                            int64_t dataOffset,
                            char* workBuffer,
                            int64_t mismatchSize
                        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCandidates);
    NN_SDK_REQUIRES(!pCandidates->empty());
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER(mismatchSize, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(dataOffset, mismatchSize);

    const auto newBuffer = workBuffer;
    const auto oldBuffer = workBuffer + m_RegionSize;
    auto newOffset = dataOffset;
    auto remainingSize = mismatchSize;
    CandidateHolder candidates(pCandidates);

    // リージョンハッシュを含めた厳密比較を行う
    while( static_cast<int64_t>(m_RegionSize) <= remainingSize )
    {
        newOffset -= m_RegionSize;

        NN_RESULT_DO(m_NewStorage.Read(newOffset, newBuffer, m_RegionSize));

        for( auto& candidate : candidates )
        {
            if( candidate.IsExcluded() )
            {
                continue;
            }

            const auto oldOffset = candidate.GetOffset() - static_cast<int64_t>(m_RegionSize);
            if( 0 <= oldOffset )
            {
                NN_RESULT_DO(m_OldStorage.Read(oldOffset, oldBuffer, m_RegionSize));

                const auto& region = candidate.MovePreviousRegion(m_RegionSize);
                NN_SDK_ASSERT(Contains(m_Regions, region));
                NN_SDK_ASSERT(region.IsInside(oldOffset));
                NN_UNUSED(region);

                const auto matchedSize = candidate.ComparePreviousData(oldBuffer, newBuffer, m_RegionSize, m_BlockSize);
                if( matchedSize < m_RegionSize )
                {
                    candidate.Exclude(); // 部分一致は候補から外す
                }
            }
            // ストレージの範囲外
            else
            {
                candidate.Exclude();
            }

            candidates.Update(candidate);
            if( candidates.empty() )
            {
                NN_RESULT_SUCCESS;
            }
        }

        remainingSize -= m_RegionSize;
    }

    // 残りのデータの部分一致を調査
    if( 0 < remainingSize )
    {
        newOffset -= remainingSize;

        const auto readSize = static_cast<size_t>(remainingSize);
        NN_RESULT_DO(m_NewStorage.Read(newOffset, newBuffer, readSize));

        for( auto& candidate : candidates )
        {
            if( candidate.IsExcluded() )
            {
                continue;
            }

            const auto oldOffset = candidate.GetOffset() - static_cast<int64_t>(readSize);
            if( 0 <= oldOffset )
            {
                NN_RESULT_DO(m_OldStorage.Read(oldOffset, oldBuffer, readSize));

                candidate.ComparePreviousData(oldBuffer, newBuffer, readSize, m_BlockSize);
            }
        }
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

// 一致する領域の後方を比較します。
nn::Result BinaryMatch::CompareNextData(
                            std::vector<Candidate>* pCandidates,
                            int64_t dataOffset,
                            const char* textBuffer,
                            char* workBuffer,
                            int64_t compareSize
                        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCandidates);
    NN_SDK_REQUIRES(!pCandidates->empty());
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(dataOffset, 0);

    CandidateHolder candidates(pCandidates);
    auto newOffset = dataOffset;
    auto remainingSize = compareSize;

    // 比較開始位置のデータを調査
    {
        const auto newBuffer = textBuffer;
        const auto oldBuffer = workBuffer;

        for( auto& candidate : candidates )
        {
            const auto oldOffset = candidate.GetOffset();
            NN_SDK_ASSERT(candidate.GetBaseRegion().IsInside(oldOffset));

            NN_RESULT_DO(m_OldStorage.Read(oldOffset, oldBuffer, m_RegionSize));

            if( std::equal(oldBuffer, oldBuffer + m_RegionSize, newBuffer) )
            {
                candidate.ExpandPositive(m_RegionSize);
            }
            else
            {
                candidate.Exclude();
                candidates.Update(candidate);
            }
        }

        if( candidates.empty() )
        {
            remainingSize = 0;
        }
        else
        {
            remainingSize -= m_RegionSize;
        }
    }
    newOffset += m_RegionSize;

    const auto hashCount = static_cast<int>(m_RegionSize / m_BlockSize);
    const auto newBuffer = workBuffer;
    const auto oldBuffer = workBuffer + m_RegionSize;

    std::vector<Candidate> backup;

    while( static_cast<int64_t>(m_RegionSize) <= remainingSize )
    {
        NN_RESULT_DO(m_NewStorage.Read(newOffset, newBuffer, m_RegionSize));

        // 比較のためのリージョンハッシュ（遅延評価）を用意
        RegionHash hash = {};
        bool isDelay = true;
        auto compareLazyHash = [=, &hash, &isDelay](const RegionHash& other) NN_NOEXCEPT -> bool
        {
            if( isDelay )
            {
                isDelay = false;
                hash.Make(newBuffer, hashCount, m_BlockSize);
            }
            return hash == other;
        };

        for( auto& candidate : candidates )
        {
            if( candidate.IsExcluded() )
            {
                continue;
            }

            const auto& region = candidate.MoveNextRegion(m_RegionSize);

            // old 側のリージョンハッシュの範囲内
            if( Contains(m_Regions, region) )
            {
                const auto oldOffset = candidate.GetEndOffset();
                NN_SDK_ASSERT(region.IsInside(oldOffset));

                NN_RESULT_DO(m_OldStorage.Read(oldOffset, oldBuffer, m_RegionSize));

                Candidate slideCandidate = candidate;

                const auto matchedSize = candidate.CompareNextData(oldBuffer, newBuffer, m_RegionSize, m_BlockSize);
                if( matchedSize < m_RegionSize )
                {
                    candidate.Exclude(); // 部分一致は候補から外す

                    // ハッシュをまとめた領域から比較が始まっている場合は、その後ろの領域と一致するか比較する。
                    // 例えば、まとめた領域が 4 つ分で、厳密一致していた領域が 3 つ分だった場合、
                    // 今までの処理は（... ？ ● ● ● × ...   ）このパターンを検出していて、
                    // この後の処理は（   ... × ● ● ● ？ ...）このパターンを調査する。
                    if( slideCandidate.IsSlideCheckNeeded() )
                    {
                        const auto& slideRegion = slideCandidate.SlideNextRegion();

                        if( Contains(m_Regions, slideRegion) && compareLazyHash(slideRegion.hash) )
                        {
                            NN_RESULT_DO(m_OldStorage.Read(slideRegion.offset, oldBuffer, m_RegionSize));

                            if( std::equal(oldBuffer, oldBuffer + m_RegionSize, newBuffer) )
                            {
                                // 元々の候補を補欠として取っておく
                                backup.push_back(candidate);

                                candidate = slideCandidate;
                                candidate.ExpandPositive(m_RegionSize);
                            }
                        }
                    }
                }
            }
            // old 側のリージョンハッシュの範囲外
            else
            {
                const auto oldOffset = candidate.GetEndOffset();
                NN_SDK_ASSERT_LESS_EQUAL(oldOffset, m_OldStorageSize);
                const auto oldSize = static_cast<size_t>(m_OldStorageSize - oldOffset);
                NN_SDK_ASSERT_LESS(oldSize, m_RegionSize);

                // old 側末尾のデータの部分一致を調査
                if( 0 < oldSize )
                {
                    NN_RESULT_DO(m_OldStorage.Read(oldOffset, oldBuffer, oldSize));

                    candidate.CompareNextData(oldBuffer, newBuffer, oldSize, m_BlockSize);
                }

                candidate.Exclude();
            }

            candidates.Update(candidate);
            if( candidates.empty() )
            {
                remainingSize = 0;
                break;
            }
        }

        newOffset += m_RegionSize;
        remainingSize -= m_RegionSize;
    }

    // new 側末尾のデータの部分一致を調査
    if( 0 < remainingSize )
    {
        const auto newSize = static_cast<size_t>(remainingSize);
        NN_RESULT_DO(m_NewStorage.Read(newOffset, newBuffer, newSize));

        for( auto& candidate : candidates )
        {
            if( candidate.IsExcluded() )
            {
                continue;
            }

            const auto oldOffset = candidate.GetEndOffset();
            NN_SDK_ASSERT_LESS_EQUAL(oldOffset, m_OldStorageSize);
            const auto oldSize = GetMinimumSize(m_OldStorageSize - oldOffset, newSize);
            if( 0 < oldSize )
            {
                NN_RESULT_DO(m_OldStorage.Read(oldOffset, oldBuffer, oldSize));

                candidate.CompareNextData(oldBuffer, newBuffer, oldSize, m_BlockSize);
            }
        }
    }

    // 補欠候補をマージ
    if( !backup.empty() )
    {
        pCandidates->insert(pCandidates->end(), backup.begin(), backup.end());

        std::sort(
            pCandidates->begin(),
            pCandidates->end(),
            [](const Candidate& lhs, const Candidate& rhs) NN_NOEXCEPT
            {
                return lhs.GetOffset() < rhs.GetOffset();
            }
        );
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

// ハッシュ比較後にデータを比較します。
nn::Result BinaryMatch::CompareSeparate(
                            std::vector<Candidate>* pCandidates,
                            int64_t dataOffset,
                            char* workBuffer,
                            int64_t mismatchSize,
                            int64_t compareSize
                        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCandidates);
    NN_SDK_REQUIRES_GREATER(pCandidates->size(), static_cast<size_t>(1));
    NN_SDK_REQUIRES_GREATER_EQUAL(dataOffset, 0);
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(mismatchSize, 0);
    NN_SDK_REQUIRES_GREATER(compareSize, 0);

    auto& candidates = *pCandidates;

    // 後方の領域のハッシュをチェック
    NN_RESULT_DO(CompareNextHash(&candidates, dataOffset, workBuffer, compareSize));

    // 優先度の高い順にソート
    {
        auto checkOffset = dataOffset;

        // データが一致しそうな位置を計算
        if( 1 < m_ResultCountCache ) // 番兵以外の結果がある場合
        {
            struct Position
            {
                int64_t oldOffset;
                int64_t newOffset;
            };
            Position previous = {};

            const auto begin = m_Result.begin();

            // CompareData() 以前に算出した結果から最適なものを選択
            const auto next = std::lower_bound(
                begin,
                begin + m_ResultCountCache,
                dataOffset,
                [](const Result& result, int64_t offset) NN_NOEXCEPT
                {
                    return result.newOffset < offset;
                }
            );

            if( next != begin )
            {
                const auto& result = *(next - 1);
                NN_SDK_ASSERT(result.newOffset < dataOffset);

                previous.oldOffset = result.oldOffset;
                previous.newOffset = result.newOffset;
            }

            // CompareData() で算出中の結果から選択してみる
            if( m_ResultCountCache < m_Result.size() )
            {
                const auto& latest = *(m_Result.end() - 1);
                NN_SDK_ASSERT_LESS(latest.newOffset, dataOffset);

                if( previous.newOffset < latest.newOffset )
                {
                    previous.oldOffset = latest.oldOffset;
                    previous.newOffset = latest.newOffset;
                }
            }

            // Result::newOffset が dataOffset 未満になる最後の結果（previous）から、
            // データが一致する可能性のある oldOffset を算出する
            checkOffset = previous.oldOffset + (dataOffset - previous.newOffset);
        }

        // サイズの降順、checkOffset の距離との昇順、オフセットの昇順にソート
        std::sort(
            candidates.begin(),
            candidates.end(),
            [=](const Candidate& lhs, const Candidate& rhs) NN_NOEXCEPT -> bool
            {
                if( lhs.GetSize() == rhs.GetSize() )
                {
                    const auto lhsDistance = std::abs(lhs.GetOffset() - checkOffset);
                    const auto rhsDistance = std::abs(rhs.GetOffset() - checkOffset);
                    if( lhsDistance == rhsDistance )
                    {
                        return lhs.GetOffset() < rhs.GetOffset();
                    }
                    return lhsDistance < rhsDistance;
                }
                return lhs.GetSize() > rhs.GetSize();
            }
        );
        NN_SDK_ASSERT_GREATER_EQUAL(candidates.front().GetSize(), candidates.back().GetSize());
    }

    const auto candidateSize = candidates.front().GetSize() + static_cast<int64_t>(m_RegionSize - m_BlockSize) * 2;

    // 最長候補が最小マッチサイズ以上になる可能性があれば厳密比較を行う
    if( m_MatchSize <= candidateSize )
    {
        // 衝突数が多い場合は候補を絞る
        if( ConflictCountMax < candidates.size() )
        {
            candidates.resize(ConflictCountMax);
        }

        for( auto& candidate : candidates )
        {
            candidate.ResetRegion(m_RegionSize);
        }

        NN_RESULT_DO(CompareDataStrict(&candidates, dataOffset, workBuffer, mismatchSize, compareSize));
    }
    else
    {
        candidates.clear();
    }

    NN_RESULT_SUCCESS;
}

// 一致した領域より後方部分のリージョンハッシュを比較します。
nn::Result BinaryMatch::CompareNextHash(
                            std::vector<Candidate>* pCandidates,
                            int64_t dataOffset,
                            char* workBuffer,
                            int64_t compareSize
                        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCandidates);
    NN_SDK_REQUIRES(!pCandidates->empty());
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(dataOffset, 0);

    CandidateHolder candidates(pCandidates);

    // 先頭のハッシュは一致済み
    for( auto& candidate : candidates )
    {
        candidate.ExpandPositive(m_RegionSize);
    }

    const auto hashCount = static_cast<int>(m_RegionSize / m_BlockSize);
    const auto newBuffer = workBuffer;
    auto newOffset = dataOffset + static_cast<int64_t>(m_RegionSize);
    auto remainingSize = compareSize - static_cast<int64_t>(m_RegionSize);
    std::vector<Candidate> backup;

    while( static_cast<int64_t>(m_RegionSize) <= remainingSize )
    {
        NN_RESULT_DO(m_NewStorage.Read(newOffset, newBuffer, m_RegionSize));

        // 調査対象のリージョンハッシュを用意
        RegionHash hash = {};
        hash.Make(newBuffer, hashCount, m_BlockSize);

        for( auto& candidate : candidates )
        {
            if( candidate.IsExcluded() )
            {
                continue;
            }

            const auto& region = candidate.MoveNextRegion(m_RegionSize);

            // old 側のリージョンハッシュの範囲内
            if( Contains(m_Regions, region) )
            {
                NN_SDK_ASSERT(region.IsInside(candidate.GetEndOffset()));

                if( hash == region.hash )
                {
                    candidate.ExpandPositive(m_RegionSize);
                }
                else
                {
                    Candidate slideCandidate = candidate;

                    candidate.Exclude();

                    // ハッシュをまとめた領域から比較が始まっている場合は、その後ろの領域と一致するか比較する。
                    // 例えば、まとめた領域が 4 つ分で、厳密一致していた領域が 3 つ分だった場合、
                    // 今までの処理は（... ？ ● ● ● × ...   ）このパターンを検出していて、
                    // この後の処理は（   ... × ● ● ● ？ ...）このパターンを調査する。
                    if( slideCandidate.IsSlideCheckNeeded() )
                    {
                        const auto& slideRegion = slideCandidate.SlideNextRegion();

                        if( Contains(m_Regions, slideRegion) && (hash == slideRegion.hash) )
                        {
                            // 元々の候補を補欠として取っておく
                            backup.push_back(candidate);

                            candidate = slideCandidate;
                            candidate.ExpandPositive(m_RegionSize);
                        }
                    }
                }
            }
            // old 側のリージョンハッシュの範囲外
            else
            {
                candidate.Exclude();
            }

            candidates.Update(candidate);
            if( candidates.empty() )
            {
                remainingSize = 0;
                break;
            }
        }

        newOffset += m_RegionSize;
        remainingSize -= m_RegionSize;
    }

    // 補欠候補をマージ
    if( !backup.empty() )
    {
        pCandidates->insert(pCandidates->end(), backup.begin(), backup.end());

        std::sort(
            pCandidates->begin(),
            pCandidates->end(),
            [](const Candidate& lhs, const Candidate& rhs) NN_NOEXCEPT
            {
                return lhs.GetOffset() < rhs.GetOffset();
            }
        );
    }

    NN_RESULT_SUCCESS;
}

// リージョンハッシュが一致した部分の厳密比較をします。
nn::Result BinaryMatch::CompareDataStrict(
                            std::vector<Candidate>* pCandidates,
                            int64_t dataOffset,
                            char* workBuffer,
                            int64_t mismatchSize,
                            int64_t compareSize
                        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCandidates);
    NN_SDK_REQUIRES(!pCandidates->empty());
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(dataOffset, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(mismatchSize, 0);
    NN_SDK_REQUIRES_GREATER(compareSize, 0);

    const auto newBuffer = workBuffer;
    const auto oldBuffer = workBuffer + m_RegionSize;
    auto& candidates = *pCandidates;
    int64_t maximumSize = 0;

    for( bool isContinue = true; isContinue; )
    {
        isContinue = false;

        for( auto& candidate : candidates )
        {
            if( candidate.IsExcluded() )
            {
                continue;
            }

            const auto baseOffset = candidate.GetBaseOffset();
            const auto range = std::make_pair(candidate.GetOffset(), candidate.GetEndOffset());

            candidate = Candidate(candidate.GetBaseRegion(), baseOffset, dataOffset, 0);

            // 前方を厳密比較
            {
                const auto endOldOffset = range.first;
                const auto endNewOffset = dataOffset - mismatchSize;
                auto oldOffset = baseOffset;
                auto newOffset = dataOffset;

                while( endOldOffset <= oldOffset )
                {
                    NN_SDK_ASSERT_GREATER_EQUAL(oldOffset, 0);
                    NN_SDK_ASSERT_GREATER_EQUAL(newOffset, endNewOffset);

                    const auto oldSize = oldOffset;
                    const auto newSize = newOffset - endNewOffset;

                    const auto readSize = GetMinimumSize(std::min(oldSize, newSize), m_RegionSize);
                    if( readSize == 0 )
                    {
                        break;
                    }

                    oldOffset -= readSize;
                    newOffset -= readSize;

                    NN_RESULT_DO(m_OldStorage.Read(oldOffset, oldBuffer, readSize));
                    NN_RESULT_DO(m_NewStorage.Read(newOffset, newBuffer, readSize));

                    const auto matchSize = candidate.ComparePreviousData(oldBuffer, newBuffer, readSize, m_BlockSize);
                    if( matchSize < m_RegionSize )
                    {
                        break;
                    }
                }
            }

            // 後方を厳密比較
            {
                const auto endOldOffset = range.second;
                const auto endNewOffset = dataOffset + compareSize;
                auto oldOffset = baseOffset;
                auto newOffset = dataOffset;

                while( oldOffset <= endOldOffset )
                {
                    NN_SDK_ASSERT_LESS_EQUAL(oldOffset, m_OldStorageSize);
                    NN_SDK_ASSERT_LESS_EQUAL(newOffset, endNewOffset);

                    const auto oldSize = m_OldStorageSize - oldOffset;
                    const auto newSize = endNewOffset - newOffset;

                    const auto readSize = GetMinimumSize(std::min(oldSize, newSize), m_RegionSize);
                    if( readSize == 0 )
                    {
                        break;
                    }

                    NN_RESULT_DO(m_OldStorage.Read(oldOffset, oldBuffer, readSize));
                    NN_RESULT_DO(m_NewStorage.Read(newOffset, newBuffer, readSize));

                    const auto matchSize = candidate.CompareNextData(oldBuffer, newBuffer, readSize, m_BlockSize);
                    if( matchSize < m_RegionSize )
                    {
                        break;
                    }

                    oldOffset += readSize;
                    newOffset += readSize;
                }
            }

            candidate.Exclude();

            // 必要な候補を選別する
            if( maximumSize < candidate.GetSize() )
            {
                static const int64_t MaximumSizeThreshold = 1024 * 1024;

                maximumSize = candidate.GetSize();

                // NOTE: 「一致したハッシュの長さ + 前後の部分一致の長さ < マッチしたデータの長さ」であれば、
                //       厳密比較の対象から外しても問題ない。
                const auto threshold = (MaximumSizeThreshold <= maximumSize)
                                            ? maximumSize
                                            : (maximumSize - static_cast<int64_t>(m_RegionSize - m_BlockSize) * 2);
                if( static_cast<int64_t>(m_RegionSize) <= threshold )
                {
                    const auto ptr = &candidate;

                    candidates.erase(
                        std::remove_if(
                            candidates.begin(),
                            candidates.end(),
                            [=](const Candidate& obj) NN_NOEXCEPT
                            {
                                return (ptr != &obj) && (obj.GetSize() <= threshold);
                            }
                        ),
                        candidates.end()
                    );
                    NN_SDK_ASSERT(!candidates.empty());
                }
            }

            isContinue = true;

            break;
        }
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

// 比較結果を生成します。
void BinaryMatch::MakeResult(int64_t storageSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(storageSize, 0);

    if( m_Result.empty() )
    {
        m_Result.push_back(BinaryMatchResult::MakeUnknown(0, storageSize));
    }
    // newStorage を網羅するように m_Result を調整
    else
    {
        const auto count = m_Result.size();

        // 先頭の領域をチェック
        {
            const auto& front = m_Result[0];
            NN_SDK_ASSERT_NOT_EQUAL(front.newOffset, Result::InvalidOffset);
            NN_SDK_ASSERT_NOT_EQUAL(front.oldOffset, Result::InvalidOffset);
            NN_SDK_ASSERT_GREATER(front.size, 0);
            NN_SDK_ASSERT_EQUAL(front.detail, BinaryMatchDetail_Match);

            // 前方に不一致領域あり
            if( 0 < front.newOffset )
            {
                m_Result.push_back(Result::MakeUnknown(0, front.newOffset));
            }
        }

        for( size_t i = 1; i < count; ++i )
        {
            const auto& previous = m_Result[i - 1];
            const auto& current = m_Result[i];
            NN_SDK_ASSERT_LESS(previous.newOffset, current.newOffset);
            NN_SDK_ASSERT_NOT_EQUAL(current.oldOffset, Result::InvalidOffset);
            NN_SDK_ASSERT_GREATER(current.size, 0);
            NN_SDK_ASSERT_EQUAL(current.detail, BinaryMatchDetail_Match);

            const auto previousOffset = previous.newOffset + previous.size;
            const auto currentOffset = current.newOffset;
            NN_SDK_ASSERT_LESS_EQUAL(previousOffset, currentOffset);

            const auto mismatchSize = currentOffset - previousOffset;

            // 不一致領域を追加
            if( 0 < mismatchSize )
            {
                m_Result.push_back(Result::MakeUnknown(previousOffset, mismatchSize));
            }
        }

        // 末尾の領域をチェック
        {
            const auto& back = m_Result[count - 1];
            const auto backOffset = back.newOffset + back.size;
            NN_SDK_ASSERT_LESS_EQUAL(backOffset, storageSize);

            // 不一致領域あり
            if( backOffset < storageSize )
            {
                m_Result.push_back(Result::MakeUnknown(backOffset, storageSize - backOffset));
            }
        }

        std::sort(m_Result.begin(), m_Result.end());
    }

    // 比較しない領域を排除
    if( !m_Result.empty() )
    {
        if( !m_ExcludeRangeNew.empty() )
        {
            ExcludeNewStorage();
        }

        if( !m_ExcludeRangeOld.empty() )
        {
            ExcludeOldStorage();
        }

        if( !m_ExcludeRangeNew.empty() || !m_ExcludeRangeOld.empty() )
        {
            Result::Merge(&m_Result);
        }
    }
}

// 比較結果から新しいストレージの比較しない領域を排除します。
void BinaryMatch::ExcludeNewStorage() NN_NOEXCEPT
{
    int startIndex = 0;
    const int endIndex = static_cast<int>(m_Result.size());

    // NOTE: 新しいストレージの領域は被っていない前提で処理をする
    for( int i = 0, count = static_cast<int>(m_ExcludeRangeNew.size()); i < count; ++i )
    {
        auto range = m_ExcludeRangeNew[i];

        // 比較しない領域を含む比較結果を検索
        const auto ptr = std::upper_bound(
            m_Result.data() + startIndex,
            m_Result.data() + endIndex,
            range.offset,
            [](int64_t offset, const Result& result) NN_NOEXCEPT
            {
                return offset < result.newOffset;
            }
        ) - 1;

        auto splitIndex = static_cast<int>(std::distance(m_Result.data(), ptr));
        NN_SDK_ASSERT_GREATER_EQUAL(splitIndex, startIndex);
        NN_SDK_ASSERT_LESS_EQUAL(ptr->newOffset, range.offset);

        // ストレージ末尾より後ろが指定されている
        if( ptr->newOffset + ptr->size < range.offset )
        {
            NN_SDK_REQUIRES_EQUAL(splitIndex, endIndex - 1);
            return;
        }

        while( 0 < range.size )
        {
            const auto split = m_Result[splitIndex];

            if( split.oldOffset != Result::InvalidOffset )
            {
                if( split.newOffset == range.offset )
                {
                    // 結果をそのまま不一致に書き換え
                    if( split.size <= range.size )
                    {
                        auto& result = m_Result[splitIndex];
                        result = Result::MakeUnknown(result.newOffset, result.size);
                    }
                    // 不一致・一致に分割
                    else
                    {
                        const auto exclude = Result::MakeUnknown(split.newOffset, range.size);
                        m_Result.push_back(exclude);

                        const auto forward = exclude.size;
                        NN_SDK_ASSERT_LESS(forward, split.size);

                        auto& result = m_Result[splitIndex];
                        result.newOffset += forward;
                        result.oldOffset += forward;
                        result.size -= forward;
                        NN_SDK_ASSERT_EQUAL(result.storageIndex, 0);

                        break; // ↑で設定した領域を次の調査の先頭にする
                    }
                }
                // 一致・不一致に分割
                else if( split.newOffset + split.size <= range.offset + range.size )
                {
                    const auto include = Result::MakeMatch(split.newOffset, split.oldOffset, range.offset - split.newOffset);
                    m_Result.push_back(include);

                    const auto forward = include.size;
                    NN_SDK_ASSERT_LESS(forward, split.size);

                    auto& result = m_Result[splitIndex];
                    result = Result::MakeUnknown(result.newOffset + forward, result.size - forward);
                }
                // 一致・不一致・一致に分割
                else
                {
                    const auto include = Result::MakeMatch(split.newOffset, split.oldOffset, range.offset - split.newOffset);
                    m_Result.push_back(include);

                    const auto exclude = Result::MakeUnknown(range.offset, range.size);
                    m_Result.push_back(exclude);

                    const auto forward = include.size + exclude.size;
                    NN_SDK_ASSERT_LESS(forward, split.size);

                    auto& result = m_Result[splitIndex];
                    result.newOffset += forward;
                    result.oldOffset += forward;
                    result.size -= forward;
                    NN_SDK_ASSERT_EQUAL(result.storageIndex, 0);

                    break; // ↑で設定した領域を次の調査の先頭にする
                }
            }
            else
            {
                // 比較しない領域が途中で終わるのでループを抜けて次の調査の先頭にする
                if( range.offset + range.size < split.newOffset + split.size )
                {
                    break;
                }
            }

            ++splitIndex;
            if( endIndex <= splitIndex )
            {
                return;
            }

            auto forward = split.size - (range.offset - split.newOffset);
            NN_SDK_ASSERT_GREATER(forward, 0);

            range.offset += forward;
            range.size -= forward;

            NN_SDK_ASSERT_EQUAL(range.offset, m_Result[splitIndex].newOffset);
        }

        startIndex = splitIndex;
    }
} // NOLINT(impl/function_size)

// 比較結果から古いストレージの比較しない領域を排除します。
void BinaryMatch::ExcludeOldStorage() NN_NOEXCEPT
{
    class ResultComparer
    {
    public:
        bool operator()(const BinaryMatchResult& lhs, const BinaryMatchResult& rhs) const NN_NOEXCEPT
        {
            if( lhs.oldOffset == BinaryMatchResult::InvalidOffset )
            {
                return rhs.oldOffset != BinaryMatchResult::InvalidOffset;
            }
            else
            {
                return rhs.oldOffset != BinaryMatchResult::InvalidOffset &&
                    lhs.oldOffset + lhs.size < rhs.oldOffset + rhs.size;
            }
        }
    };

    std::sort(m_Result.begin(), m_Result.end(), ResultComparer());

    int startIndex = 0;
    const int endIndex = static_cast<int>(m_Result.size());

    // 古いストレージの領域は被っていることを考慮して処理をする
    for( int i = 0, count = static_cast<int>(m_ExcludeRangeOld.size()); i < count; ++i )
    {
        const auto range = m_ExcludeRangeOld[i];

        // 比較しない領域を含む（であろう）比較結果を検索
        const auto ptr = std::upper_bound(
            m_Result.data() + startIndex,
            m_Result.data() + endIndex,
            range.offset,
            [](int64_t offset, const Result& result) NN_NOEXCEPT
            {
                return result.oldOffset != Result::InvalidOffset &&
                       offset < result.oldOffset + result.size;
            }
        );

        startIndex = static_cast<int>(std::distance(m_Result.data(), ptr));
        if( startIndex == endIndex )
        {
            continue;
        }

        for( int splitIndex = startIndex; splitIndex < endIndex; ++splitIndex )
        {
            const auto split = m_Result[splitIndex];

            // 比較しない領域を超えていたら終了
            if( range.offset + range.size <= split.oldOffset )
            {
                break;
            }

            // 比較しない領域に至っていない
            if( split.oldOffset + split.size <= range.offset )
            {
                // 何もしない
            }
            else if( split.oldOffset + split.size <= range.offset + range.size )
            {
                // 結果をそのまま不一致に書き換え
                if( range.offset <= split.oldOffset )
                {
                    auto& result = m_Result[splitIndex];
                    result = Result::MakeUnknown(result.newOffset, result.size);
                }
                // 一致・不一致に分割
                else
                {
                    const auto include = Result::MakeMatch(split.newOffset, split.oldOffset, range.offset - split.oldOffset);
                    m_Result.push_back(include);

                    const auto forward = include.size;
                    NN_SDK_ASSERT_LESS(forward, split.size);

                    auto& result = m_Result[splitIndex];
                    result = Result::MakeUnknown(result.newOffset + forward, result.size - forward);
                }

                // oldOffset が同値の（可能性のある）データをソートし直し
                std::sort(
                    m_Result.begin() + startIndex,
                    m_Result.begin() + splitIndex + 1,
                    ResultComparer()
                );
            }
            else
            {
                // 不一致・一致に分割
                if( range.offset <= split.oldOffset )
                {
                    const auto exclude = Result::MakeUnknown(split.newOffset, range.offset + range.size - split.oldOffset);
                    m_Result.push_back(exclude);

                    auto forward = exclude.size;
                    NN_SDK_ASSERT_LESS(forward, split.size);

                    auto& result = m_Result[splitIndex];
                    result.newOffset += forward;
                    result.oldOffset += forward;
                    result.size -= forward;
                    NN_SDK_ASSERT_EQUAL(result.storageIndex, 0);
                }
                // 一致・不一致・一致に分割
                else
                {
                    const auto include = Result::MakeMatch(split.newOffset, split.oldOffset, range.offset - split.oldOffset);
                    m_Result.push_back(include);

                    const auto exclude = Result::MakeUnknown(range.offset, range.size);
                    m_Result.push_back(exclude);

                    const auto forward = include.size + exclude.size;
                    NN_SDK_ASSERT_LESS(forward, split.size);

                    auto& result = m_Result[splitIndex];
                    result.newOffset += forward;
                    result.oldOffset += forward;
                    result.size -= forward;
                    NN_SDK_ASSERT_EQUAL(result.storageIndex, 0);
                }
            }
        }
    }
} // NOLINT(impl/function_size)

}}}
