﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fssystem/fs_PathTool.h>
#include <nn/fssystem/dbm/fs_DbmPathTool.h>
#include <nn/fssystem/save/fs_SavePath.h>

namespace nn { namespace fssystem { namespace save {

Result Path::SetChildPath(const char* childName) NN_NOEXCEPT
{
    const auto childNameLength = strnlen(childName, fs::EntryNameLengthMax);
    const auto parentPathLength = GetLength();
    const auto endsWithSeparator = 0 < parentPathLength && m_Buf[parentPathLength - 1] == '/';
    const auto fullPathLength = parentPathLength + (endsWithSeparator ? 0 : 1) + childNameLength;

    if( nn::fs::EntryNameLengthMax < fullPathLength )
    {
        return nn::fs::ResultTooLongPath();
    }

    if( !endsWithSeparator )
    {
        m_Buf[parentPathLength] = '/';
    }

    memcpy(m_Buf + parentPathLength + (endsWithSeparator ? 0 : 1), childName, childNameLength);
    m_Buf[fullPathLength] = '\0';

    NN_RESULT_SUCCESS;
}

Result Path::SetParentPath() NN_NOEXCEPT
{
    if( m_Buf[0] == '/' && m_Buf[1] != '\0' )
    {
        for( auto index = GetLength() - 1; 0 < index; --index )
        {
            if( m_Buf[index] == '/' )
            {
                m_Buf[index] = '\0';
                NN_RESULT_SUCCESS;
            }
        }

        m_Buf[1] = '\0';
        NN_RESULT_SUCCESS;
    }
    else
    {
        return nn::fs::ResultDirectoryUnobtainable();
    }
}

/**
* @brief パスが等しいかどうか比較します。
*
* @param[in] str1 パス
* @param[in] str2 パス
*
* @return パスが等しければ true、そうでなければ false
*/
bool Path::IsEqualPath(const Path& str1, const Path& str2) NN_NOEXCEPT
{
    bool result = false;
    size_t length = str1.GetLength();
    if( length == str2.GetLength() )
    {
        result = (0 == std::memcmp(str1.GetString(), str2.GetString(), length * sizeof(Path::Char)));
    }
    return result;
}

/**
* @brief パスが別のパスの部分文字列かどうか判定します。
*
* @param[in] strFullset パス
* @param[in] strSubset 含まれているかを調べるパス
*
* @return strSubset が strFullset に含まれていれば true、そうでばければ false
*/
bool Path::IsContainingPath(const Path& strFullset, const Path& strSubset) NN_NOEXCEPT
{
    // p1, p2は正規化されたパスである必要があります。
    // p1: 集合     "/dir1/dir2/dir3/file"
    // p2: 部分集合 "/dir1/dir2"
    const Path::Char* p1 = strFullset.GetString();
    const Path::Char* p2 = strSubset.GetString();

    size_t length1 = strFullset.GetLength();
    size_t length2 = strSubset.GetLength();
    if( length1 < length2 )
    {
        // 全体集合より部分集合のほうが長い場合は含まない
        return false;
    }
    if( std::memcmp(p1, p2, length2 * sizeof(*p1)) != 0 )
    {
        // 部分集合の内容が違う
        return false;
    }
    if( length1 == length2 )
    {
        // 全体集合と部分集合は完全に一緒
        return true;
    }
    if( (length2 == 1) || (p1[length2] == '/') )
    {
        // パスセパレータを検出した
        return true;
    }
    return false;
}

/**
* @brief ディレクトリと、その中に入っているファイルやディレクトリという関係かどうかを判定します。
*        (/dir1 と　/dir1/file という関係)
*
* @param[in] strDirectory ディレクトリパス
* @param[in] strSubEntry サブエントリーパス
*
* @return strSubEntry が strDirectory の子ファイル、ディレクトリであれば true、そうでなければ false
*/
bool Path::IsInsidePath(const Path& strDirectory, const Path& strSubEntry) NN_NOEXCEPT
{
    // p1, p2は正規化されたパスである必要があります
    // p1: ディレクトリ       "/dir1"
    // p2: その中のエントリー "/dir1/dir2"
    const Path::Char* p1 = strDirectory.GetString();
    const Path::Char* p2 = strSubEntry.GetString();

    size_t length1 = strDirectory.GetLength();
    size_t length2 = strSubEntry.GetLength();
    if( length1 >= length2 )
    {
        // パス名の長さが意図通りではない
        return  false;
    }
    if( std::memcmp(p1, p2, length1 * sizeof(*p1)) != 0 )
    {
        // 共通の部分が異なる
        return  false;
    }
    // これ以降、p2は"/" を含まない
    for( size_t i = length1 + 1;i < length2; ++i )
    {
        if( p2[i] == '/' )
        {
            return  false;
        }
    }
    return true;
}

}}}

