﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <cstddef>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/fs/fs_Result.h>
#include <nn/fssystem/save/fs_IInternalStorageFileSystemVisitor.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystem.h>

// SaveDataFileSystemCore::QueryTotalSize を呼び出すためにインクルードします。
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.impl.h>

namespace nn { namespace fssystem { namespace save {

namespace
{
    //! 完全性セーブデータファイルシステムの初期化済みコード 'DISF'
    const uint32_t MagicCode = 0x46534944;

    // ファイルシステムのバージョン
    // 0x00040000 : 製品リリース ～ 5NUP よりも古い本体で作成したセーブデータ
    // 0x00050000 : 5NUP 以降で作成したセーブデータ
    const uint32_t VersionCode = 0x00050000;
    const uint32_t VersionCodeMask = 0xFFFF0000;
    const uint32_t VersionSupportSaveDataMetaHash = 0x00050000;
    const uint32_t VersionSupportedMin = 0x00040000;
    const uint32_t VersionSupportedMax = 0x00050000;
    const uint32_t VersionMajorShift = 16;

    const int RemapBufferCount = 16;
    const int MetaBufferCount = 32;

    // マスターハッシュのサイズ
    const int64_t MasterHashSize = 32;

    // メタデータに対する階層ハッシュのブロックサイズ
    const size_t SaveDataMetaIntegrityBlockSize = 16 * 1024;

    // ・L2 ビットマップ × 2
    // ・完全性保証メタデータ × 2
    // ・セーブデータ本体
    const int FileSystemRemapEntryCount = 5;

    // ・ジャーナリングメタ
    // ・ジャーナリングビットマップ × 3
    // ・階層ハッシュ L1 ～ L3
    // ・セーブデータメタ
    // ・セーブデータメタの階層ハッシュ L1 ～ L2 (VersionSupportSaveDataMetaHash 以降)
    const int MetaRemapEntryCount = 10;
    const int MetaRemapEntryCountV4 = 8;

    inline int64_t AdjustAlign64(int64_t offset, int64_t align)
    {
        return ((offset + align - 1) & ~(align - 1));
    }

    inline int GetMetaRemapEntryCount(uint32_t version) NN_NOEXCEPT
    {
        if( (version & VersionCodeMask) < (VersionSupportSaveDataMetaHash & VersionCodeMask) )
        {
            return MetaRemapEntryCountV4;
        }
        return MetaRemapEntryCount;
    }

    inline int64_t GetRemappedFileSystemEntryTableSize(uint32_t version, int countExpandMax) NN_NOEXCEPT
    {
        // このリマップストレージには
        // ・L2 ビットマップ × 2
        // ・メタデータ × 2
        // ・セーブデータ本体
        // が配置されるので、ファイルシステムを拡張するためには
        // その個数回マッピングの拡張を行う必要があります。
        // また、少なくともフォーマット 1 回 + 拡張 countExpandMax 回ができる必要があります。
        NN_UNUSED(version);
        return RemapStorage::QueryEntryTableSize((1 + countExpandMax) * FileSystemRemapEntryCount);
    }

    inline int64_t GetRemappedMetaEntryTableSize(uint32_t version, int countExpandMax) NN_NOEXCEPT
    {
        // このリマップストレージには
        // ・ジャーナリングメタ
        // ・ジャーナリングビットマップ × 3、
        // ・階層ハッシュ L1 ～ L3
        // ・セーブデータメタ
        // ・セーブデータメタの階層ハッシュ L1 ～ L2 (VersionSupportSaveDataMetaHash 以降)
        // が配置されるので、ファイルシステムを拡張するためにはその個数回マッピングの拡張を
        // 行う必要があります。
        // また、少なくともフォーマット 1 回 + 拡張 countExpandMax 回ができる必要があります。
        return RemapStorage::QueryEntryTableSize((1 + countExpandMax) * GetMetaRemapEntryCount(version));
    }

    inline int GetExpandCountMaxFromRemappedFileSystem(uint32_t version, int countUpdateMax) NN_NOEXCEPT
    {
        NN_UNUSED(version);
        return (countUpdateMax / FileSystemRemapEntryCount) - 1;
    }

    inline int GetExpandCountMaxFromRemappedMeta(uint32_t version, int countUpdateMax) NN_NOEXCEPT
    {
        return (countUpdateMax / GetMetaRemapEntryCount(version)) - 1;
    }
}

NN_DEFINE_STATIC_CONSTANT( const size_t JournalIntegritySaveDataFileSystem::ControlAreaHashSize );
JournalIntegritySaveDataFileSystem::ControlAreaHolder::ControlAreaHolder() NN_NOEXCEPT
    : m_pHeader(),
    m_pStorage(),
    m_BufferSize(0),
    m_BitmapSize(0)
{
}

Result JournalIntegritySaveDataFileSystem::ControlAreaHolder::Reset(const MasterHeader* pAtomic) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAtomic);

    if( pAtomic->controlArea.header.sizeMasterHash != MasterHashSize
        || !nn::util::IsIntValueRepresentable<size_t>(pAtomic->controlArea.header.sizeBitmapL1) )
    {
        return nn::fs::ResultInvalidJournalIntegritySaveDataHashSize();
    }

    m_Version = pAtomic->controlArea.header.version;

    const auto controlAreaSize = offsetof(MasterHeader, reserved) - sizeof(MasterHeader::ControlArea);
    const auto bitmapSize = static_cast<size_t>(pAtomic->controlArea.header.sizeBitmapL1);
    const auto bufferSize = static_cast<size_t>(controlAreaSize + MasterHashSize * 2 + bitmapSize * 2);

    if( m_BufferSize == 0 )
    {
        m_BitmapSize = bitmapSize;
        m_BufferSize = bufferSize;

        m_pHeader = nn::fs::detail::MakeUnique<char[]>(m_BufferSize);
        if( m_pHeader == nullptr )
        {
            return nn::fs::ResultAllocationMemoryFailedMakeUnique();
        }

        m_pStorage.reset(new nn::fs::MemoryStorage(m_pHeader.get(), static_cast<int64_t>(bufferSize)));
        if( m_pStorage == nullptr )
        {
            return nn::fs::ResultAllocationMemoryFailedNew();
        }
    }
    else
    {
        NN_SDK_ASSERT_GREATER_EQUAL(m_BitmapSize, bitmapSize);
        NN_SDK_ASSERT_GREATER_EQUAL(m_BufferSize, bufferSize);
    }

    NN_SDK_ASSERT_NOT_NULL(m_pHeader);
    NN_SDK_ASSERT_NOT_NULL(m_pStorage);

    const auto pDst = reinterpret_cast<char*>(m_pHeader.get());
    const auto pSrc = reinterpret_cast<const char*>(pAtomic);
    int64_t offsetDst = 0;

    // 以下の順でバッファに配置します
    // ・管理領域
    // ・マスターハッシュ
    // ・L1 ビットマップ x 2
    // ・セーブデータメタのマスターハッシュ

    std::memcpy(
        pDst + offsetDst,
        pSrc + offsetof(MasterHeader, hierarchicalDuplexControlArea),
        controlAreaSize
    );
    offsetDst += controlAreaSize;
    std::memcpy(
        pDst + offsetDst,
        pSrc + pAtomic->controlArea.header.offsetMasterHash,
        MasterHashSize
    );
    offsetDst += MasterHashSize;
    std::memcpy(
        pDst + offsetDst,
        pSrc + pAtomic->controlArea.header.offsetsBitmapL1[0],
        m_BitmapSize
    );
    offsetDst += m_BitmapSize;
    std::memcpy(
        pDst + offsetDst,
        pSrc + pAtomic->controlArea.header.offsetsBitmapL1[1],
        m_BitmapSize
    );
    offsetDst += m_BitmapSize;
    std::memcpy(
        pDst + offsetDst,
        pSrc + pAtomic->controlArea.header.offsetSaveDataMetaMasterHash,
        MasterHashSize
    );
    offsetDst += MasterHashSize;

    NN_RESULT_SUCCESS;
}

void JournalIntegritySaveDataFileSystem::ControlAreaHolder::Clear() NN_NOEXCEPT
{
    m_BitmapSize = 0;
    m_BufferSize = 0;
    m_pHeader.reset();
    m_pStorage.reset();
}

void JournalIntegritySaveDataFileSystem::ControlAreaHolder::Commit(MasterHeader* pAtomic) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHeader);
    const auto pDst = reinterpret_cast<char*>(pAtomic);
    const auto pSrc = reinterpret_cast<const char*>(m_pHeader.get());
    const auto controlAreaSize = offsetof(MasterHeader, reserved) - sizeof(MasterHeader::ControlArea);

    int64_t offsetSrc = 0;

    const auto originalExtraData = pAtomic->originalExtraData;
    std::memcpy(
        pDst + offsetof(MasterHeader, hierarchicalDuplexControlArea),
        pSrc + offsetSrc,
        controlAreaSize
    );
    offsetSrc += controlAreaSize;
    pAtomic->originalExtraData = originalExtraData;

    std::memcpy(
        pDst + pAtomic->controlArea.header.offsetMasterHash,
        pSrc + offsetSrc,
        MasterHashSize
    );
    offsetSrc += MasterHashSize;
    std::memcpy(
        pDst + pAtomic->controlArea.header.offsetsBitmapL1[0],
        pSrc + offsetSrc,
        m_BitmapSize
    );
    offsetSrc += m_BitmapSize;
    std::memcpy(
        pDst + pAtomic->controlArea.header.offsetsBitmapL1[1],
        pSrc + offsetSrc,
        m_BitmapSize
    );
    offsetSrc += m_BitmapSize;
    std::memcpy(
        pDst + pAtomic->controlArea.header.offsetSaveDataMetaMasterHash,
        pSrc + offsetSrc,
        MasterHashSize
    );
    offsetSrc += MasterHashSize;
}

JournalIntegritySaveDataFileSystem::MasterHeader::CommitData* JournalIntegritySaveDataFileSystem::ControlAreaHolder::GetCommitData() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHeader);
    return reinterpret_cast<MasterHeader::CommitData*>(m_pHeader.get() + offsetof(MasterHeader, commitData) - sizeof(MasterHeader::ControlArea));
}

inline JournalIntegritySaveDataFileSystem::ExtraData* JournalIntegritySaveDataFileSystem::ControlAreaHolder::GetExtraData() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHeader);
    return reinterpret_cast<ExtraData*>(m_pHeader.get() + offsetof(MasterHeader, extraData) - sizeof(MasterHeader::ControlArea));
}

inline const nn::fs::SubStorage JournalIntegritySaveDataFileSystem::ControlAreaHolder::GetDuplexControlAreaStorage() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    return nn::fs::SubStorage(
               m_pStorage.get(),
               offsetof(MasterHeader, hierarchicalDuplexControlArea) - sizeof(MasterHeader::ControlArea),
               sizeof(decltype(MasterHeader::hierarchicalDuplexControlArea))
           );
}

inline const nn::fs::SubStorage JournalIntegritySaveDataFileSystem::ControlAreaHolder::GetFileSystemRemapControlAreaStorage() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    return nn::fs::SubStorage(
               m_pStorage.get(),
               offsetof(MasterHeader, fileSystemRemapControlArea) - sizeof(MasterHeader::ControlArea),
               sizeof(decltype(MasterHeader::fileSystemRemapControlArea))
           );
}

inline const nn::fs::SubStorage JournalIntegritySaveDataFileSystem::ControlAreaHolder::GetMetaRemapControlAreaStorage() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    return nn::fs::SubStorage(
               m_pStorage.get(),
               offsetof(MasterHeader, metaRemapControlArea) - sizeof(MasterHeader::ControlArea),
               sizeof(decltype(MasterHeader::metaRemapControlArea))
           );
}

inline const nn::fs::SubStorage JournalIntegritySaveDataFileSystem::ControlAreaHolder::GetIntegrityControlAreaStorage() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    const auto offset = offsetof(MasterHeader, hierarchicalIntegrityControlArea) - sizeof(MasterHeader::ControlArea);
    const auto size = sizeof(decltype(MasterHeader::hierarchicalIntegrityControlArea));
    return nn::fs::SubStorage(m_pStorage.get(), offset, size);
}

inline const nn::fs::SubStorage JournalIntegritySaveDataFileSystem::ControlAreaHolder::GetJournalControlAreaStorage() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    const auto offset = offsetof(MasterHeader, journalControlArea) - sizeof(MasterHeader::ControlArea);
    const auto size = sizeof(decltype(MasterHeader::journalControlArea));
    return nn::fs::SubStorage(m_pStorage.get(), offset, size);
}

inline const nn::fs::SubStorage JournalIntegritySaveDataFileSystem::ControlAreaHolder::GetSaveDataControlAreaStorage() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    const auto offset = offsetof(MasterHeader, saveDataControlArea) - sizeof(MasterHeader::ControlArea);
    const auto size = sizeof(decltype(MasterHeader::saveDataControlArea));
    return nn::fs::SubStorage(m_pStorage.get(), offset, size);
}

inline const nn::fs::SubStorage JournalIntegritySaveDataFileSystem::ControlAreaHolder::GetLayeredHashSaveDataMetaControlAreaStorage() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    const auto offset = offsetof(MasterHeader, saveDataMetaHashControlArea) - sizeof(MasterHeader::ControlArea);
    const auto size = sizeof(decltype(MasterHeader::saveDataMetaHashControlArea));
    return nn::fs::SubStorage(m_pStorage.get(), offset, size);
}

inline const nn::fs::SubStorage JournalIntegritySaveDataFileSystem::ControlAreaHolder::GetMasterHashStorage() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    // ControlAreaHolder::Reset で指定した配置に基づいてオフセットは決定しています
    const auto controlAreaSize = offsetof(MasterHeader, reserved) - sizeof(MasterHeader::ControlArea);
    const auto offset = controlAreaSize;
    const auto size = MasterHashSize;
    return nn::fs::SubStorage(m_pStorage.get(), offset, size);
}

inline const nn::fs::SubStorage JournalIntegritySaveDataFileSystem::ControlAreaHolder::GetL1BitmapStorage(int index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    NN_SDK_REQUIRES_MINMAX(index, 0, 1);
    // ControlAreaHolder::Reset で指定した配置に基づいてオフセットは決定しています
    const auto controlAreaSize = offsetof(MasterHeader, reserved) - sizeof(MasterHeader::ControlArea);
    const auto offset = controlAreaSize + MasterHashSize + m_BitmapSize * index;
    const auto size = m_BitmapSize;
    return nn::fs::SubStorage(m_pStorage.get(), offset, size);
}

inline const nn::fs::SubStorage JournalIntegritySaveDataFileSystem::ControlAreaHolder::GetSaveDataMetaMasterHashStorage() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    // ControlAreaHolder::Reset で指定した配置に基づいてオフセットは決定しています
    const auto controlAreaSize = offsetof(MasterHeader, reserved) - sizeof(MasterHeader::ControlArea);
    const auto offset = controlAreaSize + MasterHashSize + m_BitmapSize * 2;
    const auto size = MasterHashSize;
    return nn::fs::SubStorage(m_pStorage.get(), offset, size);
}

/**
* @brief        ファイルシステムをコミットします。
*
* @param[in]    counterForBundledCommit 仮コミット時に書き込むカウンタ値
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::CommitFileSystemCore(int64_t counterForBundledCommit) NN_NOEXCEPT
{
    // ファイルやディレクトリオープン中はコミットできません。
    if( m_FileSystem.HasOpenedFiles(nn::fs::OpenMode_Write) )
    {
        return nn::fs::ResultUnsupportedOperation();
    }

    // キャッシュマネージャーに対して排他制御します。
    ScopedFSLock lock(GetLocker());

    // 完全性検証+二重化レイヤーに対してコミット処理を行います。
    NN_RESULT_DO(m_IntegrityStorage.Commit());

    // 完全性検証レイヤーに対してコミット処理を行います。
    if( m_IntegrityStorageSaveDataMeta.IsInitialized() )
    {
        NN_RESULT_DO(m_IntegrityStorageSaveDataMeta.Commit());
    }

    auto needsCommitMaster = false;

    // データが更新されたなら、完全性保証メタデータのフラッシュを行います。
    auto needsFlipMasterSelectBitmap = false;
    NN_RESULT_DO(m_StorageBufferingMeta.Flush());
    if( m_DuplexStorage.NeedCommit() )
    {
        NN_RESULT_DO(m_DuplexStorage.Flush());
        needsCommitMaster = true;
        needsFlipMasterSelectBitmap = true;
        m_DuplexStorage.ClearCommitFlag();
    }

    // 以下のいずれかの場合、マスター領域を更新します。
    // ・ 拡張データ更新後のコミット時
    // ・ 仮コミット時
    // ・ 仮コミット状態からのコミット時
    if( m_IsExtraDataModified
        || counterForBundledCommit != 0
        || m_IsProvisionallyCommitted )
    {
        needsCommitMaster = true;
    }

    if( needsCommitMaster )
    {
        MasterHeaderBufferType pAtomic;
        NN_RESULT_DO(GetMasterHeader(&pAtomic, m_SaveDataMinimumVersion));
        nn::fs::MemoryStorage memoryStorageControlArea(pAtomic.get(), pAtomic.get_deleter().GetSize());

        // マスター領域に不正が無いことを検証します。
        NN_RESULT_DO(
            Verify(
                m_pMacGenerator,
                &pAtomic->controlArea.header,
                sizeof(pAtomic->controlArea.header),
                pAtomic->controlArea.signature,
                sizeof(pAtomic->controlArea.signature)
            )
        );

        // カウンタ値を書き込みます。
        m_ControlArea.GetCommitData()->counterForBundledCommit = counterForBundledCommit;

        //管理領域の変更を反映します。
        m_ControlArea.Commit(pAtomic.get());

        // 選択ビットマップを反転します。
        if( needsFlipMasterSelectBitmap )
        {
            NN_RESULT_DO(m_DuplexStorage.SwapDuplexBitmap());
            pAtomic->controlArea.header.masterSelectBitmap
                = !pAtomic->controlArea.header.masterSelectBitmap;
        }

        // 管理領域のSHAを計算します。
        nn::crypto::GenerateSha256Hash(
            pAtomic->controlArea.header.hash,
            sizeof(pAtomic->controlArea.header.hash),
            &pAtomic->hierarchicalDuplexControlArea,
            sizeof(MasterHeader) - sizeof(MasterHeader::ControlArea)
        );

        // ベースファイルのSignを用いてマスター管理領域の先頭に署名します。
        NN_RESULT_DO(
            Sign(
                &memoryStorageControlArea,
                m_pMacGenerator,
                pAtomic->controlArea.signature,
                sizeof(pAtomic->controlArea.signature),
                &pAtomic->controlArea.header,
                sizeof(pAtomic->controlArea.header)
            )
        );

        // 管理領域以外のデータをフラッシュします。
        NN_RESULT_DO(m_RemappedFileSystemStorage.Flush());

        // 管理領域を書き込みます。
        NN_RESULT_DO(
            m_BaseStorageControlArea.Write(
                m_IsFirstControlAreaMounted ? sizeof(MasterHeader) : 0,
                pAtomic.get(),
                sizeof(MasterHeader)
            )
        );
        NN_RESULT_DO(m_BaseStorageControlArea.Flush());

        NN_RESULT_DO(
            m_BaseStorageControlArea.Write(
                m_IsFirstControlAreaMounted ? 0 : sizeof(MasterHeader),
                pAtomic.get(),
                sizeof(MasterHeader)
            )
        );
        NN_RESULT_DO(m_BaseStorageControlArea.Flush());

        m_DuplexStorage.ClearCommitFlag();
        m_IsFirstControlAreaMounted = !m_IsFirstControlAreaMounted;
        m_IsProvisionallyCommitted = counterForBundledCommit != 0;

        // 本コミットの場合は更新前のデータを更新しておきます。
        if( !m_IsProvisionallyCommitted )
        {
            // 拡張データ更新
            memcpy(&pAtomic->originalExtraData, &pAtomic->extraData, sizeof(ExtraData));

            // マスターハッシュ更新
            const auto pAtomicAddr = reinterpret_cast<char*>(pAtomic.get());
            std::memcpy(
                pAtomicAddr + pAtomic->controlArea.header.offsetOriginalMasterHash,
                pAtomicAddr + pAtomic->controlArea.header.offsetMasterHash,
                static_cast<size_t>(pAtomic->controlArea.header.sizeMasterHash)
            );
            std::memcpy(
                pAtomicAddr + pAtomic->controlArea.header.offsetOriginalSaveDataMetaMasterHash,
                pAtomicAddr + pAtomic->controlArea.header.offsetSaveDataMetaMasterHash,
                static_cast<size_t>(pAtomic->controlArea.header.sizeMasterHash)
            );
        }

        StoreMasterHeader(pAtomic.release(), pAtomic.get_deleter().GetSize());
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        指定のパラメータのセーブデータにおける各データの配置を計算します。
*
* @param[out]   outFileSystemLayoutHeader           レイアウトヘッダ
* @param[in]    version                             レイアウトデータのバージョン
* @param[in]    sizeRemappedFileSystemEntryTable    ファイルシステム用論物変換エントリテーブルのサイズ
* @param[in]    sizeRemappedMetaEntryTable          メタデータ用論物変換エントリテーブルのサイズ
* @param[in]    inputParamDuplex                    二重化パラメータ
* @param[in]    inputParamIntegrity                 完全性検証パラメータ
* @param[in]    sizeBlock                           ブロックサイズ
* @param[in]    countDataBlock                      データブロック数
* @param[in]    countReservedBlock                  予約領域のブロック数
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::FormatFileSystemLayoutHeader(
           FileSystemLayoutHeader* outFileSystemLayoutHeader,
           uint32_t version,
           int64_t sizeRemappedFileSystemEntryTable,
           int64_t sizeRemappedMetaEntryTable,
           const HierarchicalDuplexStorageControlArea::InputParam& inputParamDuplex,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam& inputParamIntegrity,
           uint32_t sizeBlock,
           uint32_t countDataBlock,
           uint32_t countReservedBlock
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outFileSystemLayoutHeader);

    if( (version & VersionCodeMask) > (VersionSupportedMax & VersionCodeMask) )
    {
        // データのバージョン互換性エラーです。
        NN_RESULT_THROW(nn::fs::ResultUnsupportedVersion());
    }
    if( (version & VersionCodeMask) < (VersionSupportedMin & VersionCodeMask) )
    {
        // データのバージョン互換性エラーです。
        NN_RESULT_THROW(nn::fs::ResultUnsupportedVersion());
    }

    // データ構造
    //
    // マスター管理領域
    // {
    //     ヘッダ
    //
    //     階層二重化管理領域
    //     階層ハッシュ検証管理領域
    //     ジャーナリング管理領域
    //     セーブデータ管理領域
    //     ファイルシステム用リマップ管理領域
    //     メタデータ用リマップ管理領域
    //     仮コミットデータ
    //     拡張データ × 2
    //     セーブデータメタ用階層ハッシュ管理領域 (VersionSupportSaveDataMetaHash 以降)
    //
    //     階層二重化ビットマップ L1 × 2
    //     マスターハッシュ
    //     セーブデータメタ用マスターハッシュ (VersionSupportSaveDataMetaHash 以降)
    // } × 2
    //
    // ファイルシステム用リマップエントリーテーブル
    //
    // 完全性保証メタデータ
    // {
    //     階層二重化ビットマップ L2 × 2
    //
    //     メタデータ
    //     {
    //         メタデータ用リマップエントリーテーブル
    //         ジャーナリングメタ
    //         ジャーナリングビットマップ × 3
    //         階層ハッシュ L1, L2, L3
    //         セーブデータメタ
    //         セーブデータメタの階層ハッシュ L1, L2 (VersionSupportSaveDataMetaHash 以降)
    //     } × 2
    // }
    //
    // セーブデータ
    // {
    //     セーブデータ用領域
    //     コミットまでに書き込める領域
    // }

    outFileSystemLayoutHeader->magic = MagicCode;
    outFileSystemLayoutHeader->version = version;

    // セーブデータ領域のサイズを求めます。
    const int64_t sizeCoreData = SaveDataFileSystemCore::QueryDataSize(
                                     sizeBlock,
                                     countDataBlock
                                 );

    size_t sizeBlockMeta;
    if( (outFileSystemLayoutHeader->version & VersionCodeMask) >= (VersionSupportSaveDataMetaHash & VersionCodeMask) )
    {
        sizeBlockMeta = nn::util::align_up(
            sizeBlock,
            SaveDataMetaIntegrityBlockSize
        );
    }
    else
    {
        sizeBlockMeta = sizeBlock;
    }

    const int64_t sizeCoreMeta = SaveDataFileSystemCore::QueryMetaSize(
                                     sizeBlockMeta,
                                     countDataBlock
                                 );
    const int64_t sizeReservedArea = static_cast<int64_t>(countReservedBlock) * sizeBlock;

    // 検証機能付きジャーナリングストレージ管理領域のサイズを取得します。
    int64_t sizeJournalTable = 0;
    int64_t sizeJournalBitmapPhysical = 0;
    int64_t sizeJournalBitmapVirtual = 0;
    int64_t sizeJournalBitmapUnassigned = 0;
    HierarchicalIntegrityVerificationSizeSet integritySizes = { 0 };
    NN_RESULT_DO(
        JournalIntegritySaveDataStorage::QuerySize(
            &sizeJournalTable,
            &sizeJournalBitmapPhysical,
            &sizeJournalBitmapVirtual,
            &sizeJournalBitmapUnassigned,
            &integritySizes,
            inputParamIntegrity,
            sizeReservedArea,
            sizeBlock,
            sizeCoreData
        )
    );

    static const int64_t SizeMasterHash = static_cast<int64_t>(nn::crypto::Sha256Generator::HashSize);
    const auto sizeLayeredHashL1 = inputParamIntegrity.sizeBlockLevel[0];
    if( integritySizes.masterHashSize != static_cast<int64_t>(SizeMasterHash)
        || integritySizes.layeredHashSizes[0] != static_cast<int64_t>(sizeLayeredHashL1) )
    {
        return nn::fs::ResultInvalidSize();
    }

    // メタデータに対する階層ハッシュのサイズを計算します
    HierarchicalIntegrityVerificationSizeSet integritySaveDataMetaSizes = { 0 };
    if( (outFileSystemLayoutHeader->version & VersionCodeMask) >= (VersionSupportSaveDataMetaHash & VersionCodeMask) )
    {
        HierarchicalIntegrityVerificationStorageControlArea::InputParam inputParamSaveDataMetaIntegrity;
        for( size_t i = 0; i < nn::fssystem::save::IntegrityLayerCountSaveDataMeta; ++i )
        {
            inputParamSaveDataMetaIntegrity.sizeBlockLevel[i] = SaveDataMetaIntegrityBlockSize;
        }
        HierarchicalIntegrityVerificationStorageControlArea::QuerySize(
            &integritySaveDataMetaSizes,
            inputParamSaveDataMetaIntegrity,
            nn::fssystem::save::IntegrityLayerCountSaveDataMeta,
            sizeCoreMeta
        );
    }

    int64_t sizeNonduplicatedMeta = 0;
    {
        outFileSystemLayoutHeader->sizeJournalTable
            = nn::util::align_up(
                  sizeJournalTable,
                  RemapStorage::AlignmentSmall
              );
        sizeNonduplicatedMeta
            += outFileSystemLayoutHeader->sizeJournalTable;
        outFileSystemLayoutHeader->sizeJournalBitmapPhysical
            = nn::util::align_up(
                  sizeJournalBitmapPhysical,
                  RemapStorage::AlignmentSmall
              );
        sizeNonduplicatedMeta
            += outFileSystemLayoutHeader->sizeJournalBitmapPhysical;
        outFileSystemLayoutHeader->sizeJournalBitmapVirtual
            = nn::util::align_up(
                  sizeJournalBitmapVirtual,
                  RemapStorage::AlignmentSmall
              );
        sizeNonduplicatedMeta
            += outFileSystemLayoutHeader->sizeJournalBitmapVirtual;
        outFileSystemLayoutHeader->sizeJournalBitmapUnassigned
            = nn::util::align_up(
                  sizeJournalBitmapUnassigned,
                  RemapStorage::AlignmentSmall
              );
        sizeNonduplicatedMeta
            += outFileSystemLayoutHeader->sizeJournalBitmapUnassigned;
        outFileSystemLayoutHeader->sizeHierarchicalIntegrityLayeredHashL1
            = nn::util::align_up(
                integritySizes.layeredHashSizes[0],
                RemapStorage::AlignmentSmall
            );
        sizeNonduplicatedMeta
            += outFileSystemLayoutHeader->sizeHierarchicalIntegrityLayeredHashL1;
        outFileSystemLayoutHeader->sizeHierarchicalIntegrityLayeredHashL2
            = nn::util::align_up(
                  integritySizes.layeredHashSizes[1],
                  RemapStorage::AlignmentSmall
              );
        sizeNonduplicatedMeta
            += outFileSystemLayoutHeader->sizeHierarchicalIntegrityLayeredHashL2;
        outFileSystemLayoutHeader->sizeHierarchicalIntegrityLayeredHashL3
            = nn::util::align_up(
                  integritySizes.layeredHashSizes[2],
                  RemapStorage::AlignmentSmall
              );
        sizeNonduplicatedMeta
            += outFileSystemLayoutHeader->sizeHierarchicalIntegrityLayeredHashL3;
        outFileSystemLayoutHeader->sizeSaveDataMeta
            = nn::util::align_up(
                  sizeCoreMeta,
                  RemapStorage::AlignmentSmall
              );
        sizeNonduplicatedMeta
            += outFileSystemLayoutHeader->sizeSaveDataMeta;
        if( (outFileSystemLayoutHeader->version & VersionCodeMask) >= (VersionSupportSaveDataMetaHash & VersionCodeMask) )
        {
            outFileSystemLayoutHeader->sizeLayeredHashSaveDataMetaL1
                = nn::util::align_up(
                    integritySaveDataMetaSizes.layeredHashSizes[0],
                    RemapStorage::AlignmentSmall
                );
            sizeNonduplicatedMeta
                += outFileSystemLayoutHeader->sizeLayeredHashSaveDataMetaL1;
            outFileSystemLayoutHeader->sizeLayeredHashSaveDataMetaL2
                = nn::util::align_up(
                      integritySaveDataMetaSizes.layeredHashSizes[1],
                      RemapStorage::AlignmentSmall
                  );
            sizeNonduplicatedMeta
                += outFileSystemLayoutHeader->sizeLayeredHashSaveDataMetaL2;
        }
        sizeNonduplicatedMeta
            = nn::util::align_up(sizeNonduplicatedMeta, inputParamDuplex.sizeBlockLevel[1]);
    }

    // 二重化ビットマップのサイズを取得します。
    HierarchicalDuplexSizeSet duplexSizes = {};
    NN_RESULT_DO(
        HierarchicalDuplexStorageControlArea::QuerySize(
            &duplexSizes,
            inputParamDuplex,
            sizeNonduplicatedMeta
        )
    );

    // マスター管理領域内のオフセットとサイズを計算します。
    {
        int64_t offset = 0;
        offset += offsetof(MasterHeader, reserved);
        offset = AdjustAlign64(offset, sizeof(int64_t));

        outFileSystemLayoutHeader->offsetsBitmapL1[0] = offset;
        outFileSystemLayoutHeader->sizeBitmapL1 = duplexSizes.layeredBitmapSizes[0];
        offset += outFileSystemLayoutHeader->sizeBitmapL1;
        offset = AdjustAlign64(offset, sizeof(int64_t));

        outFileSystemLayoutHeader->offsetsBitmapL1[1] = offset;
        outFileSystemLayoutHeader->sizeBitmapL1 = duplexSizes.layeredBitmapSizes[0];
        offset += outFileSystemLayoutHeader->sizeBitmapL1;
        offset = AdjustAlign64(offset, sizeof(int64_t));

        outFileSystemLayoutHeader->offsetMasterHash = offset;
        outFileSystemLayoutHeader->sizeMasterHash = integritySizes.masterHashSize;
        offset += outFileSystemLayoutHeader->sizeMasterHash;
        offset = AdjustAlign64(offset, sizeof(int64_t));

        outFileSystemLayoutHeader->offsetOriginalMasterHash = offset;

        if( (outFileSystemLayoutHeader->version & VersionCodeMask) >= (VersionSupportSaveDataMetaHash & VersionCodeMask) )
        {
            offset += outFileSystemLayoutHeader->sizeMasterHash;
            offset = AdjustAlign64(offset, sizeof(int64_t));

            outFileSystemLayoutHeader->offsetSaveDataMetaMasterHash = offset;
            offset += outFileSystemLayoutHeader->sizeMasterHash;
            offset = AdjustAlign64(offset, sizeof(int64_t));

            outFileSystemLayoutHeader->offsetOriginalSaveDataMetaMasterHash = offset;
            offset += outFileSystemLayoutHeader->sizeMasterHash;
            offset = AdjustAlign64(offset, sizeof(int64_t));
        }

        NN_SDK_ASSERT_EQUAL(integritySizes.masterHashSize, SizeMasterHash);
    }

    // ブロックサイズの最大値を取得します。
    const int64_t sizeBlockMax =
        (static_cast<size_t>(sizeBlock - 1) |
        (inputParamDuplex.sizeBlockLevel[0] - 1) |
        (inputParamDuplex.sizeBlockLevel[1] - 1) |
        (inputParamIntegrity.sizeBlockLevel[0] - 1) |
        (inputParamIntegrity.sizeBlockLevel[1] - 1) |
        (inputParamIntegrity.sizeBlockLevel[2] - 1) |
        (inputParamIntegrity.sizeBlockLevel[3] - 1) |
        (RemapStorage::SizeBlockLarge - 1)) + 1;

    // ファイルシステム内の各領域のオフセットとサイズを計算します。
    {
        outFileSystemLayoutHeader->offsetFileSystemRemapMeta
            = sizeof(MasterHeader) * 2;
        outFileSystemLayoutHeader->sizeFileSystemRemapMeta
            = nn::util::align_up(
                  sizeRemappedFileSystemEntryTable,
                  RemapStorage::AlignmentLarge
              );
        outFileSystemLayoutHeader->offsetMetaRemapMeta
            = outFileSystemLayoutHeader->offsetFileSystemRemapMeta
            + outFileSystemLayoutHeader->sizeFileSystemRemapMeta;
        outFileSystemLayoutHeader->sizeMetaRemapMeta
            = nn::util::align_up(
                  sizeRemappedMetaEntryTable,
                  RemapStorage::AlignmentLarge
              );
        outFileSystemLayoutHeader->sizeBitmapL2
            = nn::util::align_up(
                  duplexSizes.layeredBitmapSizes[1],
                  RemapStorage::AlignmentSmall
              );
        outFileSystemLayoutHeader->sizeMeta
            = nn::util::align_up(
                  duplexSizes.bodySize,
                  RemapStorage::AlignmentSmall
              );
        outFileSystemLayoutHeader->sizeSaveDataCoreOriginal = sizeCoreData;
        outFileSystemLayoutHeader->sizeSaveDataCore
            = nn::util::align_up(
                  sizeCoreData,
                  static_cast<size_t>(sizeBlockMax)
              );
        outFileSystemLayoutHeader->sizeReservedArea
            = nn::util::align_up(
                  sizeReservedArea,
                  static_cast<size_t>(sizeBlockMax)
              );
        outFileSystemLayoutHeader->offsetFileSystemRemapData
            = outFileSystemLayoutHeader->offsetMetaRemapMeta
            + outFileSystemLayoutHeader->sizeMetaRemapMeta;
        outFileSystemLayoutHeader->sizeFileSystemRemapData
            = nn::util::align_up(
                  outFileSystemLayoutHeader->sizeBitmapL2 * 2
                      + outFileSystemLayoutHeader->sizeMeta * 2
                      + outFileSystemLayoutHeader->sizeSaveDataCore
                      + outFileSystemLayoutHeader->sizeReservedArea,
                  RemapStorage::AlignmentLarge
              );
    }

    outFileSystemLayoutHeader->masterSelectBitmap = true;

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

Result JournalIntegritySaveDataFileSystem::GetMasterHeader(
           MasterHeaderBufferType* outAtomic,
           uint32_t minimumVersion
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(minimumVersion <= VersionSupportedMax);
    NN_SDK_REQUIRES(minimumVersion >= VersionSupportedMin);

    auto buffer = m_pBufferManager->AcquireCache(m_MasterHandle);
    if( buffer.first != 0 )
    {
        *outAtomic = MasterHeaderBufferType(
            reinterpret_cast<MasterHeader*>(buffer.first),
            buffers::detail::BufferManagerBufferDeallocator(m_pBufferManager, buffer.second)
        );
        NN_RESULT_SUCCESS;
    }
    else
    {
        *outAtomic = buffers::MakeUniqueBufferFromBufferManager<MasterHeader>(m_pBufferManager, NN_CURRENT_FUNCTION_NAME);

        auto isProvisionallyComitted = false;
        NN_RESULT_DO(
            ReadMasterControlArea(
                outAtomic->get(),
                &isProvisionallyComitted,
                &m_IsFirstControlAreaMounted,
                outAtomic->get_deleter().GetSize(),
                m_BaseStorageControlArea,
                m_pMacGenerator,
                minimumVersion
            )
        );
        if( isProvisionallyComitted != m_IsProvisionallyCommitted )
        {
            NN_RESULT_THROW(nn::fs::ResultInvalidJournalIntegritySaveDataCommitState());
        }
        NN_RESULT_SUCCESS;
    }
}

void JournalIntegritySaveDataFileSystem::StoreMasterHeader(
         MasterHeader* pAtomic,
         size_t size
     ) NN_NOEXCEPT
{
    m_MasterHandle = m_pBufferManager->RegisterCache(
                         reinterpret_cast<uintptr_t>(pAtomic),
                         size,
                         IBufferManager::BufferAttribute()
                     );
}

/**
* @brief        ジャーナリング+完全性検証機能付きセーブデータ ファイルシステムとしてフォーマットします。
*
* @param[in]    storage                  ストレージ
* @param[in]    countExpandMax           拡張できる回数の最大
* @param[in]    inputParamDuplex         二重化パラメータ
* @param[in]    inputParamIntegrity      完全性検証パラメータ
* @param[in]    sizeBlock                ブロックサイズ
* @param[in]    countDataBlock           データブロック数
* @param[in]    countReservedBlock       予約領域のブロック数
* @param[in]    pIntegrityCacheBufferSet キャッシュマネージャ
* @param[in]    pDuplicateCacheBuffer    キャッシュマネージャ
* @param[in]    pLocker                  キャッシュマネージャのスレッド間同期オブジェクト
* @param[in]    pMacGenerator            MAC 生成クラス
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::Format(
           fs::SubStorage storage,
           int countExpandMax,
           const HierarchicalDuplexStorageControlArea::InputParam& inputParamDuplex,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam& inputParamIntegrity,
           uint32_t sizeBlock,
           uint32_t countDataBlock,
           uint32_t countReservedBlock,
           FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
           IBufferManager* pDuplicateCacheBuffer,
           os::Mutex* pLocker,
           IMacGenerator* pMacGenerator,
           const nn::fs::SaveDataHashSalt& hashSalt
    ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pLocker);

    // フォーマット処理中はキャッシュマネージャーに対して排他制御します。
    ScopedFSLock lock(pLocker);

    // マスター管理領域を初期化します。
    const auto pAtomic = buffers::MakeUniqueBufferFromBufferManager<MasterHeader>(pDuplicateCacheBuffer, NN_CURRENT_FUNCTION_NAME);
    NN_SDK_ASSERT_NOT_NULL(pAtomic);

    std::memset(pAtomic.get(), 0, sizeof(MasterHeader));
    FileSystemLayoutHeader& layoutHeader = pAtomic->controlArea.header;

    int64_t sizeStorage = 0;
    NN_RESULT_DO(storage.GetSize(&sizeStorage));
    if( sizeStorage < static_cast<int64_t>(sizeof(MasterHeader)) )
    {
        return nn::fs::ResultInvalidSize();
    }

    NN_RESULT_DO(storage.Write(0, pAtomic.get(), sizeof(MasterHeader)));
    NN_RESULT_DO(storage.Write(sizeof(MasterHeader), pAtomic.get(), sizeof(MasterHeader)));

    // 各種データサイズを取得します。
    NN_RESULT_DO(
        FormatFileSystemLayoutHeader(
            &layoutHeader,
            VersionCode,
            GetRemappedFileSystemEntryTableSize(VersionCode, countExpandMax),
            GetRemappedMetaEntryTableSize(VersionCode, countExpandMax),
            inputParamDuplex,
            inputParamIntegrity,
            sizeBlock,
            countDataBlock,
            countReservedBlock
        )
    );
    if( layoutHeader.offsetSaveData + layoutHeader.sizeSaveDataCore + layoutHeader.sizeReservedArea > sizeStorage )
    {
        return nn::fs::ResultInvalidSize();
    }
    NN_SDK_ASSERT((layoutHeader.version & VersionCodeMask) >= (VersionSupportSaveDataMetaHash & VersionCodeMask));

    // 管理領域ごとにストレージを分割します。
    fs::MemoryStorage storageControlArea(
                      pAtomic.get(),
                      sizeof(MasterHeader)
                  );
    fs::SubStorage storageDuplexControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, hierarchicalDuplexControlArea),
                   sizeof(decltype(MasterHeader::hierarchicalDuplexControlArea))
               );
    fs::SubStorage storageIntegrityControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, hierarchicalIntegrityControlArea),
                   sizeof(decltype(MasterHeader::hierarchicalIntegrityControlArea))
               );
    fs::SubStorage storageJournalControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, journalControlArea),
                   sizeof(decltype(MasterHeader::journalControlArea))
               );
    fs::SubStorage storageSaveDataControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, saveDataControlArea),
                   sizeof(decltype(MasterHeader::saveDataControlArea))
               );
    fs::SubStorage storageFileSystemRemapControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, fileSystemRemapControlArea),
                   sizeof(decltype(MasterHeader::fileSystemRemapControlArea))
               );
    fs::SubStorage storageMetaRemapControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, metaRemapControlArea),
                   sizeof(decltype(MasterHeader::metaRemapControlArea))
               );
    fs::SubStorage storageLayeredHashSaveDataMetaControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, saveDataMetaHashControlArea),
                   sizeof(decltype(MasterHeader::saveDataMetaHashControlArea))
               );
    fs::SubStorage storageBitmapL1A(
                   &storageControlArea,
                   layoutHeader.offsetsBitmapL1[0],
                   layoutHeader.sizeBitmapL1
               );
    fs::SubStorage storageBitmapL1B(
                   &storageControlArea,
                   layoutHeader.offsetsBitmapL1[1],
                   layoutHeader.sizeBitmapL1
               );
    fs::SubStorage storageMasterHash(
                   &storageControlArea,
                   layoutHeader.offsetMasterHash,
                   layoutHeader.sizeMasterHash
               );
    fs::SubStorage storageSaveDataMetaMasterHash(
                   &storageControlArea,
                   layoutHeader.offsetSaveDataMetaMasterHash,
                   layoutHeader.sizeMasterHash
               );

    // リマップストレージを敷きます。
    BufferedStorage storageBufferingRemap;
    storageBufferingRemap.Initialize(
        storage,
        pDuplicateCacheBuffer,
        sizeBlock,
        RemapBufferCount
    );

    fs::SubStorage storageFileSystemRemapMeta(
                   &storageBufferingRemap,
                   layoutHeader.offsetFileSystemRemapMeta,
                   layoutHeader.sizeFileSystemRemapMeta
               );
    fs::SubStorage storageMetaRemapMeta(
                   &storageBufferingRemap,
                   layoutHeader.offsetMetaRemapMeta,
                   layoutHeader.sizeMetaRemapMeta
               );
    fs::SubStorage storageFileSystemRemapData(
                   &storageBufferingRemap,
                   layoutHeader.offsetFileSystemRemapData,
                   layoutHeader.sizeFileSystemRemapData
               );
    RemapStorage storageRemappedFileSystem;

    NN_RESULT_DO(
        RemapStorage::Format(
            storageFileSystemRemapControlArea,
            FileSystemRemapEntryCount
        )
    );

    NN_RESULT_DO(
        storageRemappedFileSystem.Initialize(
            storageFileSystemRemapControlArea,
            storageFileSystemRemapMeta
        )
    );

    NN_RESULT_DO(storageRemappedFileSystem.RegisterStorage(storageFileSystemRemapData, 0));

    NN_RESULT_DO(
        storageRemappedFileSystem.RegisterMap(
            layoutHeader.offsetsBitmapL2 + 0,
            layoutHeader.sizeBitmapL2,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedFileSystem.RegisterMap(
            layoutHeader.offsetsBitmapL2 + 1,
            layoutHeader.sizeBitmapL2,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedFileSystem.RegisterMap(
            layoutHeader.offsetsMeta + 0,
            layoutHeader.sizeMeta,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedFileSystem.RegisterMap(
            layoutHeader.offsetsMeta + 1,
            layoutHeader.sizeMeta,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedFileSystem.RegisterMap(
            &layoutHeader.offsetSaveData,
            layoutHeader.sizeSaveDataCore + layoutHeader.sizeReservedArea,
            RemapStorage::AlignmentLarge,
            0
        )
    );

    NN_RESULT_DO(storageRemappedFileSystem.Flush());

    fs::SubStorage storageBitmapL2A(
                   &storageRemappedFileSystem,
                   layoutHeader.offsetsBitmapL2[0],
                   layoutHeader.sizeBitmapL2
               );
    fs::SubStorage storageBitmapL2B(
                   &storageRemappedFileSystem,
                   layoutHeader.offsetsBitmapL2[1],
                   layoutHeader.sizeBitmapL2
               );
    fs::SubStorage storageMetaA(
                   &storageRemappedFileSystem,
                   layoutHeader.offsetsMeta[0],
                   layoutHeader.sizeMeta
               );
    fs::SubStorage storageMetaB(
                   &storageRemappedFileSystem,
                   layoutHeader.offsetsMeta[1],
                   layoutHeader.sizeMeta
               );

    fs::SubStorage storageIntegritySaveData(
                   &storageRemappedFileSystem,
                   layoutHeader.offsetSaveData,
                   layoutHeader.sizeSaveDataCore + layoutHeader.sizeReservedArea
               );

    // 完全性保証メタデータ領域をフォーマットし、マウントします。
    NN_RESULT_DO(
        HierarchicalDuplexStorageControlArea::Format(
            storageDuplexControlArea,
            inputParamDuplex,
            layoutHeader.sizeMeta
        )
    );
    ScopedFinalizeObject<HierarchicalDuplexStorageControlArea> duplexControlArea;
    NN_RESULT_DO(duplexControlArea->Initialize(storageDuplexControlArea));
    NN_RESULT_DO(
        HierarchicalDuplexStorage::Format(
            duplexControlArea->GetMeta().infoDuplex,
            storageBitmapL1A,
            storageBitmapL1B,
            storageBitmapL2A,
            storageBitmapL2B,
            pDuplicateCacheBuffer
        )
    );
    HierarchicalDuplexStorage storageDuplicatedIntegrityMeta;
    NN_RESULT_DO(
        storageDuplicatedIntegrityMeta.Initialize(
            duplexControlArea->GetMeta().infoDuplex,
            storageBitmapL1A,
            storageBitmapL1B,
            storageBitmapL2A,
            storageBitmapL2B,
            storageMetaA,
            storageMetaB,
            !layoutHeader.masterSelectBitmap,
            pDuplicateCacheBuffer,
            pLocker
        )
    );

    // さらにメタデータ用リマップストレージを敷きます。
    fs::SubStorage storageMetaRemapData(
                   &storageDuplicatedIntegrityMeta,
                   0,
                   layoutHeader.sizeMeta
               );
    RemapStorage storageRemappedMeta;

    NN_RESULT_DO(
        RemapStorage::Format(
            storageMetaRemapControlArea,
            MetaRemapEntryCount
        )
    );

    NN_RESULT_DO(
        storageRemappedMeta.Initialize(
            storageMetaRemapControlArea,
            storageMetaRemapMeta
        )
    );

    NN_RESULT_DO(storageRemappedMeta.RegisterStorage(storageMetaRemapData, 0));

    NN_RESULT_DO(
        storageRemappedMeta.RegisterMap(
            &layoutHeader.offsetJournalTable,
            layoutHeader.sizeJournalTable,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.RegisterMap(
            &layoutHeader.offsetJournalBitmapPhysical,
            layoutHeader.sizeJournalBitmapPhysical,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.RegisterMap(
            &layoutHeader.offsetJournalBitmapVirtual,
            layoutHeader.sizeJournalBitmapVirtual,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.RegisterMap(
            &layoutHeader.offsetJournalBitmapUnassigned,
            layoutHeader.sizeJournalBitmapUnassigned,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.RegisterMap(
            &layoutHeader.offsetHierarchicalIntegrityLayeredHashL1,
            layoutHeader.sizeHierarchicalIntegrityLayeredHashL1,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.RegisterMap(
            &layoutHeader.offsetHierarchicalIntegrityLayeredHashL2,
            layoutHeader.sizeHierarchicalIntegrityLayeredHashL2,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.RegisterMap(
            &layoutHeader.offsetHierarchicalIntegrityLayeredHashL3,
            layoutHeader.sizeHierarchicalIntegrityLayeredHashL3,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.RegisterMap(
            &layoutHeader.offsetSaveDataMeta,
            layoutHeader.sizeSaveDataMeta,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.RegisterMap(
            &layoutHeader.offsetLayeredHashSaveDataMetaL1,
            layoutHeader.sizeLayeredHashSaveDataMetaL1,
            RemapStorage::AlignmentSmall,
            0
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.RegisterMap(
            &layoutHeader.offsetLayeredHashSaveDataMetaL2,
            layoutHeader.sizeLayeredHashSaveDataMetaL2,
            RemapStorage::AlignmentSmall,
            0
        )
    );

    NN_RESULT_DO(storageRemappedMeta.Flush());

    // メタデータ内をさらに分割します。
    fs::SubStorage storageJournalTable(
                   &storageRemappedMeta,
                   layoutHeader.offsetJournalTable,
                   layoutHeader.sizeJournalTable
               );
    fs::SubStorage storageJournalBitmapPhysical(
                   &storageRemappedMeta,
                   layoutHeader.offsetJournalBitmapPhysical,
                   layoutHeader.sizeJournalBitmapPhysical
               );
    fs::SubStorage storageJournalBitmapVirtual(
                   &storageRemappedMeta,
                   layoutHeader.offsetJournalBitmapVirtual,
                   layoutHeader.sizeJournalBitmapVirtual
               );
    fs::SubStorage storageJournalBitmapUnassigend(
                   &storageRemappedMeta,
                   layoutHeader.offsetJournalBitmapUnassigned,
                   layoutHeader.sizeJournalBitmapUnassigned
               );
    fs::SubStorage storageIntegrityLayeredHashL1(
                   &storageRemappedMeta,
                   layoutHeader.offsetHierarchicalIntegrityLayeredHashL1,
                   layoutHeader.sizeHierarchicalIntegrityLayeredHashL1
               );
    fs::SubStorage storageIntegrityLayeredHashL2(
                   &storageRemappedMeta,
                   layoutHeader.offsetHierarchicalIntegrityLayeredHashL2,
                   layoutHeader.sizeHierarchicalIntegrityLayeredHashL2
               );
    fs::SubStorage storageIntegrityLayeredHashL3(
                   &storageRemappedMeta,
                   layoutHeader.offsetHierarchicalIntegrityLayeredHashL3,
                   layoutHeader.sizeHierarchicalIntegrityLayeredHashL3
               );
    fs::SubStorage storageSaveDataMeta(
                   &storageRemappedMeta,
                   layoutHeader.offsetSaveDataMeta,
                   layoutHeader.sizeSaveDataMeta
               );
    fs::SubStorage storageSaveDataMetaLayeredHashL1(
                   &storageRemappedMeta,
                   layoutHeader.offsetLayeredHashSaveDataMetaL1,
                   layoutHeader.sizeLayeredHashSaveDataMetaL1
               );
    fs::SubStorage storageSaveDataMetaLayeredHashL2(
                   &storageRemappedMeta,
                   layoutHeader.offsetLayeredHashSaveDataMetaL2,
                   layoutHeader.sizeLayeredHashSaveDataMetaL2
               );

    // 階層ハッシュ検証付きジャーナルストレージをフォーマットし、マウントします。
    ScopedFinalizeObject<JournalIntegritySaveDataStorage> storageIntegrity;
    {
        HierarchicalIntegrityVerificationMetaInformation metaIntegrity;
        metaIntegrity.Format(hashSalt);
        metaIntegrity.sizeMasterHash = static_cast<uint32_t>(layoutHeader.sizeMasterHash);

        metaIntegrity.infoLevelHash.info[0].offset = layoutHeader.offsetHierarchicalIntegrityLayeredHashL1;
        metaIntegrity.infoLevelHash.info[0].size = layoutHeader.sizeHierarchicalIntegrityLayeredHashL1;
        metaIntegrity.infoLevelHash.info[0].orderBlock
            = ILog2(static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[0]));

        metaIntegrity.infoLevelHash.info[1].offset = layoutHeader.offsetHierarchicalIntegrityLayeredHashL2;
        metaIntegrity.infoLevelHash.info[1].size = layoutHeader.sizeHierarchicalIntegrityLayeredHashL2;
        metaIntegrity.infoLevelHash.info[1].orderBlock
            = ILog2(static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[1]));

        metaIntegrity.infoLevelHash.info[2].offset = layoutHeader.offsetHierarchicalIntegrityLayeredHashL3;
        metaIntegrity.infoLevelHash.info[2].size = layoutHeader.sizeHierarchicalIntegrityLayeredHashL3;
        metaIntegrity.infoLevelHash.info[2].orderBlock
            = ILog2(static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[2]));

        metaIntegrity.infoLevelHash.info[3].offset = 0;
        metaIntegrity.infoLevelHash.info[3].size = layoutHeader.sizeSaveDataCoreOriginal;
        metaIntegrity.infoLevelHash.info[3].orderBlock
            = ILog2(static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[3]));

        NN_RESULT_DO(
            JournalIntegritySaveDataStorage::Format(
                storageJournalControlArea,
                storageJournalTable,
                storageJournalBitmapPhysical,
                storageJournalBitmapVirtual,
                storageJournalBitmapUnassigend,
                storageIntegrityControlArea,
                storageMasterHash,
                layoutHeader.sizeReservedArea,
                sizeBlock,
                metaIntegrity,
                layoutHeader.sizeSaveDataCoreOriginal
            )
        );

        HierarchicalIntegrityVerificationInformation infoIntegrity;
        {
            ScopedFinalizeObject<HierarchicalIntegrityVerificationStorageControlArea> controlArea;
            NN_RESULT_DO(controlArea->Initialize(storageIntegrityControlArea));
            controlArea->GetLevelHashInfo(&infoIntegrity);
            infoIntegrity.maxLayers = nn::fssystem::save::IntegrityLayerCountSave;
        }

        NN_RESULT_DO(
            storageIntegrity->Initialize(
                storageJournalControlArea,
                storageJournalTable,
                storageJournalBitmapPhysical,
                storageJournalBitmapVirtual,
                storageJournalBitmapUnassigend,
                storageMasterHash,
                storageIntegrityLayeredHashL1,
                storageIntegrityLayeredHashL2,
                storageIntegrityLayeredHashL3,
                storageIntegritySaveData,
                infoIntegrity,
                pIntegrityCacheBufferSet,
                pLocker
            )
        );
    }

    // アロケーションテーブルの階層ハッシュをフォーマット、マウントする
    ScopedFinalizeObject<HierarchicalIntegrityVerificationStorage> storageIntegritySaveDataMeta;
    {
        HierarchicalIntegrityVerificationMetaInformation metaIntegrity;
        metaIntegrity.Format();
        metaIntegrity.sizeMasterHash = static_cast<uint32_t>(layoutHeader.sizeMasterHash);

        metaIntegrity.infoLevelHash.info[0].offset = layoutHeader.offsetLayeredHashSaveDataMetaL1;
        metaIntegrity.infoLevelHash.info[0].size = layoutHeader.sizeLayeredHashSaveDataMetaL1;
        metaIntegrity.infoLevelHash.info[0].orderBlock = ILog2(SaveDataMetaIntegrityBlockSize);

        metaIntegrity.infoLevelHash.info[1].offset = layoutHeader.offsetLayeredHashSaveDataMetaL2;
        metaIntegrity.infoLevelHash.info[1].size = layoutHeader.sizeLayeredHashSaveDataMetaL2;
        metaIntegrity.infoLevelHash.info[1].orderBlock = ILog2(SaveDataMetaIntegrityBlockSize);

        metaIntegrity.infoLevelHash.info[2].offset = layoutHeader.offsetSaveDataMeta;
        metaIntegrity.infoLevelHash.info[2].size = layoutHeader.sizeSaveDataMeta;
        metaIntegrity.infoLevelHash.info[2].orderBlock = ILog2(SaveDataMetaIntegrityBlockSize);

        NN_RESULT_DO(
            HierarchicalIntegrityVerificationStorageControlArea::Format(
                storageLayeredHashSaveDataMetaControlArea,
                metaIntegrity
            )
        );

        HierarchicalIntegrityVerificationInformation infoLayeredHashSaveDataMeta;
        {
            ScopedFinalizeObject<HierarchicalIntegrityVerificationStorageControlArea> controlArea;
            NN_RESULT_DO(controlArea->Initialize(storageLayeredHashSaveDataMetaControlArea));
            controlArea->GetLevelHashInfo(&infoLayeredHashSaveDataMeta);
            infoLayeredHashSaveDataMeta.maxLayers = nn::fssystem::save::IntegrityLayerCountSaveDataMeta;
        }

        HierarchicalIntegrityVerificationStorage::HierarchicalStorageInformation storageInfo;
        storageInfo.SetMasterHashStorage(storageSaveDataMetaMasterHash);
        storageInfo.SetLayer1HashStorage(storageSaveDataMetaLayeredHashL1);
        storageInfo.SetLayer2HashStorage(storageSaveDataMetaLayeredHashL2);
        storageInfo.SetDataStorage(storageSaveDataMeta);
        NN_RESULT_DO(
            storageIntegritySaveDataMeta->Initialize(
                infoLayeredHashSaveDataMeta,
                storageInfo,
                pIntegrityCacheBufferSet,
                pLocker,
                nn::fs::StorageType_SaveData
            )
        );
    }

    {
        // セーブデータ領域をフォーマットします。
        NN_RESULT_DO(
            SaveDataFileSystemCore::Format(
                storageSaveDataControlArea,
                fs::SubStorage(
                    storageIntegritySaveDataMeta.Ptr(),
                    0,
                    layoutHeader.sizeSaveDataMeta
                ),
                fs::SubStorage(
                    storageIntegrity.Ptr(),
                    0,
                    SaveDataFileSystemCore::QueryDataSize(sizeBlock, countDataBlock)
                ),
                sizeBlock,
                pDuplicateCacheBuffer
            )
        );

        NN_RESULT_DO(storageJournalTable.Flush());

        NN_RESULT_DO(storageIntegritySaveDataMeta->Commit());
        NN_RESULT_DO(storageIntegrity->Commit());
    }
    NN_RESULT_DO(
        storageDuplicatedIntegrityMeta.Flush()
    );
    // カウンタ値を書き込みます。
    pAtomic->commitData.counterForBundledCommit = 0;

    // 管理領域のSHAを計算します。
    nn::crypto::GenerateSha256Hash(
        pAtomic->controlArea.header.hash,
        sizeof(pAtomic->controlArea.header.hash),
        &pAtomic->hierarchicalDuplexControlArea,
        sizeof(MasterHeader) - sizeof(MasterHeader::ControlArea)
    );

    // ベースファイルのSignを用いてマスター管理領域の先頭に署名します。
    NN_RESULT_DO(Sign(&storage, pMacGenerator, pAtomic->controlArea.signature, sizeof(pAtomic->controlArea.signature), &pAtomic->controlArea.header, sizeof(pAtomic->controlArea.header)));

    // マスター管理領域をベースファイルに書き込みます。
    NN_RESULT_DO(storage.Write(0, pAtomic.get(), sizeof(MasterHeader)));
    NN_RESULT_DO(storage.Write(sizeof(MasterHeader), pAtomic.get(), sizeof(MasterHeader)));
    NN_RESULT_DO(storage.Flush());

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        ジャーナリング+完全性検証機能付きセーブデータファイルシステムを拡張します。
*
* @param[in]    storage ストレージ
* @param[in]    sizeBlock ブロックサイズ
* @param[in]    countDataBlock 拡張後のデータブロック数
* @param[in]    countlReservedBlock 拡張後の予約領域のブロック数
* @param[in]    pDuplicateCacheBuffer キャッシュマネージャ
* @param[in]    pLocker キャッシュマネージャのスレッド間同期オブジェクト
* @param[in]    pMacGenerator            MAC 生成クラス
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::Expand(
            fs::SubStorage storage,
            uint32_t sizeBlock,
            uint32_t countDataBlock,
            uint32_t countReservedBlock,
            IBufferManager* pDuplicateCacheBuffer,
            os::Mutex* pLocker,
            IMacGenerator* pMacGenerator,
            uint32_t minimumVersion
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pLocker);
    NN_SDK_REQUIRES(minimumVersion <= (VersionSupportedMax >> VersionMajorShift));
    NN_SDK_REQUIRES(minimumVersion >= (VersionSupportedMin >> VersionMajorShift));
    uint32_t saveDataMinimumVersion = (minimumVersion << VersionMajorShift);

    // 拡張処理中はキャッシュマネージャーに対して排他制御します。
    ScopedFSLock lock(pLocker);

    // マスター管理領域を読み込みます。
    const auto pAtomic = buffers::MakeUniqueBufferFromBufferManager<MasterHeader>(pDuplicateCacheBuffer, NN_CURRENT_FUNCTION_NAME);
    NN_SDK_ASSERT_NOT_NULL(pAtomic);

    auto isProvisionallyCommitted = false;
    auto isFirstControlAreaMounted = false;
    NN_RESULT_DO(
        ReadMasterControlArea(
            pAtomic.get(),
            &isProvisionallyCommitted,
            &isFirstControlAreaMounted,
            sizeof(MasterHeader),
            storage,
            pMacGenerator,
            saveDataMinimumVersion
        )
    );
    if( isProvisionallyCommitted )
    {
        // TODO: 適切なリザルトに置き換える
        return nn::fs::ResultUnsupportedOperation();
    }

    auto& layoutHeader = pAtomic->controlArea.header;

    const auto sizeData = static_cast<int64_t>(countDataBlock) * sizeBlock;
    const auto sizeReservedArea = static_cast<int64_t>(countReservedBlock) * sizeBlock;

    if( sizeBlock != pAtomic->saveDataControlArea.sizeBlocks
        || sizeData < layoutHeader.sizeSaveDataCore
        || sizeReservedArea < layoutHeader.sizeReservedArea )
    {
        return nn::fs::ResultInvalidSize();
    }

    // 各種データサイズを取得します。
    FileSystemLayoutHeader layoutHeaderNew = {};
    const auto sizeArchiveBlock = static_cast<uint32_t>(pAtomic->saveDataControlArea.sizeBlocks);
    {
        const auto sizeRemappedFileSystemEntryTable = layoutHeader.sizeFileSystemRemapMeta;
        const auto sizeRemappedMetaEntryTable = layoutHeader.sizeMetaRemapMeta;
        HierarchicalDuplexStorageControlArea::InputParam inputParamDuplex = {};
        HierarchicalIntegrityVerificationStorageControlArea::InputParam inputParamIntegrity = {};

        const auto& infoDuplex = pAtomic->hierarchicalDuplexControlArea.infoDuplex.info;
        inputParamDuplex.sizeBlockLevel[0] = sizeof(char) << infoDuplex[1].orderBlock;
        inputParamDuplex.sizeBlockLevel[1] = sizeof(char) << infoDuplex[2].orderBlock;

        const auto& infoIntegrity = pAtomic->hierarchicalIntegrityControlArea.infoLevelHash.info;
        inputParamIntegrity.sizeBlockLevel[0] = sizeof(char) << infoIntegrity[0].orderBlock;
        inputParamIntegrity.sizeBlockLevel[1] = sizeof(char) << infoIntegrity[1].orderBlock;
        inputParamIntegrity.sizeBlockLevel[2] = sizeof(char) << infoIntegrity[2].orderBlock;
        inputParamIntegrity.sizeBlockLevel[3] = sizeof(char) << infoIntegrity[3].orderBlock;

        // 拡張後のファイルシステム配置を決定します。
        // 現在は、バージョンを維持します
        // メタデータに対する階層ハッシュ未対応バージョンでは
        // 階層ハッシュが配置されません。
        NN_RESULT_DO(
            FormatFileSystemLayoutHeader(
                &layoutHeaderNew,
                layoutHeader.version,
                sizeRemappedFileSystemEntryTable,
                sizeRemappedMetaEntryTable,
                inputParamDuplex,
                inputParamIntegrity,
                sizeArchiveBlock,
                countDataBlock,
                countReservedBlock
            )
        );
    }

    // 拡張後に必要なサイズ分足りているかチェック
    {
        int64_t sizeStorage = 0;
        NN_RESULT_DO(storage.GetSize(&sizeStorage));
        if( layoutHeaderNew.offsetSaveData + layoutHeaderNew.sizeSaveDataCore + layoutHeaderNew.sizeReservedArea > sizeStorage )
        {
            return nn::fs::ResultInvalidSize();
        }
    }

    NN_SDK_ASSERT_EQUAL(layoutHeaderNew.sizeBitmapL1, layoutHeader.sizeBitmapL1);
    NN_SDK_ASSERT_EQUAL(layoutHeaderNew.sizeMasterHash, layoutHeader.sizeMasterHash);

    // 管理領域ごとにストレージを分割します。
    fs::MemoryStorage storageControlArea(
                      pAtomic.get(),
                      sizeof(MasterHeader)
                  );
    fs::SubStorage storageSaveDataControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, saveDataControlArea),
                   sizeof(decltype(MasterHeader::saveDataControlArea))
               );
    fs::SubStorage storageFileSystemRemapControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, fileSystemRemapControlArea),
                   sizeof(decltype(MasterHeader::fileSystemRemapControlArea))
               );

    BufferedStorage storageBufferingRemap;
    storageBufferingRemap.Initialize(
        storage,
        pDuplicateCacheBuffer,
        sizeBlock,
        RemapBufferCount
    );

    // ファイルシステム用リマップストレージを敷きます。
    fs::SubStorage storageFileSystemRemapMeta(
                   &storageBufferingRemap,
                   layoutHeader.offsetFileSystemRemapMeta,
                   layoutHeader.sizeFileSystemRemapMeta
               );
    fs::SubStorage storageMetaRemapMeta(
                   &storageBufferingRemap,
                   layoutHeader.offsetMetaRemapMeta,
                   layoutHeader.sizeMetaRemapMeta
               );
    fs::SubStorage storageFileSystemRemapData(
                   &storageBufferingRemap,
                   layoutHeader.offsetFileSystemRemapData,
                   layoutHeaderNew.sizeFileSystemRemapData
               );
    RemapStorage storageRemappedFileSystem;

    NN_RESULT_DO(
        storageRemappedFileSystem.Initialize(
            storageFileSystemRemapControlArea,
            storageFileSystemRemapMeta
        )
    );

    NN_RESULT_DO(storageRemappedFileSystem.RegisterStorage(storageFileSystemRemapData, 0));

    NN_RESULT_DO(
        storageRemappedFileSystem.ExpandMap(
            layoutHeader.offsetsBitmapL2[0],
            layoutHeaderNew.sizeBitmapL2
        )
    );
    NN_RESULT_DO(
        storageRemappedFileSystem.ExpandMap(
            layoutHeader.offsetsBitmapL2[1],
            layoutHeaderNew.sizeBitmapL2
        )
    );
    NN_RESULT_DO(
        storageRemappedFileSystem.ExpandMap(
            layoutHeader.offsetsMeta[0],
            layoutHeaderNew.sizeMeta
        )
    );
    NN_RESULT_DO(
        storageRemappedFileSystem.ExpandMap(
            layoutHeader.offsetsMeta[1],
            layoutHeaderNew.sizeMeta
        )
    );
    NN_RESULT_DO(
        storageRemappedFileSystem.ExpandMap(
            layoutHeader.offsetSaveData,
            layoutHeaderNew.sizeSaveDataCore + layoutHeaderNew.sizeReservedArea
        )
    );

    NN_RESULT_DO(storageRemappedFileSystem.Flush());

    // メタデータを拡張します。
    auto needsCommit = false;
    NN_RESULT_DO(
        ExpandMeta(
            &needsCommit,
            storageMetaRemapMeta,
            fs::SubStorage(&storageRemappedFileSystem, 0, std::numeric_limits<int64_t>::max()),
            fs::SubStorage(&storageControlArea, 0, sizeof(MasterHeader)),
            layoutHeader,
            layoutHeaderNew,
            sizeArchiveBlock,
            pDuplicateCacheBuffer,
            pLocker
        )
    );
    if( needsCommit )
    {
        layoutHeader.masterSelectBitmap = !layoutHeader.masterSelectBitmap;
    }

    // セーブデータコアの管理領域を拡張します。
    NN_RESULT_DO(
        SaveDataFileSystemCore::ExpandControlArea(
            storageSaveDataControlArea,
            layoutHeaderNew.sizeSaveDataCoreOriginal
        )
    );

    // レイアウトヘッダを更新します。
    layoutHeader.sizeFileSystemRemapData = layoutHeaderNew.sizeFileSystemRemapData;
    layoutHeader.sizeBitmapL2 = layoutHeaderNew.sizeBitmapL2;
    layoutHeader.sizeMeta = layoutHeaderNew.sizeMeta;
    layoutHeader.sizeSaveDataCoreOriginal = layoutHeaderNew.sizeSaveDataCoreOriginal;
    layoutHeader.sizeSaveDataCore = layoutHeaderNew.sizeSaveDataCore;
    layoutHeader.sizeReservedArea = layoutHeaderNew.sizeReservedArea;
    layoutHeader.sizeJournalTable = layoutHeaderNew.sizeJournalTable;
    layoutHeader.sizeJournalBitmapPhysical = layoutHeaderNew.sizeJournalBitmapPhysical;
    layoutHeader.sizeJournalBitmapVirtual = layoutHeaderNew.sizeJournalBitmapVirtual;
    layoutHeader.sizeJournalBitmapUnassigned = layoutHeaderNew.sizeJournalBitmapUnassigned;
    layoutHeader.sizeHierarchicalIntegrityLayeredHashL1
        = layoutHeaderNew.sizeHierarchicalIntegrityLayeredHashL1;
    layoutHeader.sizeHierarchicalIntegrityLayeredHashL2
        = layoutHeaderNew.sizeHierarchicalIntegrityLayeredHashL2;
    layoutHeader.sizeHierarchicalIntegrityLayeredHashL3
        = layoutHeaderNew.sizeHierarchicalIntegrityLayeredHashL3;
    layoutHeader.sizeSaveDataMeta = layoutHeaderNew.sizeSaveDataMeta;
    layoutHeader.sizeLayeredHashSaveDataMetaL1 = layoutHeaderNew.sizeLayeredHashSaveDataMetaL1;
    layoutHeader.sizeLayeredHashSaveDataMetaL2 = layoutHeaderNew.sizeLayeredHashSaveDataMetaL2;

    // カウンタ値を書き込みます。
    pAtomic->commitData.counterForBundledCommit = 0;

    // 管理領域のSHAを計算します。
    nn::crypto::GenerateSha256Hash(
        pAtomic->controlArea.header.hash,
        sizeof(pAtomic->controlArea.header.hash),
        &pAtomic->hierarchicalDuplexControlArea,
        sizeof(MasterHeader) - sizeof(MasterHeader::ControlArea)
    );

    // ベースファイルのSignを用いてマスター管理領域の先頭に暗号化をかけます。
    NN_RESULT_DO(Sign(&storage, pMacGenerator, pAtomic->controlArea.signature, sizeof(pAtomic->controlArea.signature), &pAtomic->controlArea.header, sizeof(pAtomic->controlArea.header)));

    // マスター管理領域をベースファイルに書き込みます。
    NN_RESULT_DO(storage.Write(0, pAtomic.get(), sizeof(MasterHeader)));
    NN_RESULT_DO(storage.Write(sizeof(MasterHeader), pAtomic.get(), sizeof(MasterHeader)));
    NN_RESULT_DO(storage.Flush());

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        拡張データを取得します。
*
* @param[out]   outData         拡張データ
* @param[in]    storage         ストレージ
* @param[in]    pBufferManager  バッファマネージャ
* @param[in]    pMacGenerator   MAC 生成クラス
* @param[in]    minimumVersion  最低バージョン
*
* @return       関数の処理結果を返します。
*
* @details      最後にコミットした際の拡張データを読み込みます。
*               すなわち、WriteExtraData で書き込んだデータは
*               コミットするまでこの関数では読み込めません。
*               マウントしていなくても実行可能です。
*/
Result JournalIntegritySaveDataFileSystem::ReadExtraData(
           ExtraData* outData,
           fs::SubStorage storage,
           IBufferManager* pBufferManager,
           IMacGenerator* pMacGenerator,
           uint32_t minimumVersion
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outData);
    NN_SDK_REQUIRES(minimumVersion <= (VersionSupportedMax >> VersionMajorShift));
    NN_SDK_REQUIRES(minimumVersion >= (VersionSupportedMin >> VersionMajorShift));
    uint32_t saveDataMinimumVersion = (minimumVersion << VersionMajorShift);

    // マスター管理領域を読み込みます。
    const auto pAtomic = buffers::MakeUniqueBufferFromBufferManager<MasterHeader>(pBufferManager, NN_CURRENT_FUNCTION_NAME);
    NN_SDK_ASSERT_NOT_NULL(pAtomic);

    auto isProvisionallyCommitted = false;
    auto isFirstControlAreaMounted = false;
    NN_RESULT_DO(
        ReadMasterControlArea(
            pAtomic.get(),
            &isProvisionallyCommitted,
            &isFirstControlAreaMounted,
            sizeof(MasterHeader),
            storage,
            pMacGenerator,
            saveDataMinimumVersion
        )
    );

    std::memcpy(
        outData,
        &pAtomic->originalExtraData,
        sizeof(ExtraData)
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief        管理領域のパラメータを抽出します。
*
* @param[out]   outParamDuplex          二重化のパラメータ
* @param[out]   outParamIntegrity       完全性検証のパラメータ
* @param[out]   outCountDataBlock       データ領域のブロック数
* @param[out]   outCountJournalBlock    ジャーナル領域のブロック数
* @param[out]   outBlockSize            ブロックサイズ
* @param[out]   outCountExpandMax       拡張回数の最大
* @param[out]   outVersion              レイアウトデータのバージョン
*/
void JournalIntegritySaveDataFileSystem::ExtractParameters(
    HierarchicalDuplexStorageControlArea::InputParam* outParamDuplex,
    HierarchicalIntegrityVerificationStorageControlArea::InputParam* outParamIntegrity,
    uint32_t* outCountDataBlock,
    uint32_t* outCountJournalBlock,
    int32_t* outBlockSize,
    int* outCountExpandMax,
    uint32_t* outVersion
    ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCountDataBlock);
    NN_SDK_REQUIRES_NOT_NULL(outCountJournalBlock);
    NN_SDK_REQUIRES_NOT_NULL(outBlockSize);
    NN_SDK_REQUIRES_NOT_NULL(outCountExpandMax);

    m_DuplexStorage.GetParametners(outParamDuplex);
    m_IntegrityStorage.GetParameters(outParamIntegrity);

    *outBlockSize
        = static_cast<uint32_t>(m_IntegrityStorage.GetBlockSize());
    *outCountDataBlock
        = static_cast<uint32_t>(m_IntegrityStorage.GetDataAreaSize() / *outBlockSize);
    *outCountJournalBlock
        = static_cast<uint32_t>(m_IntegrityStorage.GetReservedAreaSize() / *outBlockSize);

    *outVersion = m_ControlArea.GetVersion();

    const auto updateCountMaxFileSystem
        = static_cast<int>(m_RemappedFileSystemStorage.GetMapUpdateCount());
    const auto updateCountMaxMeta
        = static_cast<int>(m_RemappedMetaStorage.GetMapUpdateCount());
    *outCountExpandMax = std::min(
        GetExpandCountMaxFromRemappedFileSystem(*outVersion, updateCountMaxFileSystem),
        GetExpandCountMaxFromRemappedMeta(*outVersion, updateCountMaxMeta));

    NN_SDK_ASSERT_EQUAL(
        nn::util::align_up(
            GetRemappedFileSystemEntryTableSize(*outVersion, *outCountExpandMax),
            RemapStorage::AlignmentLarge),
        m_RemappedFileSystemStorage.GetEntryTableSize());
    NN_SDK_ASSERT_EQUAL(
        nn::util::align_up(
            GetRemappedMetaEntryTableSize(*outVersion, *outCountExpandMax),
            RemapStorage::AlignmentLarge),
        m_RemappedMetaStorage.GetEntryTableSize());
}

/**
* @brief        メタデータを拡張します。
*
* @param[out]   outNeedsCommit              コミット処理を行う必要があるかどうかを格納する領域
* @param[in]    storageMetaRemapMeta        メタデータのリマップストレージのメタストレージ
* @param[in]    storageRemappedFileSystem   リマップが敷かれたファイルシステムストレージ
* @param[in]    storageControlArea          マスター管理領域ストレージ
* @param[in]    layoutHeader                拡張前のレイアウトヘッダ
* @param[in]    layoutHeaderNew             拡張後のレイアウトヘッダ
* @param[in]    sizeArchiveBlock            ファイルシステムコアのブロックサイズ
* @param[in]    pDuplicateCacheBuffer       キャッシュマネージャ
* @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
*
* @return       関数の処理結果を返します。
*
* @pre
*               - outNeedsCommit != nullptr
*/
Result JournalIntegritySaveDataFileSystem::ExpandMeta(
           bool* outNeedsCommit,
           fs::SubStorage storageMetaRemapMeta,
           fs::SubStorage storageRemappedFileSystem,
           fs::SubStorage storageControlArea,
           const FileSystemLayoutHeader& layoutHeader,
           const FileSystemLayoutHeader& layoutHeaderNew,
           uint32_t sizeArchiveBlock,
           IBufferManager* pDuplicateCacheBuffer,
           os::Mutex* pLocker
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outNeedsCommit);

    fs::SubStorage storageDuplexControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, hierarchicalDuplexControlArea),
                   sizeof(decltype(MasterHeader::hierarchicalDuplexControlArea))
               );
    fs::SubStorage storageIntegrityControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, hierarchicalIntegrityControlArea),
                   sizeof(decltype(MasterHeader::hierarchicalIntegrityControlArea))
               );
    fs::SubStorage storageJournalControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, journalControlArea),
                   sizeof(decltype(MasterHeader::journalControlArea))
               );
    fs::SubStorage storageMetaRemapControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, metaRemapControlArea),
                   sizeof(decltype(MasterHeader::metaRemapControlArea))
               );
    fs::SubStorage storageControlAreaSaveDataMetaHash(
                   &storageControlArea,
                   offsetof(MasterHeader, saveDataMetaHashControlArea),
                   sizeof(decltype(MasterHeader::saveDataMetaHashControlArea))
               );
    fs::SubStorage storageBitmapL1A(
                   &storageControlArea,
                   layoutHeader.offsetsBitmapL1[0],
                   layoutHeaderNew.sizeBitmapL1
               );
    fs::SubStorage storageBitmapL1B(
                   &storageControlArea,
                   layoutHeader.offsetsBitmapL1[1],
                   layoutHeaderNew.sizeBitmapL1
               );
    fs::SubStorage storageMasterHash(
                   &storageControlArea,
                   layoutHeader.offsetMasterHash,
                   layoutHeaderNew.sizeMasterHash
               );
    fs::SubStorage storageSaveDataMetaMasterHash(
                   &storageControlArea,
                   layoutHeader.offsetSaveDataMetaMasterHash,
                   layoutHeaderNew.sizeMasterHash
               );
    fs::SubStorage storageBitmapL2A(
                   &storageRemappedFileSystem,
                   layoutHeader.offsetsBitmapL2[0],
                   layoutHeaderNew.sizeBitmapL2
               );
    fs::SubStorage storageBitmapL2B(
                   &storageRemappedFileSystem,
                   layoutHeader.offsetsBitmapL2[1],
                   layoutHeaderNew.sizeBitmapL2
               );
    fs::SubStorage storageMetaA(
                   &storageRemappedFileSystem,
                   layoutHeader.offsetsMeta[0],
                   layoutHeaderNew.sizeMeta
               );
    fs::SubStorage storageMetaB(
                   &storageRemappedFileSystem,
                   layoutHeader.offsetsMeta[1],
                   layoutHeaderNew.sizeMeta
               );
    fs::SubStorage storageIntegritySaveData(
                   &storageRemappedFileSystem,
                   layoutHeader.offsetSaveData,
                   layoutHeaderNew.sizeSaveDataCore + layoutHeaderNew.sizeReservedArea
               );

    // 完全性保証メタデータ領域を拡張し、マウントします。
    nn::fssystem::save::HierarchicalDuplexInformation infoDuplexOld = {};
    nn::fssystem::save::HierarchicalDuplexInformation infoDuplexNew = {};
    {
        ScopedFinalizeObject<HierarchicalDuplexStorageControlArea> duplexControlArea;
        NN_RESULT_DO(duplexControlArea->Initialize(storageDuplexControlArea));
        infoDuplexOld = duplexControlArea->GetMeta().infoDuplex;
    }
    NN_RESULT_DO(
        HierarchicalDuplexStorageControlArea::Expand(
            storageDuplexControlArea,
            layoutHeaderNew.sizeMeta
        )
    );
    ScopedFinalizeObject<HierarchicalDuplexStorageControlArea> duplexControlArea;
    NN_RESULT_DO(duplexControlArea->Initialize(storageDuplexControlArea));
    infoDuplexNew = duplexControlArea->GetMeta().infoDuplex;
    NN_RESULT_DO(
        HierarchicalDuplexStorage::Expand(
            infoDuplexOld,
            infoDuplexNew,
            storageBitmapL1A,
            storageBitmapL1B,
            storageBitmapL2A,
            storageBitmapL2B,
            pDuplicateCacheBuffer
        )
    );

    HierarchicalDuplexStorage storageDuplicatedIntegrityMeta;
    NN_RESULT_DO(
        storageDuplicatedIntegrityMeta.Initialize(
            duplexControlArea->GetMeta().infoDuplex,
            storageBitmapL1A,
            storageBitmapL1B,
            storageBitmapL2A,
            storageBitmapL2B,
            storageMetaA,
            storageMetaB,
            layoutHeader.masterSelectBitmap,
            pDuplicateCacheBuffer,
            pLocker
        )
    );

    // さらにメタデータ用リマップストレージを敷きます。
    fs::SubStorage storageMetaRemapData(
                   &storageDuplicatedIntegrityMeta,
                   0,
                   layoutHeaderNew.sizeMeta
               );
    RemapStorage storageRemappedMeta;

    NN_RESULT_DO(
        storageRemappedMeta.Initialize(
            storageMetaRemapControlArea,
            storageMetaRemapMeta
        )
    );

    NN_RESULT_DO(storageRemappedMeta.RegisterStorage(storageMetaRemapData, 0));

    NN_RESULT_DO(
        storageRemappedMeta.ExpandMap(
            layoutHeader.offsetJournalTable,
            layoutHeaderNew.sizeJournalTable
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.ExpandMap(
            layoutHeader.offsetJournalBitmapPhysical,
            layoutHeaderNew.sizeJournalBitmapPhysical
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.ExpandMap(
            layoutHeader.offsetJournalBitmapVirtual,
            layoutHeaderNew.sizeJournalBitmapVirtual
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.ExpandMap(
            layoutHeader.offsetJournalBitmapUnassigned,
            layoutHeaderNew.sizeJournalBitmapUnassigned
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.ExpandMap(
            layoutHeader.offsetHierarchicalIntegrityLayeredHashL1,
            layoutHeaderNew.sizeHierarchicalIntegrityLayeredHashL1
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.ExpandMap(
            layoutHeader.offsetHierarchicalIntegrityLayeredHashL2,
            layoutHeaderNew.sizeHierarchicalIntegrityLayeredHashL2
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.ExpandMap(
            layoutHeader.offsetHierarchicalIntegrityLayeredHashL3,
            layoutHeaderNew.sizeHierarchicalIntegrityLayeredHashL3
        )
    );
    NN_RESULT_DO(
        storageRemappedMeta.ExpandMap(
            layoutHeader.offsetSaveDataMeta,
            layoutHeaderNew.sizeSaveDataMeta
        )
    );
    if( (layoutHeader.version & VersionCodeMask) >= (VersionSupportSaveDataMetaHash & VersionCodeMask) )
    {
        NN_RESULT_DO(
            storageRemappedMeta.ExpandMap(
                layoutHeader.offsetLayeredHashSaveDataMetaL1,
                layoutHeaderNew.sizeLayeredHashSaveDataMetaL1
            )
        );
        NN_RESULT_DO(
            storageRemappedMeta.ExpandMap(
                layoutHeader.offsetLayeredHashSaveDataMetaL2,
                layoutHeaderNew.sizeLayeredHashSaveDataMetaL2
            )
        );
    }
    NN_RESULT_DO(storageRemappedMeta.Flush());

    // メタデータ内をさらに分割します。
    fs::SubStorage storageJournalTable(
                   &storageRemappedMeta,
                   layoutHeader.offsetJournalTable,
                   layoutHeaderNew.sizeJournalTable
               );
    fs::SubStorage storageJournalBitmapPhysical(
                   &storageRemappedMeta,
                   layoutHeader.offsetJournalBitmapPhysical,
                   layoutHeaderNew.sizeJournalBitmapPhysical
               );
    fs::SubStorage storageJournalBitmapVirtual(
                   &storageRemappedMeta,
                   layoutHeader.offsetJournalBitmapVirtual,
                   layoutHeaderNew.sizeJournalBitmapVirtual
               );
    fs::SubStorage storageJournalBitmapUnassigend(
                   &storageRemappedMeta,
                   layoutHeader.offsetJournalBitmapUnassigned,
                   layoutHeaderNew.sizeJournalBitmapUnassigned
               );
    fs::SubStorage storageSaveDataMeta(
                   &storageRemappedMeta,
                   layoutHeader.offsetSaveDataMeta,
                   layoutHeaderNew.sizeSaveDataMeta
               );
    fs::SubStorage storageSaveDataMetaLayeredHashL1(
                   &storageRemappedMeta,
                   layoutHeader.offsetLayeredHashSaveDataMetaL1,
                   layoutHeaderNew.sizeLayeredHashSaveDataMetaL1
               );
    fs::SubStorage storageSaveDataMetaLayeredHashL2(
                   &storageRemappedMeta,
                   layoutHeader.offsetLayeredHashSaveDataMetaL2,
                   layoutHeaderNew.sizeLayeredHashSaveDataMetaL2
               );

    // 階層ハッシュ検証付きジャーナルストレージを拡張します。
    HierarchicalIntegrityVerificationMetaInformation metaIntegrity;
    metaIntegrity.Format();
    metaIntegrity.sizeMasterHash = static_cast<uint32_t>(layoutHeader.sizeMasterHash);

    {
        ScopedFinalizeObject<HierarchicalIntegrityVerificationStorageControlArea> controlArea;
        NN_RESULT_DO(controlArea->Initialize(storageIntegrityControlArea));
        controlArea->GetLevelHashInfo(&metaIntegrity.infoLevelHash);
    }

    metaIntegrity.infoLevelHash.info[0].offset = layoutHeader.offsetHierarchicalIntegrityLayeredHashL1;
    metaIntegrity.infoLevelHash.info[0].size = layoutHeaderNew.sizeHierarchicalIntegrityLayeredHashL1;

    metaIntegrity.infoLevelHash.info[1].offset = layoutHeader.offsetHierarchicalIntegrityLayeredHashL2;
    metaIntegrity.infoLevelHash.info[1].size = layoutHeaderNew.sizeHierarchicalIntegrityLayeredHashL2;

    metaIntegrity.infoLevelHash.info[2].offset = layoutHeader.offsetHierarchicalIntegrityLayeredHashL3;
    metaIntegrity.infoLevelHash.info[2].size = layoutHeaderNew.sizeHierarchicalIntegrityLayeredHashL3;

    metaIntegrity.infoLevelHash.info[3].offset = 0;
    metaIntegrity.infoLevelHash.info[3].size = layoutHeaderNew.sizeSaveDataCoreOriginal;

    NN_RESULT_DO(
        JournalIntegritySaveDataStorage::Expand(
            storageJournalControlArea,
            storageJournalTable,
            storageJournalBitmapPhysical,
            storageJournalBitmapVirtual,
            storageJournalBitmapUnassigend,
            storageIntegrityControlArea,
            storageMasterHash,
            layoutHeaderNew.sizeReservedArea,
            metaIntegrity,
            layoutHeaderNew.sizeSaveDataCoreOriginal
        )
    );

    // セーブデータメタの階層ハッシュストレージを拡張します。
    HierarchicalIntegrityVerificationStorage storageIntegritySaveDataMeta;
    ScopedFinalizer<HierarchicalIntegrityVerificationStorage> intSaveDataMetaStorageFinalizer(&storageIntegritySaveDataMeta);
    if( (layoutHeader.version & VersionCodeMask) >= (VersionSupportSaveDataMetaHash & VersionCodeMask) )
    {
        HierarchicalIntegrityVerificationMetaInformation metaIntegritySaveDataMetaHash;
        metaIntegritySaveDataMetaHash.Format();
        {
            ScopedFinalizeObject<HierarchicalIntegrityVerificationStorageControlArea> controlArea;
            NN_RESULT_DO(controlArea->Initialize(storageControlAreaSaveDataMetaHash));
            controlArea->GetLevelHashInfo(&metaIntegritySaveDataMetaHash.infoLevelHash);
        }
        metaIntegritySaveDataMetaHash.sizeMasterHash = static_cast<uint32_t>(layoutHeader.sizeMasterHash);

        metaIntegritySaveDataMetaHash.infoLevelHash.info[0].offset = layoutHeader.offsetLayeredHashSaveDataMetaL1;
        metaIntegritySaveDataMetaHash.infoLevelHash.info[0].size = layoutHeaderNew.sizeLayeredHashSaveDataMetaL1;

        metaIntegritySaveDataMetaHash.infoLevelHash.info[1].offset = layoutHeader.offsetLayeredHashSaveDataMetaL2;
        metaIntegritySaveDataMetaHash.infoLevelHash.info[1].size = layoutHeaderNew.sizeLayeredHashSaveDataMetaL2;

        metaIntegritySaveDataMetaHash.infoLevelHash.info[2].offset = layoutHeader.offsetSaveDataMeta;
        metaIntegritySaveDataMetaHash.infoLevelHash.info[2].size = layoutHeaderNew.sizeSaveDataMeta;

        NN_RESULT_DO(
            HierarchicalIntegrityVerificationStorageControlArea::Expand(
                storageControlAreaSaveDataMetaHash,
                metaIntegritySaveDataMetaHash
            )
        );
        HierarchicalIntegrityVerificationInformation infoLayeredHashSaveDataMeta;
        {
            ScopedFinalizeObject<HierarchicalIntegrityVerificationStorageControlArea> controlArea;
            NN_RESULT_DO(controlArea->Initialize(storageControlAreaSaveDataMetaHash));
            controlArea->GetLevelHashInfo(&metaIntegritySaveDataMetaHash.infoLevelHash);
            controlArea->GetLevelHashInfo(&infoLayeredHashSaveDataMeta);
            infoLayeredHashSaveDataMeta.maxLayers = nn::fssystem::save::IntegrityLayerCountSaveDataMeta;
        }

        HierarchicalIntegrityVerificationStorage::HierarchicalStorageInformation storageInfoSaveDataMetaHash;
        storageInfoSaveDataMetaHash.SetMasterHashStorage(storageSaveDataMetaMasterHash);
        storageInfoSaveDataMetaHash.SetLayer1HashStorage(storageSaveDataMetaLayeredHashL1);
        storageInfoSaveDataMetaHash.SetLayer2HashStorage(storageSaveDataMetaLayeredHashL2);
        storageInfoSaveDataMetaHash.SetDataStorage(storageSaveDataMeta);
        FilesystemBufferManagerSet integrityCacheBufferSet;
        integrityCacheBufferSet.pBuffer[0] = pDuplicateCacheBuffer;
        integrityCacheBufferSet.pBuffer[1] = pDuplicateCacheBuffer;
        integrityCacheBufferSet.pBuffer[2] = pDuplicateCacheBuffer;
        NN_RESULT_DO(
            storageIntegritySaveDataMeta.Initialize(
                infoLayeredHashSaveDataMeta,
                storageInfoSaveDataMetaHash,
                &integrityCacheBufferSet,
                pLocker,
                nn::fs::StorageType_SaveData
            )
        );

        // セーブデータ領域をフォーマットします。
        NN_RESULT_DO(
            SaveDataFileSystemCore::ExpandMeta(
                fs::SubStorage(&storageIntegritySaveDataMeta, 0, layoutHeaderNew.sizeSaveDataMeta),
                sizeArchiveBlock,
                layoutHeader.sizeSaveDataCoreOriginal,
                layoutHeaderNew.sizeSaveDataCoreOriginal
            )
        );
    }
    else
    {
        // セーブデータ領域をフォーマットします。
        NN_RESULT_DO(
            SaveDataFileSystemCore::ExpandMeta(
                storageSaveDataMeta,
                sizeArchiveBlock,
                layoutHeader.sizeSaveDataCoreOriginal,
                layoutHeaderNew.sizeSaveDataCoreOriginal
            )
        );
    }

    *outNeedsCommit = storageDuplicatedIntegrityMeta.NeedCommit();

    if (storageIntegritySaveDataMeta.IsInitialized())
    {
        NN_RESULT_DO(storageIntegritySaveDataMeta.Commit());
    }
    NN_RESULT_DO(storageDuplicatedIntegrityMeta.Flush());
    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        ジャーナリング+完全性検証機能付きセーブデータファイルシステムをフォーマットするために必要なサイズを取得します。
*
* @param[out]   outSize             ジャーナリング+完全性検証に必要なサイズを取得します。
* @param[in]    countExpandMax      拡張できる回数の最大
* @param[in]    inputParamDuplex    二重化パラメータ
* @param[in]    inputParamIntegrity 完全性検証パラメータ
* @param[in]    sizeBlock           ブロックサイズ
* @param[in]    countDataBlock      データブロック数
* @param[in]    countlReservedBlock 予約領域のブロック数
* @param[in]    version             セーブデータバージョン
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::QuerySize(
           int64_t* outSize,
           int countExpandMax,
           const HierarchicalDuplexStorageControlArea::InputParam& inputParamDuplex,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam&
           inputParamIntegrity,
           uint32_t sizeBlock,
           uint32_t countDataBlock,
           uint32_t countReservedBlock,
           uint32_t version
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);

    FileSystemLayoutHeader layoutHeader;

    // ジャーナリング+完全性検証に必要なサイズを求めます。
    const auto result = FormatFileSystemLayoutHeader(
                            &layoutHeader,
                            version,
                            GetRemappedFileSystemEntryTableSize(version, countExpandMax),
                            GetRemappedMetaEntryTableSize(version, countExpandMax),
                            inputParamDuplex,
                            inputParamIntegrity,
                            sizeBlock,
                            countDataBlock,
                            countReservedBlock
                        );
    if( result.IsSuccess() )
    {
        *outSize = layoutHeader.offsetFileSystemRemapData + layoutHeader.sizeFileSystemRemapData;
    }
    return result;
}

/**
* @brief        ジャーナリング+完全性検証機能付きセーブデータファイルシステムをフォーマットするために必要なサイズを取得します。
*
* @param[out]   outSize             ジャーナリング+完全性検証に必要なサイズを取得します。
* @param[in]    countExpandMax      拡張できる回数の最大
* @param[in]    inputParamDuplex    二重化パラメータ
* @param[in]    inputParamIntegrity 完全性検証パラメータ
* @param[in]    sizeBlock           ブロックサイズ
* @param[in]    countDataBlock      データブロック数
* @param[in]    countlReservedBlock 予約領域のブロック数
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::QuerySize(
           int64_t* outSize,
           int countExpandMax,
           const HierarchicalDuplexStorageControlArea::InputParam& inputParamDuplex,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam&
           inputParamIntegrity,
           uint32_t sizeBlock,
           uint32_t countDataBlock,
           uint32_t countReservedBlock
       ) NN_NOEXCEPT
{
    return QuerySize(
               outSize,
               countExpandMax,
               inputParamDuplex,
               inputParamIntegrity,
               sizeBlock,
               countDataBlock,
               countReservedBlock,
               VersionCode
           );
}

/**
* @brief        署名を付けます。
*
* @param[in]    pStorage    計算したハッシュを格納するストレージ
* @param[in]    pHash       計算したハッシュを格納するバッファ
* @param[in]    pData       計算対象データ
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::Sign(
           fs::IStorage* pStorage,
           IMacGenerator* pMacGenerator,
           void* pHash,
           size_t hashSize,
           const void* pData,
           size_t dataSize
       ) NN_NOEXCEPT
{
    NN_UNUSED(pStorage);
    NN_UNUSED(pData);
    NN_SDK_REQUIRES_EQUAL(hashSize, static_cast<size_t>(MasterHeader::SignatureSize));
    NN_SDK_REQUIRES_NOT_NULL(pMacGenerator);

    pMacGenerator->Generate(pHash, hashSize, pData, dataSize);

    NN_RESULT_DO(pStorage->Write(0, pHash, hashSize));
    NN_RESULT_DO(pStorage->Flush());
    NN_RESULT_SUCCESS;
}

/**
* @brief        署名の正当性を検証します。
*
* @param[in]    pData         計算対象データ
* @param[in]    pSignature    比較に使用する署名
* @param[in]    signatureSize pSignature のサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::Verify(
           IMacGenerator* pMacGenerator,
           const void* pData,
           size_t dataSize,
           const void* pSignature,
           size_t signatureSize
       ) NN_NOEXCEPT
{
    NN_UNUSED(signatureSize);
    NN_SDK_REQUIRES_EQUAL(signatureSize, static_cast<size_t>(MasterHeader::SignatureSize));
    NN_SDK_REQUIRES_NOT_NULL(pMacGenerator);

    char buf[MasterHeader::SignatureSize];
    pMacGenerator->Generate(buf, sizeof(buf), pData, dataSize);

    if( !nn::crypto::IsSameBytes(pSignature, buf, sizeof(buf)) )
    {
        return nn::fs::ResultJournalIntegritySaveDataMasterSignatureVerificationFailed();
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        マスターヘッダの正当性を検証します。
*
* @param[in]    pMasterHeader   マスターヘッダ
*
* @return       関数の処理結果を返します。
*/
bool JournalIntegritySaveDataFileSystem::VerifyMasterHeaderContent(
         MasterHeader* pMasterHeader
     ) NN_NOEXCEPT
{
    if(
        (pMasterHeader->controlArea.header.magic != MagicCode)
        || ((pMasterHeader->controlArea.header.version & VersionCodeMask)
             > (VersionSupportedMax & VersionCodeMask))
        || ((pMasterHeader->controlArea.header.version & VersionCodeMask)
             < (VersionSupportedMin & VersionCodeMask))
        || (pMasterHeader->controlArea.header.sizeSaveDataCore < 0)
        || (pMasterHeader->controlArea.header.sizeReservedArea < 0)
        || (pMasterHeader->controlArea.header.offsetsBitmapL1[1]
            < (pMasterHeader->controlArea.header.offsetsBitmapL1[0]
                + pMasterHeader->controlArea.header.sizeBitmapL1))
        || (pMasterHeader->controlArea.header.offsetMasterHash
            < (pMasterHeader->controlArea.header.offsetsBitmapL1[1]
                + pMasterHeader->controlArea.header.sizeBitmapL1))
        || (static_cast<int64_t>(sizeof(MasterHeader))
            < (pMasterHeader->controlArea.header.offsetMasterHash
                + pMasterHeader->controlArea.header.sizeMasterHash))
        )
    {
        return false;
    }

    // 読み込んだ管理領域に対してSHA検証を行います。
    char hashControlArea[ControlAreaHashSize];
    nn::crypto::GenerateSha256Hash(
        hashControlArea,
        sizeof(hashControlArea),
        &pMasterHeader->hierarchicalDuplexControlArea,
        sizeof(MasterHeader) - sizeof(MasterHeader::ControlArea)
    );

    return nn::crypto::IsSameBytes(
               pMasterHeader->controlArea.header.hash,
               hashControlArea,
               ControlAreaHashSize
           );
}

/**
* @brief        コンストラクタ
*/
JournalIntegritySaveDataFileSystem::JournalIntegritySaveDataFileSystem() NN_NOEXCEPT
    : m_pCommitLocker(nullptr)
    , m_IsInitialized(false)
    , m_SaveDataMinimumVersion(VersionSupportSaveDataMetaHash)
{
}

/**
* @brief        デストラクタ
*/
JournalIntegritySaveDataFileSystem::~JournalIntegritySaveDataFileSystem() NN_NOEXCEPT
{
    Finalize();
}

/**
* @brief        マスター管理領域を読み込みます。
*
* @param[out]   outHeader                    マスター管理領域の読み込み先
* @param[out]   outIsProvisionallyCommitted  仮コミット中かどうか
* @param[out]   outIsFirstControlAreaMounted 最初の管理領域を読み込んだかどうか
* @param[in]    headerBufferSize             読み込み先バッファのサイズ
* @param[in]    storage                      マスター管理領域を読み込むストレージ
* @param[in]    pMacGenerator                MAC 生成クラス
* @param[in]    minimumVersion               最低バージョン
*
* @return       関数の処理結果を返します。
*
* @pre
*               - outHeader != nullptr
*               - outIsProvisionallyCommitted != nullptr
*               - outIsFirstControlAreaMounted != nullptr
*/
Result JournalIntegritySaveDataFileSystem::ReadMasterControlArea(
           MasterHeader* outHeader,
           bool* outIsProvisionallyCommitted,
           bool* outIsFirstControlAreaMounted,
           size_t headerBufferSize,
           fs::SubStorage storage,
           IMacGenerator* pMacGenerator,
           uint32_t minimumVersion
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outHeader);
    NN_SDK_REQUIRES_NOT_NULL(outIsProvisionallyCommitted);
    NN_SDK_REQUIRES_NOT_NULL(outIsFirstControlAreaMounted);
    NN_SDK_REQUIRES_GREATER_EQUAL(headerBufferSize, sizeof(MasterHeader));
    NN_SDK_REQUIRES(minimumVersion <= VersionSupportedMax);
    NN_SDK_REQUIRES(minimumVersion >= VersionSupportedMin);
    NN_UNUSED(headerBufferSize);

    static const auto ControlAreaCount = 2;
    {
        int64_t sizeStorage = 0;
        NN_RESULT_DO(storage.GetSize(&sizeStorage));
        if( sizeStorage < static_cast<int64_t>(sizeof(MasterHeader::ControlArea) * ControlAreaCount) )
        {
            return nn::fs::ResultInvalidJournalIntegritySaveDataControlAreaSize();
        }
    }

    bool isValidControlArea[ControlAreaCount] = { false, false };
    bool isSupportedVersion[ControlAreaCount] = { true, true };
    for( auto indexControlArea = 0; indexControlArea < ControlAreaCount; ++indexControlArea )
    {
        NN_RESULT_DO(
            storage.Read(
                sizeof(MasterHeader) * indexControlArea,
                outHeader,
                sizeof(MasterHeader)
            )
        );

        // マスター領域に不正が無いことを検証します。
        NN_RESULT_TRY(
            Verify(
                pMacGenerator,
                &outHeader->controlArea.header,
                sizeof(outHeader->controlArea.header),
                outHeader->controlArea.signature,
                sizeof(outHeader->controlArea.signature)
            )
        )
        NN_RESULT_CATCH(nn::fs::ResultJournalIntegritySaveDataMasterSignatureVerificationFailed)
        {
            continue;
        }
        NN_RESULT_END_TRY;

        if( !VerifyMasterHeaderContent(outHeader) )
        {
            continue;
        }

        if( (outHeader->controlArea.header.version & VersionCodeMask) < (minimumVersion & VersionCodeMask) )
        {
            // 最低バージョンよりも古いセーブデータは使用できません。
            isSupportedVersion[indexControlArea] = false;
            continue;
        }

        // 検証に成功したら読み込みを終了します。
        isValidControlArea[indexControlArea] = true;
        break;
    }

    if( !isValidControlArea[0] && !isValidControlArea[1] )
    {
        if( !isSupportedVersion[0] || !isSupportedVersion[1] )
        {
            // 最低バージョンよりも古いセーブデータは使用できません。
            NN_RESULT_THROW(nn::fs::ResultUnsupportedVersion());
        }
        // 全ての管理領域が壊れているなら修復できません。
        NN_RESULT_THROW(nn::fs::ResultJournalIntegritySaveDataControlAreaVerificationFailed());
    }

    // 状態を設定します。
    *outIsProvisionallyCommitted = outHeader->commitData.counterForBundledCommit != 0;
    if( isValidControlArea[0] )
    {
        *outIsFirstControlAreaMounted = true;
    }
    else
    {
        NN_SDK_ASSERT(isValidControlArea[1]);
        *outIsFirstControlAreaMounted = false;
    }

    // 本コミットの場合は更新前のデータを更新しておきます。
    if( !*outIsProvisionallyCommitted )
    {
        // 更新前の拡張データを更新
        std::memcpy(&outHeader->originalExtraData, &outHeader->extraData, sizeof(ExtraData));

        // 更新前のマスターハッシュを更新
        const auto pAtomicAddr = reinterpret_cast<char*>(outHeader);
        std::memcpy(
            pAtomicAddr + outHeader->controlArea.header.offsetOriginalMasterHash,
            pAtomicAddr + outHeader->controlArea.header.offsetMasterHash,
            static_cast<size_t>(outHeader->controlArea.header.sizeMasterHash)
        );
        if( outHeader->controlArea.header.offsetOriginalSaveDataMetaMasterHash > 0 )
        {
            std::memcpy(
                pAtomicAddr + outHeader->controlArea.header.offsetOriginalSaveDataMetaMasterHash,
                pAtomicAddr + outHeader->controlArea.header.offsetSaveDataMetaMasterHash,
                static_cast<size_t>(outHeader->controlArea.header.sizeMasterHash)
            );
        }
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        マスター管理領域の MAC を更新します。
*
* @param[in]    storage                      マスター管理領域のストレージ
* @param[in]    pMacGenerator                MAC 生成クラス
* @param[in]    minimumVersion               最低バージョン
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::UpdateMasterControlAreaMac(
    fs::SubStorage storage,
    IMacGenerator* pMacGenerator,
    uint32_t minimumVersion
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(minimumVersion <= (VersionSupportedMax >> VersionMajorShift));
    NN_SDK_REQUIRES(minimumVersion >= (VersionSupportedMin >> VersionMajorShift));
    uint32_t saveDataMinimumVersion = (minimumVersion << VersionMajorShift);

    const int ControlAreaCount = 2;
    {
        int64_t sizeStorage = 0;
        NN_RESULT_DO(storage.GetSize(&sizeStorage));
        if (sizeStorage < static_cast<int64_t>(sizeof(MasterHeader::ControlArea) * ControlAreaCount))
        {
            return nn::fs::ResultInvalidJournalIntegritySaveDataControlAreaSize();
        }
    }

    // 各マスターヘッダの MAC を更新します。
    for (int index = 0; index < ControlAreaCount; ++index)
    {
        fs::SubStorage storageControlArea(
            &storage,
            sizeof(MasterHeader) * index,
            sizeof(MasterHeader)
        );

        const int HashTargetSize = sizeof(FileSystemLayoutHeader);
        NN_STATIC_ASSERT(HashTargetSize <= 512);
        char header[HashTargetSize];

        const int HashTargetOffset = offsetof(MasterHeader, controlArea.header);
        NN_RESULT_DO(storageControlArea.Read(HashTargetOffset, header, sizeof(header)));

        FileSystemLayoutHeader* pLayoutHeader = reinterpret_cast<FileSystemLayoutHeader*>(&header);
        if( (pLayoutHeader->version & VersionCodeMask) < (saveDataMinimumVersion & VersionCodeMask) )
        {
            // 最低バージョンよりも古いセーブデータは使用できません。
            NN_RESULT_THROW(nn::fs::ResultUnsupportedVersion());
        }

        NN_STATIC_ASSERT(MasterHeader::SignatureSize <= 256);
        char hash[MasterHeader::SignatureSize];

        NN_RESULT_DO(Sign(&storageControlArea, pMacGenerator, hash, sizeof(hash), header, sizeof(header)));
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        セーブデータをマウントします。
*
* @param[in]    layoutHeader                レイアウトヘッダ
* @param[in]    storageFileSystem           セーブデータファイルシステムをマウントするストレージ
* @param[in]    pIntegrityCacheBufferSet    キャッシュマネージャ
* @param[in]    pDuplicateCacheBuffer       キャッシュマネージャ
* @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::MountSaveData(
           const FileSystemLayoutHeader& layoutHeader,
           fs::SubStorage storageFileSystem,
           size_t sizeBlock,
           FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
           IBufferManager* pDuplicateCacheBuffer,
           os::Mutex* pLocker
       ) NN_NOEXCEPT
{
    // 管理領域ごとにストレージを分割します。
    fs::SubStorage storageDuplexControlArea
        = m_ControlArea.GetDuplexControlAreaStorage();
    fs::SubStorage storageFileSystemRemapControlArea
        = m_ControlArea.GetFileSystemRemapControlAreaStorage();
    fs::SubStorage storageMetaRemapControlArea
        = m_ControlArea.GetMetaRemapControlAreaStorage();
    fs::SubStorage storageBitmapL1A
        = m_ControlArea.GetL1BitmapStorage(0);
    fs::SubStorage storageBitmapL1B
        = m_ControlArea.GetL1BitmapStorage(1);

    // NOTE: リマップの下に敷くバッファの初期化
    NN_RESULT_DO(
        m_StorageBufferingRemap.Initialize(
            storageFileSystem,
            pDuplicateCacheBuffer,
            sizeBlock,
            RemapBufferCount
        )
    );

    // ファイルシステム用リマップストレージを敷きます。
    fs::SubStorage storageFileSystemRemapMeta(
                   &m_StorageBufferingRemap,
                   layoutHeader.offsetFileSystemRemapMeta,
                   layoutHeader.sizeFileSystemRemapMeta
               );
    fs::SubStorage storageMetaRemapMeta(
                   &m_StorageBufferingRemap,
                   layoutHeader.offsetMetaRemapMeta,
                   layoutHeader.sizeMetaRemapMeta
               );
    fs::SubStorage storageFileSystemRemapData(
                   &m_StorageBufferingRemap,
                   layoutHeader.offsetFileSystemRemapData,
                   layoutHeader.sizeFileSystemRemapData
               );

    NN_RESULT_DO(
        m_RemappedFileSystemStorage.Initialize(
            storageFileSystemRemapControlArea,
            storageFileSystemRemapMeta
        )
    );
    NN_RESULT_DO(m_RemappedFileSystemStorage.RegisterStorage(storageFileSystemRemapData, 0));

    fs::SubStorage storageBitmapL2A(
                   &m_RemappedFileSystemStorage,
                   layoutHeader.offsetsBitmapL2[0],
                   layoutHeader.sizeBitmapL2
               );
    fs::SubStorage storageBitmapL2B(
                   &m_RemappedFileSystemStorage,
                   layoutHeader.offsetsBitmapL2[1],
                   layoutHeader.sizeBitmapL2
               );
    fs::SubStorage storageMetaA(
                   &m_RemappedFileSystemStorage,
                   layoutHeader.offsetsMeta[0],
                   layoutHeader.sizeMeta
               );
    fs::SubStorage storageMetaB(
                   &m_RemappedFileSystemStorage,
                   layoutHeader.offsetsMeta[1],
                   layoutHeader.sizeMeta
               );

    ScopedFinalizeObject<HierarchicalDuplexStorageControlArea> duplexControlArea;
    NN_RESULT_DO(duplexControlArea->Initialize(storageDuplexControlArea));
    NN_RESULT_DO(
        m_DuplexStorage.Initialize(
            duplexControlArea->GetMeta().infoDuplex,
            storageBitmapL1A,
            storageBitmapL1B,
            storageBitmapL2A,
            storageBitmapL2B,
            storageMetaA,
            storageMetaB,
            layoutHeader.masterSelectBitmap,
            pDuplicateCacheBuffer,
            pLocker
        )
    );

    // さらに完全性保証メタデータ用リマップストレージを敷きます。
    fs::SubStorage storageMetaRemapData(
                   &m_DuplexStorage,
                   0,
                   layoutHeader.sizeMeta
               );

    NN_RESULT_DO(
        m_RemappedMetaStorage.Initialize(
            storageMetaRemapControlArea,
            storageMetaRemapMeta
        )
    );
    NN_RESULT_DO(m_RemappedMetaStorage.RegisterStorage(storageMetaRemapData, 0));

    fs::SubStorage storageBufferingMeta(
        &m_RemappedMetaStorage,
        0,
        std::numeric_limits<int64_t>::max()
    );
    NN_RESULT_DO(
        m_StorageBufferingMeta.Initialize(
            storageBufferingMeta,
            pDuplicateCacheBuffer,
            sizeBlock,
            MetaBufferCount
        )
    );

    // 完全性検証+ジャーナリングセーブデータをマウントします。
    return MountIntegritySaveData(
               layoutHeader,
               pIntegrityCacheBufferSet,
               pDuplicateCacheBuffer,
               pLocker
           );
} // NOLINT(impl/function_size)

/**
* @brief        完全性検証+ジャーナリングセーブデータをマウントします。
*
* @param[in]    layoutHeader                レイアウトヘッダ
* @param[in]    pIntegrityCacheBufferSet    キャッシュマネージャ
* @param[in]    pSaveDataCoreCacheBufferSet キャッシュマネージャ
* @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::MountIntegritySaveData(
           const FileSystemLayoutHeader& layoutHeader,
           FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
           IBufferManager* pSaveDataCoreCacheBuffer,
           os::Mutex* pLocker
       ) NN_NOEXCEPT
{
    // 領域ごとにストレージを分割します。
    // NOTE: 完全性保証メタデータ上のデータ
    fs::SubStorage storageIntegrityControlArea = m_ControlArea.GetIntegrityControlAreaStorage();
    fs::SubStorage storageJournalControlArea = m_ControlArea.GetJournalControlAreaStorage();
    fs::SubStorage storageSaveDataControlArea = m_ControlArea.GetSaveDataControlAreaStorage();
    fs::SubStorage storageLayeredHashSaveDataMetaControlArea = m_ControlArea.GetLayeredHashSaveDataMetaControlAreaStorage();
    fs::SubStorage storageMasterHash = m_ControlArea.GetMasterHashStorage();
    fs::SubStorage storageSaveDataMetaMasterHash = m_ControlArea.GetSaveDataMetaMasterHashStorage();

    fs::SubStorage storageIntegritySaveData(
                   &m_RemappedFileSystemStorage,
                   layoutHeader.offsetSaveData,
                   layoutHeader.sizeSaveDataCore + layoutHeader.sizeReservedArea
               );

    fs::SubStorage storageJournalTable(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetJournalTable,
                   layoutHeader.sizeJournalTable
               );
    fs::SubStorage storageJournalBitmapPhysical(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetJournalBitmapPhysical,
                   layoutHeader.sizeJournalBitmapPhysical
               );
    fs::SubStorage storageJournalBitmapVirtual(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetJournalBitmapVirtual,
                   layoutHeader.sizeJournalBitmapVirtual
               );
    fs::SubStorage storageJournalBitmapUnassigend(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetJournalBitmapUnassigned,
                   layoutHeader.sizeJournalBitmapUnassigned
               );
    fs::SubStorage storageIntegrityLayeredHashL1(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetHierarchicalIntegrityLayeredHashL1,
                   layoutHeader.sizeHierarchicalIntegrityLayeredHashL1
               );
    fs::SubStorage storageIntegrityLayeredHashL2(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetHierarchicalIntegrityLayeredHashL2,
                   layoutHeader.sizeHierarchicalIntegrityLayeredHashL2
               );
    fs::SubStorage storageIntegrityLayeredHashL3(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetHierarchicalIntegrityLayeredHashL3,
                   layoutHeader.sizeHierarchicalIntegrityLayeredHashL3
               );
    fs::SubStorage storageSaveDataMeta(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetSaveDataMeta,
                   layoutHeader.sizeSaveDataMeta
               );
    fs::SubStorage storageLayeredHashSaveDataMetaL1(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetLayeredHashSaveDataMetaL1,
                   layoutHeader.sizeLayeredHashSaveDataMetaL1
               );
    fs::SubStorage storageLayeredHashSaveDataMetaL2(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetLayeredHashSaveDataMetaL2,
                   layoutHeader.sizeLayeredHashSaveDataMetaL2
               );

    // 階層ハッシュ情報初期化
    HierarchicalIntegrityVerificationInformation infoIntegrity;
    {
        ScopedFinalizeObject<HierarchicalIntegrityVerificationStorageControlArea> controlArea;
        NN_RESULT_DO(controlArea->Initialize(storageIntegrityControlArea));
        controlArea->GetLevelHashInfo(&infoIntegrity);
        infoIntegrity.maxLayers = nn::fssystem::save::IntegrityLayerCountSave;
    }

    // ジャーナリングつき階層ハッシュストレージをマウント
    ScopedFinalizer<JournalIntegritySaveDataStorage> intStorageFinalizer(&m_IntegrityStorage);
    NN_RESULT_DO(
        m_IntegrityStorage.Initialize(
            storageJournalControlArea,
            storageJournalTable,
            storageJournalBitmapPhysical,
            storageJournalBitmapVirtual,
            storageJournalBitmapUnassigend,
            storageMasterHash,
            storageIntegrityLayeredHashL1,
            storageIntegrityLayeredHashL2,
            storageIntegrityLayeredHashL3,
            storageIntegritySaveData,
            infoIntegrity,
            pIntegrityCacheBufferSet,
            pLocker
        )
    );

    if( (layoutHeader.version & VersionCodeMask) >= VersionSupportSaveDataMetaHash )
    {
        // アロケーションテーブルに完全検証レイヤーを被せる
        ScopedFinalizer<HierarchicalIntegrityVerificationStorage> intSaveDataMetaStorageFinalizer(&m_IntegrityStorageSaveDataMeta);
        {
            HierarchicalIntegrityVerificationInformation infoIntegritySaveDataMeta;
            {
                ScopedFinalizeObject<HierarchicalIntegrityVerificationStorageControlArea> controlArea;
                NN_RESULT_DO(controlArea->Initialize(storageLayeredHashSaveDataMetaControlArea));
                controlArea->GetLevelHashInfo(&infoIntegritySaveDataMeta);
                infoIntegritySaveDataMeta.maxLayers = nn::fssystem::save::IntegrityLayerCountSaveDataMeta;
            }

            HierarchicalIntegrityVerificationStorage::HierarchicalStorageInformation storageInfoSaveDataMeta;
            storageInfoSaveDataMeta.SetMasterHashStorage(storageSaveDataMetaMasterHash);
            storageInfoSaveDataMeta.SetLayer1HashStorage(storageLayeredHashSaveDataMetaL1);
            storageInfoSaveDataMeta.SetLayer2HashStorage(storageLayeredHashSaveDataMetaL2);
            storageInfoSaveDataMeta.SetDataStorage(storageSaveDataMeta);
            NN_RESULT_DO(
                m_IntegrityStorageSaveDataMeta.Initialize(
                    infoIntegritySaveDataMeta,
                    storageInfoSaveDataMeta,
                    pIntegrityCacheBufferSet,
                    pLocker,
                    nn::fs::StorageType_SaveData
                )
            );
        }

        // 完全性検証付きジャーナリングストレージ上にファイルシステムをマウントします。
        NN_RESULT_DO(
            m_FileSystem.Initialize(
                storageSaveDataControlArea,
                fs::SubStorage(
                    &m_IntegrityStorageSaveDataMeta,
                    0,
                    layoutHeader.sizeSaveDataMeta
                ),
                fs::SubStorage(
                    &m_IntegrityStorage,
                    0,
                    layoutHeader.sizeSaveDataCoreOriginal
                ),
                pSaveDataCoreCacheBuffer
            )
        );

        intSaveDataMetaStorageFinalizer.Release();
    }
    else
    {
        // 完全性検証付きジャーナリングストレージ上にファイルシステムをマウントします。
        NN_RESULT_DO(
            m_FileSystem.Initialize(
                storageSaveDataControlArea,
                storageSaveDataMeta,
                fs::SubStorage(
                    &m_IntegrityStorage,
                    0,
                    layoutHeader.sizeSaveDataCoreOriginal
                ),
                pSaveDataCoreCacheBuffer
            )
        );
    }

    intStorageFinalizer.Release();

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        セーブデータを再マウントします。
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::RemountSaveData() NN_NOEXCEPT
{
    const auto pBufferManagerSet = m_IntegrityStorage.GetBufferManagerSet();
    const auto pBufferManager = m_StorageBufferingMeta.GetBufferManager();

    MasterHeaderBufferType pAtomic;
    NN_RESULT_DO(GetMasterHeader(&pAtomic, m_SaveDataMinimumVersion));

    // 一度アンマウントします。
    {
        ScopedFSLock lock(GetLocker());

        m_FileSystem.Finalize();
        m_IntegrityStorageSaveDataMeta.Finalize();
        m_IntegrityStorage.Finalize();
    }

    // マウントし直します。
    NN_RESULT_DO(
        MountIntegritySaveData(
            pAtomic->controlArea.header,
            pBufferManagerSet,
            pBufferManager,
            m_pCommitLocker
        )
    );

    StoreMasterHeader(pAtomic.release(), pAtomic.get_deleter().GetSize());

    NN_RESULT_SUCCESS;
}

/**
* @brief        ジャーナリング+完全性検証付きセーブデータファイルシステムをマウントします。
*
* @param[in]    storage                     ストレージ
* @param[in]    pIntegrityCacheBufferSet    キャッシュマネージャ
* @param[in]    pDuplicateCacheBuffer       キャッシュマネージャ
* @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
* @param[in]    pMacGenerator               MAC 生成クラス
* @param[in]    minimumVersion              最低バージョン
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::Initialize(
           fs::SubStorage storage,
           FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
           IBufferManager* pDuplicateCacheBuffer,
           os::Mutex* pLocker,
           IMacGenerator* pMacGenerator,
           uint32_t minimumVersion
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pIntegrityCacheBufferSet);
    NN_SDK_ASSERT_NOT_NULL(pDuplicateCacheBuffer);
    NN_SDK_ASSERT_NOT_NULL(pLocker);
    NN_SDK_ASSERT_NOT_NULL(pMacGenerator);
    NN_SDK_REQUIRES(minimumVersion <= (VersionSupportedMax >> VersionMajorShift));
    NN_SDK_REQUIRES(minimumVersion >= (VersionSupportedMin >> VersionMajorShift));

    m_pCommitLocker = pLocker;
    m_pMacGenerator = pMacGenerator;
    m_SaveDataMinimumVersion = (minimumVersion << VersionMajorShift);

    // キャッシュマネージャーに対して排他制御
    ScopedFSLock lock(GetLocker());

    m_pBufferManager = pDuplicateCacheBuffer;
    m_MasterHandle = 0;

    fs::SubStorage subStorage = storage;

    // マスター管理領域をRAMへ読み出します。
    auto pAtomic = buffers::MakeUniqueBufferFromBufferManager<MasterHeader>(pDuplicateCacheBuffer, NN_CURRENT_FUNCTION_NAME);
    NN_SDK_ASSERT_NOT_NULL(pAtomic);

    NN_RESULT_DO(
        ReadMasterControlArea(
            pAtomic.get(),
            &m_IsProvisionallyCommitted,
            &m_IsFirstControlAreaMounted,
            sizeof(MasterHeader),
            subStorage,
            m_pMacGenerator,
            m_SaveDataMinimumVersion
        )
    );

    const auto& layoutHeader = pAtomic->controlArea.header;

    // 初期化チェックします。
    if( layoutHeader.magic != MagicCode )
    {
        // フォーマットされていません。
        return nn::fs::ResultIncorrectJournalIntegritySaveDataMagicCode();
    }
    if( (layoutHeader.version & VersionCodeMask) < (m_SaveDataMinimumVersion & VersionCodeMask) )
    {
        // 特定バージョンよりも古いセーブデータは使用できません。
        NN_RESULT_THROW(nn::fs::ResultUnsupportedVersion());
    }
    if( (layoutHeader.version & VersionCodeMask) > (VersionSupportedMax & VersionCodeMask) )
    {
        // データのバージョン互換性エラーです。
        return nn::fs::ResultUnsupportedVersion();
    }

    NN_RESULT_DO(m_ControlArea.Reset(pAtomic.get()));

    // セーブデータストレージをマウントします。
    NN_RESULT_DO(
        MountSaveData(
            pAtomic->controlArea.header,
            storage,
            static_cast<size_t>(pAtomic->saveDataControlArea.sizeBlocks),
            pIntegrityCacheBufferSet,
            pDuplicateCacheBuffer,
            pLocker
        )
    );

    m_BaseStorageControlArea = fs::SubStorage(&storage, 0, sizeof(MasterHeader) * 2);

    m_IsExtraDataModified = false;

    StoreMasterHeader(pAtomic.release(), pAtomic.get_deleter().GetSize());

    m_IsInitialized = true;

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        ジャーナリング+完全性検証付きセーブデータファイルシステムをアンマウントします。
*               アンマウント時に自動Commitは行いません。
*/
void JournalIntegritySaveDataFileSystem::Finalize() NN_NOEXCEPT
{
    if( m_IsInitialized )
    {
        // キャッシュマネージャーに対して排他制御
        ScopedFSLock lock(GetLocker());

        const auto cache = m_pBufferManager->AcquireCache(m_MasterHandle);
        if( cache.first != 0 )
        {
            m_pBufferManager->DeallocateBuffer(cache.first, cache.second);
        }
        m_pBufferManager = nullptr;

        m_FileSystem.Finalize();

        m_IntegrityStorage.Finalize();
        m_IntegrityStorageSaveDataMeta.Finalize();
        m_DuplexStorage.Finalize();

        m_StorageBufferingMeta.Finalize();

        m_RemappedMetaStorage.Finalize();
        m_RemappedFileSystemStorage.Finalize();

        m_StorageBufferingRemap.Finalize();

        m_BaseStorageControlArea = fs::SubStorage();

        m_ControlArea.Clear();

        m_IsInitialized = false;
    }

    m_pCommitLocker = nullptr;
}

/**
* @brief        ファイルシステムを巻き戻します。
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::RollbackFileSystem() NN_NOEXCEPT
{
    // ファイルやディレクトリオープン中はロールバックできません。
    if( m_FileSystem.HasOpenedEntries() )
    {
        return nn::fs::ResultUnsupportedOperation();
    }

    MasterHeaderBufferType pAtomic;
    NN_RESULT_DO(GetMasterHeader(&pAtomic, m_SaveDataMinimumVersion));

    bool needsFlipMasterSelectBitmap = true;
    if( m_IntegrityStorageSaveDataMeta.IsInitialized() )
    {
        // 書き込みが発生しなかった場合はマスター選択ビットマップの反転は不要です。
        if( !m_IntegrityStorageSaveDataMeta.IsWrittenForRollback() )
        {
            needsFlipMasterSelectBitmap = false;
        }

        // 現在書き込み途中のデータを破棄します。
        NN_RESULT_DO(m_IntegrityStorageSaveDataMeta.OnRollback());
    }
    NN_RESULT_DO(m_IntegrityStorage.OnRollback());

    // 拡張データを巻き戻します。
    memcpy(&pAtomic->extraData, &pAtomic->originalExtraData, sizeof(ExtraData));

    if( m_IsProvisionallyCommitted )
    {
        // 二重化ビットマップを反転し、マウント時のデータが読み込まれるようにします。
        m_StorageBufferingMeta.InvalidateCaches();
        if( needsFlipMasterSelectBitmap )
        {
            NN_RESULT_DO(m_DuplexStorage.SwapDuplexBitmap());
            pAtomic->controlArea.header.masterSelectBitmap
                = !pAtomic->controlArea.header.masterSelectBitmap;
        }

        // マスターハッシュを巻き戻します。
        const auto pAtomicAddr = reinterpret_cast<char*>(pAtomic.get());
        std::memcpy(
            pAtomicAddr + pAtomic->controlArea.header.offsetMasterHash,
            pAtomicAddr + pAtomic->controlArea.header.offsetOriginalMasterHash,
            static_cast<size_t>(pAtomic->controlArea.header.sizeMasterHash)
        );
        std::memcpy(
            pAtomicAddr + pAtomic->controlArea.header.offsetSaveDataMetaMasterHash,
            pAtomicAddr + pAtomic->controlArea.header.offsetOriginalSaveDataMetaMasterHash,
            static_cast<size_t>(pAtomic->controlArea.header.sizeMasterHash)
        );

        // 仮コミットデータをクリアします。
        pAtomic->commitData.counterForBundledCommit = 0;

        // 署名し直します。
        nn::crypto::GenerateSha256Hash(
            pAtomic->controlArea.header.hash,
            sizeof(pAtomic->controlArea.header.hash),
            &pAtomic->hierarchicalDuplexControlArea,
            sizeof(MasterHeader) - sizeof(MasterHeader::ControlArea)
        );
        NN_RESULT_DO(Sign(&m_BaseStorageControlArea, m_pMacGenerator, pAtomic->controlArea.signature, sizeof(pAtomic->controlArea.signature), &pAtomic->controlArea.header, sizeof(pAtomic->controlArea.header)));

        // 管理領域を書き込みます。
        NN_RESULT_DO(
            m_BaseStorageControlArea.Write(
                m_IsFirstControlAreaMounted ? sizeof(MasterHeader) : 0,
                pAtomic.get(),
                sizeof(MasterHeader)
            )
        );
        NN_RESULT_DO(m_BaseStorageControlArea.Flush());
        NN_RESULT_DO(
            m_BaseStorageControlArea.Write(
                m_IsFirstControlAreaMounted ? 0 : sizeof(MasterHeader),
                pAtomic.get(),
                sizeof(MasterHeader)
            )
        );
        NN_RESULT_DO(m_BaseStorageControlArea.Flush());

        m_IsProvisionallyCommitted = false;
    }
    else
    {
        // 二重化ビットマップを再マウントし、マウント時のデータが読み込まれるようにします。
        m_StorageBufferingMeta.InvalidateCaches();
        m_DuplexStorage.Remount();
    }

    m_IsExtraDataModified = false;

    NN_RESULT_DO(m_ControlArea.Reset(pAtomic.get()));
    StoreMasterHeader(pAtomic.release(), pAtomic.get_deleter().GetSize());

    // 巻き戻ったメタデータを元にデータをマウントし直します。
    return RemountSaveData();
}

/**
* @brief        ジャーナリング+完全性検証付きセーブデータファイルシステムの署名 (MAC) を更新します。
*
* @param[in]    storage                     セーブデータのストレージ
* @param[in]    pMacGenerator               MAC 生成クラス
* @param[in]    minimumVersion              最低バージョン
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::UpdateMac(
    fs::SubStorage storage,
    IMacGenerator* pMacGenerator,
    uint32_t minimumVersion
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pMacGenerator);
    NN_RESULT_DO(UpdateMasterControlAreaMac(storage, pMacGenerator, minimumVersion));
    NN_RESULT_SUCCESS;
}

/**
* @brief        セーブデータのマスターヘッダーを修復します。
*
* @param[in]    storage                      セーブデータのストレージ
* @param[in]    pDuplicateCacheBuffer        キャッシュマネージャ
* @param[in]    pMacGenerator                MAC 生成クラス
* @param[in]    minimumVersion               最低バージョン
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::RecoverMasterHeader(
    fs::SubStorage storage,
    IBufferManager* pDuplicateCacheBuffer,
    IMacGenerator* pMacGenerator,
    uint32_t minimumVersion
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(minimumVersion <= (VersionSupportedMax >> VersionMajorShift));
    NN_SDK_REQUIRES(minimumVersion >= (VersionSupportedMin >> VersionMajorShift));
    uint32_t saveDataMinimumVersion = (minimumVersion << VersionMajorShift);

    auto pMasterHeader = buffers::MakeUniqueBufferFromBufferManager<MasterHeader>(pDuplicateCacheBuffer, NN_CURRENT_FUNCTION_NAME);

    static const auto ControlAreaCount = 2;
    {
        int64_t sizeStorage = 0;
        NN_RESULT_DO(storage.GetSize(&sizeStorage));
        if( sizeStorage < static_cast<int64_t>(sizeof(MasterHeader::ControlArea) * ControlAreaCount) )
        {
            NN_RESULT_THROW(nn::fs::ResultInvalidJournalIntegritySaveDataControlAreaSize());
        }
    }

    bool isValidControlArea[ControlAreaCount];
    bool isSupportedVersion[ControlAreaCount];
    for( auto indexControlArea = 0; indexControlArea < ControlAreaCount; ++indexControlArea )
    {
        isValidControlArea[indexControlArea] = false;
        isSupportedVersion[indexControlArea] = true;

        NN_RESULT_DO(
            storage.Read(
                sizeof(MasterHeader) * indexControlArea,
                pMasterHeader.get(),
                sizeof(MasterHeader)
            )
        );

        // マスター領域に不正が無いことを検証します。
        fs::SubStorage storageControlArea(
            &storage,
            sizeof(MasterHeader) * indexControlArea,
            sizeof(MasterHeader)
        );
        NN_RESULT_TRY(
            Verify(
                pMacGenerator,
                &pMasterHeader->controlArea.header,
                sizeof(pMasterHeader->controlArea.header),
                pMasterHeader->controlArea.signature,
                sizeof(pMasterHeader->controlArea.signature)
            )
        )
            NN_RESULT_CATCH(nn::fs::ResultJournalIntegritySaveDataMasterSignatureVerificationFailed)
            {
                continue;
            }
        NN_RESULT_END_TRY;

        if( !VerifyMasterHeaderContent(pMasterHeader.get()) )
        {
            continue;
        }

        if( (pMasterHeader->controlArea.header.version & VersionCodeMask) < (saveDataMinimumVersion & VersionCodeMask) )
        {
            // 最低バージョンよりも古いセーブデータは使用できません。
            isSupportedVersion[indexControlArea] = false;
            continue;
        }

        isValidControlArea[indexControlArea] = true;
    }

    if( !isValidControlArea[0] && !isValidControlArea[1] )
    {
        if( !isSupportedVersion[0] || !isSupportedVersion[1] )
        {
            // 最低バージョンよりも古いセーブデータは使用できません。
            NN_RESULT_THROW(nn::fs::ResultUnsupportedVersion());
        }
        // 全ての管理領域が壊れているなら修復できません。
        NN_RESULT_THROW(nn::fs::ResultJournalIntegritySaveDataControlAreaVerificationFailed());
    }

    if( isValidControlArea[0] && isValidControlArea[1] )
    {
        // 壊れてないなら何もしません
        NN_RESULT_SUCCESS;
    }

    if( isValidControlArea[0] )
    {
        // 0 -> 1
        NN_RESULT_DO(
            storage.Read(
                sizeof(MasterHeader) * 0,
                pMasterHeader.get(),
                sizeof(MasterHeader)
            )
        );
        NN_RESULT_DO(
            storage.Write(
                sizeof(MasterHeader) * 1,
                pMasterHeader.get(),
                sizeof(MasterHeader)
            )
        );
        NN_RESULT_DO(storage.Flush());
    }
    else
    {
        // 1 -> 0
        NN_RESULT_DO(
            storage.Read(
                sizeof(MasterHeader) * 1,
                pMasterHeader.get(),
                sizeof(MasterHeader)
            )
        );
        NN_RESULT_DO(
            storage.Write(
                sizeof(MasterHeader) * 0,
                pMasterHeader.get(),
                sizeof(MasterHeader)
            )
        );
        NN_RESULT_DO(storage.Flush());
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        拡張データを更新します。
*
* @param[in]    extraData   拡張データ
*
* @return       関数の処理結果を返します。
*               データが書き込まれるのは本コミットをした後です。
*/
Result JournalIntegritySaveDataFileSystem::WriteExtraData(
           const ExtraData& extraData
       ) NN_NOEXCEPT
{
    // 操作中はロックします。
    ScopedFSLock lock(GetLocker());

    // 拡張データを更新します。
    std::memcpy(m_ControlArea.GetExtraData(), &extraData, sizeof(ExtraData));

    m_IsExtraDataModified = true;

    NN_RESULT_SUCCESS;
}

/**
* @brief        拡張データを取得します。
*
* @param[out]   outData     拡張データ
*
* @return       関数の処理結果を返します。
*
* @details      WriteExtraData で書き込んだ値が読み込まれます。
*/
void JournalIntegritySaveDataFileSystem::ReadExtraData(
         ExtraData* outData
     ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outData);
    std::memcpy(outData, m_ControlArea.GetExtraData(), sizeof(ExtraData));
}

/**
* @brief        ファイルを作成します。
*
* @param[in]    path    ファイルパス
* @param[in]    size    ファイルサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::CreateFile(
           const Path& path,
           int64_t size
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    return m_FileSystem.CreateFile(path, size);
}

/**
* @brief        ディレクトリを作成します。
*
* @param[in]    path ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::CreateDirectory(
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    return m_FileSystem.CreateDirectory(path);
}

/**
* @brief        ファイルを開きます。
*
* @param[out]   outValue    ファイルオブジェクト
* @param[in]    path        ファイルパス
* @param[in]    mode        オープンモード
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::OpenFile(
           IFile** outValue,
           const Path& path,
           int mode
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);

    // 操作中はロックします。
    ScopedFSLock lock(GetLocker());

    IFile* pInnerFile = nullptr;
    Result result = m_FileSystem.OpenFile(&pInnerFile, path, mode);
    if( result.IsFailure() )
    {
        return result;
    }

    IFile* pFileWrapper = new IntegrityFilteredFile(pInnerFile, this);
    if( pFileWrapper == nullptr )
    {
        m_FileSystem.CloseFile(pInnerFile);

        // メモリ不足です。
        return nn::fs::ResultAllocationMemoryFailedNew();
    }

    *outValue = pFileWrapper;

    NN_RESULT_SUCCESS;
}

/**
* @brief ディレクトリを開きます。
*
* @param[out] outValue ディレクトリオブジェクト
* @param[in] path パス
*
* @return 関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::OpenDirectory(
           IDirectory** outValue,
           const Path& path
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);

    // 操作中はロックします。
    ScopedFSLock lock(GetLocker());

    IDirectory* pInnerDirectory = nullptr;
    NN_RESULT_DO(m_FileSystem.OpenDirectory(&pInnerDirectory, path));

    IDirectory* pDirectoryWrapper = new IntegrityFilteredDirectory(pInnerDirectory, this);
    if( pDirectoryWrapper == nullptr )
    {
        m_FileSystem.CloseDirectory(pInnerDirectory);

        // メモリ不足です。
        return nn::fs::ResultAllocationMemoryFailedNew();
    }

    *outValue = pDirectoryWrapper;

    NN_RESULT_SUCCESS;
}

/**
* @brief ファイルを閉じます。
*
* @param[in] pFile ファイルオブジェクト
*/
void JournalIntegritySaveDataFileSystem::CloseFile(IFile* pFile) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pFile);

    // 操作中はロックします。
    ScopedFSLock lock(GetLocker());

    IntegrityFilteredFile* pFile2 = reinterpret_cast<IntegrityFilteredFile*>(pFile);
    m_FileSystem.CloseFile(pFile2->GetFile());
    delete pFile;
}

/**
* @brief ディレクトリを閉じます。
*
* @param[in] pDirectory ディレクトリオブジェクト
*/
void JournalIntegritySaveDataFileSystem::CloseDirectory(IDirectory* pDirectory) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDirectory);

    // 操作中はロックします。
    ScopedFSLock lock(GetLocker());

    IntegrityFilteredDirectory* pDirectory2 = reinterpret_cast<IntegrityFilteredDirectory*>(pDirectory);
    m_FileSystem.CloseDirectory(pDirectory2->GetDirectory());
    delete pDirectory;
}

/**
* @brief        ファイルを削除します。
*
* @param[in]    path    ファイルパス
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::DeleteFile(
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    return m_FileSystem.DeleteFile(path);
}

/**
* @brief        ディレクトリを削除します。
*
* @param[in]    path    ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::DeleteDirectory(
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    return m_FileSystem.DeleteDirectory(path);
}

/**
* @brief        ディレクトリを再帰的に削除します。
*
* @param[in]    path    ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::DeleteDirectoryRecursively(
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    return m_FileSystem.DeleteDirectoryRecursively(path);
}

/**
* @brief        全ての子エントリを削除します。
*
* @param[in]    path    ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::CleanDirectoryRecursively(
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    return m_FileSystem.CleanDirectoryRecursively(path);
}

/**
* @brief        ファイルをリネームします。
*
* @param[in]    oldPath 変更対象のファイルパス
* @param[in]    newPath 変更後のファイルパス
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::RenameFile(
           const Path& oldPath,
           const Path& newPath
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    return m_FileSystem.RenameFile(oldPath, newPath);
}

/**
* @brief        ディレクトリをリネームします。
*
* @param[in]    oldPath 変更対象のディレクトリパス
* @param[in]    newPath 変更後のディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::RenameDirectory(
           const Path& oldPath,
           const Path& newPath
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    return m_FileSystem.RenameDirectory(oldPath, newPath);
}

/**
* @brief        指定したファイルが存在しているかどうかを取得します。
*
* @param[out]   outValue    ファイルが存在しているかどうか
* @param[in]    path        ファイルパス
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::HasFile(
           bool* outValue,
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    return m_FileSystem.HasFile(outValue, path);
}

/**
* @brief        指定したディレクトリが存在しているかどうかを取得します。
*
* @param[out]   outValue    ディレクトリが存在しているかどうか
* @param[in]    path        ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::HasDirectory(
           bool* outValue,
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    return m_FileSystem.HasDirectory(outValue, path);
}

/**
* @brief        空き領域のサイズをバイト数単位で取得します。
*
* @param[out]   outValue    空き領域のサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::GetFreeBytes(int64_t* outValue) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    return m_FileSystem.GetFreeBytes(outValue);
}

/**
* @brief        実データ領域のサイズをバイト数単位で取得します。
*
* @param[out]   outValue    実データ領域のサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataFileSystem::GetDataAreaBytes(int64_t* outValue) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    return m_FileSystem.GetDataAreaBytes(outValue);
}

/**
* @brief        乱数列生成関数をセットします。
*
* @return       なし
*/
void JournalIntegritySaveDataFileSystem::SetGenerateRandomFunction(
            GenerateRandomFunction function
        ) NN_NOEXCEPT
{
    HierarchicalIntegrityVerificationStorage::SetGenerateRandomFunction(function);
}

// セーブデータ内部ストレージアクセス用の Vistor を受け入れます。
nn::Result JournalIntegritySaveDataFileSystem::AcceptVisitor(IInternalStorageFileSystemVisitor* pVisitor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pVisitor);
    NN_RESULT_DO(m_IntegrityStorage.AcceptVisitor(pVisitor));
    NN_RESULT_DO(m_FileSystem.AcceptVisitor(pVisitor));

    fs::SubStorage storageIntegrityControlArea = m_ControlArea.GetIntegrityControlAreaStorage();
    const auto offset = offsetof(HierarchicalIntegrityVerificationMetaInformation, infoLevelHash) +
                        offsetof(HierarchicalIntegrityVerificationInformation, seed);
    fs::SubStorage storageIntegrityControlAreaSeed(&storageIntegrityControlArea, offset, sizeof(nn::fs::HashSalt));
    NN_RESULT_DO(pVisitor->VisitStorage(InternalStorageFileNameIntegritySeed, &storageIntegrityControlAreaSeed));

    NN_RESULT_SUCCESS;
}

const char JournalIntegritySaveDataFileSystem::InternalStorageFileNameIntegritySeed[]         = "InternalStorageFileNameIntegritySeed";

/**
* @brief        ファイルの内容をバッファに読み込みます。
*
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result IntegrityFilteredFile::ReadBytes(
           int64_t offset,
           void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);

    // 操作中はロックします。
    ScopedFSLock lock(m_pFileSystem->GetLocker());

    NN_RESULT_DO(FileInFile::ReadBytes(offset, buffer, size));

    NN_RESULT_SUCCESS;
}

/**
* @brief        コンストラクタ
*/
IntegrityFilteredFile::IntegrityFilteredFile(
    IFile* pFile,
    JournalIntegritySaveDataFileSystem* pFileSystem
) NN_NOEXCEPT
: m_pFileSystem(pFileSystem)
{
    NN_SDK_ASSERT_NOT_NULL(pFileSystem);
    FileInFile::Initialize(pFile);
}

/**
* @brief        デストラクタ
*/
IntegrityFilteredFile::~IntegrityFilteredFile() NN_NOEXCEPT
{
}

/**
* @brief        下位ファイルオブジェクトへの参照を解除します。
*/
void IntegrityFilteredFile::Finalize() NN_NOEXCEPT
{
    FileInFile::Finalize();
    m_pFileSystem = nullptr;
}
/**
* @brief        バッファの内容をファイルに書き込みます。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むデータサイズ
* @param[in]    option  書き込みオプション
*
* @return       関数の処理結果を返します。
*/
Result IntegrityFilteredFile::WriteBytes(
           int64_t offset,
           const void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    // 操作中はロックします。
    ScopedFSLock lock(m_pFileSystem->GetLocker());

    // 仮コミット中は書き込めません。
    if( m_pFileSystem->m_IsProvisionallyCommitted )
    {
        return nn::fs::ResultUnexpectedInJournalIntegritySaveDataFileSystemC();
    }

    NN_RESULT_DO(FileInFile::WriteBytes(offset, buffer, size));

    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルサイズを変更します。
*
* @param[in]    size    変更後のファイルサイズ
*
* @return       失敗の結果を返します。
*/
Result IntegrityFilteredFile::SetSize(int64_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);

    // 操作中はロックします。
    ScopedFSLock lock(m_pFileSystem->GetLocker());

    NN_RESULT_DO(FileInFile::SetSize(size));

    NN_RESULT_SUCCESS;
}

/**
* @brief        コンストラクタ
*/
IntegrityFilteredDirectory::IntegrityFilteredDirectory(
    IDirectory* pDirectory,
    JournalIntegritySaveDataFileSystem* pFileSystem
) NN_NOEXCEPT
    : m_pFileSystem(pFileSystem),
      m_pDirectory(pDirectory)
{
}

/**
* @brief        デストラクタ
*/
IntegrityFilteredDirectory::~IntegrityFilteredDirectory() NN_NOEXCEPT
{
}

/**
* @brief        下位ディレクトリオブジェクトへの参照を解除します。
*/
void IntegrityFilteredDirectory::Finalize() NN_NOEXCEPT
{
    m_pFileSystem = nullptr;
    m_pDirectory = nullptr;
}

/**
* @brief        ディレクトリ内の子ディレクトリエントリーを取得します。
*
* @param[out]   outNumEntries   取得したエントリー数
* @param[out]   outEntries      取得したエントリー情報
*                               配列のサイズは numEntries 以上である必要があります。
* @param[in]    numEntries      取得するエントリー数
*
* @return       関数の処理結果を返します。
*/
Result IntegrityFilteredDirectory::Read(
           int32_t* outNumEntries,
           nn::fs::DirectoryEntry outEntries[],
           int32_t numEntries
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
    ScopedFSLock lock(m_pFileSystem->GetLocker());
    return m_pDirectory->Read(outNumEntries, outEntries, numEntries);
}

}}}
