﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_TFormatString.h>

#include <nn/fs/fs_Result.h>
#include <nn/fssystem/fs_PathTool.h>
#include <nn/fssystem/save/fs_ISaveFileSystemDriver.h>

namespace nn { namespace fssystem { namespace save {

/**
* @brief        指定したファイルが存在しているかどうかを取得します。
*
* @param[out]   outValue    ファイルが存在しているかどうか
* @param[in]    path        ファイルパス
*
* @return       関数の処理結果を返します。
*/
Result IFileSystem::HasFile(bool* outValue, const Path& path) NN_NOEXCEPT
{
    IFile* p;
    Result result = OpenFile(&p, path, nn::fs::OpenMode_Read);
    if( result.IsFailure() )
    {
        if( nn::fs::ResultNotFound::Includes(result) )
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return result;
        }
    }
    CloseFile(p);
    *outValue = true;

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したディレクトリが存在しているかどうかを取得します。
*
* @param[out]   outValue    ディレクトリが存在しているかどうか
* @param[in]    path        ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result IFileSystem::HasDirectory(bool* outValue, const Path& path) NN_NOEXCEPT
{
    IDirectory* p;
    Result result = OpenDirectory(&p, path);
    if( result.IsFailure() )
    {
        if( nn::fs::ResultNotFound::Includes(result) )
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return result;
        }
    }
    CloseDirectory(p);
    *outValue = true;

    NN_RESULT_SUCCESS;
}

/**
* @brief        ディレクトリを再帰的に削除します。
*
* @param[in]    path            ディレクトリパス
* @param[in]    deleteItself    指定したパス自身を削除するかどうか
*
* @return       関数の処理結果を返します。
*/
Result IFileSystem::DeleteDirectoryRecursivelyInternal(
           const Path& path,
           bool deleteItself
       ) NN_NOEXCEPT
{
    const std::unique_ptr<Path> workPath(new Path);
    NN_RESULT_THROW_UNLESS(
        workPath != nullptr,
        nn::fs::ResultAllocationMemoryFailedInISaveFileSystemA()
    );
    NN_RESULT_DO(workPath->InitializeNormalized(path.GetString()));

    const auto workEntry = nn::fs::detail::MakeUnique<nn::fs::DirectoryEntry>();
    NN_RESULT_THROW_UNLESS(
        workEntry != nullptr,
        nn::fs::ResultAllocationMemoryFailedInISaveFileSystemB()
    );

    return DeleteDirectoryRecursivelyInternal(workPath.get(), workEntry.get(), deleteItself);
}

/**
* @brief        ディレクトリを再帰的に削除します。
*
* @param[in]    pPath           ディレクトリパス
* @param[in]    pEntry          ディレクトリエントリ
* @param[in]    deleteItself    指定したパス自身を削除するかどうか
*
* @return       関数の処理結果を返します。
*/
Result IFileSystem::DeleteDirectoryRecursivelyInternal(
                        Path* pPath,
                        nn::fs::DirectoryEntry* pEntry,
                        bool deleteItself
                    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPath);
    NN_SDK_REQUIRES_NOT_NULL(pEntry);

    {
        ScopedDirectoryAccessor directory(this);
        NN_RESULT_DO(directory.Open(*pPath));

        for( ; ; )
        {
            int32_t readCount = 0;
            NN_RESULT_DO(directory->Read(&readCount, pEntry, 1));
            if( readCount == 0 )
            {
                break;
            }
            if( strncmp(pEntry->name, ".", 2) == 0 || strncmp(pEntry->name, "..", 3) == 0 )
            {
                continue;
            }

            NN_RESULT_DO(pPath->SetChildPath(pEntry->name));
            if( pEntry->directoryEntryType != nn::fs::DirectoryEntryType_Directory )
            {
                NN_RESULT_DO(DeleteFile(*pPath));
            }
            else
            {
                NN_RESULT_DO(DeleteDirectoryRecursivelyInternal(pPath, pEntry, true));
            }
            NN_RESULT_DO(pPath->SetParentPath());
        }
    }

    if( deleteItself )
    {
        NN_RESULT_DO(DeleteDirectory(*pPath));
    }

    NN_RESULT_SUCCESS;
}

}}}

