﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <limits>
#include <memory>
#include <algorithm>
#include <functional>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Mutex.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/save/fs_ISaveFileSystemDriver.h>
#include <nn/fssystem/save/fs_IntegrityVerificationStorage.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>
#include <nn/fssystem/buffers/fs_BufferManagerUtility.h>

namespace nn { namespace fssystem { namespace save {

/**
* @brief        コンストラクタです。
*/
BlockCacheBufferedStorage::BlockCacheBufferedStorage() NN_NOEXCEPT
    : m_pBufferManager(nullptr),
      m_pLocker(nullptr),
      m_pEntry(),
      m_pStorageData(nullptr),
      m_LastResult(ResultSuccess()),
      m_SizeBytesData(0),
      m_SizeBytesVerificationBlock(0),
      m_ShiftBytesVerificationBlock(0),
      m_IndexInvalidate(0),
      m_MaxCacheEntryCount(0),
      m_Flags(0),
      m_BufferLevel(-1)
{
}

/**
* @brief        デストラクタです。
*/
BlockCacheBufferedStorage::~BlockCacheBufferedStorage() NN_NOEXCEPT
{
    Finalize();
}

/**
* @brief        バッファストレージを初期化します。
*
* @param[in]    pBuffer                     バッファマネージャ
* @param[in]    pLocker                     排他制御用のロックオブジェクト
* @param[in]    pData                       データを含んだストレージ
* @param[in]    sizeBytesData               ストレージのサイズ
* @param[in]    sizeBytesVerificationBlock  ブロックのサイズ
* @param[in]    maxCacheEntries             キャッシュするエントリー数
* @param[in]    bufferLevel                 バッファの登録レベル
* @param[in]    isKeepBurstMode             バースト転送維持モード
* @param[in]    storageType                 使用するストレージのタイプ
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::Initialize(
           IBufferManager* pBuffer,
           nn::os::Mutex* pLocker,
           IStorage* pData,
           int64_t sizeBytesData,
           size_t sizeBytesVerificationBlock,
           int maxCacheEntries,
           bool isRealDataCache,
           int8_t bufferLevel,
           bool isKeepBurstMode,
           fs::StorageType storageType
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pData);
    NN_SDK_ASSERT_NOT_NULL(pBuffer);
    NN_SDK_ASSERT_NOT_NULL(pLocker);
    NN_SDK_ASSERT(m_pBufferManager == nullptr);
    NN_SDK_ASSERT(m_pLocker == nullptr);
    NN_SDK_ASSERT(m_pStorageData == nullptr);
    NN_SDK_ASSERT(m_pEntry == nullptr);
    NN_SDK_ASSERT(maxCacheEntries > 0);

    // メモリ確保の必要がある処理を他のメンバ変数の初期化前に行います。
    m_pEntry = nn::fs::detail::MakeUnique<CacheEntry[]>(static_cast<size_t>(maxCacheEntries));
    if( m_pEntry == nullptr )
    {
        return nn::fs::ResultAllocationMemoryFailedInBlockCacheBufferedStorageA();
    }

    // 以降は途中抜けの可能性はありません。
    m_pBufferManager = pBuffer;
    m_pLocker = pLocker;
    m_pStorageData = pData;
    m_SizeBytesData = sizeBytesData;
    m_SizeBytesVerificationBlock = sizeBytesVerificationBlock;
    m_LastResult = ResultSuccess();
    m_IndexInvalidate = 0;
    m_MaxCacheEntryCount = maxCacheEntries;
    m_Flags = 0;
    m_BufferLevel = bufferLevel;
    m_StorageType = storageType;

    // ブロックサイズからシフト値を求めておきます。
    m_ShiftBytesVerificationBlock = ILog2(static_cast<uint32_t>(sizeBytesVerificationBlock));
    NN_SDK_ASSERT(static_cast<size_t>(1LL << m_ShiftBytesVerificationBlock)
        == m_SizeBytesVerificationBlock);

    // エントリーをクリアしておきます。
    std::memset(m_pEntry.get(), 0, sizeof(CacheEntry) * maxCacheEntries);

    // バースト転送維持モードを設定します。
    SetKeepBurstMode(isKeepBurstMode);

    // 実データに対するキャッシュかどうかを設定します。
    SetRealDataCache(isRealDataCache);

    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージを閉じます。
*/
void BlockCacheBufferedStorage::Finalize() NN_NOEXCEPT
{
    if (m_pEntry != nullptr)
    {
        // キャッシュしているエントリーのうち、
        // ディスクへの書き出しが必要なものはこのタイミングで行います。
        // バッファの返却もあわせて行います。
        // 正しく実装されていれば書き込みは行われず、エントリの返却のみが行われます。
        (void)InvalidateAllCacheEntries();

        m_pEntry.reset();
        m_pStorageData = nullptr;
        m_pBufferManager = nullptr;
        m_pLocker = nullptr;
        m_SizeBytesData = 0;
        m_SizeBytesVerificationBlock = 0;
        m_ShiftBytesVerificationBlock = 0;
        m_IndexInvalidate = 0;
        m_MaxCacheEntryCount = 0;
    }
}

/**
* @brief        ストレージのサイズを取得します。
*
* @param[out]   outValue    ストレージのサイズ
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);
    NN_SDK_ASSERT(m_pStorageData);
    *outValue = m_SizeBytesData;
    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージからデータを読み込みます。
*
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::Read(
           int64_t offset,
           void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pStorageData);
    NN_SDK_ASSERT_NOT_NULL(m_pBufferManager);

    // 継続不能な失敗をした後は新たな処理は行わず以前の失敗をそのまま返します。
    NN_RESULT_DO(m_LastResult);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    if( buffer == nullptr )
    {
        return nn::fs::ResultNullptrArgument();
    }

    int64_t readOffset = offset;
    size_t readSize = size;

    // 上限チェック
    if( readOffset > m_SizeBytesData )
    {
        return nn::fs::ResultInvalidOffset();
    }
    if( static_cast<int64_t>(readOffset + readSize) > m_SizeBytesData )
    {
        readSize = static_cast<size_t>(m_SizeBytesData - readOffset);
    }

    const size_t blockAlignment = m_SizeBytesVerificationBlock;
    const int64_t blockAlignmentMask = ~static_cast<int64_t>(blockAlignment - 1);

    int64_t offsetAligned = readOffset & blockAlignmentMask;
    int64_t offsetAlignedEnd = (readOffset + readSize + blockAlignment - 1) & blockAlignmentMask;
    NN_SDK_ASSERT(offsetAligned >= 0);
    NN_SDK_ASSERT(offsetAlignedEnd
        <= static_cast<int64_t>((m_SizeBytesData + blockAlignment - 1) & blockAlignmentMask));
    char* pDst = static_cast<char*>(buffer);

    {
        static const auto BulkReadSizeMax = 2 * 1024 * 1024;
        const bool isBulkReadEnabled =
            (readOffset != offsetAligned
                || static_cast<int64_t>(readOffset + readSize) != offsetAlignedEnd)
            && offsetAlignedEnd - offsetAligned <= BulkReadSizeMax;

        // 先頭がキャッシュされていればそれを使用します。
        CacheEntry entryHead = {};
        MemoryRange memoryRangeHead = {};
        bool isHeadCacheNeeded = true;

        NN_RESULT_DO(ReadHeadCache(
            &memoryRangeHead,
            &entryHead,
            &isHeadCacheNeeded,
            &readOffset,
            &offsetAligned,
            offsetAlignedEnd,
            &pDst,
            &readSize));

        if( offsetAligned >= offsetAlignedEnd )
        {
            // 読み込む領域が全てキャッシュされていました。
            NN_RESULT_SUCCESS;
        }

        bool isDestroyHeadBufferNeeded = true;
        NN_UTIL_SCOPE_EXIT
        {
            if( isDestroyHeadBufferNeeded )
            {
                DestroyBuffer(&entryHead, memoryRangeHead);
            }
        };

        // 末尾がキャッシュされていればそれを使用します。
        CacheEntry entryTail = {};
        MemoryRange memoryRangeTail = {};
        bool isTailCacheNeeded = true;

        NN_RESULT_DO(ReadTailCache(
            &memoryRangeTail,
            &entryTail,
            &isTailCacheNeeded,
            readOffset,
            offsetAligned,
            &offsetAlignedEnd,
            pDst,
            &readSize));

        if( offsetAligned >= offsetAlignedEnd )
        {
            // 読み込む領域が全てキャッシュされていました。
            NN_RESULT_SUCCESS;
        }

        bool isDestroyTailBufferNeeded = true;
        NN_UTIL_SCOPE_EXIT
        {
            if( isDestroyTailBufferNeeded )
            {
                DestroyBuffer(&entryTail, memoryRangeTail);
            }
        };

        if( isBulkReadEnabled )
        {
            isDestroyHeadBufferNeeded = false;
            isDestroyTailBufferNeeded = false;

            const auto result = BulkRead(
                readOffset,
                pDst,
                readSize,
                &memoryRangeHead,
                &memoryRangeTail,
                &entryHead,
                &entryTail,
                isHeadCacheNeeded,
                isTailCacheNeeded);
            if( !nn::fs::ResultAllocationPooledBufferNotEnoughSize::Includes(result) )
            {
                NN_RESULT_DO(result);
                NN_RESULT_SUCCESS;
            }
        }
    }

    while( offsetAligned < offsetAlignedEnd )
    {
        NN_SDK_ASSERT(readSize > 0);

        // アドレス境界にそろっており、読み込みサイズが十分なときにはバーストリードを維持します。
        // 但し、ストレージキャッシュモード時はアドレス境界を意識しません。
        // このモードで読み込んだデータはキャッシュしません。
        // キャッシュに含まれている場合はキャッシュを破棄します。
        if( IsEnabledKeepBurstMode()
         && (readOffset == offsetAligned)
         && (blockAlignment * 2 <= readSize) )
        {
            const size_t sizeAligned = readSize & blockAlignmentMask;

            // バーストリードの結果と不整合を起こさないよう、キャッシュを掃き出します。
            NN_RESULT_DO(
                UpdateLastResult(
                    FlushRangeCacheEntries(readOffset, sizeAligned, false)
                )
            );

            // このキャッシュエントリを有効なもの、とするため
            // 検証レイヤーに読み取り要求を発行します。
            NN_RESULT_DO(
                UpdateLastResult(
                    m_pStorageData->Read(readOffset, pDst, sizeAligned)
                )
            );

            // パラメータのみ更新します。
            pDst += sizeAligned;
            readOffset += sizeAligned;
            readSize -= sizeAligned;

            // このバッファはキャッシュと関連付けしません。
            offsetAligned += sizeAligned;
        }
        else
        {
            CacheEntry entry;
            MemoryRange memoryRange;

            // ブロック単位でアクセスするためのバッファを用意します。
            // entry.offsetとoffsetAlignedは異なるものが得られる可能性があります。
            // (既存のバッファが取れるため)
            // TODO: バッファを 2 ブロック分用意すべきだが 1 ブロックしか確保できないときに非効率なのを改善する
            NN_RESULT_DO(
                UpdateLastResult(
                    GetAssociateBuffer(
                        &memoryRange,
                        &entry,
                        offsetAligned,
                        static_cast<size_t>(offsetAlignedEnd - offsetAligned),
                        true
                    )
                )
            );
            char* pSrc = reinterpret_cast<char*>(memoryRange.first);

            // 中途半端なサイズは帰ってこないはずです。
            NN_SDK_ASSERT((entry.size & (blockAlignment - 1)) == 0);

            // キャッシュされているものではない場合
            if( !entry.isCached )
            {
                // このキャッシュエントリを有効なもの、とするため
                // 検証レイヤーに読み取り要求を発行します。
                Result result = m_pStorageData->Read(entry.offset, pSrc, entry.size);
                if( result.IsFailure() )
                {
                    // 致命的な読み込みエラーです。
                    DestroyBuffer(&entry, memoryRange);
                    NN_RESULT_THROW(UpdateLastResult(result));
                }
                entry.isCached = true;
            }

            NN_SDK_ASSERT(static_cast<int64_t>(entry.offset) <= offsetAligned);
            NN_SDK_ASSERT(offsetAligned < static_cast<int64_t>(entry.offset + entry.size));
            NN_SDK_ASSERT(offsetAligned <= readOffset);

            // ユーザーバッファにコピーします。
            {
                // 転送サイズを求めます。
                const int64_t offsetBuffer = readOffset - entry.offset;
                size_t sizeCopy = static_cast<size_t>(entry.offset + entry.size - readOffset);
                if( sizeCopy > readSize )
                {
                    sizeCopy = readSize;
                }

                std::memcpy(pDst, pSrc + offsetBuffer, sizeCopy);

                pDst += sizeCopy;
                readOffset += sizeCopy;
                readSize -= sizeCopy;
            }

            // 検証済みのこのバッファをキャッシュと関連付けします。
            NN_RESULT_DO(UpdateLastResult(StoreOrDestroyBuffer(memoryRange, &entry)));
            offsetAligned = entry.offset + entry.size;
        }
    }
    NN_SDK_ASSERT(readSize == 0);

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        ストレージにデータを書き込みます。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::Write(
           int64_t offset,
           const void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pStorageData);
    NN_SDK_ASSERT_NOT_NULL(m_pBufferManager);

    // 継続不能な失敗をした後は新たな処理は行わず以前の失敗をそのまま返します。
    NN_RESULT_DO(m_LastResult);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    if( buffer == nullptr )
    {
        return nn::fs::ResultNullptrArgument();
    }

    // 上限チェック
    if( offset > m_SizeBytesData )
    {
        return nn::fs::ResultInvalidOffset();
    }
    if( static_cast<int64_t>(offset + size) > m_SizeBytesData )
    {
        size = static_cast<size_t>(m_SizeBytesData - offset);

        // 丸められた結果 size == 0 となった場合には即時復帰(明示的な size == 0 の場合と切り分け)
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
    }

    const size_t blockAlignment = m_SizeBytesVerificationBlock;
    const int64_t blockAlignmentMask = ~static_cast<int64_t>(blockAlignment - 1);

    // ハッシュブロック単位に分けて書き込みます。
    int64_t offsetAligned = offset & blockAlignmentMask;
    int64_t offsetAlignedEnd = (offset + size + blockAlignment - 1) & blockAlignmentMask;
    NN_SDK_ASSERT(offsetAligned >= 0);
    NN_SDK_ASSERT(offsetAlignedEnd
        <= static_cast<int64_t>((m_SizeBytesData + blockAlignment - 1) & blockAlignmentMask));

    const uint8_t* pSrc = static_cast<const uint8_t*>(buffer);

    while( offsetAligned < offsetAlignedEnd )
    {
        if( IsEnabledKeepBurstMode()
         && (offset == offsetAligned)
         && (blockAlignment * 2 <= size) )
        {
            // アドレス境界にそろっており、サイズが十分なときにはバーストライトを維持します。
            // 但し、ストレージキャッシュモード時はアドレス境界を意識しません。
            // このモードで読み込んだデータはキャッシュしません。
            // キャッシュに含まれている場合はキャッシュを破棄します。
            size_t sizeAligned = size & blockAlignmentMask;

            // バーストライトの結果と不整合を起こさないよう、キャッシュを無効化します
            NN_RESULT_DO(
                UpdateLastResult(
                    FlushRangeCacheEntries(offset, sizeAligned, true)
                )
            );

            // 即座に検証レイヤーへストアします。
            NN_RESULT_DO(
                UpdateLastResult(
                    m_pStorageData->Write(offset, pSrc, sizeAligned)
                )
            );

            // 検証レイヤーへの書き込みがあったら、それ以後はバッファ確保でリトライします。
            buffers::EnableBlockingBufferManagerAllocation();

            // パラメータのみ更新します。
            pSrc += sizeAligned;
            offset += sizeAligned;
            size -= sizeAligned;
            offsetAligned += sizeAligned;
        }
        else
        {
            CacheEntry entry;
            MemoryRange memoryRange;

            // バッファを用意します。
            NN_RESULT_DO(
                UpdateLastResult(
                    GetAssociateBuffer(
                        &memoryRange,
                        &entry,
                        offsetAligned,
                        static_cast<size_t>(offsetAlignedEnd - offsetAligned),
                        true
                    )
                )
            );
            char* pDst = reinterpret_cast<char*>(memoryRange.first);

            // キャッシュされているものではなく、かつ書き込み対象データも"端"が有る場合は
            // 検証レイヤーからデータを読み込みます。
            if( !entry.isCached
             && ((offset != entry.offset) || (offset + size < entry.offset + entry.size)) )
            {
                // 検証レイヤーに読み取り要求を発行します。
                Result result = m_pStorageData->Read(entry.offset, pDst, entry.size);
                if( result.IsFailure() )
                {
                    // 致命的なエラーです。
                    DestroyBuffer(&entry, memoryRange);
                    NN_RESULT_THROW(UpdateLastResult(result));
                }
            }

            entry.isCached = true;

            NN_SDK_ASSERT(static_cast<int64_t>(entry.offset) <= offsetAligned);
            NN_SDK_ASSERT(offsetAligned < static_cast<int64_t>(entry.offset + entry.size));
            NN_SDK_ASSERT(offsetAligned <= offset);

            // キャッシュバッファーにコピーします。
            {
                // 転送サイズを求めます。
                int64_t offsetBuffer = offset - entry.offset;
                size_t sizeCopy = static_cast<size_t>(entry.offset + entry.size - offset);
                if( sizeCopy > size )
                {
                    sizeCopy = size;
                }

                std::memcpy(pDst + offsetBuffer, pSrc, sizeCopy);

                pSrc += sizeCopy;
                offset += sizeCopy;
                size -= sizeCopy;
            }

            // 可能な限り書き込みを遅らせます。
            entry.isWriteBack = true;

            // キャッシュへの書き込みがあったら、それ以後はバッファ確保でリトライします。
            buffers::EnableBlockingBufferManagerAllocation();

            // 検証済みのこのバッファをキャッシュと関連付けします。
            CacheIndex indexCache;
            NN_RESULT_DO(
                UpdateLastResult(
                    StoreAssociateBuffer(&indexCache, memoryRange, entry)
                )
            );
            offsetAligned = entry.offset + entry.size;

            // キャッシュにヒットしてしまうと IVF 層まで書き込みが届かないので
            // システムオプションが有効な場合は、書き出しと共にエントリも破棄します。
            if( ((0 <= indexCache)
             && IsEnabledKeepBurstMode()
             && (offset == offsetAligned)
             && (blockAlignment * 2 <= size)) )
            {
                NN_RESULT_DO(
                    UpdateLastResult(
                        FlushCacheEntry(indexCache, false)
                    )
                );
            }
        }
    }

    NN_RESULT_DO(m_LastResult);
    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        フラッシュします。
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::Flush() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pStorageData);
    NN_SDK_ASSERT(m_pBufferManager);

    // 継続不能な失敗をした後は新たな処理は行わず以前の失敗をそのまま返します。
    NN_RESULT_DO(m_LastResult);

    // ディスクへ書き出しします。
    NN_RESULT_DO(
        UpdateLastResult(
            FlushAllCacheEntries()
        )
    );

    // 下層レイヤにフラッシュを伝搬
    NN_RESULT_DO(
        UpdateLastResult(
            m_pStorageData->Flush()
        )
    );

    // 検証レイヤーへの書き込みがあったら、それ以後はバッファ確保でリトライします。
    buffers::EnableBlockingBufferManagerAllocation();

    NN_RESULT_SUCCESS;
}

/**
* @brief       範囲指定処理を行います。
*
* @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
* @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
* @param[in]   operationId      範囲指定処理の種類
* @param[in]   offset           範囲指定処理開始位置
* @param[in]   size             範囲指定処理を行うデータサイズ
* @param[in]   inBuffer         範囲指定処理に渡すバッファ
* @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
*
* @return      関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    NN_UNUSED(inBuffer);
    NN_UNUSED(inBufferSize);

    NN_SDK_ASSERT(m_pStorageData);

    switch( operationId )
    {
    case fs::OperationId::FillZero:
        {
            NN_RESULT_DO(FillZeroImpl(offset, size));
            NN_RESULT_SUCCESS;
        }

    case fs::OperationId::DestroySignature:
        {
            NN_RESULT_DO(DestroySignatureImpl(offset, size));
            NN_RESULT_SUCCESS;
        }

    case fs::OperationId::Invalidate:
        {
            if( m_StorageType == fs::StorageType_SaveData )
            {
                NN_RESULT_THROW(nn::fs::ResultUnsupportedOperation());
            }

            NN_RESULT_DO(InvalidateImpl(offset, size));
            NN_RESULT_SUCCESS;
        }

    case nn::fs::OperationId::QueryRange:
        {
            NN_RESULT_DO(QueryRangeImpl(outBuffer, outBufferSize, offset, size));
            NN_RESULT_SUCCESS;
        }

    default:
        return nn::fs::ResultUnsupportedOperation();
    }
}

/**
* @brief        コミット処理を行います。
*
* @param[in]    option  書き込みオプション
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::Commit() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pStorageData);
    NN_SDK_ASSERT(m_pBufferManager);

    // 継続不能な失敗をした後は新たな処理は行わず以前の失敗をそのまま返します。
    NN_RESULT_DO(m_LastResult);

    // ディスクへ書き出しします。
    NN_RESULT_DO(
        UpdateLastResult(
            FlushAllCacheEntries()
        )
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief        ロールバック時の処理です。
*
* @return       関数の処理結果を返します。
*
* @details      全てのキャッシュをストアせずに破棄します。
*/
Result BlockCacheBufferedStorage::OnRollback() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBufferManager);

    // 継続不能な失敗をした後は新たな処理は行わず以前の失敗をそのまま返します。
    NN_RESULT_DO(m_LastResult);

    const auto countMaxCacheEntries = GetMaxCacheEntryCount();
    for( int index = 0; index < countMaxCacheEntries; index++ )
    {
        const auto& entry = m_pEntry[index];
        if( entry.isValid )
        {
            if( entry.isWriteBack )
            {
                NN_SDK_ASSERT(entry.memoryAddress != 0 && entry.handle == 0);
                m_pBufferManager->DeallocateBuffer(entry.memoryAddress, entry.memorySize);
            }
            else
            {
                NN_SDK_ASSERT(entry.memoryAddress == 0 && entry.handle != 0);
                const auto memRange = m_pBufferManager->AcquireCache(entry.handle);
                if( memRange.first != 0 )
                {
                    m_pBufferManager->DeallocateBuffer(memRange.first, memRange.second);
                }
            }
        }
    }
    std::memset(m_pEntry.get(), 0, sizeof(CacheEntry) * countMaxCacheEntries);

    NN_RESULT_SUCCESS;
}

/**
* @brief       範囲を指定してゼロフィルします
*
* @param[in]   offset  ゼロフィル開始位置
* @param[in]   size    ゼロフィルするデータサイズ
*
* @return      関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::FillZeroImpl(
           int64_t offset,
           int64_t size
       ) NN_NOEXCEPT
{
    // 継続不能な失敗をした後は新たな処理は行わず以前の失敗をそのまま返します。
    NN_RESULT_DO(m_LastResult);

    int64_t sizeStorageData = 0;
    NN_RESULT_DO(
        UpdateLastResult(
            m_pStorageData->GetSize(&sizeStorageData)
        )
    );
    if( (offset < 0) || (sizeStorageData < offset) )
    {
        return nn::fs::ResultInvalidOffset();
    }

    const auto sizeBlock = m_SizeBytesVerificationBlock;
    auto offsetBegin = nn::util::align_down(offset, sizeBlock);
    auto offsetEnd = nn::util::align_up(std::min(offset + size, sizeStorageData), sizeBlock);

    // フラッシュします
    NN_RESULT_DO(
        UpdateLastResult(
            FlushRangeCacheEntries(offset, size, true)
        )
    );

    // 前後の端の実データをゼロ埋めします。
    if( offsetBegin < offset || offset + size < offsetEnd )
    {
        std::unique_ptr<char[], nn::fs::detail::Deleter> buf
            = nn::fs::detail::MakeUnique<char[]>(sizeBlock);

        if( buf == nullptr )
        {
            NN_RESULT_THROW(nn::fs::ResultAllocationMemoryFailedInBlockCacheBufferedStorageB());
        }

        if( offsetBegin < offset )
        {
            NN_RESULT_DO(
                UpdateLastResult(
                    m_pStorageData->Read(offsetBegin, buf.get(), sizeBlock)
                )
            );

            const auto offsetFill = static_cast<size_t>(offset - offsetBegin);
            const auto sizeFill
                = static_cast<size_t>(
                      std::min(
                          static_cast<int64_t>(sizeBlock - offsetFill),
                          size
                      )
                  );
            std::memset(buf.get() + offsetFill, 0, sizeFill);

            NN_RESULT_DO(
                UpdateLastResult(
                    m_pStorageData->Write(offsetBegin, buf.get(), sizeBlock)
                )
            );
            offsetBegin += sizeBlock;

            // 検証レイヤーへの書き込みがあったら、それ以後はバッファ確保でリトライします。
            buffers::EnableBlockingBufferManagerAllocation();
        }

        if( offsetBegin < offset + size && offset + size < offsetEnd )
        {
            const auto offsetLast = offsetEnd - sizeBlock;
            NN_RESULT_DO(
                UpdateLastResult(
                    m_pStorageData->Read(offsetLast, buf.get(), sizeBlock)
                )
            );

            const auto sizeFill = static_cast<size_t>((offset + size) - offsetLast);
            std::memset(buf.get(), 0, sizeFill);

            NN_RESULT_DO(
                UpdateLastResult(
                    m_pStorageData->Write(offsetLast, buf.get(), sizeBlock)
                )
            );
            offsetEnd -= sizeBlock;

            // 検証レイヤーへの書き込みがあったら、それ以後はバッファ確保でリトライします。
            buffers::EnableBlockingBufferManagerAllocation();
        }
    }

    if( offsetEnd - offsetBegin == 0 )
    {
        NN_RESULT_SUCCESS;
    }

    // ハッシュをゼロ埋めします。
    NN_RESULT_DO(
        UpdateLastResult(
            m_pStorageData->OperateRange(
                fs::OperationId::FillZero,
                offsetBegin,
                offsetEnd - offsetBegin
            )
        )
    );

    // ハッシュへの書き込みがあったら、それ以後はバッファ確保でリトライします。
    buffers::EnableBlockingBufferManagerAllocation();

    NN_RESULT_SUCCESS;
}

/**
* @brief       範囲を指定してハッシュを破壊します
*
* @param[in]   offset  ハッシュを破壊する開始位置
* @param[in]   size    ハッシュを破壊するデータサイズ
*
* @return      関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::DestroySignatureImpl(
           int64_t offset,
           int64_t size
       ) NN_NOEXCEPT
{
    // 継続不能な失敗をした後は新たな処理は行わず以前の失敗をそのまま返します。
    NN_RESULT_DO(m_LastResult);

    int64_t sizeStorageData = 0;
    NN_RESULT_DO(
        UpdateLastResult(
            m_pStorageData->GetSize(&sizeStorageData)
        )
    );
    if( (offset < 0) || (sizeStorageData < offset) )
    {
        return nn::fs::ResultInvalidOffset();
    }

    const auto sizeBlock = m_SizeBytesVerificationBlock;
    const auto offsetBegin = nn::util::align_up(offset, sizeBlock);
    const auto offsetEnd
        = nn::util::align_down(std::min(offset + size, sizeStorageData), sizeBlock);

    NN_RESULT_DO(
        UpdateLastResult(
            FlushRangeCacheEntries(offset, size, true)
        )
    );

    NN_RESULT_DO(
        UpdateLastResult(
            m_pStorageData->OperateRange(
                nn::fs::OperationId::DestroySignature,
                offsetBegin,
                offsetEnd - offsetBegin
            )
        )
    );

    // ハッシュへの書き込みがあったら、それ以後はバッファ確保でリトライします。
    buffers::EnableBlockingBufferManagerAllocation();

    NN_RESULT_SUCCESS;
}

/**
* @brief       範囲を指定してキャッシュを無効化します。
*
* @param[in]   offset  キャッシュ無効化開始位置
* @param[in]   size    無効化するデータサイズ
*
* @return      関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::InvalidateImpl(
           int64_t offset,
           int64_t size
       ) NN_NOEXCEPT
{
    InvalidateRangeCacheEntries(offset, size);

    int64_t storageSize = 0;
    NN_RESULT_DO(GetSize(&storageSize));

    const auto accessibleSize
        = std::min(size, storageSize - offset);
    const auto alignedOffset
        = nn::util::align_down(offset, m_SizeBytesVerificationBlock);
    const auto alignedOffsetEnd
        = nn::util::align_up(offset + accessibleSize, m_SizeBytesVerificationBlock);
    const auto alignedSize
        = alignedOffsetEnd - alignedOffset;
    {
        auto result = m_pStorageData->OperateRange(
            fs::OperationId::Invalidate,
            alignedOffset,
            alignedSize);
        NN_SDK_ASSERT(!nn::fs::ResultBufferAllocationFailed::Includes(result));
        NN_RESULT_DO(result);
    }

    if( nn::fs::ResultIntegrityVerificationStorageCorrupted::Includes(m_LastResult) )
    {
        m_LastResult = ResultSuccess();
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief       指定範囲にアクセスする際に使用する機能の情報を取得します。
*
* @param[out]  outBuffer        情報を保存するバッファ
* @param[in]   outBufferSize    情報を保存するバッファのサイズ
* @param[in]   offset           範囲指定の開始位置
* @param[in]   size             範囲指定のデータサイズ
*
* @return      関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::QueryRangeImpl(
           void* outBuffer,
           size_t outBufferSize,
           int64_t offset,
           int64_t size
       ) NN_NOEXCEPT
{
    int64_t storageSize = 0;
    NN_RESULT_DO(GetSize(&storageSize));

    const auto accessibleSize
        = std::min(size, storageSize - offset);
    const auto alignedOffset
        = nn::util::align_down(offset, m_SizeBytesVerificationBlock);
    const auto alignedOffsetEnd
        = nn::util::align_up(offset + accessibleSize, m_SizeBytesVerificationBlock);
    const auto alignedSize
        = alignedOffsetEnd - alignedOffset;

    NN_RESULT_DO(
        UpdateLastResult(m_pStorageData->OperateRange(
            outBuffer,
            outBufferSize,
            nn::fs::OperationId::QueryRange,
            alignedOffset,
            alignedSize,
            nullptr,
            0)
        )
    );
    NN_RESULT_SUCCESS;
}

/**
* @brief        指定のキャッシュエントリと重複するエントリが存在するか確認します。
*
* @param[in]    entry   キャッシュエントリ
*
* @return       重複エントリが見つかれば true を返します。
*/
bool BlockCacheBufferedStorage::ExistsRedundantCacheEntry(
         const CacheEntry& entry
     ) const NN_NOEXCEPT
{
    const int64_t offset = entry.offset;
    const size_t size = entry.size;

    // キャッシュエントリ配列を排他制御
    std::lock_guard<os::Mutex> locker(*m_pLocker);

    for( int index = 0; index < GetMaxCacheEntryCount(); ++index )
    {
        const CacheEntry *pEntry = &m_pEntry[index];
        if( pEntry->isValid
            && (m_pEntry[index].isWriteBack
            ? (m_pEntry[index].memoryAddress != 0)
            : (m_pEntry[index].handle != 0)) )
        {
            if( (pEntry->offset < static_cast<int64_t>(offset + size))
             && (offset < static_cast<int64_t>(pEntry->offset + pEntry->size)) )
            {
                // 重複するエントリが見つかりました。
                return true;
            }
        }
    }

    return false;
}

/**
* @brief        指定したオフセット、サイズを含んだキャッシュエントリとそのエントリーが示す範囲を取得します。
*
* @param[out]   outRange    範囲オブジェクト
* @param[out]   outEntry    キャッシュエントリ
* @param[in]    offset      領域の開始オフセット
* @param[in]    sizeIdeal   領域のサイズ
* @param[in]    isAllocateForWrite 書き込み用かどうか
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::GetAssociateBuffer(
           MemoryRange* outRange,
           CacheEntry* outEntry,
           int64_t offset,
           size_t sizeIdeal,
           bool isAllocateForWrite
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pStorageData);
    NN_SDK_ASSERT_NOT_NULL(m_pBufferManager);
    NN_SDK_ASSERT_NOT_NULL(outRange);
    NN_SDK_ASSERT_NOT_NULL(outEntry);
    NN_UNUSED(isAllocateForWrite);

    auto sizeActual = sizeIdeal;

    // キャッシュエントリ配列を排他制御
    std::lock_guard<os::Mutex> locker(*m_pLocker);

    // 最大キャッシュエントリ数を取得しておきます。
    const int numMaxCacheEntries = GetMaxCacheEntryCount();

    // 既に登録していたらそのエントリーを利用します。
    // 登録していない場合に備え、sizeActual を調整します
    int index;
    for( index = 0; index < numMaxCacheEntries; ++index )
    {
        if( m_pEntry[index].isValid
            && (m_pEntry[index].isWriteBack
            ? (m_pEntry[index].memoryAddress != 0)
            : (m_pEntry[index].handle != 0)) )
        {
            int64_t indexOffset = m_pEntry[index].offset;
            if( (indexOffset <= offset)
                && (offset < static_cast<int64_t>(indexOffset + m_pEntry[index].size)) )
            {
                // 始点を含む物が見つかりました。
                break;
            }
            if( (offset <= indexOffset)
                && (indexOffset < static_cast<int64_t>(offset + sizeActual)) )
            {
                // 途中が見つかったので sizeActual を切り詰めます。
                sizeActual = static_cast<size_t>(indexOffset - offset);
            }
        }
    }

    outRange->first = 0;
    outRange->second = 0;

    if( index != numMaxCacheEntries )
    {
        // 登録エントリーが見つかりました。
        if( m_pEntry[index].isWriteBack )
        {
            // 書き込み待ちなのでキャッシュとして登録されていません。
            *outRange = std::make_pair(m_pEntry[index].memoryAddress, m_pEntry[index].memorySize);
        }
        else
        {
            // キャッシュとして登録されているはずですが、
            // この場合でも、既に引き剥がされているために取れない可能性があります。
            *outRange = m_pBufferManager->AcquireCache(m_pEntry[index].handle);
        }

        *outEntry = m_pEntry[index];
        NN_SDK_ASSERT(outEntry->isValid);
        NN_SDK_ASSERT(outEntry->isCached);

        sizeActual = m_pEntry[index].size - static_cast<size_t>(offset - m_pEntry[index].offset);

        // リスト上の、このエントリーは無効にします。
        m_pEntry[index].isValid = false;
        m_pEntry[index].handle = 0ULL;
        m_pEntry[index].memoryAddress = 0;
        m_pEntry[index].memorySize = 0;

        // エントリのハンドルは無効化します。
        outEntry->isValid = true;
        outEntry->handle = 0ULL;
        outEntry->memoryAddress = 0;
        outEntry->memorySize = 0;
    }

    if( (*outRange).first == 0 )
    {
        const auto sizeThreshold = m_pBufferManager->GetTotalSize() / 8;

        // 空きが少なくなってきたらバッファを新規確保する前にフラッシュします。
        if( m_pBufferManager->GetTotalAllocatableSize() < sizeThreshold )
        {
            NN_RESULT_DO(FlushAllCacheEntries());
        }

        const size_t blockAlignment = m_SizeBytesVerificationBlock;

        // 新規にバッファを確保します。
        // 素直に sizeIdeal 分確保すると、大き過ぎるブロック割り当たるケースがあります。
        // そこで一度に割り当てるのは最大 2 ブロックとします。
        {
            NN_SDK_ASSERT(sizeActual >= 1);
            sizeActual = std::min(sizeActual, blockAlignment * 2);
        }

        // 書き込みに使うバッファーを用意します。
        NN_SDK_ASSERT(blockAlignment <= sizeActual);
        NN_RESULT_DO(buffers::AllocateBufferUsingBufferManagerContext(
            outRange,
            m_pBufferManager,
            sizeActual,
            IBufferManager::BufferAttribute(m_BufferLevel),
            [=](const MemoryRange& buffer) NN_NOEXCEPT
            {
                return buffer.first != 0 && blockAlignment <= buffer.second;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        if( outRange->second < sizeActual )
        {
            sizeActual = outRange->second;
        }

        // 新規に取れたメモリとします。
        // キャッシュとして登録されるまでは無効エントリとします。
        outEntry->isValid = true;
        outEntry->isWriteBack = false;
        outEntry->isCached = false;
        outEntry->isFlushing = false;
        outEntry->handle = 0ULL;
        outEntry->memoryAddress = 0;
        outEntry->memorySize = 0;
        outEntry->offset = offset;
        outEntry->size = sizeActual;
    }

    NN_SDK_ASSERT(outRange->second >= outEntry->size);

    NN_RESULT_SUCCESS;
}

/**
* @brief        バッファを破棄します。
*
* @param[in]    pEntry  破棄するキャッシュエントリ
* @param[in]    range   破棄するバッファの範囲
*
* @return       関数の処理結果を返します。
*/
void BlockCacheBufferedStorage::DestroyBuffer(
         CacheEntry* pEntry,
         const MemoryRange& range
     ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBufferManager);
    NN_SDK_ASSERT_NOT_NULL(pEntry);
    pEntry->isCached = false;
    pEntry->isValid = false;
    m_pBufferManager->DeallocateBuffer(range.first, range.second);
}

/**
* @brief        検証済みのバッファをキャッシュと関連付けします。
*
* @param[out]   outValue    関連付けられたキャッシュエントリのインデクス
* @param[in]    range       関連付けするバッファの範囲
* @param[in]    pEntry      関連付けするキャッシュエントリ
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::StoreAssociateBuffer(
           CacheIndex* outValue,
           const MemoryRange& memoryRange,
           const CacheEntry& entry
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);

    Result result = ResultSuccess();

    // キャッシュエントリ配列を排他制御
    std::lock_guard<os::Mutex> locker(*m_pLocker);

    if( entry.isWriteBack )
    {
        NN_RESULT_DO(ControlDirtiness());
    }

    // 最大キャッシュエントリ数を取得しておきます。
    const CacheIndex numMaxCacheEntries = static_cast<CacheIndex>(GetMaxCacheEntryCount());
    NN_SDK_ASSERT(numMaxCacheEntries > 0);

    // 空きエントリーを一つ取得します。
    CacheIndex index;
    for( index = 0; index < numMaxCacheEntries; ++index )
    {
        if( !m_pEntry[index].isValid )
        {
            break;
        }
    }

    // まだエントリーが取れていない場合
    if( index == numMaxCacheEntries )
    {
        // 既にあるエントリーから一つ要素を削除します。
        // 今は暫定実装を入れておきます。
        m_IndexInvalidate = (m_IndexInvalidate + 1) % numMaxCacheEntries;

        const CacheEntry *pEntryInvalidate = &m_pEntry[m_IndexInvalidate];
        NN_UNUSED(pEntryInvalidate);

        NN_SDK_ASSERT(pEntryInvalidate->isValid);
        NN_SDK_ASSERT(!pEntryInvalidate->isFlushing);

        // Flush書き込みに失敗しても、必ず無効化されていることが保証されます。
        result = FlushCacheEntry(m_IndexInvalidate, true);

        // キャッシュエントリは登録せずにエラーを返します。
        NN_RESULT_DO(result);

        NN_SDK_ASSERT(!pEntryInvalidate->isValid);
        NN_SDK_ASSERT(!pEntryInvalidate->isFlushing);

        index = m_IndexInvalidate;
    }

    // キャッシュしておきます。
    CacheEntry *pEntry = &m_pEntry[index];
    *pEntry = entry;

    NN_SDK_ASSERT(pEntry->isValid);
    NN_SDK_ASSERT(pEntry->isCached);
    NN_SDK_ASSERT(pEntry->handle == 0LL);
    NN_SDK_ASSERT(pEntry->memoryAddress == 0);

    // 重複エントリの登録をブロックします。
    if( !ExistsRedundantCacheEntry(*pEntry) )
    {
        if( pEntry->isWriteBack )
        {
            // 書き込み待ちであれば登録しません。
            pEntry->handle = 0LL;
            pEntry->memoryAddress = memoryRange.first;
            pEntry->memorySize = memoryRange.second;
        }
        else
        {
            // メモリレンジを登録しなおします。
            pEntry->handle = m_pBufferManager->RegisterCache(
                                 memoryRange.first,
                                 memoryRange.second,
                                 IBufferManager::BufferAttribute(m_BufferLevel)
                             );
            pEntry->memoryAddress = 0;
            pEntry->memorySize = 0;
        }

        NN_SDK_ASSERT(pEntry->isValid);
        *outValue = index;
        m_IndexInvalidate = index;
    }
    else
    {
        // バッファをマネージャーに返します。
        m_pBufferManager->DeallocateBuffer(memoryRange.first, memoryRange.second);
        pEntry->isValid = false;
        *outValue = -1;
    }

    NN_RESULT_DO(result);
    NN_RESULT_SUCCESS;
}

/**
* @brief        キャッシュエントリをフラッシュします。
*
* @param[in]    index               フラッシュするキャッシュエントリ
* @param[in]    isWithInvalidate    フラッシュしたキャッシュを無効かするかどうか
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::FlushCacheEntry(
           CacheIndex index,
           bool isWithInvalidate
       ) NN_NOEXCEPT
{
    // 書き込みに失敗してもInvalidate処理は必ず行います。

    // キャッシュエントリ配列を排他制御
    std::lock_guard<os::Mutex> locker(*m_pLocker);

    MemoryRange memoryRange;
    CacheEntry* pEntry = &m_pEntry[index];

    NN_SDK_ASSERT(pEntry->isValid);
    NN_SDK_ASSERT(!pEntry->isFlushing);

    if( !pEntry->isWriteBack )
    {
        NN_SDK_ASSERT(isWithInvalidate);

        // ストレージへ書き出す必要がない場合は、単にバッファを返却します。
        memoryRange = m_pBufferManager->AcquireCache(pEntry->handle);
        if( memoryRange.first != 0 )
        {
            // キャッシュは既に引き剥がされている可能性があります。
            m_pBufferManager->DeallocateBuffer(memoryRange.first, memoryRange.second);
        }

        pEntry->isValid = false;

        NN_RESULT_SUCCESS;
    }

    // キャッシュの取得
    // 必ず取れる必要があります。取れない場合はプログラムのミス
    pEntry->isFlushing = true;
    memoryRange = std::make_pair(pEntry->memoryAddress, pEntry->memorySize);
    NN_SDK_ASSERT(memoryRange.first != 0);
    NN_SDK_ASSERT(memoryRange.second >= pEntry->size);

    // データの診断
    NN_SDK_ASSERT(0 <= pEntry->offset);
    NN_SDK_ASSERT(pEntry->offset < m_SizeBytesData);
    NN_SDK_ASSERT((pEntry->offset & (m_SizeBytesVerificationBlock - 1)) == 0);

    // 最後のエラー状態
    Result result = nn::ResultSuccess();

    // データ書き込み
    size_t writeSize = pEntry->size;

    // エラーが発生していた場合は書き出しを行いません。
    // エラー状態の場合は書き出したことにして先に進めます。
    if( !m_LastResult.IsFailure() )
    {
        // pEntry は無効になるため、この書き込みは失敗が許されません。
        buffers::EnableBlockingBufferManagerAllocation();

        result = m_pStorageData->Write(
                     pEntry->offset,
                     reinterpret_cast<const void *>(memoryRange.first),
                     writeSize
                 );

        NN_SDK_ASSERT(!nn::fs::ResultBufferAllocationFailed::Includes(result));
    }
    else
    {
        result = m_LastResult;
    }

    pEntry->isWriteBack = false;

    if( isWithInvalidate )
    {
        // バッファをマネージャーに返します。
        m_pBufferManager->DeallocateBuffer(memoryRange.first, memoryRange.second);
        pEntry->isValid = false;
        pEntry->isFlushing = false;
    }
    else
    {
        // ディスクへのフラッシュのみ行います。キャッシュは無効化しません。
        NN_SDK_ASSERT(pEntry->isValid);

        pEntry->handle = m_pBufferManager->RegisterCache(
                             memoryRange.first,
                             memoryRange.second,
                             IBufferManager::BufferAttribute(m_BufferLevel)
                         );
        pEntry->memoryAddress = 0;
        pEntry->memorySize = 0;
        pEntry->isFlushing = false;
    }

    NN_RESULT_DO(result);
    NN_RESULT_SUCCESS;
}

/**
* @brief        範囲に含まれるキャッシュエントリをフラッシュします。
*
* @param[in]    offset              フラッシュ範囲の始点を示すオフセット
* @param[in]    size                フラッシュ範囲
* @param[in]    isWithInvalidate    フラッシュしたキャッシュを無効かするかどうか
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::FlushRangeCacheEntries(
           int64_t offset,
           int64_t size,
           bool isWithInvalidate
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pStorageData);
    NN_SDK_ASSERT_NOT_NULL(m_pBufferManager);

    Result result = ResultSuccess();

    const auto countMaxCacheEntries = GetMaxCacheEntryCount();

    for( int index = 0; index < countMaxCacheEntries; index++ )
    {
        const CacheEntry& entry = m_pEntry[index];
        if( entry.isValid
         && (entry.isWriteBack || isWithInvalidate)
         && (entry.offset < (offset + size))
         && (offset < static_cast<int64_t>(entry.offset + entry.size)) )
        {
            const auto curResult = FlushCacheEntry(index, isWithInvalidate);

            if( curResult.IsFailure() && result.IsSuccess() )
            {
                result = curResult;
            }
        }
    }

    NN_RESULT_DO(result);
    NN_RESULT_SUCCESS;
}

/**
* @brief        範囲に含まれるキャッシュエントリを無効化します。
*
* @param[in]    offset              無効化範囲の始点を示すオフセット
* @param[in]    size                無効化範囲
*
* @return       関数の処理結果を返します。
*
* @details      キャッシュをフラッシュしません。
*               フラッシュしてから無効化するためには FlushRangeCacheEntries() を使用します。
*/
void BlockCacheBufferedStorage::InvalidateRangeCacheEntries(
         int64_t offset,
         int64_t size
     ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pStorageData);
    NN_SDK_ASSERT_NOT_NULL(m_pBufferManager);

    const auto countMaxCacheEntries = GetMaxCacheEntryCount();

    for( int index = 0; index < countMaxCacheEntries; index++ )
    {
        auto& entry = m_pEntry[index];
        if( entry.isValid
         && (entry.offset < (offset + size))
         && (offset < static_cast<int64_t>(entry.offset + entry.size)) )
        {
            if( entry.isWriteBack )
            {
                NN_SDK_ASSERT(entry.memoryAddress != 0 && entry.handle == 0);
                m_pBufferManager->DeallocateBuffer(entry.memoryAddress, entry.memorySize);
            }
            else
            {
                // 単にバッファを返却します。
                NN_SDK_ASSERT(entry.memoryAddress == 0 && entry.handle != 0);
                const auto memoryRange = m_pBufferManager->AcquireCache(entry.handle);
                if( memoryRange.first != 0 )
                {
                    // キャッシュは既に引き剥がされている可能性があります。
                    m_pBufferManager->DeallocateBuffer(memoryRange.first, memoryRange.second);
                }
            }

            entry.isValid = false;
            entry.isWriteBack = false;
            entry.isFlushing = false;
        }
    }
}

/**
* @brief        全てのキャッシュエントリをフラッシュします。
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::FlushAllCacheEntries() NN_NOEXCEPT
{
    NN_RESULT_DO(
        FlushRangeCacheEntries(
            0,
            std::numeric_limits<int64_t>::max(),
            false
        )
    );
    NN_RESULT_SUCCESS;
}

/**
* @brief        全てのキャッシュエントリを無効化します。
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::InvalidateAllCacheEntries() NN_NOEXCEPT
{
    NN_RESULT_DO(
        FlushRangeCacheEntries(
            0,
            std::numeric_limits<int64_t>::max(),
            true
        )
    );
    NN_RESULT_SUCCESS;
}

/**
* @brief        ダーティバッファをため込み過ぎないように、必要ならキャッシュをフラッシュします。
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::ControlDirtiness() NN_NOEXCEPT
{
    const auto countCacheEntriesMax = GetMaxCacheEntryCount();
    NN_SDK_ASSERT_GREATER(countCacheEntriesMax, 0);

    const auto sizeTotal = m_pBufferManager->GetTotalSize();
    const auto sizeAllocatable = m_pBufferManager->GetTotalAllocatableSize();
    auto threshold = countCacheEntriesMax;

    // 空きバッファが少ない場合、ダーティエントリ数を制限します
    if( sizeAllocatable < sizeTotal / 4 )
    {
        threshold = 2;
    }
    else
    {
        NN_RESULT_SUCCESS;
    }

    auto countDirty = 0;
    auto indexFlushed = m_IndexInvalidate;

    // 本関数の呼び出し直後に登録されるキャッシュと合わせて threshold 個までダーティエントリを許可します
    // それを超える数のダーティエントリはここでフラッシュします
    for( auto count = 0; count < countCacheEntriesMax; ++count )
    {
        auto index = (count + m_IndexInvalidate + 1) % countCacheEntriesMax;
        if( m_pEntry[index].isValid && m_pEntry[index].isWriteBack )
        {
            ++countDirty;
            if( threshold <= countDirty )
            {
                NN_RESULT_DO(FlushCacheEntry(index, false));
                indexFlushed = index;
            }
        }
    }

    m_IndexInvalidate = indexFlushed;

    NN_RESULT_SUCCESS;
}

/**
* @brief        読み書き結果の検査関数
*
* @param[in]    result        発生したリザルト
*
* @return       関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::UpdateLastResult(nn::Result result) NN_NOEXCEPT
{
    // 発生したリザルトがエラーである場合、m_LastResult を更新する
    if( result.IsFailure()
        && !nn::fs::ResultBufferAllocationFailed::Includes(result)
        && m_LastResult.IsSuccess() )
    {
        m_LastResult = result;
    }

    NN_RESULT_DO(result);
    NN_RESULT_SUCCESS;
}

/**
* @brief      読み込む領域の先頭にある連続キャッシュを読み込み、読み込む領域を狭めます。
*
* @param[out] pOutMemoryRange       最後に読み込んだキャッシュのバッファ範囲
* @param[out] pOutEntry             最後に読み込んだキャッシュのエントリー
* @param[out] pOutIsCacheNeeded     最後に読み込んだキャッシュを新たにキャッシュすべきか
* @param[in]  pOffset               読み込み開始位置のポインタ
* @param[in]  pOffsetAligned        アライメントを揃えた読み込み開始位置のポインタ
* @param[in]  offsetAlignedEnd      アライメントを揃えた読み込み終了位置
* @param[in]  pBuffer               読み込んだ内容をコピーするバッファのポインタ
* @param[in]  pSize                 読み込むデータサイズのポインタ
*
* @return     関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::ReadHeadCache(
           MemoryRange* pOutMemoryRange,
           CacheEntry* pOutEntry,
           bool* pOutIsCacheNeeded,
           int64_t* pOffset,
           int64_t* pOffsetAligned,
           int64_t offsetAlignedEnd,
           char** pBuffer,
           size_t* pSize
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMemoryRange);
    NN_SDK_REQUIRES_NOT_NULL(pOutEntry);
    NN_SDK_REQUIRES_NOT_NULL(pOutIsCacheNeeded);
    NN_SDK_REQUIRES_NOT_NULL(pOffset);
    NN_SDK_REQUIRES_NOT_NULL(pOffsetAligned);
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(*pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pSize);

    NN_SDK_ASSERT_LESS(*pOffsetAligned, offsetAlignedEnd);

    const size_t blockAlignment = m_SizeBytesVerificationBlock;

    CacheEntry entry = {};
    MemoryRange memoryRange = {};
    *pOutIsCacheNeeded = true;

    while( *pOffsetAligned < offsetAlignedEnd )
    {
        NN_RESULT_DO(
            UpdateLastResult(
                GetAssociateBuffer(
                    &memoryRange,
                    &entry,
                    *pOffsetAligned,
                    blockAlignment,
                    true
                )
            )
        );

        if( entry.isCached )
        {
            *pOutIsCacheNeeded = false;

            // 転送サイズを求めます。
            const int64_t offsetBuffer = *pOffset - entry.offset;
            size_t sizeCopy = static_cast<size_t>(entry.offset + entry.size - *pOffset);
            if( sizeCopy > *pSize )
            {
                sizeCopy = *pSize;
            }

            std::memcpy(
                *pBuffer, reinterpret_cast<void*>(memoryRange.first + offsetBuffer), sizeCopy);

            *pBuffer += sizeCopy;
            *pOffset += sizeCopy;
            *pSize -= sizeCopy;
            *pOffsetAligned = entry.offset + entry.size;

            NN_RESULT_DO(UpdateLastResult(StoreOrDestroyBuffer(memoryRange, &entry)));
        }
        else
        {
            break;
        }
    }

    *pOutEntry = entry;
    *pOutMemoryRange = memoryRange;

    NN_RESULT_SUCCESS;
}

/**
* @brief      読み込む領域の末尾にある連続キャッシュを読み込み、読み込む領域を狭めます。
*
* @param[out] pOutMemoryRange       最後に読み込んだキャッシュのバッファ範囲
* @param[out] pOutEntry             最後に読み込んだキャッシュのエントリー
* @param[out] pOutIsCacheNeeded     最後に読み込んだキャッシュを新たにキャッシュすべきか
* @param[in]  offset                読み込み開始位置のポインタ
* @param[in]  offsetAligned         アライメントを揃えた読み込み開始位置
* @param[in]  pOffsetAlignedEnd     アライメントを揃えた読み込み終了位置のポインタ
* @param[in]  buffer                読み込んだ内容をコピーするバッファ
* @param[in]  pSize                 読み込むデータサイズのポインタ
*
* @return     関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::ReadTailCache(
           MemoryRange* pOutMemoryRange,
           CacheEntry* pOutEntry,
           bool* pOutIsCacheNeeded,
           int64_t offset,
           int64_t offsetAligned,
           int64_t* pOffsetAlignedEnd,
           char* buffer,
           size_t* pSize
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMemoryRange);
    NN_SDK_REQUIRES_NOT_NULL(pOutEntry);
    NN_SDK_REQUIRES_NOT_NULL(pOutIsCacheNeeded);
    NN_SDK_REQUIRES_NOT_NULL(pOffsetAlignedEnd);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_NOT_NULL(pSize);

    NN_SDK_ASSERT_LESS(offsetAligned, *pOffsetAlignedEnd);

    const size_t blockAlignment = m_SizeBytesVerificationBlock;

    CacheEntry entry = {};
    MemoryRange memoryRange = {};
    *pOutIsCacheNeeded = true;

    while( offsetAligned < *pOffsetAlignedEnd )
    {
        NN_RESULT_DO(
            UpdateLastResult(
                GetAssociateBuffer(
                    &memoryRange,
                    &entry,
                    *pOffsetAlignedEnd - blockAlignment,
                    blockAlignment,
                    true
                )
            )
        );
        if( entry.isCached )
        {
            *pOutIsCacheNeeded = false;

            // 転送サイズを求めます。
            const int64_t offsetBuffer = std::max(static_cast<int64_t>(0), offset - entry.offset);
            const size_t sizeCopy
                = std::min(*pSize, static_cast<size_t>(offset + *pSize - entry.offset));

            std::memcpy(
                buffer + *pSize - sizeCopy,
                reinterpret_cast<void*>(memoryRange.first + offsetBuffer),
                sizeCopy);

            *pSize -= sizeCopy;
            *pOffsetAlignedEnd = entry.offset;

            NN_RESULT_DO(UpdateLastResult(StoreOrDestroyBuffer(memoryRange, &entry)));
        }
        else
        {
            break;
        }
    }

    *pOutEntry = entry;
    *pOutMemoryRange = memoryRange;

    NN_RESULT_SUCCESS;
}

/**
* @brief      ストレージの内容をバッファに一括読み込みします。
*
* @param[in]  offset  読み込み開始位置
* @param[out] buffer  読み込んだ内容をコピーするバッファ
* @param[in]  size    読み込むデータサイズ
*
* @return     関数の処理結果を返します。
*/
Result BlockCacheBufferedStorage::BulkRead(
           int64_t offset,
           void* buffer,
           size_t size,
           MemoryRange* pMemoryRangeHead,
           MemoryRange* pMemoryRangeTail,
           CacheEntry* pEntryHead,
           CacheEntry* pEntryTail,
           bool isHeadCacheNeeded,
           bool isTailCacheNeeded
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_NOT_NULL(pMemoryRangeHead);
    NN_SDK_REQUIRES_NOT_NULL(pMemoryRangeTail);
    NN_SDK_REQUIRES_NOT_NULL(pEntryHead);
    NN_SDK_REQUIRES_NOT_NULL(pEntryTail);

    const size_t blockAlignment = m_SizeBytesVerificationBlock;
    const int64_t blockAlignmentMask = ~static_cast<int64_t>(blockAlignment - 1);

    int64_t readOffset = offset;
    size_t readSize = size;
    int64_t offsetAligned = readOffset & blockAlignmentMask;
    int64_t offsetAlignedEnd = (readOffset + readSize + blockAlignment - 1) & blockAlignmentMask;
    char* pDst = static_cast<char*>(buffer);
    bool isHeadCacheDestroyed = false;
    bool isTailCacheDestroyed = false;

    NN_UTIL_SCOPE_EXIT
    {
        if( !isHeadCacheDestroyed )
        {
            DestroyBuffer(pEntryHead, *pMemoryRangeHead);
        }
        if( !isTailCacheDestroyed )
        {
            DestroyBuffer(pEntryTail, *pMemoryRangeTail);
        }
    };

    NN_RESULT_DO(
        UpdateLastResult(
            FlushRangeCacheEntries(offsetAligned, offsetAlignedEnd - offsetAligned, false)
        )
    );

    PooledBuffer pooledBuffer;
    const size_t bufferSize = static_cast<size_t>(offsetAlignedEnd - offsetAligned);
    char* readBuffer = nullptr;
    if( readOffset == offsetAligned && readSize == bufferSize )
    {
        // アラインメントが揃っているので PooledBuffer を介さず直接一括読み込みします。
        readBuffer = pDst;
    }
    else if( isTailCacheNeeded
        && pEntryTail->offset == offsetAligned
        && pEntryTail->size == bufferSize )
    {
        // 末尾のキャッシュバッファを使って一括読み込みします。
        readBuffer = reinterpret_cast<char*>(pMemoryRangeTail->first);
    }
    else if( isHeadCacheNeeded
        && pEntryHead->offset == offsetAligned
        && pEntryHead->size == bufferSize )
    {
        // 先頭のキャッシュバッファを使って一括読み込みします。
        readBuffer = reinterpret_cast<char*>(pMemoryRangeHead->first);
    }
    else
    {
        // PooledBuffer で一括読み込みします。
        pooledBuffer.AllocateParticularlyLarge(bufferSize, 1);
        if( pooledBuffer.GetSize() < bufferSize )
        {
            NN_RESULT_THROW(nn::fs::ResultAllocationPooledBufferNotEnoughSize());
        }
        readBuffer = pooledBuffer.GetBuffer();
    }
    NN_RESULT_DO(
        m_pStorageData->Read(offsetAligned, readBuffer, bufferSize));
    if( pDst != readBuffer )
    {
        std::memcpy(pDst, readBuffer + readOffset - offsetAligned, readSize);
    }

    const auto funcCopyFromPooledBuffer
        = [&](CacheEntry* pEntry, MemoryRange* pMemoryRange) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pEntry);
        NN_SDK_ASSERT_NOT_NULL(pMemoryRange);

        if( offsetAligned <= pEntry->offset
            && pEntry->offset + pEntry->size <= offsetAligned + bufferSize )
        {
            NN_SDK_ASSERT(!pEntry->isCached);
            if( reinterpret_cast<void*>(pMemoryRange->first) != readBuffer )
            {
                std::memcpy(
                    reinterpret_cast<void*>(pMemoryRange->first),
                    readBuffer + pEntry->offset - offsetAligned,
                    pEntry->size);
            }
            pEntry->isCached = true;
        }
    };

    if( isTailCacheNeeded )
    {
        // 読み込む領域の末尾をキャッシュします。
        funcCopyFromPooledBuffer(pEntryTail, pMemoryRangeTail);
    }

    if( isHeadCacheNeeded )
    {
        // 読み込む領域の先頭をキャッシュします。
        funcCopyFromPooledBuffer(pEntryHead, pMemoryRangeHead);
    }

    // どちらかのキャッシュがもう一方のキャッシュに内包される場合は破棄します。
    if( pEntryHead->isCached && pEntryTail->isCached )
    {
        if( pEntryTail->offset <= pEntryHead->offset
            && pEntryHead->offset + pEntryHead->size <= pEntryTail->offset + pEntryTail->size )
        {
            pEntryHead->isCached = false;
        }
        else if( pEntryHead->offset <= pEntryTail->offset
            && pEntryTail->offset + pEntryTail->size <= pEntryHead->offset + pEntryHead->size )
        {
            pEntryTail->isCached = false;
        }
    }

    // 末尾キャッシュを破棄またはストアします。
    isTailCacheDestroyed = true;

    if( pEntryTail->isCached )
    {
        NN_RESULT_DO(UpdateLastResult(StoreOrDestroyBuffer(*pMemoryRangeTail, pEntryTail)));
    }
    else
    {
        DestroyBuffer(pEntryTail, *pMemoryRangeTail);
    }

    // 先頭キャッシュを破棄またはストアします。
    isHeadCacheDestroyed = true;

    if( pEntryHead->isCached )
    {
        NN_RESULT_DO(UpdateLastResult(StoreOrDestroyBuffer(*pMemoryRangeHead, pEntryHead)));
    }
    else
    {
        DestroyBuffer(pEntryHead, *pMemoryRangeHead);
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

}}}
