﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fssystem/fs_StorageLayoutTypeSetter.h>

namespace nn { namespace fssystem {

StorageLayoutTypeSetStorage::~StorageLayoutTypeSetStorage() NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    m_pBaseStorage.reset();
}

Result StorageLayoutTypeSetStorage::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseStorage->Read(offset, buffer, size));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetStorage::Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseStorage->Write(offset, buffer, size));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetStorage::Flush() NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseStorage->Flush());
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetStorage::SetSize(int64_t size) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseStorage->SetSize(size));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseStorage->GetSize(outValue));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetStorage::OperateRange(
    void* outBuffer,
    size_t outBufferSize,
    nn::fs::OperationId operationId,
    int64_t offset,
    int64_t size,
    const void* inBuffer,
    size_t inBufferSize) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseStorage->OperateRange(outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize));
    NN_RESULT_SUCCESS;
}

StorageLayoutTypeSetFile::~StorageLayoutTypeSetFile() NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    m_pBaseFile = nullptr;
    m_pUniqueBaseFile.reset();
    m_pSharedBaseFile.reset();
}

Result StorageLayoutTypeSetFile::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFile->Read(outValue, offset, buffer, size, option));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFile::DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFile->Write(offset, buffer, size, option));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFile::DoFlush() NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFile->Flush());
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFile::DoSetSize(int64_t size) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFile->SetSize(size));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFile::DoGetSize(int64_t* outValue) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFile->GetSize(outValue));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFile::DoOperateRange(
    void* outBuffer,
    size_t outBufferSize,
    nn::fs::OperationId operationId,
    int64_t offset,
    int64_t size,
    const void* inBuffer,
    size_t inBufferSize) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFile->OperateRange(outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize));
    NN_RESULT_SUCCESS;
}

class StorageLayoutTypeSetDirectory : public nn::fs::fsa::IDirectory, public nn::fs::detail::Newable
{
public:
    StorageLayoutTypeSetDirectory(std::unique_ptr<IDirectory>&& pBaseDirectory, int32_t storageFlag) NN_NOEXCEPT
        : m_pBaseDirectory(std::move(pBaseDirectory))
        , m_StorageFlag(storageFlag)
    {
    }

    virtual ~StorageLayoutTypeSetDirectory() NN_NOEXCEPT NN_OVERRIDE
    {
        ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
        m_pBaseDirectory.reset();
    }

private:
    virtual Result DoRead(int64_t* outValue, nn::fs::DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT NN_OVERRIDE
    {
        ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
        NN_RESULT_DO(m_pBaseDirectory->Read(outValue, outEntries, count));
        NN_RESULT_SUCCESS;
    }

    virtual Result DoGetEntryCount(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
        NN_RESULT_DO(m_pBaseDirectory->GetEntryCount(outValue));
        NN_RESULT_SUCCESS;
    }

private:
    std::unique_ptr<IDirectory> m_pBaseDirectory;
    int32_t m_StorageFlag;
};

StorageLayoutTypeSetFileSystem::~StorageLayoutTypeSetFileSystem() NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    m_pBaseFileSystem.reset();
}

Result StorageLayoutTypeSetFileSystem::DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->CreateFile(path, size, option));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoDeleteFile(const char* path) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->DeleteFile(path));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoCreateDirectory(const char* path) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->CreateDirectory(path));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoDeleteDirectory(const char* path) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->DeleteDirectory(path));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->DeleteDirectoryRecursively(path));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->CleanDirectoryRecursively(path));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->RenameFile(currentPath, newPath));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->RenameDirectory(currentPath, newPath));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->GetEntryType(outValue, path));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->GetFreeSpaceSize(outValue, path));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->GetTotalSpaceSize(outValue, path));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);

    std::unique_ptr<nn::fs::fsa::IFile> baseFile;
    NN_RESULT_DO(m_pBaseFileSystem->OpenFile(&baseFile, path, mode));

    std::unique_ptr<nn::fs::fsa::IFile> file(new StorageLayoutTypeSetFile(std::move(baseFile), m_StorageFlag));
    NN_RESULT_THROW_UNLESS(file, nn::fs::ResultAllocationMemoryFailedNew());
    *outValue = std::move(file);

    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);

    std::unique_ptr<nn::fs::fsa::IDirectory> baseDirectory;
    NN_RESULT_DO(m_pBaseFileSystem->OpenDirectory(&baseDirectory, path, mode));

    std::unique_ptr<nn::fs::fsa::IDirectory> directory(new StorageLayoutTypeSetDirectory(std::move(baseDirectory), m_StorageFlag));
    NN_RESULT_THROW_UNLESS(directory, nn::fs::ResultAllocationMemoryFailedNew());
    *outValue = std::move(directory);

    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoCommit() NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->Commit());
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoCommitProvisionally(int64_t counter) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->CommitProvisionally(counter));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoRollback() NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->Rollback());
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoFlush() NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->Flush());
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->GetFileTimeStampRaw(outTimeStamp, path));
    NN_RESULT_SUCCESS;
}

Result StorageLayoutTypeSetFileSystem::DoQueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, nn::fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT
{
    ScopedStorageLayoutTypeSetter scopedContext(m_StorageFlag);
    NN_RESULT_DO(m_pBaseFileSystem->QueryEntry(outBuffer, outBufferSize, inBuffer, inBufferSize, queryId, path));
    NN_RESULT_SUCCESS;
}


}}
