﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>

#include "fs_SaveDataFileSystemResultHandlingUtility.h"

namespace nn { namespace fssystem { namespace detail {

namespace {

    nn::Result ConvertIntegrityVerificationStorageCorruptedResult(nn::Result result) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(nn::fs::ResultIntegrityVerificationStorageCorrupted::Includes(result));

        if( nn::fs::ResultIncorrectIntegrityVerificationMagicCode::Includes(result) )
        {
            return nn::fs::ResultIncorrectSaveDataIntegrityVerificationMagicCode();
        }
        else if( nn::fs::ResultInvalidZeroHash::Includes(result) )
        {
            return nn::fs::ResultInvalidSaveDataZeroHash();
        }
        else if( nn::fs::ResultNonRealDataVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultSaveDataNonRealDataVerificationFailed();
        }
        else if( nn::fs::ResultInvalidHierarchicalIntegrityVerificationLayerCount::Includes(result) )
        {
            return nn::fs::ResultInvalidSaveDataHierarchicalIntegrityVerificationLayerCount();
        }
        else if( nn::fs::ResultClearedRealDataVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultClearedSaveDataRealDataVerificationFailed();
        }
        else if( nn::fs::ResultUnclearedRealDataVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultUnclearedSaveDataRealDataVerificationFailed();
        }
        else if( nn::fs::ResultIntegrityVerificationStorageCorrupted::Includes(result) )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultSaveDataIntegrityVerificationStorageCorrupted();
        }
        else
        {
            NN_SDK_ASSERT(false);
            return result;
        }
    }

    nn::Result ConvertBuiltInStorageCorruptedResult(nn::Result result) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(nn::fs::ResultBuiltInStorageCorrupted::Includes(result));

        if( nn::fs::ResultGptHeaderSignatureVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultSaveDataGptHeaderSignatureVerificationFailed();
        }
        else if( nn::fs::ResultBuiltInStorageCorrupted::Includes(result) )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultSaveDataBuiltInStorageCorrupted();
        }
        else
        {
            NN_SDK_ASSERT(false);
            return result;
        }
    }

    nn::Result ConvertHostFileSystemCorruptedResult(nn::Result result) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(nn::fs::ResultHostFileSystemCorrupted::Includes(result));

        if( nn::fs::ResultHostEntryCorrupted::Includes(result) )
        {
            return nn::fs::ResultSaveDataHostEntryCorrupted();
        }
        else if( nn::fs::ResultHostFileDataCorrupted::Includes(result) )
        {
            return nn::fs::ResultSaveDataHostFileDataCorrupted();
        }
        else if( nn::fs::ResultHostFileCorrupted::Includes(result) )
        {
            return nn::fs::ResultSaveDataHostFileCorrupted();
        }
        else if( nn::fs::ResultInvalidHostHandle::Includes(result) )
        {
            return nn::fs::ResultInvalidSaveDataHostHandle();
        }
        else if( nn::fs::ResultHostFileSystemCorrupted::Includes(result) )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultSaveDataHostFileSystemCorrupted();
        }
        else
        {
            NN_SDK_ASSERT(false);
            return result;
        }
    }

    nn::Result ConvertDatabaseCorruptedResult(nn::Result result) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(nn::fs::ResultDatabaseCorrupted::Includes(result));

        if( nn::fs::ResultInvalidAllocationTableBlock::Includes(result) )
        {
            return nn::fs::ResultInvalidSaveDataAllocationTableBlock();
        }
        else if( nn::fs::ResultInvalidKeyValueListElementIndex::Includes(result) )
        {
            return nn::fs::ResultInvalidSaveDataKeyValueListElementIndex();
        }
        else if( nn::fs::ResultDatabaseCorrupted::Includes(result) )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultSaveDataDatabaseCorrupted();
        }
        else
        {
            NN_SDK_ASSERT(false);
            return result;
        }
    }

}

nn::Result ConvertSaveFsDriverPrivateResult(nn::Result result) NN_NOEXCEPT
{
    // through success
    if( result.IsSuccess() )
    {
        NN_RESULT_SUCCESS;
    }

    if( nn::fs::ResultUnsupportedVersion::Includes(result) )
    {
        return nn::fs::ResultUnsupportedSaveDataVersion();
    }
    else if( nn::fs::ResultIntegrityVerificationStorageCorrupted::Includes(result) )
    {
        return ConvertIntegrityVerificationStorageCorruptedResult(result);
    }
    else if( nn::fs::ResultBuiltInStorageCorrupted::Includes(result) )
    {
        return ConvertBuiltInStorageCorruptedResult(result);
    }
    else if( nn::fs::ResultFatFileSystemCorrupted::Includes(result) )
    {
        // FAT のエラーはどのパーティションで問題があったかが重要なのでリザルトを振り分けずにそのまま返す
        return result;
    }
    else if( nn::fs::ResultHostFileSystemCorrupted::Includes(result) )
    {
        return ConvertHostFileSystemCorruptedResult(result);
    }
    else if( nn::fs::ResultDatabaseCorrupted::Includes(result) )
    {
        return ConvertDatabaseCorruptedResult(result);
    }
    else if( nn::fs::ResultNotFound::Includes(result) )
    {
        return nn::fs::ResultPathNotFound();
    }
    else if( nn::fs::ResultAllocationTableFull::Includes(result) )
    {
        return nn::fs::ResultUsableSpaceNotEnough();
    }
    else if( nn::fs::ResultAlreadyExists::Includes(result) )
    {
        return nn::fs::ResultPathAlreadyExists();
    }
    else if( nn::fs::ResultInvalidOffset::Includes(result) )
    {
        return nn::fs::ResultOutOfRange();
    }
    else if( nn::fs::ResultPermissionDenied::Includes(result) )
    {
        return nn::fs::ResultTargetLocked();
    }
    else if( nn::fs::ResultFileNotFound::Includes(result)
                || nn::fs::ResultIncompatiblePath::Includes(result) )
    {
        return nn::fs::ResultPathNotFound();
    }

    return result;
}

}}}
