﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/detail/fs_Newable.h>

#include <nn/fssystem/fs_DbmRomTypes.h>
#include <nn/fssystem/fs_DbmHierarchicalRomFileTableTemplate.h>
#include <nn/fssystem/fs_DbmHierarchicalRomFileTableTemplate.impl.h>
#include <nn/fssystem/fs_RomFsFileSystem.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/buffers/fs_BufferManagerUtility.h>

using namespace nn::fs;

namespace nn { namespace fssystem {


namespace {

    size_t CalculateRequiredWorkingMemorySize(const RomFileSystemInformation* pHeader) NN_NOEXCEPT
    {
        return static_cast<size_t>(pHeader->sizeBucketDirectory + pHeader->sizeDirectoryEntry + pHeader->sizeBucketFile + pHeader->sizeFileEntry);
    }

    class RomFsFile : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
    {
    private:

        RomFsFileSystem* m_pParent;
        int64_t m_Head;
        int64_t m_Tail;

        int64_t GetSize() const NN_NOEXCEPT
        {
            return m_Tail - m_Head;
        }

    public:

        RomFsFile(RomFsFileSystem* pParent, int64_t head, int64_t tail) NN_NOEXCEPT
            : m_pParent(pParent)
            , m_Head(head)
            , m_Tail(tail)
        {
        }

        virtual ~RomFsFile() NN_NOEXCEPT
        {
        }

        virtual Result DoRead(size_t* outValue, int64_t offset, void *buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
                [=]() NN_NOEXCEPT -> Result
                {
                    size_t readSize = 0;
                    NN_RESULT_DO(DryRead(&readSize, offset, size, option, nn::fs::OpenMode_Read));

                    NN_RESULT_DO(m_pParent->GetBaseFile()->Read(offset + m_Head, buffer, readSize));
                    *outValue = readSize;

                    NN_RESULT_SUCCESS;
                },
                NN_CURRENT_FUNCTION_NAME
            ));
            NN_RESULT_SUCCESS;
        }

        virtual Result DoGetSize(int64_t *outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = GetSize();
            NN_RESULT_SUCCESS;
        }

        virtual Result DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            switch( operationId )
            {
            case OperationId::Invalidate:
            case OperationId::QueryRange:
                {
                    NN_RESULT_THROW_UNLESS(offset >= 0, ResultOutOfRange());
                    NN_RESULT_THROW_UNLESS(GetSize() >= offset, ResultOutOfRange());

                    auto operateSize = size;
                    if( offset + operateSize > GetSize() || offset + operateSize < offset )
                    {
                        operateSize = GetSize() - offset;
                    }

                    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
                        [=]() NN_NOEXCEPT -> Result
                        {
                            NN_RESULT_DO(
                                m_pParent->GetBaseFile()->OperateRange(
                                    outBuffer,
                                    outBufferSize,
                                    operationId,
                                    m_Head + offset,
                                    operateSize,
                                    inBuffer,
                                    inBufferSize
                                )
                            );
                            NN_RESULT_SUCCESS;
                        },
                        NN_CURRENT_FUNCTION_NAME
                    ));

                    NN_RESULT_SUCCESS;
                }
            default:
                NN_RESULT_THROW(nn::fs::ResultUnsupportedOperation());
            }
        }

        virtual Result DoWrite(int64_t offset, const void *buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(offset);
            NN_UNUSED(buffer);
            NN_UNUSED(size);
            NN_UNUSED(option);
            NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "Write is not supported.");
        }
        virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }
        virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(size);
            NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "SetSize is not supported.");
        }

    };

    class RomFsDirectory : public nn::fs::fsa::IDirectory, public nn::fs::detail::Newable
    {

    private:
        typedef RomFsFileSystem::RomFileTable::FindPosition FindPosition;

        RomFsFileSystem* m_pParent;
        FindPosition m_FindPosition;
        FindPosition m_FirstFindPosition;
        nn::fs::OpenDirectoryMode m_Mode;


        Result ReadInternal(int64_t *outValue, FindPosition* pFindPosition, nn::fs::DirectoryEntry *entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT
        {
            static const size_t NameLength = nn::fs::EntryNameLengthMax + 1;
            RomPathChar name[nn::fs::EntryNameLengthMax + 1];
            int32_t i = 0;

            if ((m_Mode & nn::fs::OpenDirectoryMode_Directory) != 0)
            {
                for (; i < entryBufferCount || entryBuffer == nullptr ; ++i)
                {
                    NN_RESULT_TRY(m_pParent->GetRomFileTable()->FindNextDirectory(name, pFindPosition))
                        NN_RESULT_CATCH(ResultDbmFindFinished)
                        {
                            break;
                        }
                    NN_RESULT_END_TRY

                    if (entryBuffer)
                    {
                        NN_SDK_ASSERT( i < entryBufferCount);
                        NN_RESULT_THROW_UNLESS(
                            strnlen(name, NameLength) < NameLength,
                            nn::fs::ResultTooLongPath());
                        strncpy(entryBuffer[i].name, name, nn::fs::EntryNameLengthMax);
                        entryBuffer[i].name[nn::fs::EntryNameLengthMax] = '\0';
                        entryBuffer[i].directoryEntryType = nn::fs::DirectoryEntryType_Directory;
                        entryBuffer[i].fileSize = 0;
                    }
                }
            }

            if ((m_Mode & nn::fs::OpenDirectoryMode_File) != 0)
            {
                for (; i < entryBufferCount || entryBuffer == nullptr ; ++i)
                {
                    RomFsFileSystem::RomFileTable::StoragePosition filePosition = pFindPosition->nextPositionFile;

                    NN_RESULT_TRY(m_pParent->GetRomFileTable()->FindNextFile(name, pFindPosition))
                        NN_RESULT_CATCH(ResultDbmFindFinished)
                        {
                            break;
                        }
                    NN_RESULT_END_TRY

                    if (entryBuffer)
                    {
                        NN_SDK_ASSERT( i < entryBufferCount);
                        NN_RESULT_THROW_UNLESS(
                            strnlen(name, NameLength) < NameLength,
                            nn::fs::ResultTooLongPath());
                        strncpy(entryBuffer[i].name, name, nn::fs::EntryNameLengthMax);
                        entryBuffer[i].name[nn::fs::EntryNameLengthMax] = '\0';
                        RomFsFileSystem::RomFileTable::FileInfo fi;
                        NN_RESULT_DO(m_pParent->GetRomFileTable()->OpenFile(&fi, m_pParent->GetRomFileTable()->PositionToFileId(filePosition)));
                        entryBuffer[i].directoryEntryType = nn::fs::DirectoryEntryType_File;
                        entryBuffer[i].fileSize = fi.size.Get();
                    }
                }
            }

            *outValue = i;
            NN_RESULT_SUCCESS;
        }


    public:

        RomFsDirectory(RomFsFileSystem* pParent, const RomFsFileSystem::RomFileTable::FindPosition& fp, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
            : m_pParent(pParent)
            , m_FindPosition(fp)
            , m_FirstFindPosition(fp)
            , m_Mode(mode)
        {
        }

        Result DoRead(int64_t *outValue, nn::fs::DirectoryEntry *entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT
        {
            NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
                [=]() NN_NOEXCEPT -> Result
                {
                    NN_RESULT_DO(ReadInternal(outValue, &m_FindPosition, entryBuffer, entryBufferCount));
                    NN_RESULT_SUCCESS;
                },
                NN_CURRENT_FUNCTION_NAME
            ));
            NN_RESULT_SUCCESS;
        }

        Result DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT
        {
            // FIXME: 毎回カウントする必要はない
            FindPosition findPosition = m_FirstFindPosition;

            NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
                [&]() NN_NOEXCEPT -> Result
                {
                    NN_RESULT_DO(ReadInternal(outValue, &findPosition, nullptr, 0));
                    NN_RESULT_SUCCESS;
                },
                NN_CURRENT_FUNCTION_NAME
            ));

            NN_RESULT_SUCCESS;
        }

    };

} // namespace

    RomFsFileSystem::RomFsFileSystem() NN_NOEXCEPT
        : m_pBaseFile(nullptr)
    {
    }

    RomFsFileSystem::~RomFsFileSystem() NN_NOEXCEPT
    {
    }

    fs::IStorage* RomFsFileSystem::GetBaseFile() NN_NOEXCEPT
    {
        return m_pBaseFile;
    }
    RomFsFileSystem::RomFileTable* RomFsFileSystem::GetRomFileTable() NN_NOEXCEPT
    {
        return &m_RomFileTable;
    }

    Result RomFsFileSystem::GetRequiredWorkingMemorySize(size_t* outValue, fs::IStorage* pFile) NN_NOEXCEPT
    {
        RomFileSystemInformation header;
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [&]() NN_NOEXCEPT -> Result
            {
                NN_RESULT_DO(pFile->Read(0, &header, sizeof(RomFileSystemInformation)));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));

        *outValue = CalculateRequiredWorkingMemorySize(&header);
        NN_RESULT_SUCCESS;
    }

    Result RomFsFileSystem::Initialize(std::shared_ptr<fs::IStorage> pBaseFile, void* workingMemory, size_t workingMemorySize, bool useCache) NN_NOEXCEPT
    {
        m_pSharedBaseFile = std::move(pBaseFile);
        return Initialize(m_pSharedBaseFile.get(), workingMemory, workingMemorySize, useCache);
    }

    Result RomFsFileSystem::Initialize(fs::IStorage* pBaseFile, void* workingMemory, size_t workingMemorySize, bool useCache) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pBaseFile);
        NN_SDK_ASSERT(!useCache || workingMemory != nullptr);

        buffers::ScopedBufferManagerContextRegistration scopedRegistration;
        buffers::EnableBlockingBufferManagerAllocation();

        Bit8* buf = static_cast<Bit8*>(workingMemory);

        RomFileSystemInformation header;

        NN_RESULT_DO(pBaseFile->Read(0, &header, sizeof(RomFileSystemInformation)));

        // ファイルシステムのヘッダ情報をキャッシュします
        if (useCache)
        {
            // 必要なワーキングメモリサイズのチェック
            size_t requiredWorkingMemorySize = CalculateRequiredWorkingMemorySize(&header);
            if (requiredWorkingMemorySize > workingMemorySize)
            {
                NN_RESULT_THROW(nn::fs::ResultAllocationMemoryFailedInRomFsFileSystemA());
            }

            auto pBufDirectoryBucket = buf;
            NN_RESULT_DO(pBaseFile->Read(header.offsetBucketDirectory, pBufDirectoryBucket, static_cast<size_t>(header.sizeBucketDirectory)));

            buf += header.sizeBucketDirectory;
            auto pBufDirectoryEntry = buf;
            NN_RESULT_DO(pBaseFile->Read(header.offsetDirectoryEntry, pBufDirectoryEntry, static_cast<size_t>(header.sizeDirectoryEntry)));

            buf += header.sizeDirectoryEntry;
            auto pBufFileBucket = buf;
            NN_RESULT_DO(pBaseFile->Read(header.offsetBucketFile, pBufFileBucket, static_cast<size_t>(header.sizeBucketFile)));

            buf += header.sizeBucketFile;
            auto pBufFileEntry = buf;
            NN_RESULT_DO(pBaseFile->Read(header.offsetFileEntry, pBufFileEntry, static_cast<size_t>(header.sizeFileEntry)));

            m_pStorageDirectoryBucket.reset(new MemoryStorage(pBufDirectoryBucket, header.sizeBucketDirectory));
            m_pStorageDirectoryEntry.reset( new MemoryStorage(pBufDirectoryEntry,  header.sizeDirectoryEntry));
            m_pStorageFileBucket.reset(     new MemoryStorage(pBufFileBucket,      header.sizeBucketFile));
            m_pStorageFileEntry.reset(      new MemoryStorage(pBufFileEntry,       header.sizeFileEntry));
        }
        else
        {
            m_pStorageDirectoryBucket.reset(new SubStorage(pBaseFile, header.offsetBucketDirectory, header.sizeBucketDirectory));
            m_pStorageDirectoryEntry.reset( new SubStorage(pBaseFile, header.offsetDirectoryEntry, header.sizeDirectoryEntry));
            m_pStorageFileBucket.reset(     new SubStorage(pBaseFile, header.offsetBucketFile, header.sizeBucketFile));
            m_pStorageFileEntry.reset(      new SubStorage(pBaseFile, header.offsetFileEntry, header.sizeFileEntry));
        }

        if( m_pStorageDirectoryBucket == nullptr ||
            m_pStorageDirectoryEntry  == nullptr ||
            m_pStorageFileBucket      == nullptr ||
            m_pStorageFileEntry       == nullptr )
        {
            NN_RESULT_THROW(nn::fs::ResultAllocationMemoryFailedInRomFsFileSystemB());
        }

        // ファイルテーブルをマウントします。
        NN_RESULT_DO(m_RomFileTable.Initialize(
                     m_pStorageDirectoryBucket.get(),
                     0,
                     static_cast<uint32_t>(header.sizeBucketDirectory),
                     m_pStorageDirectoryEntry.get(),
                     0,
                     static_cast<uint32_t>(header.sizeDirectoryEntry),
                     m_pStorageFileBucket.get(),
                     0,
                     static_cast<uint32_t>(header.sizeBucketFile),
                     m_pStorageFileEntry.get(),
                     0,
                     static_cast<uint32_t>(header.sizeFileEntry)
            ));

        m_EntrySize = header.offsetFileBody;
        m_pBaseFile = pBaseFile;

        NN_RESULT_SUCCESS;
    }

    // TODO: パッチ対応で断片化するようになったらリスト出力に変更
    Result RomFsFileSystem::GetFileBaseOffset(int64_t* outValue, const char* path) NN_NOEXCEPT
    {
        RomFsFileSystem::RomFileTable::FileInfo fi;
        NN_RESULT_DO(GetFileInfo(&fi, path));
        *outValue = m_EntrySize + fi.offset.Get();
        NN_RESULT_SUCCESS;
    }

    Result RomFsFileSystem::DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(mode == nn::fs::OpenMode_Read, ResultInvalidArgument());
        NN_UNUSED(mode);

        RomFsFileSystem::RomFileTable::FileInfo fi;
        NN_RESULT_DO(GetFileInfo(&fi, path));

        std::unique_ptr<RomFsFile> file(new RomFsFile(this, m_EntrySize + fi.offset.Get(), m_EntrySize + fi.offset.Get() + fi.size.Get() ));
        if (file == nullptr)
        {
            return nn::fs::ResultAllocationMemoryFailedInRomFsFileSystemC();
        }

        *outValue = std::move(file);
        NN_RESULT_SUCCESS;
    }

    Result RomFsFileSystem::DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
    {
        RomFsFileSystem::RomFileTable::FindPosition fp;
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [&]() NN_NOEXCEPT -> Result
            {
                NN_RESULT_TRY(m_RomFileTable.FindOpen(&fp, path))
                    NN_RESULT_CATCH(ResultDbmNotFound)
                    {
                        NN_RESULT_THROW(nn::fs::ResultPathNotFound());
                    }
                NN_RESULT_END_TRY;
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));

        std::unique_ptr<RomFsDirectory> directory(new RomFsDirectory(this, fp, mode));
        if (directory == nullptr)
        {
            NN_RESULT_THROW(nn::fs::ResultAllocationMemoryFailedInRomFsFileSystemD());
        }

        *outValue = std::move(directory);
        NN_RESULT_SUCCESS;
    }

    Result RomFsFileSystem::DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char *path) NN_NOEXCEPT
    {
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                RomDirectoryInfo directoryInfo;

                NN_RESULT_TRY(m_RomFileTable.GetDirectoryInformation(&directoryInfo, path))
                    NN_RESULT_CATCH(ResultDbmNotFound) // 親ディレクトリが見つからない場合
                    {
                        NN_RESULT_THROW(nn::fs::ResultPathNotFound());
                    }
                    NN_RESULT_CATCH(ResultDbmInvalidOperation) // 対象がファイルである場合（を含む）
                    {
                        RomFsFileSystem::RomFileTable::FileInfo fi;
                        NN_RESULT_DO(GetFileInfo(&fi, path));

                        *outValue = nn::fs::DirectoryEntryType_File;
                        NN_RESULT_SUCCESS;
                    }
                NN_RESULT_END_TRY;

                *outValue = nn::fs::DirectoryEntryType_Directory;

                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }


    Result RomFsFileSystem::DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_UNUSED(size);
        NN_UNUSED(option);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "CreateFile is not supported.");
    }
    Result RomFsFileSystem::DoDeleteFile(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "DeleteFile is not supported.");
    }
    Result RomFsFileSystem::DoCreateDirectory(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "CreateDirectory is not supported.");
    }
    Result RomFsFileSystem::DoDeleteDirectory(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "DeleteDirectory is not supported.");
    }
    Result RomFsFileSystem::DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "DeleteDirectoryRecursively is not supported.");
    }
    Result RomFsFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "CleanDirectory is not supported.");
    }
    Result RomFsFileSystem::DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
    {
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "RenameFile is not supported.");
    }
    Result RomFsFileSystem::DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT
    {
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "RenameDirectory is not supported.");
    }
    Result RomFsFileSystem::DoCommit() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    Result RomFsFileSystem::GetFileInfo(RomFileTable::FileInfo* pOutValue, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                NN_RESULT_TRY(m_RomFileTable.OpenFile(pOutValue, path))
                    NN_RESULT_CATCH(ResultDbmNotFound)
                    {
                        NN_RESULT_THROW(nn::fs::ResultPathNotFound());
                    }
                NN_RESULT_END_TRY;

                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

}}
