﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/fs_PartitionFileSystemMeta.h>

namespace nn { namespace fssystem {

    using namespace nn::fs;

namespace {
    const int VersionSignatureSize = 4;
}

    const char detail::PartitionFileSystemFormat::VersionSignature[VersionSignatureSize]       = {'P', 'F', 'S', '0'};
    const char detail::Sha256PartitionFileSystemFormat::VersionSignature[VersionSignatureSize] = {'H', 'F', 'S', '0'};

    template <typename Format>
    struct PartitionFileSystemMetaCore<Format>::PartitionFileSystemHeader
    {
        char formatVersionSignature[VersionSignatureSize];
        uint32_t partitionCount;
        uint32_t nameTableSize;
        uint32_t reserved;
    };
    NN_STATIC_ASSERT(std::is_pod<PartitionFileSystemMetaCore<detail::PartitionFileSystemFormat>::PartitionFileSystemHeader>::value);
    NN_STATIC_ASSERT(std::is_pod<PartitionFileSystemMetaCore<detail::Sha256PartitionFileSystemFormat>::PartitionFileSystemHeader>::value);

namespace {
    template <typename Format>
    size_t CalcMetaDataRoundUpSize(size_t entryTableSize, size_t nameTableSize, int fileDataAlignmentSize)
    {
        size_t metaDataSize = sizeof(typename PartitionFileSystemMetaCore<Format>::PartitionFileSystemHeader) + entryTableSize + nameTableSize;
        size_t metaDataSizeRoundUp = (metaDataSize + (fileDataAlignmentSize - 1)) & ~(fileDataAlignmentSize - 1);
        return metaDataSizeRoundUp - metaDataSize;
    }


    void SetEntry(detail::Sha256PartitionFileSystemFormat::PartitionEntry* pDst, const detail::Sha256PartitionFileSystemFormat::FileEntryForConstruct* pSrc, uint32_t nameOffset)
    {
        pDst->offset           = pSrc->offset;
        pDst->size             = pSrc->size;
        pDst->hashTargetSize   = pSrc->hashTargetSize;
        pDst->hashTargetOffset = pSrc->hashTargetOffset;
        pDst->nameOffset       = nameOffset;
        std::memcpy(pDst->hash, pSrc->hash, detail::Sha256PartitionFileSystemFormat::HashSize);
    }

    void SetEntry(detail::PartitionFileSystemFormat::PartitionEntry* pDst, const detail::PartitionFileSystemFormat::FileEntryForConstruct* pSrc, uint32_t nameOffset)
    {
        pDst->offset     = pSrc->offset;
        pDst->size       = pSrc->size;
        pDst->nameOffset = nameOffset;
    }

};

    template <typename Format>
    PartitionFileSystemMetaCore<Format>::PartitionFileSystemMetaCore()
        : m_IsInitialized(false)
        , m_pAllocator(nullptr)
        , m_pBufferToBeFreed(nullptr)
    {
    }

    template <typename Format>
    PartitionFileSystemMetaCore<Format>::~PartitionFileSystemMetaCore()
    {
        DeallocateBuffer();
    }

    template <typename Format>
    void PartitionFileSystemMetaCore<Format>::DeallocateBuffer() NN_NOEXCEPT
    {
        if( m_pBufferToBeFreed )
        {
            NN_SDK_ASSERT(m_pAllocator != nullptr);
            m_pAllocator->deallocate(m_pBufferToBeFreed, m_MetaDataSize);
            m_pBufferToBeFreed = nullptr;
        }
    }

    template <typename Format>
    Result PartitionFileSystemMetaCore<Format>::Initialize(fs::IStorage* pBaseStorage, void* metaDataBuffer, size_t metaDataBufferSize) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(metaDataBufferSize >= sizeof(PartitionFileSystemHeader), ResultInvalidSize());

        NN_RESULT_DO(pBaseStorage->Read(0, metaDataBuffer, sizeof(PartitionFileSystemHeader)));

        m_pHeader    = reinterpret_cast<PartitionFileSystemHeader*>(metaDataBuffer);
        NN_RESULT_THROW_UNLESS( nn::crypto::IsSameBytes(m_pHeader->formatVersionSignature, Format::VersionSignature, VersionSignatureSize), typename Format::ResultPartitionSignatureVerificationFailed() );

        int entryTableSize = sizeof(PartitionEntry) * m_pHeader->partitionCount;
        m_pEntries   = reinterpret_cast<PartitionEntry*>(static_cast<char*>(metaDataBuffer) + sizeof(PartitionFileSystemHeader));
        m_pNameTable = static_cast<char*>(metaDataBuffer) + sizeof(PartitionFileSystemHeader) + entryTableSize;

        NN_RESULT_THROW_UNLESS(metaDataBufferSize >= sizeof(PartitionFileSystemHeader) + entryTableSize + m_pHeader->nameTableSize, ResultInvalidSize());
        NN_RESULT_DO(pBaseStorage->Read(sizeof(PartitionFileSystemHeader), m_pEntries, entryTableSize + m_pHeader->nameTableSize));

        m_IsInitialized = true;
        NN_RESULT_SUCCESS;
    }

    template <typename Format>
    Result PartitionFileSystemMetaCore<Format>::Initialize(fs::IStorage* pBaseStorage, MemoryResource* pAllocator) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pAllocator != nullptr);
        NN_RESULT_DO(QueryMetaDataSize(&m_MetaDataSize, pBaseStorage));

        DeallocateBuffer();
        m_pAllocator = pAllocator;
        m_pBufferToBeFreed = static_cast<char*>(m_pAllocator->allocate(m_MetaDataSize));
        NN_RESULT_THROW_UNLESS( m_pBufferToBeFreed != nullptr, ResultAllocationMemoryFailedInPartitionFileSystemMetaA() );

        return Initialize(pBaseStorage, m_pBufferToBeFreed, m_MetaDataSize);
    }

    template <typename Format>
    Result PartitionFileSystemMetaCore<Format>::QueryMetaDataSize(size_t* outValue, fs::IStorage* pBaseStorage) NN_NOEXCEPT
    {
        PartitionFileSystemHeader header;
        NN_RESULT_DO(pBaseStorage->Read(0, &header, sizeof(PartitionFileSystemHeader)));

        NN_RESULT_THROW_UNLESS( nn::crypto::IsSameBytes(header.formatVersionSignature, Format::VersionSignature, VersionSignatureSize), typename Format::ResultPartitionSignatureVerificationFailed() );

        *outValue = sizeof(PartitionFileSystemHeader)
            + sizeof(PartitionEntry) * header.partitionCount
            + header.nameTableSize;

        NN_RESULT_SUCCESS;
    }

    template <typename Format>
    Result PartitionFileSystemMetaCore<Format>::QueryMetaDataSize(size_t* outValue, const FileEntryForConstruct* pEntryArray, int entryCount) NN_NOEXCEPT
    {
        size_t entryTableSize = sizeof(PartitionEntry) * entryCount;
        size_t nameTableSize = 0;

        for(int i = 0; i < entryCount; ++i)
        {
            size_t nameLength = strnlen(pEntryArray[i].name, sizeof(pEntryArray[i].name)) + 1; // NULL を含む
            NN_FSP_REQUIRES(nameLength <= EntryNameLengthMax + 1, ResultTooLongPath());
            nameTableSize += nameLength;
        }

        size_t metaDataRoundUpSize = CalcMetaDataRoundUpSize<Format>(entryTableSize, nameTableSize, FileDataAlignmentSize);
        nameTableSize += metaDataRoundUpSize;

        *outValue = sizeof(PartitionFileSystemHeader) + entryTableSize + nameTableSize;

        NN_RESULT_SUCCESS;
    }

    template <typename Format>
    int PartitionFileSystemMetaCore<Format>::GetEntryIndex(const char* path) const NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(m_IsInitialized, ResultPreconditionViolation());
        for(int index = 0; index < static_cast<int>(m_pHeader->partitionCount); ++index)
        {
            NN_RESULT_THROW_UNLESS(m_pEntries[index].nameOffset < m_pHeader->nameTableSize, ResultInvalidPartitionEntryOffset());

            uint32_t maxNameLength = m_pHeader->nameTableSize - m_pEntries[index].nameOffset;
            if( strncmp(&m_pNameTable[m_pEntries[index].nameOffset], path, maxNameLength) == 0 )
            {
                return index;
            }
        }

        return -1;
    }

    template <typename Format>
    const typename PartitionFileSystemMetaCore<Format>::PartitionEntry* PartitionFileSystemMetaCore<Format>::GetEntry(int index) const NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(m_IsInitialized, ResultPreconditionViolation());
        NN_FSP_REQUIRES(0 <= index && index < static_cast<int>(m_pHeader->partitionCount), ResultPreconditionViolation());
        return &m_pEntries[index];
    }

    template <typename Format>
    int32_t PartitionFileSystemMetaCore<Format>::GetEntryCount() const NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(m_IsInitialized, ResultPreconditionViolation());
        return m_pHeader->partitionCount;
    }

    template <typename Format>
    const char* PartitionFileSystemMetaCore<Format>::GetEntryName(int index) const NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(m_IsInitialized, ResultPreconditionViolation());
        NN_FSP_REQUIRES(index < static_cast<int>(m_pHeader->partitionCount), ResultPreconditionViolation());
        return &m_pNameTable[GetEntry(index)->nameOffset];
    }

    template <typename Format>
    size_t PartitionFileSystemMetaCore<Format>::GetMetaDataSize() const NN_NOEXCEPT
    {
        return m_MetaDataSize;
    }

    template <typename Format>
    Result PartitionFileSystemMetaCore<Format>::ConstructMetaData(void* metaDataBuffer, size_t metaDataBufferSize, const FileEntryForConstruct* pEntryArray, int entryCount) NN_NOEXCEPT
    {
        int entryTableSize = sizeof(PartitionEntry) * entryCount;

        size_t metaDataSize;
        NN_RESULT_DO(QueryMetaDataSize(&metaDataSize, pEntryArray, entryCount));
        NN_FSP_REQUIRES(metaDataBufferSize >= metaDataSize, ResultInvalidSize());

        auto pHeader = reinterpret_cast<PartitionFileSystemHeader*>(metaDataBuffer);
        auto pEntries = reinterpret_cast<PartitionEntry*>(static_cast<char*>(metaDataBuffer) + sizeof(PartitionFileSystemHeader));
        auto pNameTable = static_cast<char*>(metaDataBuffer) + sizeof(PartitionFileSystemHeader) + entryTableSize;

        uint32_t nameTableSize = 0;
        int entryIndex = 0;

        // エントリテーブル、名前テーブルに追加
        for(int i = 0; i < entryCount; ++i)
        {
            auto nameLength = static_cast<uint32_t>(strnlen(pEntryArray[i].name, sizeof(pEntryArray[i].name)) + 1); // NULL を含む

            NN_FSP_REQUIRES(nameLength <= sizeof(pEntryArray[i].name), ResultTooLongPath());
            NN_FSP_REQUIRES(sizeof(PartitionFileSystemHeader) + entryTableSize + nameTableSize + nameLength <= metaDataBufferSize, ResultPreconditionViolation());

            SetEntry(&pEntries[entryIndex], &pEntryArray[i], nameTableSize);

            std::memcpy(pNameTable, pEntryArray[i].name, nameLength);

            pNameTable    += nameLength;
            nameTableSize += nameLength;
            ++entryIndex;
        }

        // パディングしアラインをとる
        size_t metaDataRoundUpSize = CalcMetaDataRoundUpSize<Format>(entryTableSize, nameTableSize, FileDataAlignmentSize);
        nameTableSize += static_cast<uint32_t>(metaDataRoundUpSize);
        memset(pNameTable, 0x00, metaDataRoundUpSize);

        // ヘッダ作成
        std::memcpy(pHeader->formatVersionSignature, Format::VersionSignature, VersionSignatureSize);
        pHeader->nameTableSize = nameTableSize;
        pHeader->partitionCount = entryCount;
        pHeader->reserved = 0x00000000;

        NN_RESULT_SUCCESS;
    }


    template class PartitionFileSystemMetaCore<detail::PartitionFileSystemFormat>;
    template class PartitionFileSystemMetaCore<detail::Sha256PartitionFileSystemFormat>;


    Result Sha256PartitionFileSystemMeta::Initialize(fs::IStorage* pBaseStorage, MemoryResource* pAllocator, const void* pHash, size_t hashSize) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(hashSize == crypto::Sha256Generator::HashSize, ResultPreconditionViolation());
        NN_UNUSED(hashSize);

        NN_RESULT_DO(QueryMetaDataSize(&m_MetaDataSize, pBaseStorage));

        DeallocateBuffer();
        m_pAllocator = pAllocator;
        m_pBufferToBeFreed = static_cast<char*>(m_pAllocator->allocate(m_MetaDataSize));
        NN_RESULT_THROW_UNLESS(m_pBufferToBeFreed != nullptr, nn::fs::ResultAllocationMemoryFailedInPartitionFileSystemMetaB());

        NN_RESULT_DO(pBaseStorage->Read(0, m_pBufferToBeFreed, m_MetaDataSize));

        char hash[crypto::Sha256Generator::HashSize];
        crypto::GenerateSha256Hash(hash, sizeof(hash), m_pBufferToBeFreed, m_MetaDataSize);
        NN_RESULT_THROW_UNLESS(nn::crypto::IsSameBytes(pHash, hash, sizeof(hash)), nn::fs::ResultSha256PartitionHashVerificationFailed());

        m_pHeader = reinterpret_cast<PartitionFileSystemHeader*>(m_pBufferToBeFreed);
        NN_RESULT_THROW_UNLESS(nn::crypto::IsSameBytes(m_pHeader->formatVersionSignature, detail::Sha256PartitionFileSystemFormat::VersionSignature, VersionSignatureSize), nn::fs::ResultSha256PartitionSignatureVerificationFailed());

        int entryTableSize = sizeof(PartitionEntry) * m_pHeader->partitionCount;
        NN_RESULT_THROW_UNLESS(m_MetaDataSize >= sizeof(PartitionFileSystemHeader) + entryTableSize + m_pHeader->nameTableSize, nn::fs::ResultInvalidSha256PartitionMetaDataSize());
        m_pEntries = reinterpret_cast<PartitionEntry*>(static_cast<char*>(m_pBufferToBeFreed) + sizeof(PartitionFileSystemHeader));
        m_pNameTable = static_cast<char*>(m_pBufferToBeFreed) + sizeof(PartitionFileSystemHeader) + entryTableSize;

        m_IsInitialized = true;
        NN_RESULT_SUCCESS;
    }

}}

