﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os.h>
#include <nn/crypto.h>
#include <nn/fs/fs_Result.h>
#include <nn/fssystem/fs_IntegrityRomFsStorage.h>

namespace nn { namespace fssystem {

/**
* @brief        コンストラクタ
*/
IntegrityRomFsStorage::IntegrityRomFsStorage() NN_NOEXCEPT
    : m_Locker(true)
{
}

/**
* @brief        デストラクタ
*/
IntegrityRomFsStorage::~IntegrityRomFsStorage() NN_NOEXCEPT
{
    Finalize();
}

/**
* @brief        階層署名付きRomFsをマウントします。
*
* @param[in]    hashStorage             メタデータ、階層ハッシュ
* @param[in]    bodyStorage             実データ
* @param[in]    pBufferManager          キャッシュ用バッファマネージャー
*
* @return      関数の処理結果を返します。
*/
Result IntegrityRomFsStorage::Initialize(
    fs::SubStorage hashStorage,
    fs::SubStorage bodyStorage,
    IBufferManager* pBufferManager
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBufferManager);

    save::HierarchicalIntegrityVerificationStorageControlArea controlArea;
    fs::SubStorage storageMeta(&hashStorage, 0, sizeof(save::HierarchicalIntegrityVerificationMetaInformation));
    NN_RESULT_DO(controlArea.Initialize(storageMeta));

    int64_t offset = sizeof(save::HierarchicalIntegrityVerificationMetaInformation);
    save::HierarchicalIntegrityVerificationStorage::HierarchicalStorageInformation storageInfo;

    fs::SubStorage storageMasterHash(&hashStorage, offset, controlArea.GetMasterHashSize());
    storageInfo.SetMasterHashStorage(storageMasterHash);
    offset += controlArea.GetMasterHashSize();

    save::HierarchicalIntegrityVerificationInformation infoLevelHash;
    controlArea.GetLevelHashInfo(&infoLevelHash);

    fs::SubStorage storageHash(&hashStorage, offset, infoLevelHash.GetLayeredHashSize());
    for( int32_t layer = 0; layer < static_cast<int32_t>(infoLevelHash.maxLayers - 2); ++layer )
    {
        fs::SubStorage storageHashLayer(&storageHash, infoLevelHash.info[layer].offset, infoLevelHash.info[layer].size);
        storageInfo[layer + 1] = storageHashLayer;
    }
    offset += infoLevelHash.GetDataOffset();

    fs::SubStorage storageData(&bodyStorage, 0, infoLevelHash.GetDataSize());
    storageInfo.SetDataStorage(storageData);

    for( size_t i = 0; i < sizeof(m_Buffers.pBuffer) / sizeof(m_Buffers.pBuffer[0]); ++i )
    {
        m_Buffers.pBuffer[i] = pBufferManager;
    }

    return m_IntegrityStorage.Initialize(infoLevelHash, storageInfo, &m_Buffers, &m_Locker, fs::StorageType::StorageType_Authoring);
}

/**
* @brief        階層署名付きRomFsをマウントします。
*
* @param[in]    infoLevelHash          メタデータ
* @param[in]    masterHash             マスターハッシュ
* @param[in]    storageInfo            階層ハッシュ、実データ
* @param[in]    pBufferManager         キャッシュ用バッファマネージャー
*
* @return       関数の処理結果を返します。
*/
Result IntegrityRomFsStorage::Initialize(
    save::HierarchicalIntegrityVerificationInformation infoLevelHash,
    Hash masterHash,
    save::HierarchicalIntegrityVerificationStorage::HierarchicalStorageInformation storageInfo,
    IBufferManager* pBufferManager
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBufferManager);

    // マスターハッシュ設定
    m_MasterHash = masterHash;
    m_pMasterHashStorage.reset(new fs::MemoryStorage(&m_MasterHash, sizeof(Hash)));
    NN_RESULT_THROW_UNLESS(m_pMasterHashStorage != nullptr, fs::ResultAllocationMemoryFailedInIntegrityRomFsStorageA());

    fs::SubStorage masterHashStorage(m_pMasterHashStorage.get(), 0, sizeof(Hash));
    storageInfo[0] = masterHashStorage;

    for( size_t i = 0; i < sizeof(m_Buffers.pBuffer) / sizeof(m_Buffers.pBuffer[0]); ++i )
    {
        m_Buffers.pBuffer[i] = pBufferManager;
    }

    return m_IntegrityStorage.Initialize(infoLevelHash, storageInfo, &m_Buffers, &m_Locker, fs::StorageType::StorageType_RomFs);
}


/**
* @brief        ストレージを閉じます。
*/
void IntegrityRomFsStorage::Finalize() NN_NOEXCEPT
{
    m_IntegrityStorage.Finalize();
}

}}

