﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_AsynchronousRunner.h>
#include <nn/fssystem/fs_ThreadPriorityChanger.h>

namespace nn { namespace fssystem {

ThreadPool::ThreadPool(PooledThread* pThreads, int threadCount) NN_NOEXCEPT
    : m_pThreads(pThreads)
    , m_ThreadCount(threadCount)
    , m_Semaphore(threadCount, threadCount)
    , m_StackBuffer(nullptr)
    , m_StackBufferSize(0)
    , m_StackDeallocator(nullptr)
    , m_pStackArgument(nullptr)
{
    NN_SDK_REQUIRES_NOT_NULL(pThreads);
    NN_SDK_REQUIRES_GREATER(threadCount, 0);
}

Result ThreadPool::Initialize(
                       char* stackBuffer,
                       size_t stackBufferSize,
                       StackDeallocator stackDeallocator,
                       void* pStackArgument
                   ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(stackBuffer);
    NN_SDK_REQUIRES_ALIGNED(stackBuffer, os::ThreadStackAlignment);
    NN_SDK_REQUIRES_GREATER(stackBufferSize, static_cast<size_t>(0));
    NN_SDK_REQUIRES_NOT_NULL(stackDeallocator);

    m_StackBuffer = stackBuffer;
    m_StackBufferSize = stackBufferSize;
    m_StackDeallocator = stackDeallocator;
    m_pStackArgument = pStackArgument;

    const auto stackSize = stackBufferSize / m_ThreadCount;
    NN_SDK_REQUIRES_ALIGNED(stackSize, os::ThreadStackAlignment);

    for( auto i = 0; i < m_ThreadCount; ++i )
    {
        NN_RESULT_DO(m_pThreads[i].Initialize(
            this,
            stackBuffer + stackSize * i,
            stackSize,
            PooledThread::ThreadFunction
        ));
    }

    NN_RESULT_SUCCESS;
}

void ThreadPool::Finalize() NN_NOEXCEPT
{
    for( auto i = 0; i < m_ThreadCount; ++i )
    {
        m_pThreads[i].Finalize();
    }

    if( m_StackBuffer != nullptr )
    {
        m_StackDeallocator(m_StackBuffer, m_StackBufferSize, m_pStackArgument);
        m_StackBuffer = nullptr;
    }
}

void ThreadPool::ActivateAsynchronousRunnersImpl(int64_t* outActiveRunnerCount, AsynchronousRunner** pRunnerList, int64_t runnerListCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outActiveRunnerCount);
    NN_SDK_REQUIRES_NOT_NULL(pRunnerList);
    NN_SDK_ASSERT_LESS(0, runnerListCount);

    *outActiveRunnerCount = 0;

    int acquireCount = 0;
    m_Semaphore.Acquire();
    do
    {
        for( auto i = 0; i < m_ThreadCount; ++i )
        {
            auto& thread = m_pThreads[i];
            if( thread.TryAcquire() )
            {
                thread.SetRunner(pRunnerList[acquireCount]);
                acquireCount++;
                break;
            }
        }
        if( acquireCount >= runnerListCount )
        {
            break;
        }
    } while( m_Semaphore.TryAcquire() );

    NN_SDK_ASSERT_LESS(0, acquireCount);
    *outActiveRunnerCount = acquireCount;
}

void PooledThread::Start() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pOwner);

    m_ThreadPriority = os::GetThreadPriority(os::GetCurrentThread());

    auto pContext = GetServiceContext();
    if( pContext != nullptr )
    {
        m_ServiceContext = *pContext;
    }

    m_StartEvent.Signal();
}

void PooledThread::Invoke() NN_NOEXCEPT
{
    RegisterServiceContext(&m_ServiceContext);

    for( ; ; )
    {
        m_StartEvent.Wait();

        if( m_pRunner == nullptr )
        {
            break;
        }
        NN_SDK_ASSERT_EQUAL(m_IsRunning, true);

        ScopedThreadPriorityChanger priority(
            m_ThreadPriority, ScopedThreadPriorityChanger::Mode::Absolute);

        m_pRunner->Invoke();

        m_EndEvent.Signal();
    }
}

void PooledThread::SetRunner(AsynchronousRunner* pRunner) NN_NOEXCEPT
{
    m_pRunner = pRunner;
    pRunner->SetThread(this);
}

}}
