﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_BitTypes.h>

#include <nn/fssystem/dbm/fs_DuplexBitmap.h>
#include <nn/fssystem/dbm/fs_DbmUtils.h>

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief        指定したブロック数(ビット数)の二重化ビットマップに必要なデータサイズを取得します。
*
* @param[in]    blockCount      ブロック数
*
* @return       指定したブロック数の二重化ビットマップに必要なデータサイズ。
*
* @details      指定したブロック数(ビット数)の二重化ビットマップに必要なデータサイズを取得します。
*               FormatCacheSize バイト単位で切り上げを行います。
*/
int64_t DuplexBitmap::QuerySize(uint32_t bitCount) NN_NOEXCEPT
{
    return nn::util::align_up<int64_t>(bitCount, FormatCacheSize * NN_BITSIZEOF(char)) / NN_BITSIZEOF(char);
}

/**
* @brief        メタデータストレージをフォーマットします。
*
* @param[in]    bitCount            総ビット数
* @param[in]    table               メタデータ用テーブル
* @param[in]    originalTable       オリジナルメタデータ用テーブル
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       正常にフォーマットできました。
* @retval       上記以外            ストレージへの書き込みに失敗しました。
*
* @details      メタデータストレージをフォーマットします。
*/
Result DuplexBitmap::Format(
           uint32_t bitCount,
           fs::SubStorage table,
           fs::SubStorage originalTable
       ) NN_NOEXCEPT
{
    uint8_t bits[FormatCacheSize] = {};
    const size_t length = sizeof(bits);

    const size_t size0 = static_cast<size_t>(QuerySize(bitCount));
    NN_SDK_ASSERT((size0 % length) == 0);

    {
        size_t size = size0;
        int64_t offsetBytes = 0;
        while( size > 0 )
        {
            NN_RESULT_DO(table.Write(offsetBytes, bits, length));
            size -= length;
            offsetBytes += length;
        }
    }

    {
        size_t size = size0;
        int64_t originalOffsetBytes = 0;
        while( size > 0 )
        {
            NN_RESULT_DO(originalTable.Write(originalOffsetBytes, bits, length));
            size -= length;
            originalOffsetBytes += length;
        }
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        メタデータストレージを拡張します。
*
* @param[in]    bitCountOld         拡張前の総ビット数
* @param[in]    bitCountNew         拡張後の総ビット数
* @param[in]    table               メタデータ用テーブル
* @param[in]    originalTable       オリジナルメタデータ用テーブル
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に拡張できました。
* @retval       上記以外        ストレージへの書き込みに失敗しました。
*
* @pre          bitCountNew >= bitCountOld
*
* @details      メタデータストレージを拡張します。
*/
Result DuplexBitmap::Expand(
           uint32_t bitCountOld,
           uint32_t bitCountNew,
           fs::SubStorage table,
           fs::SubStorage originalTable
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(bitCountOld, bitCountNew);

    const size_t oldSize = static_cast<size_t>(QuerySize(bitCountOld));
    const size_t newSize = static_cast<size_t>(QuerySize(bitCountNew));

    if( oldSize < newSize )
    {
        char bits[FormatCacheSize] = {};
        const size_t length = sizeof(bits);

        // 拡張して増えるサイズがキャッシュサイズの倍数になっていることを確認します。
        NN_SDK_ASSERT((newSize - oldSize) % length == 0);

        // メタデータ
        {
            size_t expandSize = newSize - oldSize;
            int64_t offsetBytes = oldSize;
            while( expandSize > 0 )
            {
                NN_RESULT_DO(table.Write(offsetBytes, bits, length));
                expandSize -= length;
                offsetBytes += length;
            }
        }

        // オリジナルメタデータ
        {
            size_t expandSize = newSize - oldSize;
            int64_t originalOffsetBytes = oldSize;
            while( expandSize > 0 )
            {
                NN_RESULT_DO(originalTable.Write(originalOffsetBytes, bits, length));
                expandSize -= length;
                originalOffsetBytes += length;
            }
        }
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        コンストラクタです。
*
* @details      コンストラクタです。
*/
DuplexBitmap::DuplexBitmap() NN_NOEXCEPT
    : m_BitCount(0),
      m_Table(),
      m_OriginalTable()
{
}

/**
* @brief        メタデータストレージをマウントします。
*
* @param[in]    blockCount          総ビット数
* @param[in]    table               メタデータ用ストレージ
* @param[in]    originalTable       オリジナルメタデータ用ストレージ
*
* @details      メタデータストレージをマウントします。
*/
void DuplexBitmap::Initialize(
         uint32_t bitCount,
         fs::SubStorage table,
         fs::SubStorage originalTable
     ) NN_NOEXCEPT
{
    m_BitCount = bitCount;
    m_Table = table;
    m_OriginalTable = originalTable;
}

/**
* @brief        メタデータストレージをアンマウントします。
*
* @details      メタデータストレージをアンマウントします。
*/
void DuplexBitmap::Finalize() NN_NOEXCEPT
{
    m_BitCount = 0;
}

/**
* @brief        オリジナルメタデータビットマップに対してイテレーションを開始します。
*
* @param[out]   outIter         イテレータ
* @param[in]    index           インデックス
* @param[in]    count           ストレージ
*
* @pre          マウントしている。
* @pre          index + count が範囲内である。
*
* @details      オリジナルメタデータビットマップに対してイテレーションを開始します。
*/
void DuplexBitmap::IterateOriginalBegin(
         Iterator* outIter,
         uint32_t index,
         size_t count
     ) const NN_NOEXCEPT
{
    // イテレーション範囲をチェックします。
    NN_SDK_REQUIRES_GREATER_EQUAL(m_BitCount, index + count);

    outIter->index = index;
    outIter->indexEnd = static_cast<uint32_t>(index + count);
}

/**
* @brief        指定範囲に対してビットデータを反転します。
*
* @param[in]    index           インデックス
* @param[in]    count           ビット数
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に処理が終了しました。
* @retval       上記以外        ストレージでの読み書きに失敗しました。
*
* @pre          マウントしている。
* @pre          index + count が ブロック数より少ない。
*
* @details      指定範囲に対してビットデータを反転します。
*/
Result DuplexBitmap::MarkModified(
           uint32_t index0,
           size_t count0
       ) NN_NOEXCEPT
{
    uint32_t index = index0;
    size_t count = count0;

    // 範囲チェックを行います。
    NN_SDK_REQUIRES_GREATER_EQUAL(m_BitCount, index + count);

    size_t offset = index % NN_BITSIZEOF(uint32_t);

    // 4 バイト単位で書き込みを行います。
    while( count > 0 )
    {
        // 更新先のビットマップデータを読み込みます。
        uint8_t bits[sizeof(uint32_t)];
        uint32_t Write;
        NN_RESULT_DO(
            ReadBlock(
                bits,
                &Write,
                sizeof(uint32_t),
                index,
                &m_Table
            )
        );

        // 更新元のビットマップデータを読み込みます。
        uint8_t originalBits[sizeof(uint32_t)];
        uint32_t originalWrite;
        NN_RESULT_DO(
            ReadBlock(
                originalBits,
                &originalWrite,
                sizeof(uint32_t),
                index,
                &m_OriginalTable
            )
        );

        // オリジナルの対象範囲をビットマップを反転したものと
        // 実際のビットマップの論理和を取得します。
        uint32_t orgX = ReadU32(bits, 0);
        uint32_t x = ReadU32(originalBits, 0);
        x = ~x;

        size_t backOffset = NN_BITSIZEOF(uint32_t) - offset;
        if( offset > 0 )
        {
            // 先頭 offset ビット以前は実際のビットマップを使用するようにします。
            x = (orgX & ~((1 << backOffset) - 1)) | (x & ((1 << backOffset) - 1));
        }
        if( (offset + count > 0) && (count < backOffset) )
        {
            // offset + count ビット以降は実際のビットマップを使用するようにします。
            int64_t bitMask = (1 << (backOffset - count)) - 1;
            x = (orgX & bitMask) | (x & (~bitMask));
        }

        WriteU32(bits, 0, x);

        // 反転したデータを保存します。
        int64_t writeOffset = (index / NN_BITSIZEOF(uint32_t)) * sizeof(uint32_t);
        NN_RESULT_DO(m_Table.Write(writeOffset, bits, sizeof(uint32_t)));
        // 書き込み位置、残り書き込み個数を更新します。
        index += static_cast<uint32_t>(NN_BITSIZEOF(uint32_t) - offset);
        if( count + offset > NN_BITSIZEOF(uint32_t) )
        {
            count -= (NN_BITSIZEOF(uint32_t) - offset);
        }
        else
        {
            count = 0;
        }
        offset = 0;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        メタデータビットデータをフラッシュします。
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常にフラッシュできました。
* @retval       上記以外        ストレージへの書き込みに失敗しました。
*
* @pre          マウントしている。
*
* @details      メタデータビットデータをフラッシュします。
*/
Result DuplexBitmap::Flush() NN_NOEXCEPT
{
    return m_Table.Flush();
}

/**
* @brief        キャッシュを無効化します。
*
* @param[in]    bitIndex        開始インデックス
* @param[in]    bitCount        キャッシュを無効化するビットの数
*
* @return       関数の処理結果を返します。
*
* @pre          マウントしている。
*
* @details      フラッシュせずにキャッシュを無効化します。
*/
Result DuplexBitmap::Invalidate(
           uint32_t bitIndex,
           size_t bitCount
       ) NN_NOEXCEPT
{
    const auto alignedIndexBegin = nn::util::align_down(bitIndex, NN_BITSIZEOF(char));
    const auto alignedIndexEnd = nn::util::align_up(bitIndex + bitCount, NN_BITSIZEOF(char));
    const auto offset = alignedIndexBegin / NN_BITSIZEOF(char);
    const auto size = (alignedIndexEnd - alignedIndexBegin) / NN_BITSIZEOF(char);
    NN_RESULT_DO(
        m_Table.OperateRange(
            fs::OperationId::Invalidate,
            offset,
            size
        )
    );
    NN_RESULT_DO(
        m_OriginalTable.OperateRange(
            fs::OperationId::Invalidate,
            offset,
            size
        )
    );
    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージからデータを読み込みします。
*
* @param[out]   outBuf          読み込んだデータを格納するバッファ
* @param[out]   outReadSize     読み込んだバイト数
* @param[in]    maxBytes        読み込み先のバッファサイズ。4 の倍数である必要があります。
* @param[in]    index           読み込み開始インデックス(bit)
* @param[in]    pStorage        ストレージ
* @param[in]    readOffset      ストレージの読み込みオフセット
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に処理が終了しました。
* @retval       上記以外        ストレージからの読み込みに失敗しました。
*
* @pre          マウントしている。
* @pre          outBuf が NULL ではない。
* @pre          outReadSize が NULL ではない。
* @pre          pStorage が NULL ではない。
* @pre          maxBytes が4の倍数。
*
* @details      ストレージからデータを読み込みします。
*               読み込みは 32 ビット単位で行われます。
*               読込先のバッファサイズは 4 の倍数にしておく必要があります。
*/
Result DuplexBitmap::ReadBlock(
           uint8_t* outBuf,
           uint32_t* outReadSize,
           size_t maxBytes,
           uint32_t index,
           fs::SubStorage* pStorage
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuf);
    NN_SDK_REQUIRES_NOT_NULL(outReadSize);
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    // 読み込み開始アドレスは 4 バイト境界に配置し、
    // 読み込みは 4 バイト単位で行います。
    NN_SDK_REQUIRES_EQUAL(0U, maxBytes & 0x3);

    NN_RESULT_DO(
        pStorage->Read(
            nn::util::align_down(index, NN_BITSIZEOF(uint32_t)) / NN_BITSIZEOF(char),
            outBuf,
            maxBytes
        )
    );

    *outReadSize = static_cast<uint32_t>(maxBytes);

    NN_RESULT_SUCCESS;
}

/**
* @brief        連続しているビット数を計測します。
*
* @param[out]   outTotalCount       連続しているビット数
* @param[out]   outIter             イテレータ
* @param[out]   outBuf              読み込んだデータを格納しているバッファ
* @param[out]   outReadIndex        探索済みのデータサイズ
* @param[out]   outRead             読み込んだデータサイズ
* @param[in]    isHeadZero          連続している 0 を計測する場合は true、
*                                   連続している 1 を計測する場合は false
* @param[in]    pStorage            ストレージ
* @param[in]    readOffset          ストレージの読み込みオフセット
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       正常に処理が終了しました。
* @retval       上記以外            ストレージからの読み込みに失敗しました。
*
* @pre          マウントしている。
* @pre          outTotalCount が NULL ではない。
* @pre          outIter が NULL ではない。
* @pre          outBuf が NULL ではない。
* @pre          outReadIndex が NULL ではない。
* @pre          outRead が NULL ではない。
* @pre          pStorage が NULL ではない。
*
* @details      連続しているビット数を計測します。
*/
Result DuplexBitmap::FindBitCount(
           size_t* outTotalCount,
           Iterator* outIter,
           uint8_t* outBuf,
           uint32_t* outReadIndex,
           uint32_t* outRead,
           bool isHeadZero,
           fs::SubStorage* pStorage
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTotalCount);
    NN_SDK_REQUIRES_NOT_NULL(outIter);
    NN_SDK_REQUIRES_NOT_NULL(outBuf);
    NN_SDK_REQUIRES_NOT_NULL(outReadIndex);
    NN_SDK_REQUIRES_NOT_NULL(outRead);
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    uint32_t totalCount = 0;
    bool bFindFinished = false;
    while( outIter->index < outIter->indexEnd )
    {
        uint32_t* pReadIndex = outReadIndex;
        // 4 バイト単位でビットの連続出現数を計測します。
        while( *pReadIndex < *outRead )
        {
            // ビット列先頭から目的のビットが連続している個数を取得します。
            uint32_t x = ReadU32(outBuf, *pReadIndex);
            int shiftCount = outIter->index % NN_BITSIZEOF(uint32_t);
            x <<= shiftCount;
            int count;
            if( isHeadZero )
            {
                count = CountLeadingZeros(x);
            }
            else
            {
                count = CountLeadingOnes(x);
            }

            // 実際見るべき範囲は以下のとおりです。
            bFindFinished = false;
            if( count >= NN_BITSIZEOF(uint32_t) - shiftCount )
            {
                // データの末尾まで等しいデータが並んでいました
                count = NN_BITSIZEOF(uint32_t) - shiftCount;
            }
            else
            {
                bFindFinished = true;
            }

            // 個数を残りのビット数に丸め込みます。
            int leftBitCount = outIter->indexEnd - outIter->index;
            if( count >= leftBitCount )
            {
                count = leftBitCount;
                bFindFinished = true;
            }

            totalCount += static_cast<uint32_t>(count);
            outIter->index += static_cast<uint32_t>(count);

            if( bFindFinished )
            {
                break;
            }

            // 次の 4 バイトに対して連続出現数を計測します。
            *pReadIndex += sizeof(uint32_t);
        }

        if( bFindFinished )
        {
            break;
        }

        // バッファに次のデータを読み込みます。
        size_t startOffset = nn::util::align_down(outIter->index, NN_BITSIZEOF(uint32_t));
        size_t endOffset = nn::util::align_up(outIter->indexEnd, NN_BITSIZEOF(uint32_t));
        size_t readRequestBytes = (endOffset - startOffset) / NN_BITSIZEOF(char);
        if( readRequestBytes > CacheSize )
        {
            readRequestBytes = CacheSize;
        }
        *pReadIndex = 0;
        NN_RESULT_DO(
            ReadBlock(
                outBuf,
                outRead,
                readRequestBytes,
                outIter->index,
                pStorage
            )
        );
    }

    *outTotalCount = totalCount;

    NN_RESULT_SUCCESS;
}

/**
* @brief        二重化ビットマップに対してイテレーションを行います。
*
* @param[out]   outCountZero        0 が連続している回数
* @param[out]   outCountOne         1 が連続している回数
* @param[in]    pIter               イテレータ
* @param[in]    index               インデックス
* @param[in]    pStorage            ストレージ
* @param[in]    readOffset          オフセット
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess      正常に処理が終了しました。
* @retval       上記以外           ストレージからの読み込みに失敗しました。
*
* @pre          マウントしている。
* @pre          outZeroCount が NULL ではない。
* @pre          outOneCount が NULL ではない。
* @pre          pIter が NULL ではない。
* @pre          pStorage が NULL ではない。
* @pre          pIter->index が範囲内。
* @post         イテレーションが終了したら outCountZero = 0、outCountOne = 0。
*
* @details      二重化ビットマップに対してイテレーションを行います。
*/
Result DuplexBitmap::IterateNextImpl(
           size_t* outCountZero,
           size_t* outCountOne,
           Iterator* pIter,
           fs::SubStorage* pStorage
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCountZero);
    NN_SDK_REQUIRES_NOT_NULL(outCountOne);
    NN_SDK_REQUIRES_NOT_NULL(pIter);
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    // イテレーションが終了したかどうかをチェックします。
    NN_SDK_REQUIRES_LESS_EQUAL(pIter->index, pIter->indexEnd);
    if( pIter->index >= pIter->indexEnd )
    {
        // イテレーションは終了しました。
        *outCountZero = 0;
        *outCountOne = 0;
        NN_RESULT_SUCCESS;
    }

    uint8_t readBuf[CacheSize];

    // バッファに次のデータを読み込みます。
    size_t startOffset = nn::util::align_down(pIter->index, NN_BITSIZEOF(uint32_t));
    size_t endOffset = nn::util::align_up(pIter->indexEnd, NN_BITSIZEOF(uint32_t));
    size_t readRequestBytes = (endOffset - startOffset) / NN_BITSIZEOF(char);
    if( readRequestBytes > CacheSize )
    {
        readRequestBytes = CacheSize;
    }
    uint32_t readResult;
    NN_RESULT_DO(
        ReadBlock(
            readBuf,
            &readResult,
            readRequestBytes,
            pIter->index,
            pStorage
        )
    );

    uint32_t readIndex = 0;

    // 0 が連続している個数を取得します。
    NN_RESULT_DO(
        FindBitCount(
            outCountZero,
            pIter,
            readBuf,
            &readIndex,
            &readResult,
            true,
            pStorage
        )
    );

    // 1 が連続している個数を取得します。
    NN_RESULT_DO(
        FindBitCount(
            outCountOne,
            pIter,
            readBuf,
            &readIndex,
            &readResult,
            false,
            pStorage
        )
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief        更新側メタデータビットマップから、32 ビット分のビットマップを取得します。
*
* @param[out]   outBitmap           ビットマップ
* @param[in]    index               インデックス
* @param[in]    pStorage            ストレージ
* @param[in]    readOffset          読み込みオフセット
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       正常に取得できました。
* @retval       上記以外            ストレージからの読み込みに失敗しました。
*
* @pre          マウントしている。
* @pre          outBitmap が NULL ではない。
* @pre          pStorage が NULL ではない。
*
* @details      更新側メタデータビットマップから、32 ビット分のビットマップを取得します。
*/
Result DuplexBitmap::ReadBitmap32Impl(
           Bit32* outBitmap,
           uint32_t index,
           fs::SubStorage* pStorage
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBitmap);
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    uint32_t indexAligned = nn::util::align_down(index, NN_BITSIZEOF(Bit32));
    // インデックスがアラインされてない場合は、2倍読み込んでそのうち必要な 32 ビットを取得する
    size_t sizeReadBit = (indexAligned == index) ? NN_BITSIZEOF(Bit32) : (NN_BITSIZEOF(Bit32) * 2);
    size_t sizeBlockCount = nn::util::align_up(m_BitCount, NN_BITSIZEOF(Bit32));
    if( indexAligned + sizeReadBit > sizeBlockCount )
    {
        sizeReadBit = NN_BITSIZEOF(Bit32);
    }

    // 更新先のビットマップデータを読み込みます。(4 or 8 バイト)
    Bit32 bits[2] = {0};

    *outBitmap = 0;

    uint32_t readResult;
    NN_RESULT_DO(
        ReadBlock(
            reinterpret_cast<uint8_t*>(bits),
            &readResult,
            sizeReadBit / NN_BITSIZEOF(char),
            indexAligned, // bit単位の指定
            pStorage
        )
    );

    // シフトして値を作ります。
    if( indexAligned == index )
    {
        *outBitmap = bits[0];
    }
    else
    {
        uint32_t index31 = index & (NN_BITSIZEOF(Bit32) - 1);
        nn::Bit32 bitmapBefore = bits[0] << index31;
        nn::Bit32 bitmapAfter = bits[1] >> (NN_BITSIZEOF(Bit32) - index31);
        *outBitmap = bitmapBefore | bitmapAfter;
    }

    // もしm_BitCountにかかる場所であった場合、マスクしておく
    if( index + NN_BITSIZEOF(Bit32) > m_BitCount )
    {
        Bit32 mask = ~((1u << (index + NN_BITSIZEOF(Bit32) - m_BitCount)) - 1);
        *outBitmap = (*outBitmap) & mask;
    }

    NN_RESULT_SUCCESS;
}

}}}
