﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/buffers/fs_FileSystemBuddyHeap.h>

namespace nn { namespace fssystem {

 /*--------------------------------------------------------------------------------*
    @class FileSystemBuddyHeap::PageList
 *--------------------------------------------------------------------------------*/

/*!
* @brief       ページリストの先頭から 1 ページ取得します。
*
* @return      ページリストの先頭要素。
*
* @pre
*              - リストに少なくとも 1 つのエントリが登録されている
*/
FileSystemBuddyHeap::PageEntry* FileSystemBuddyHeap::PageList::PopFront() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(0 < m_EntryCount);

    // ページリストの最初の1ページ分を取得します
    auto pPageEntry = m_pFirstPageEntry;

    // ページリストをひとつ進めます
    m_pFirstPageEntry = pPageEntry->pNext;
    pPageEntry->pNext = nullptr;

    // エントリ数を更新します
    --m_EntryCount;
    NN_SDK_ASSERT_GREATER_EQUAL(m_EntryCount, 0);

    // ページリストが空になっていたら末尾を指すポインタをヌルにします
    if( m_EntryCount == 0 )
    {
        m_pLastPageEntry = nullptr;
    }

    return pPageEntry;
}

/*!
* @brief       ページリストの最後尾に 1 ページ追加します。
*
* @param[in]   pPageEntry  追加対象のページ
*
* @pre
*              - pPageEntry != nullptr
*              - pPageEntry がどのリストにも登録されていない
*/
void FileSystemBuddyHeap::PageList::PushBack(PageEntry* pPageEntry) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPageEntry);

    // ページリストは空か?
    if( IsEmpty() )
    {
        // 先頭要素として追加します
        m_pFirstPageEntry = pPageEntry;
    }
    else
    {
        // リストの末尾に追加します
        NN_SDK_ASSERT_NOT_EQUAL(pPageEntry, m_pLastPageEntry);
        m_pLastPageEntry->pNext = pPageEntry;
    }

    pPageEntry->pNext = nullptr;
    m_pLastPageEntry = pPageEntry;
    ++m_EntryCount;
    NN_SDK_ASSERT_GREATER(m_EntryCount, 0);
}

/*!
* @brief       ページリストから 1 ページ削除します。
*
* @param[in]   pPageEntry  削除対象のページ
*
* @return      削除に成功したかどうかを返します。
* @retval      true        正常に削除できた。
* @retval      false       ページが存在しない。
*
* @pre
*              - pPageEntry != nullptr
*/
bool FileSystemBuddyHeap::PageList::Remove(PageEntry* pPageEntry) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPageEntry);

    // ページリストは空か?
    if( IsEmpty() )
    {
        return false;
    }

    PageEntry* pPreviousPageEntry = nullptr;
    auto pCurPageEntry = m_pFirstPageEntry;

    for( ; ; )
    {
        if( pCurPageEntry == pPageEntry )
        {
            // 削除対象のページが見つかった場合
            if( pCurPageEntry == m_pFirstPageEntry )
            {
                // 削除対象のページがページリストの先頭だったため
                // リストをひとつ進めつつ要素を削除します
                m_pFirstPageEntry = pCurPageEntry->pNext;
            }
            else if( pCurPageEntry == m_pLastPageEntry )
            {
                // 削除対象のページがページリストの末尾だったため
                // 末尾の要素をひとつ戻します
                m_pLastPageEntry = pPreviousPageEntry;
                m_pLastPageEntry->pNext = nullptr;
            }
            else
            {
                // 先頭～末尾の間にあるページのため
                // リンクリストをつなぎなおします
                pPreviousPageEntry->pNext = pCurPageEntry->pNext;
            }

            pCurPageEntry->pNext = nullptr;
            --m_EntryCount;
            NN_SDK_ASSERT_GREATER_EQUAL(m_EntryCount, 0);

            return true;
        }

        if( pCurPageEntry->pNext == nullptr )
        {
            return false;
        }

        pPreviousPageEntry = pCurPageEntry;
        pCurPageEntry = pCurPageEntry->pNext;
    }

    // not reached
}

 /*--------------------------------------------------------------------------------*
    @class FileSystemBuddyHeap
 *--------------------------------------------------------------------------------*/

NN_DEFINE_STATIC_CONSTANT( const size_t FileSystemBuddyHeap::BufferAlignment );
NN_DEFINE_STATIC_CONSTANT( const size_t FileSystemBuddyHeap::BlockSizeMin );
NN_DEFINE_STATIC_CONSTANT( const int FileSystemBuddyHeap::OrderUpperLimit );

/*!
* @brief       総サイズとブロックサイズから割り当て可能な最大のオーダーを取得します。
*
* @param[in]   size        ヒープに割り当てる領域のサイズ
* @param[in]   blockSize   ブロックのサイズ
*
* @return      割り当て可能な最大のオーダーを返します。
*
* @details     本関数によって取得した値をそのまま Initialize() に渡すことができます。
*
* @pre
*              - blockSize <= size
*              - blockSize = 2^m (std::log2(BlockSizeMin) <= m)
*/
int FileSystemBuddyHeap::QueryOrderMax(size_t size, size_t blockSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(size, blockSize);
    NN_SDK_REQUIRES_GREATER_EQUAL(blockSize, BlockSizeMin);
    NN_SDK_REQUIRES(nn::util::ispow2(blockSize));

    const auto blockCount = static_cast<int>((size + (blockSize - 1U)) / blockSize);
    for( auto order = 1; ; ++order )
    {
        if( blockCount <= GetBlockCountFromOrder(order) )
        {
            return order;
        }
    }
}

/*!
* @brief       バディヒープを初期化します。
*
* @param[in]   addr        ヒープに割り当てる領域のアドレス
* @param[in]   size        ヒープに割り当てる領域のサイズ
* @param[in]   blockSize   ブロックのサイズ
* @param[in]   orderMax    最大ページ数のオーダー
*
* @pre
*              - 未初期化である
*              - addr != nullptr
*              - nn::util::is_aligned(addr, BufferAlignment);
*              - blockSize <= size
*              - blockSize = 2^m (std::log2(BlockSizeMin) <= m)
*              - 0 < orderMax < OrderUpperLimit
*/
nn::Result FileSystemBuddyHeap::Initialize(
               uintptr_t addr,
               size_t size,
               size_t blockSize,
               int orderMax
           ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pFreeAreas == nullptr);
    NN_SDK_REQUIRES_NOT_EQUAL(addr, 0U);
    NN_SDK_REQUIRES_ALIGNED(addr, BufferAlignment);
    NN_SDK_REQUIRES_GREATER_EQUAL(blockSize, BlockSizeMin);
    NN_SDK_REQUIRES(nn::util::ispow2(blockSize));
    NN_SDK_REQUIRES_GREATER_EQUAL(size, blockSize);
    NN_SDK_REQUIRES_GREATER(orderMax, 0);
    NN_SDK_REQUIRES_LESS(orderMax, OrderUpperLimit);

    m_BlockSize = blockSize;
    m_OrderMax  = orderMax;
    m_HeapStart = addr;
    m_HeapSize  = (size / m_BlockSize) * m_BlockSize;
    m_TotalFreeSize = 0;

    const auto pageSizeMax    = m_BlockSize << m_OrderMax;
    const auto ceiledHeapSize = (m_HeapSize + (pageSizeMax - 1)) & ~(pageSizeMax - 1);
    const auto maxPageCount   = ceiledHeapSize / pageSizeMax;
    NN_SDK_ASSERT_GREATER(maxPageCount, 0U);

    // 管理領域の確保
    if( m_pExternalFreeAreas != nullptr )
    {
        NN_SDK_ASSERT(m_pInternalFreeAreas == nullptr);
        m_pFreeAreas = m_pExternalFreeAreas;
    }
    else
    {
        m_pInternalFreeAreas.reset(new PageList[m_OrderMax + 1]);
        m_pFreeAreas = m_pInternalFreeAreas.get();
        if( m_pFreeAreas == nullptr )
        {
            NN_RESULT_THROW(nn::fs::ResultAllocationMemoryFailedInFileSystemBuddyHeapA());
        }
    }

    // 各ページの開始アドレスをフリーリストに登録します
    for( auto maxPageIndex = 0U; maxPageIndex < maxPageCount - 1; ++maxPageIndex )
    {
        auto pPageEntry = GetPageEntryFromAddress(m_HeapStart + maxPageIndex * pageSizeMax);
        m_pFreeAreas[m_OrderMax].PushBack(pPageEntry);
    }
    m_TotalFreeSize += m_pFreeAreas[m_OrderMax].GetSize() * GetBytesFromOrder(m_OrderMax);

    // 最後のページのみ使用不可領域を考慮して登録します
    {
        auto remainSize = m_HeapSize - (maxPageCount - 1) * pageSizeMax;
        auto curAddr = m_HeapStart + (maxPageCount - 1) * pageSizeMax;
        NN_SDK_ASSERT_ALIGNED(remainSize, m_BlockSize);

        do
        {
            auto order = GetOrderFromBytes(remainSize + 1);
            if( order < 0 )
            {
                NN_SDK_ASSERT_EQUAL(m_OrderMax, GetOrderFromBytes(remainSize));
                order = m_OrderMax + 1;
            }
            NN_SDK_ASSERT_GREATER(order, 0);
            NN_SDK_ASSERT_LESS_EQUAL(order, m_OrderMax + 1);

            // 使用可能領域を細切れにして空きページリストへ登録します
            const auto pDividedPageEntry = GetPageEntryFromAddress(curAddr);
            m_pFreeAreas[order - 1].PushBack(pDividedPageEntry);
            m_TotalFreeSize += GetBytesFromOrder(order - 1);

            // 残りサイズを更新
            const auto pageSize = GetBytesFromOrder(order - 1);
            curAddr += pageSize;
            remainSize -= pageSize;
        } while( m_BlockSize <= remainSize );
    }
    NN_RESULT_SUCCESS;
}

/*!
* @brief       バディヒープを破棄します。
*
* @details     割り当てられたまま解放されていないメモリがあったとしても何もしません。
*
* @pre
*              - 初期化済みである
*/
void FileSystemBuddyHeap::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFreeAreas);
    m_pFreeAreas = nullptr;
    m_pExternalFreeAreas = nullptr;
    m_pInternalFreeAreas.reset();
}

/*!
* @brief       オーダーを指定してページを確保します。
*
* @param[in]   order   確保するページのオーダー
*
* @return      確保したページへのポインタ
* @retval      nullptr 空き領域が不足しています。
*
* @pre
*              - 初期化済みである
*              - 0 <= order <= GetOrderMax()
*/
void* FileSystemBuddyHeap::AllocateByOrder(int order) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFreeAreas);
    NN_SDK_REQUIRES_GREATER_EQUAL(order, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(order, GetOrderMax());

    const auto pPageEntry = GetFreePageEntry(order);

    if( nullptr != pPageEntry )
    {
        NN_SDK_ASSERT(pPageEntry->pNext == nullptr);
        const auto addr = GetAddressFromPageEntry(*pPageEntry);
        return reinterpret_cast<void*>(addr);
    }
    else
    {
        return nullptr;
    }
}

/*!
* @brief       メモリを解放します。
*
* @param[in]   ptr     解放するページへのポインタ
* @param[in]   order   解放するページのオーダー
*
* @pre
*              - 初期化済みである
*              - ptr がこのヒープによって確保された未解放の領域を指している
*              - 0 <= order <= GetOrderMax()
*
* @details     ptr に nullptr を指定すると何もしません。
*/
void FileSystemBuddyHeap::Free(void* ptr, int order) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFreeAreas);
    NN_SDK_REQUIRES_GREATER_EQUAL(order, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(order, GetOrderMax());

    if( nullptr == ptr )
    {
        return;
    }

    NN_SDK_ASSERT_ALIGNED(reinterpret_cast<uintptr_t>(ptr) - m_HeapStart, GetBlockSize());

    auto pPageEntry = GetPageEntryFromAddress(reinterpret_cast<uintptr_t>(ptr));
    NN_SDK_ASSERT(IsAlignedToOrder(pPageEntry, order));

    JoinBuddies(pPageEntry, order);
}

/*!
* @brief       ヒープの空き容量の合計を取得します。
*
* @return      ヒープの空き容量の合計のバイト数
*
* @pre
*              - 初期化済みである
*/
size_t FileSystemBuddyHeap::GetTotalFreeSize() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFreeAreas);
    return m_TotalFreeSize;
}

/*!
* @brief       ヒープから確保可能なページサイズの最大を取得します。
*
* @return      ヒープから確保可能なページサイズの最大のバイト数
*
* @pre
*              - 初期化済みである
*/
size_t FileSystemBuddyHeap::GetAllocatableSizeMax() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFreeAreas);
    for( auto order = GetOrderMax(); 0 <= order; --order )
    {
        // より大きな空きメモリブロックサイズを返します
        if( !m_pFreeAreas[order].IsEmpty() )
        {
            return GetBytesFromOrder(order);
        }
    }
    return 0U;
}

/*!
* @brief       ヒープの管理状態をログに出力します。
*
* @details     デバッグ用の関数です。
*
* @pre
*              - 初期化済みである
*/
void FileSystemBuddyHeap::Dump() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFreeAreas);
    NN_SDK_LOG("heap start: 0x%x\n", m_HeapStart);
    NN_SDK_LOG("heap size : 0x%x\n", m_HeapSize);
    NN_SDK_LOG("order max : %d\n", GetOrderMax());
    NN_SDK_LOG("block size: 0x%x", GetBlockSize());
    for( auto order = 0; order <= GetOrderMax(); ++order )
    {
        NN_SDK_LOG("\n[%d]", order);
        for( auto pPageEntry = m_pFreeAreas[order].GetFront();
             pPageEntry != nullptr;
             pPageEntry = pPageEntry->pNext )
        {
            NN_SDK_LOG(" %p", pPageEntry);
        }
    }
    NN_SDK_LOG("\n");
}

/*!
* @brief       大きなページから必要な分だけを切り出します。
*
* @param[in]   pPageEntry      分割対象のページエントリー
* @param[in]   demandedOrder   必要なオーダー
* @param[in]   wholeOrder      分割対象全体のオーダー
*
* @pre
*              - pPageEntry != nullptr
*              - 0 <= demandedOrder < wholeOrder <= GetOrderMax()
*/
void FileSystemBuddyHeap::DivideBuddies(
         PageEntry* pPageEntry,
         int demandedOrder,
         int wholeOrder
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPageEntry);
    NN_SDK_REQUIRES_GREATER_EQUAL(demandedOrder, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(wholeOrder, demandedOrder);
    NN_SDK_REQUIRES_LESS_EQUAL(wholeOrder, GetOrderMax());

    auto addr = GetAddressFromPageEntry(*pPageEntry) + GetBytesFromOrder(wholeOrder);

    for( auto order = wholeOrder; demandedOrder < order; --order )
    {
        addr -= GetBytesFromOrder(order - 1);
        auto pDividedPageEntry = GetPageEntryFromAddress(addr);

        // 細切れにして空きページリストへ登録します
        // 4KB の要求に対して 16KB の大きな空きページが見つかった場合、差し引きして割り当てを必要としない
        // 12KB (16KB-4KB) を 4KB, 8KB の 2 つのページリストに分け、フリーリストへ追加します。
        m_pFreeAreas[order - 1].PushBack(pDividedPageEntry);
        m_TotalFreeSize += GetBytesFromOrder(order - 1);
    }
}

/*!
* @brief       空きページリストへ登録します。あわせてメモリの連結処理を行います。
*
* @param[in]   pPageEntry  追加対象のページエントリー
* @param[in]   order       オーダー
*
* @pre
*              - pPageEntry != nullptr
*              - 0 <= order <= GetOrderMax()
*/
void FileSystemBuddyHeap::JoinBuddies(PageEntry* pPageEntry, int order) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPageEntry);
    NN_SDK_REQUIRES_GREATER_EQUAL(order, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(order, GetOrderMax());

    auto pCurPageEntry = pPageEntry;
    auto curOrder = order;

    // 最上位のオーダーの 1 つ前まで
    while( curOrder < GetOrderMax() )
    {
        // バディ(相方)となるページを求めます
        const auto pBuddyPageEntry = GetBuddy(pCurPageEntry, curOrder);

        // フリーリストにバディがいるか
        if( pBuddyPageEntry != nullptr && m_pFreeAreas[curOrder].Remove(pBuddyPageEntry) )
        {
            m_TotalFreeSize -= GetBytesFromOrder(curOrder);
            // いたら、フリーリストから削除して（＝ジョイントした）１つ上のオーダーでジョイントを試みる
            // ページが １ つ上のオーダーでアラインしていなかったら、アドレスを Buddy に
            if( !IsAlignedToOrder(pCurPageEntry, curOrder + 1) )
            {
                pCurPageEntry = pBuddyPageEntry;
            }
            ++curOrder;
        }
        else
        {
            // いなかったら、ジョイントできなかったので、ページをフリーリストに追加する
            break;
        }
    }

    // Buddy が利用中だったか、最上位オーダーまでジョイントした
    m_pFreeAreas[curOrder].PushBack(pCurPageEntry);
    m_TotalFreeSize += GetBytesFromOrder(curOrder);
}

/*!
* @brief       指定したオーダーで指定したページとバディ関係にあるページエントリーを取得します。
*
* @param[in]   pPageEntry  ページエントリー
* @param[in]   order       オーダー
*
* @return      指定したオーダーで pPageEntry とバディ関係にあるページエントリーを返します。
* @retval      nullptr     バディが見つかりませんでした。
*
* @pre
*              - pPageEntry != nullptr
*              - 0 <= order <= GetOrderMax()
*/
FileSystemBuddyHeap::PageEntry* FileSystemBuddyHeap::GetBuddy(
                                    PageEntry* pPageEntry,
                                    int order
                                ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPageEntry);
    NN_SDK_REQUIRES_GREATER_EQUAL(order, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(order, GetOrderMax());

    const auto addr = GetAddressFromPageEntry(*pPageEntry);
    const auto offset = GetBlockCountFromOrder(order) * GetBlockSize();

    // ページのインデックスが上のオーダーで割り切れるなら、右隣が Buddy
    if( IsAlignedToOrder(pPageEntry, order + 1) )
    {
        return addr + offset < m_HeapStart + m_HeapSize ? GetPageEntryFromAddress(addr + offset) : nullptr;
    }
    else // でなければ、左隣が Buddy
    {
        return m_HeapStart <= addr - offset ? GetPageEntryFromAddress(addr - offset) : nullptr;
    }
}

/*!
* @brief       指定のオーダーを持つ未割当ページのエントリーを取得します。
*
* @param[in]   order   オーダー
*
* @return      ページエントリー
*
* @pre
*              - 0 <= order <= GetOrderMax()
*/
FileSystemBuddyHeap::PageEntry* FileSystemBuddyHeap::GetFreePageEntry(int order) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(order, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(order, GetOrderMax());

    for( auto curOrder = order; curOrder <= GetOrderMax(); ++curOrder )
    {
        // 指定のオーダーを持つページのリストを探索します
        // もしそのページに空きページが登録さていない場合は、ひとつ上のオーダーのページを探索します
        auto& freeArea = m_pFreeAreas[curOrder];
        if( !freeArea.IsEmpty() )
        {
            // 空きメモリを見つけました
            PageEntry* pPageEntry = freeArea.PopFront();
            NN_SDK_ASSERT_NOT_NULL(pPageEntry);
            m_TotalFreeSize -= GetBytesFromOrder(curOrder);

            // 要求に対して大きすぎるメモリを割り当てする場合
            // 使用しない部分を分割し、空きページリストへ登録しなおします
            DivideBuddies(pPageEntry, order, curOrder);
            NN_SDK_ASSERT(pPageEntry->pNext == nullptr);

            return pPageEntry;
        }
    }

    return nullptr;
}

/*!
* @brief       指定した値以上のブロック数からなるページのオーダーの最小を取得します。
*
* @param[in]   blockCount  ブロック数
*
* @return      指定ブロック数以上を持つページのオーダーの最小を返します。
*              最大オーダーを超える場合は負の値を返します。
*
* @pre
*              - 0 <= blockCount
*/
int FileSystemBuddyHeap::GetOrderFromBlockCount(int blockCount) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(blockCount, 0);
    for( auto order = 0; order <= GetOrderMax(); order++ )
    {
        if( blockCount <= GetBlockCountFromOrder(order) )
        {
            return order;
        }
    }
    return -1;
}

}}
