﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/fssrv/fssrv_SaveDataIndexerManager.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_Optional.h>
#include "detail/fssrv_SdmmcStorageService.h"

namespace nn { namespace fssrv {

    // TODO: クラス化
    namespace {
        fs::SaveDataId  g_IndexerSaveDataId;
        MemoryResource* g_pMemoryResource;

        nn::os::Mutex g_BisIndexerMutex(false);
        util::optional<SaveDataIndexer> g_BisIndexer;

        nn::os::Mutex g_TempIndexerMutex(false);
        SaveDataIndexerLite g_TempIndexer;

        nn::os::Mutex g_SdIndexerMutex(false);
        util::optional<SaveDataIndexer> g_SdCardIndexer;
        detail::SdCardHandle g_SdCardHandle;

        IDeviceHandleManager* g_pSdHandleManager = nullptr;

        nn::os::Mutex g_ProperBisIndexerMutex(false);
        util::optional<SaveDataIndexer> g_ProperBisIndexer;

        nn::os::Mutex g_SafeIndexerMutex(false);
        util::optional<SaveDataIndexer> g_SafeIndexer;

        bool g_IsBisUserRedirectionEnabled = false;
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    namespace detail {
        // Death テストなどで同じ実行バイナリを複数のプロセスで走らせることがある
        // fork したプロセスによって更新された Indexer の情報を元のプロセスに反映させるため、
        // Indexer を全て無効にする
        void InvalidateAllIndexers() NN_NOEXCEPT
        {
            g_BisIndexer = util::nullopt;
            g_SdCardIndexer = util::nullopt;
            g_ProperBisIndexer = util::nullopt;
            g_SafeIndexer = util::nullopt;
        }
    }
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

    void InvalidateSdCardSaveDataIndexer() NN_NOEXCEPT
    {
        g_SdCardIndexer = util::nullopt;
    }

    void InitializeSaveDataIndexerManager(fs::SaveDataId indexerSaveDataId, MemoryResource* pMemoryResource, IDeviceHandleManager* pSdHandleManager, bool enableBisUserRedirection) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMemoryResource == nullptr);
        g_pMemoryResource = pMemoryResource;
        g_IndexerSaveDataId = indexerSaveDataId;
        g_pSdHandleManager = pSdHandleManager;
        g_IsBisUserRedirectionEnabled = enableBisUserRedirection;
    }

    void ResetTemporaryStorageIndexer() NN_NOEXCEPT
    {
        auto result = g_TempIndexer.Reset();
        NN_SDK_ASSERT(result.IsSuccess());
        NN_UNUSED(result);
    }


    nn::Result SaveDataIndexerAccessor::Initialize(fs::SaveDataSpaceId id) NN_NOEXCEPT
    {
        if (g_IsBisUserRedirectionEnabled && id == fs::SaveDataSpaceId::User)
        {
            // safemode 起動時はアプリ系セーブは常に本来のシステムパーティションの indexer を参照する
            id = fs::SaveDataSpaceId::ProperSystem;
        }

        switch (id)
        {
            case fs::SaveDataSpaceId::System:
            case fs::SaveDataSpaceId::User:
                m_Mutex = std::unique_lock<os::Mutex>(g_BisIndexerMutex);

                if (g_BisIndexer == util::nullopt)
                {
                    g_BisIndexer.emplace("saveDataIxrDb", nn::fs::SaveDataSpaceId::System, g_IndexerSaveDataId, g_pMemoryResource);
                }

                m_pIndexer = &g_BisIndexer.value();
                break;

            case fs::SaveDataSpaceId::Temporary:
                m_Mutex = std::unique_lock<os::Mutex>(g_TempIndexerMutex);
                m_pIndexer = &g_TempIndexer;
                break;

            case fs::SaveDataSpaceId::SdSystem:
            case fs::SaveDataSpaceId::SdUser:
            {
                m_Mutex = std::unique_lock<os::Mutex>(g_SdIndexerMutex);

                if (!g_pSdHandleManager->IsValid(g_SdCardHandle))
                {
                    // Indexer 破棄
                    g_SdCardIndexer = util::nullopt;
                }

                if (g_SdCardIndexer == util::nullopt)
                {
                    // 生成
                    g_SdCardIndexer.emplace("saveDataIxrDbSd", nn::fs::SaveDataSpaceId::SdSystem, g_IndexerSaveDataId, g_pMemoryResource);
                    g_pSdHandleManager->GetHandle(&g_SdCardHandle); // TODO: nn::fs API レベルで DeviceHandle を扱う
                }

                m_pIndexer = &g_SdCardIndexer.value();
                break;
            }
            case fs::SaveDataSpaceId::ProperSystem:
                m_Mutex = std::unique_lock<os::Mutex>(g_ProperBisIndexerMutex);

                if (g_ProperBisIndexer == util::nullopt)
                {
                    g_ProperBisIndexer.emplace("saveDataIxrDbPr", nn::fs::SaveDataSpaceId::ProperSystem, g_IndexerSaveDataId, g_pMemoryResource);
                }

                m_pIndexer = &g_ProperBisIndexer.value();
                break;

            case fs::SaveDataSpaceId::SafeMode:
                m_Mutex = std::unique_lock<os::Mutex>(g_SafeIndexerMutex);

                if (g_SafeIndexer == util::nullopt)
                {
                    g_SafeIndexer.emplace("saveDataIxrDbSf", nn::fs::SaveDataSpaceId::SafeMode, g_IndexerSaveDataId, g_pMemoryResource);
                }

                m_pIndexer = &g_SafeIndexer.value();
                break;

            default:
                NN_RESULT_THROW(nn::fs::ResultInvalidArgument());
        }
        NN_RESULT_SUCCESS;
    }

    SaveDataIndexerAccessor::~SaveDataIndexerAccessor() NN_NOEXCEPT
    {
    }

    ISaveDataIndexer* SaveDataIndexerAccessor::GetInterface() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pIndexer);
        return m_pIndexer;
    }

}}
